/*
**	$Id: image.c 1086 2007-07-31 16:30:44Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	handle images
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <config.h>
#include "gip.h"

#if HAVE_LIBMAGICK
#define SCALE_FILTER_BEST	GaussianFilter
#define SCALE_FILTER_FAST	PointFilter
#endif
#if HAVE_LIBIL
#define SCALE_FILTER_BEST	ILU_SCALE_MITCHELL
#define SCALE_FILTER_FAST	ILU_NEAREST
#endif

#if HAVE_LIBMAGICK
#define LOG_IMAGE_INFO
#endif
#if HAVE_LIBIL
#define LOG_IMAGE_INFO \
{ \
	ILinfo info; \
	iluGetImageInfo(&info); \
	LOG("Id        : %u\n",info.Id); \
	LOG("Data      : %p\n",info.Data); \
	LOG("Width     : %u\n",info.Width); \
	LOG("Height    : %u\n",info.Height); \
	LOG("Depth     : %u\n",info.Depth); \
	LOG("Bpp       : %u\n",info.Bpp); \
	LOG("SizeOfData: %u\n",info.SizeOfData); \
	LOG("Format    : 0x%x\n",info.Format); \
	LOG("Type      : 0x%x\n",info.Type); \
	LOG("Origin    : 0x%x\n",info.Origin); \
	LOG("Palette   : %p\n",info.Palette); \
	LOG("PalType   : 0x%x\n",info.PalType); \
	LOG("PalSize   : %d\n",info.PalSize); \
	LOG("CubeFlags : 0x%x\n",info.CubeFlags); \
	LOG("NumNext   : %d\n",info.NumNext); \
	LOG("NumMips   : %d\n",info.NumMips); \
	LOG("NumLayers : %d\n",info.NumLayers); \
}
#endif

/*
**	module specific data
*/
static char *_cwd = NULL;
static int _filter = 0;

/*
**	do some initialization stuff
*/
int GIP_image_init(void)
{
	DEBUG("\n");

	/*
	**	what filter should be used for image scaling
	*/
	_filter = (_gip_setup.quick_mode) ? SCALE_FILTER_FAST : SCALE_FILTER_BEST;

#if HAVE_LIBMAGICK
	/*
	**	check for a filter specified in the environment
	*/
	if (_gip_setup.resize_filter) {
#define IMAGE_FILTER_CHECK(filter)	\
			else if (!strcasecmp(# filter,_gip_setup.resize_filter)) \
				_filter = filter
		if (0)
			;
		IMAGE_FILTER_CHECK(PointFilter);
		IMAGE_FILTER_CHECK(BoxFilter);
		IMAGE_FILTER_CHECK(TriangleFilter);
		IMAGE_FILTER_CHECK(HermiteFilter);
		IMAGE_FILTER_CHECK(HanningFilter);
		IMAGE_FILTER_CHECK(HammingFilter);
		IMAGE_FILTER_CHECK(BlackmanFilter);
		IMAGE_FILTER_CHECK(GaussianFilter);
		IMAGE_FILTER_CHECK(QuadraticFilter);
		IMAGE_FILTER_CHECK(CubicFilter);
		IMAGE_FILTER_CHECK(CatromFilter);
		IMAGE_FILTER_CHECK(MitchellFilter);
		IMAGE_FILTER_CHECK(LanczosFilter);
		IMAGE_FILTER_CHECK(BesselFilter);
		IMAGE_FILTER_CHECK(SincFilter);
		else
			CRITICAL("got unknown filter: %s\n",_gip_setup.resize_filter);
#undef IMAGE_FILTER_CHECK
	}

	/*
	**	initialize the ImageMagick lib
	*/
	InitializeMagick(_cwd = getcwd(NULL,0));
#endif
#if HAVE_LIBIL
	/*
	**	initialize the DevIL lib
	*/
	ilInit();
	iluInit();
#endif

	return 1;
}

/*
**	shutdown the API
*/
int GIP_image_exit(void)
{
	DEBUG("\n");
#if HAVE_LIBMAGICK
	if (_cwd) {
		free(_cwd);
		_cwd = NULL;
	}
	DestroyMagick();
#endif
	return 1;
}

/*
**	allocate a new image and assign an image ID
*/
static GIP_IMAGE_T *alloc_image(void)
{
	GIP_IMAGE_T *img;
	unsigned char pixels[4];
#if HAVE_LIBMAGICK
	ExceptionInfo exception;
#endif

	DEBUG("\n");

	/*
	**	initialize the pixel data
	*/
	memset(pixels,0,sizeof(pixels));

	/*
	**	allocate a new image
	*/
	if (!(img = malloc(sizeof(GIP_IMAGE_T))))
		CRITICAL("out of memory\n");
	memset(img,0,sizeof(GIP_IMAGE_T));
	img->width = 1;
	img->height = 1;

#if HAVE_LIBMAGICK
	GetExceptionInfo(&exception);
	if (!(img->image = ConstituteImage(1,1,"RGB",CharPixel,pixels,&exception)))
		CRITICAL("ConstituteImage() failed\n");
	img->image->compression = UndefinedCompression;
	img->image->depth = 8;
	DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
	/*
	**	initialize the image for our needs
	*/
	ilGenImages(1,&img->id);
	ilBindImage(img->id);
	ilTexImage(1,1,1,4,IL_RGBA,IL_UNSIGNED_BYTE,pixels);
#endif

	DEBUG("size:%dx%dpx\n",img->width,img->height);

	//	DUMP_IMAGE_INFO;

	return img;
}

/*
**	load an image from a file
*/
GIP_IMAGE_T *GIP_image_load(const char *file)
{
	GIP_IMAGE_T *img;
#if HAVE_LIBMAGICK
	ImageInfo *info;
	ExceptionInfo exception;
#endif

	DEBUG("file:%s\n",file);

	img = alloc_image();

	/*
	**	do the load
	*/
#if HAVE_LIBMAGICK
	GetExceptionInfo(&exception);
	if (!(info = CloneImageInfo((ImageInfo *) NULL)))
		CRITICAL("CloneImageInfo() failed\n");
	strcpy(info->filename,file);
	if (img->image)
		DestroyImage(img->image);
	if (!(img->image = ReadImage(info,&exception)))
		CRITICAL("ReadImage(%s) failed\n",file);
	img->width = img->image->columns;
	img->height = img->image->rows;
	DestroyImageInfo(info);
	DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
	ilBindImage(img->id);
	if (!ilLoadImage((char *) file))
		CRITICAL("ilLoadImage(%s) failed\n",file);
	if (!ilConvertImage(IL_RGBA,IL_UNSIGNED_BYTE))
		CRITICAL("ilConvert(%s) failed\n",file);
	img->width = ilGetInteger(IL_IMAGE_WIDTH);
	img->height = ilGetInteger(IL_IMAGE_HEIGHT);
#endif

	DEBUG("file:%s  size:%dx%dpx\n",file,img->width,img->height);

	LOG_IMAGE_INFO;

	return img;
}

/*
**	load an image from a stream
*/
GIP_IMAGE_T *GIP_image_loadfd(FILE *stream)
{
	GIP_IMAGE_T *img;
#if HAVE_LIBMAGICK
	ImageInfo *info;
	ExceptionInfo exception;
#endif

	DEBUG("stream:%p\n",stream);

	img = alloc_image();

	/*
	**	do the load
	*/
#if HAVE_LIBMAGICK
	GetExceptionInfo(&exception);
	if (!(info = CloneImageInfo((ImageInfo *) NULL)))
		CRITICAL("CloneImageInfo() failed\n");
    strcpy(info->filename,"");
	if (img->image)
		DestroyImage(img->image);
	if (!(img->image = ReadImage(info,&exception)))
		CRITICAL("ReadImage() failed\n");
	img->width = img->image->columns;
	img->height = img->image->rows;
	DestroyImageInfo(info);
	DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
	ilBindImage(img->id);
	if (!ilLoadF(IL_PNM,(char *) stream))
		CRITICAL("ilLoadF(%p) failed\n",stream);
	if (!ilConvertImage(IL_RGBA,IL_UNSIGNED_BYTE))
		CRITICAL("ilConvert() failed\n");
	img->width = ilGetInteger(IL_IMAGE_WIDTH);
	img->height = ilGetInteger(IL_IMAGE_HEIGHT);
#endif

	DEBUG("stream:%p  size:%dx%dpx\n",stream,img->width,img->height);

	LOG_IMAGE_INFO;

	return img;
}

/*
**	save an image to the given file
*/
int GIP_image_save(const GIP_IMAGE_T *img,const char *file)
{
#if HAVE_LIBMAGICK
	ImageInfo *info = NULL;
#endif

	DEBUG("img:%p  file:%s\n",img,file);

	/*
	**	write the frame to the stream
	*/
#if HAVE_LIBMAGICK
	img->image->compression = UndefinedCompression;
	strcpy(img->image->filename,file);
	strcpy(img->image->magick,"PPM");
	img->image->depth = 8;
	if (WriteImage(info,img->image) != MagickTrue)
		CRITICAL("WriteImage() failed\n");
#endif
#if HAVE_LIBIL
	if (!ilSave(IL_PNM,file))
		CRITICAL("ilSave() failed\n");
#endif
	return 1;
}

/*
**	save an image to the given stream
*/
int GIP_image_savefd(const GIP_IMAGE_T *img,FILE *stream)
{
#if DEBUGGING
	static int frame = 0;
	static int width = -1,height = -1;
#endif
#if HAVE_LIBMAGICK
	ImageInfo *info = NULL;
#endif

	DEBUG("img:%p  stream:%p\n",img,stream);

#if DEBUGGING
	if (!frame++) {
		/*
		**	store the size of the first frame to
		**	compare all further frames against
		*/
#if HAVE_LIBMAGICK
		width = img->image->columns;
		height = img->image->rows;
#endif
#if HAVE_LIBIL
		ilBindImage(img->id);
		width = ilGetInteger(IL_IMAGE_WIDTH);
		height = ilGetInteger(IL_IMAGE_HEIGHT);
#endif
		DEBUG("expected size for further frames is %dx%dpx\n",
				width,height);
	}

	/*
	**	check if this frame got the expected dimensions
	*/
#if HAVE_LIBMAGICK
	if (width != img->image->columns || height != img->image->rows)
		CRITICAL("image with size of %dx%dpx != expected size of %dx%dpx\n",
				(int) img->image->columns,
				(int) img->image->rows,
				width,height);
#endif
#if HAVE_LIBIL
	ilBindImage(img->id);
	if (width != ilGetInteger(IL_IMAGE_WIDTH) || height != ilGetInteger(IL_IMAGE_HEIGHT))
		CRITICAL("image with size of %dx%dpx != expected size of %dx%dpx\n",
				ilGetInteger(IL_IMAGE_WIDTH),
				ilGetInteger(IL_IMAGE_HEIGHT),
				width,height);
#endif
#endif

	/*
	**	write the frame to the stream
	*/
#if HAVE_LIBMAGICK
	if (!(info = CloneImageInfo(NULL)))
		CRITICAL("CloneImageInfo() failed\n");
	info->file = stream;
	strcpy(info->magick,"PPM");
	info->compression = UndefinedCompression;
	img->image->compression = UndefinedCompression;
	strcpy(img->image->filename,"");
	strcpy(img->image->magick,"PPM");
	img->image->depth = 8;
	if (WriteImage(info,img->image) != MagickTrue)
		CRITICAL("WriteImage() failed\n");
	DestroyImageInfo(info);
#endif
#if HAVE_LIBIL
	if (!ilSaveF(IL_PNM,stream))
		CRITICAL("ilSaveF() failed\n");
#endif
	return 1;
}

/*
**	create a solid image by color
*/
GIP_IMAGE_T *GIP_image_create(const char *rgbcolor,int width,int height)
{
	GIP_IMAGE_T *img;
	int r,g,b;
#if HAVE_LIBMAGICK
	PixelPacket *pixels;
	Image *image;
	ExceptionInfo exception;
#endif
#if HAVE_LIBIL
	ILubyte *pixels;
	ILinfo info;
#endif

	DEBUG("color:%s  width:%d  height:%d\n",rgbcolor,width,height);

	/*
	**	allocate the image
	*/
	img = alloc_image();

	/*
	**	fill it with the given color
	*/
	GIP_color_str2rgb(rgbcolor,&r,&g,&b);

	/*
	**	create an empty image of the requested color
	*/
#if HAVE_LIBMAGICK
	GetExceptionInfo(&exception);
	if (!(pixels = GetImagePixels(img->image,0,0,1,1)))
		CRITICAL("GetImagePixels() failed\n");
	pixels->red = r;
	pixels->green = g;
	pixels->blue = b;
	pixels->opacity = 255;
	if (SyncImagePixels(img->image) != MagickTrue)
		CRITICAL("SyncImagePixels() failed\n");
	if (!(image = ResizeImage(img->image,width,height,SCALE_FILTER_FAST,1.0,&exception)))
		CRITICAL("ResizeImage() failed\n");
	DestroyImage(img->image);
	img->image = image;
	img->width = img->image->columns;
	img->height = img->image->rows;
	DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
	ilBindImage(img->id);
	iluGetImageInfo(&info);
	pixels = (ILubyte *) ilGetData();
	pixels[0] = r;
	pixels[1] = g;
	pixels[2] = b;
	pixels[3] = 255;
	iluImageParameter(ILU_FILTER,SCALE_FILTER_FAST);
	if (!iluScale(width,height,ilGetInteger(IL_IMAGE_DEPTH)))
		CRITICAL("iluScale() failed\n");
	img->width = ilGetInteger(IL_IMAGE_WIDTH);
	img->height = ilGetInteger(IL_IMAGE_HEIGHT);
#endif

	if (img->width != width || img->height != height)
		CRITICAL("actual size of %dx%dpx is not the expected size of %dx%dpx\n",
			img->width,img->height,width,height);

	LOG_IMAGE_INFO;

	return img;
}

/*
**	free an image
*/
int GIP_image_free(GIP_IMAGE_T *img)
{
	DEBUG("img:%p\n",img);

	/*
	**	mark this one as unused again
	*/
#if HAVE_LIBMAGICK
	DestroyImage(img->image);
#endif
#if HAVE_LIBIL
	ilDeleteImages(1,&img->id);
#endif

	/*
	**	free the image struct
	*/
	free((void *) img);
	return 1;
}

/*
**	duplicate an image
*/
GIP_IMAGE_T *GIP_image_duplicate(const GIP_IMAGE_T *src)
{
	GIP_IMAGE_T *dst;
#if HAVE_LIBMAGICK
	ExceptionInfo exception;
#endif

	DEBUG("src:%p\n",src);

	dst = alloc_image();
#if HAVE_LIBMAGICK
	GetExceptionInfo(&exception);
	if (dst->image)
		DestroyImage(dst->image);
	if (!(dst->image = CloneImage(src->image,0,0,1,&exception)))
		CRITICAL("CloneImage() failed\n");
	DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
	ilBindImage(dst->id);
	ilCopyImage(src->id);
#endif
	dst->width = src->width;
	dst->height = src->height;

	return dst;
}

/*
**	do a bitblit operation on two images
*/
int GIP_image_bitblit(GIP_IMAGE_T *dst,int dx,int dy,const GIP_IMAGE_T *src,int sx,int sy,int w,int h)
{
#if HAVE_LIBMAGICK
	Image *cropped = NULL;
	Image *source = src->image;
	ExceptionInfo exception;
#endif

	DEBUG("dst:%p  dx:%d  dy:%d  src:%p  sx:%d  sy:%d  w:%d  h:%d\n",
			dst,dx,dy,
			src,sx,sy,
			w,h);

#if HAVE_LIBMAGICK
	GetExceptionInfo(&exception);
	if (sx != 0 || sy != 0 || w != src->width || h != src->height) {
		/*
		**	we have to crop the source image
		*/
		RectangleInfo geometry;

		geometry.x = sx;
		geometry.y = sy;
		geometry.width = w;
		geometry.height = h;
		if (!(source = cropped = CropImage(src->image,&geometry,&exception)))
			CRITICAL("CropImage() failed\n");
	}
	if (CompositeImage(dst->image,SrcOverCompositeOp,source,dx,dy) != MagickTrue)
		CRITICAL("CompositeImage() failed\n");
	if (cropped)
		DestroyImage(cropped);
	DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
	ilBindImage(dst->id);
	if (!ilBlit(src->id,dx,dy,0,
			sx,sy,0,
			w,h,
			ilGetInteger(IL_IMAGE_DEPTH))
		 )
		CRITICAL("ilBlit() failed\n");
#endif
	return 1;
}

/*
**	overlay the complete src image onto the destination image
*/
int GIP_image_overlay(GIP_IMAGE_T *dst,int dx,int dy,const GIP_IMAGE_T *src)
{
	DEBUG("dst:%p  dx:%d  dy:%d  src:%p\n",
			dst,dx,dy,src);

	return GIP_image_bitblit(dst,dx,dy,src,0,0,src->width,src->height);
}

/*
**	fade src0 ti src1 and drop the resulting image in dst
**
**	all image must have the same size
*/
int GIP_image_blend(GIP_IMAGE_T *dst,const GIP_IMAGE_T *src0,const GIP_IMAGE_T *src1,float a)
{
#if HAVE_LIBMAGICK
	PixelPacket *src0_data,*src1_data,*dst_data;
	int x,y,d;
#endif
#if HAVE_LIBIL
	ILubyte *src0_data,*src1_data,*dst_data;
	ILubyte *src0_line,*src1_line,*dst_line;
	ILint src0_origin,src1_origin,dst_origin;
	ILuint size,line_size,x,y;
	ILint d;
	ILinfo info;
#endif

	DEBUG("dst:%p  src0:%p  src1:%p  a:%f\n",
		dst,src0,src1,a);

	/*
	**	check if the sizes match
	**/
	if (src0->width != src1->width || src0->height != src1->height)
		CRITICAL("src0 size %dx%dpx != src1 size %dx%dpx\n",
			src0->width,src0->height,src1->width,src1->height);
	if (dst->width != src0->width || dst->height != src0->height)
		CRITICAL("dst size %dx%dpx != src0 size %dx%dpx\n",
			dst->width,dst->height,src0->width,src0->height);

#if HAVE_LIBMAGICK
	/*
	**	get the data pointers and the size
	*/
	if (!(src0_data = GetImagePixels(src0->image,0,0,src0->width,src0->height)))
		CRITICAL("GetImagePixels() failed\n");
	if (!(src1_data = GetImagePixels(src1->image,0,0,src1->width,src1->height)))
		CRITICAL("GetImagePixels() failed\n");
	if (!(dst_data = GetImagePixels(dst->image,0,0,dst->width,dst->height)))
		CRITICAL("GetImagePixels() failed\n");

	/*
	**	do the blend itself
	*/
	for (y = 0;y < dst->height;y++) {
		/*
		**	do it on every row
		*/
#define BLEND_PIXEL_COLOR(color0,color1) \
			d = (int) ((1 - (a)) * (color0) + (a) * (color1)); \
			if (d < 0) \
				d = 0; \
			if (d > 255) \
				d = 255;
#define BLEND_PIXEL(dst,src0,src1) \
			(dst)->red   = BLEND_PIXEL_COLOR((src0)->red  ,(src1)->red  ); \
			(dst)->blue  = BLEND_PIXEL_COLOR((src0)->blue ,(src1)->blue ); \
			(dst)->green = BLEND_PIXEL_COLOR((src0)->green,(src1)->green);
		for (x = 0;x < dst->width;x++) {
			BLEND_PIXEL(dst_data,src0_data,src1_data);
			src0_data++;
			src1_data++;
			dst_data++;
		}
#undef BLEND_PIXEL
#undef BLEND_PIXEL_COLOR
	}
#endif
#if HAVE_LIBIL
	/*
	**	get the origins, data pointers and the size
	*/
	ilBindImage(src0->id);
	iluGetImageInfo(&info);
	src0_data = info.Data;
	src0_origin = info.Origin;
	ilBindImage(src1->id);
	iluGetImageInfo(&info);
	src1_data = info.Data;
	src1_origin = info.Origin;
	ilBindImage(dst->id);
	iluGetImageInfo(&info);
	dst_data = info.Data;
	dst_origin = info.Origin;
	size = info.SizeOfData;
	line_size = info.Width * info.Bpp;
	size = info.SizeOfData;
	line_size = info.Width * info.Bpp;

	/*
	**	do the blend itself
	*/
	for (y = 0;y < dst->height;y++) {
		/*
		**	set the line pointers
		*/
		src0_line = (src0_origin == IL_ORIGIN_UPPER_LEFT)
						? &src0_data[y * line_size]
						: &src0_data[(dst->height - 1 - y) * line_size];
		src1_line = (src1_origin == IL_ORIGIN_UPPER_LEFT)
						? &src1_data[y * line_size]
						: &src1_data[(dst->height - 1 - y) * line_size];
		dst_line  = (dst_origin  == IL_ORIGIN_UPPER_LEFT)
						? &dst_data[y * line_size]
						: &dst_data[(dst->height - 1 - y) * line_size];

		/*
		**	do it on every row
		*/
		for (x = 0;x < line_size;x++) {
			d = (ILint) ((1 - a) * *src0_line++ + a * *src1_line++);
			if (d < 0) 
				d = 0;
			else if (d > 255)
				d = 255;
			*dst_line++ = d;
		}
	}
#endif
	return 1;
}

/*
**	scale the given image
*/
int GIP_image_scale(GIP_IMAGE_T *img,int width,int height)
{
#if HAVE_LIBMAGICK
	Image *image;
	ExceptionInfo exception;
#endif
	DEBUG("img:%p  size:%dx%dpx => %dx%dpx\n",img,img->width,img->height,width,height);

	if (img->width != width || img->height != height) {
#if HAVE_LIBMAGICK
		GetExceptionInfo(&exception);
		if (!(image = ResizeImage(img->image,width,height,_filter,1.0,&exception)))
			CRITICAL("ResizeImage() failed (width=%d, height=%d)\n",width,height);
		DestroyImage(img->image);
		img->image = image;
		img->width = img->image->columns;
		img->height = img->image->rows;
		DestroyExceptionInfo(&exception);
#endif
#if HAVE_LIBIL
		ilBindImage(img->id);
		DEBUG("actual size:%dx%dpx\n",
				ilGetInteger(IL_IMAGE_WIDTH),
				ilGetInteger(IL_IMAGE_HEIGHT));
		iluImageParameter(ILU_FILTER,_filter);
		if (!iluScale(width,height,ilGetInteger(IL_IMAGE_DEPTH)))
			CRITICAL("iluScale() failed\n");
		img->width = ilGetInteger(IL_IMAGE_WIDTH);
		img->height = ilGetInteger(IL_IMAGE_HEIGHT);
#endif
		if (img->width != width || img->height != height)
			CRITICAL("actual size of %dx%dpx is not the expected size of %dx%dpx\n",
				img->width,img->height,width,height);
	}
	return 1;
}

/*
**	do a scaled bitblit
*/
int GIP_image_scaleblit(GIP_IMAGE_T *dimg,int dx,int dy,int dw,int dh,const GIP_IMAGE_T *simg,int sx,int sy,int sw,int sh)
{
	GIP_IMAGE_T *img;

	DEBUG("simg: %dx%d+%d+%d => %dx%d+%d+%d\n",sw,sh,sx,sy,dw,dh,dx,dy);

	/*
	**	scale the source image
	*/
	img = GIP_image_geoscale(simg,sx,sy,sw,sh,dw,dh);

	/*
	**	bitblit onto the destination image
	*/
	GIP_image_bitblit(dimg,dx,dy,img,0,0,dw,dh);

	/*
	**	drop the interim image
	*/
	GIP_image_free(img);

	return 1;
}

/*
**	create a new image with the given gemetry
*/
GIP_IMAGE_T *GIP_image_geoscale(const GIP_IMAGE_T *simg,int x,int y,int w,int h,int width,int height)
{
	GIP_IMAGE_T *img;

	DEBUG("simg: geo:%dx%d+%d+%d => %dx%d\n",w,h,x,y,width,height);

	img = GIP_image_create(GIP_COLOR_STR_BLACK,w,h);

	/*
	**	copy the area out of the source image ...
	*/
	GIP_image_bitblit(img,0,0,simg,x,y,w,h);

	/*
	**	and scale it to the correct output size
	*/
	GIP_image_scale(img,width,height);

	return img;
}

/*
**	add border to an image
**
**	new width = width + 2 * bw
**	new height = height + 2 * bh
*/
GIP_IMAGE_T *GIP_image_border(const GIP_IMAGE_T *simg,const char *rgbcolor,int bw,int bh)
{
	GIP_IMAGE_T *img;

	DEBUG("simg: %p => %dx%d\n",simg,bw,bh);

	img = GIP_image_create(rgbcolor,simg->width + 2 * bw,simg->height + 2 * bh);

	/*
	**  copy the area out of the source image ...
	*/
	GIP_image_bitblit(img,bw,bh,simg,0,0,simg->width,simg->height);

	return img;
}/**/
