/*
**	$Id: index.c 1091 2007-09-15 09:57:41Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	handle index files
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <config.h>
#include <pthread.h>
#include <math.h>
#include "gip.h"

#define min(a,b)	(((a) < (b)) ? (a) : (b))
#define max(a,b)	(((a) > (b)) ? (a) : (b))

/*
**	the scale up factor gurantees that source images
**	are scaled up too be at least FACTOR * output_[width|height]
**	if images are already large enough, no up-scaling is done
*/
#define PRESCALE_FACTOR		2

/*
**	depending on the target frame dimensions, loaded images
**	will never we scaled larger that frame_width * MAXUPSCALE
**	and frame_height * MAXUPSCALE
*/
#define MAXUPSCALE_FACTOR	4

/*
**	sync mutex for threads
*/
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
static GIP_FRAME_T *_thread;
static int _max_threads = 0;
static int _prescale_factor = PRESCALE_FACTOR;
static int _maxupscale_factor = MAXUPSCALE_FACTOR;

/*
**	init the API
*/
int GIP_index_init(void)
{
	_prescale_factor = (_gip_setup.prescale_factor) ? _gip_setup.prescale_factor : PRESCALE_FACTOR;
	_maxupscale_factor = (_gip_setup.maxupscale_factor) ? _gip_setup.maxupscale_factor : MAXUPSCALE_FACTOR;
	return 1;
}

/*
**	shutdown the API
*/
int GIP_index_exit(void)
{
	return 1;
}

/*
**	convert an index type into the corresponding type code
*/
int GIP_index_str2type(const char *s)
{
#define GIP_INDEX_STR2TYPE(type)	if (!strcasecmp(s,# type) || !strcasecmp(s,"GIP_INDEX_TYPE_" # type)) \
										return GIP_INDEX_TYPE_ ## type;
	GIP_INDEX_STR2TYPE(AUDIO);
	GIP_INDEX_STR2TYPE(TRANSITION);
	GIP_INDEX_STR2TYPE(SOLID);
	GIP_INDEX_STR2TYPE(IMAGE);
	GIP_INDEX_STR2TYPE(SCROLL);
	GIP_INDEX_STR2TYPE(OVERLAY);
	return GIP_INDEX_TYPE_UNKNOWN;
#undef GIP_INDEX_STR2TYPE
}

/*
**	convert an imgindex type code into the corresponding name
*/
const char *GIP_index_type2str(int type)
{
#define GIP_INDEX_TYPE2STR(type)	case GIP_INDEX_TYPE_ ## type:	return # type;
	switch (type) {
		GIP_INDEX_TYPE2STR(AUDIO);
		GIP_INDEX_TYPE2STR(TRANSITION);
		GIP_INDEX_TYPE2STR(SOLID);
		GIP_INDEX_TYPE2STR(IMAGE);
		GIP_INDEX_TYPE2STR(SCROLL);
		GIP_INDEX_TYPE2STR(OVERLAY);
	}
	return NULL;
#undef GIP_INDEX_TYPE2STR
}

/*
**	generate an "uncomputed" index (image durations are not set)
*/
GIP_INDEX_T *GIP_index_create(void)
{
	GIP_INDEX_T *idx;

	if (!(idx = malloc(sizeof(GIP_INDEX_T))))
		CRITICAL("out of memory\n");
	memset(idx,0,sizeof(GIP_INDEX_T));
	return idx;
}

/*
**	add a new file entry
*/
GIP_INDEX_ITEM_T *new_index_item(GIP_INDEX_T *idx,int type)
{
	GIP_INDEX_ITEM_T *item,*last = NULL;
	GIP_INDEX_ITEM_T **anchor = NULL;

	DEBUG("idx:%p\n",idx);

	/*
	**	allocate a new image
	*/
	if (!(item = malloc(sizeof(GIP_INDEX_ITEM_T))))
		CRITICAL("out of memory\n");
	memset(item,0,sizeof(GIP_INDEX_ITEM_T));

	/*
	**	get the anchor
	*/
	switch (type) {
		case GIP_INDEX_AUDIO:
			anchor = &(idx->audio);
			break;
		case GIP_INDEX_VIDEO:
			anchor = &(idx->video);
			break;
		case GIP_INDEX_OVERLAY:
			anchor = &(idx->overlay);
			break;
		default:
			CRITICAL("unknown index type\n");
	}

	/*
	**	find the current end of the list
	*/
	for (last = *anchor;last && last->next;last = last->next)
		;

	/*
	**	book-keeping
	*/
	if (!*anchor)
		*anchor = item;
	if (last)
		last->next = item;
	item->prev = last;
	item->next = NULL;

	/*
	**	setup some defaults
	*/
	item->nr = (last) ? last->nr + 1 : 0;
	item->frames = -1;

	return item;
}

/*
**	add a new entry for audio files
*/
GIP_INDEX_ITEM_T *GIP_index_add_audio(GIP_INDEX_T *idx,const char *file)
{
	GIP_INDEX_ITEM_T *item;

	DEBUG("idx:%p  file:%s\n",idx,file);

	if ((item = new_index_item(idx,GIP_INDEX_AUDIO))) {
		item->type = GIP_INDEX_TYPE_AUDIO;
		if (file)
			item->it.audio.file = strdup(file);
	}
	return item;
}

/*
**	add a new entry for a transition
*/
GIP_INDEX_ITEM_T *GIP_index_add_transition(GIP_INDEX_T *idx,int type)
{
	GIP_INDEX_ITEM_T *item;

	DEBUG("idx:%p  type:%d\n",idx,type);

	if ((item = new_index_item(idx,GIP_INDEX_VIDEO))) {
		item->type = GIP_INDEX_TYPE_TRANSITION;
		item->it.transition.type = type;
	}
	return item;
}

/*
**	add a new entry for solid images
*/
GIP_INDEX_ITEM_T *GIP_index_add_solid(GIP_INDEX_T *idx,const char *rgb)
{
	GIP_INDEX_ITEM_T *item;

	DEBUG("idx:%p  rgb:%s\n",idx,rgb);

	if ((item = new_index_item(idx,GIP_INDEX_VIDEO))) {
		item->type = GIP_INDEX_TYPE_SOLID;
		item->frames = 0;
		if (rgb)
			item->it.solid.rgb = strdup(rgb);
	}
	return item;
}

/*
**	add a new entry for images
*/
GIP_INDEX_ITEM_T *GIP_index_add_image(GIP_INDEX_T *idx,const char *file,int kenburns,int aspectcorrection)
{
	GIP_INDEX_ITEM_T *item;

	DEBUG("idx:%p  file:%s\n",idx,file);

	if ((item = new_index_item(idx,GIP_INDEX_VIDEO))) {
		item->type = GIP_INDEX_TYPE_IMAGE;
		if (file)
			item->it.image.file = strdup(file);
		if (kenburns)
			item->it.image.to_geo = strdup(GIP_GEOMETRY_PERCENT_KENBURNS);
		item->it.image.aspectcorrection = aspectcorrection;
	}
	return item;
}

/*
**	add a new entry for scrolling images
*/
GIP_INDEX_ITEM_T *GIP_index_add_scroll(GIP_INDEX_T *idx,const char *file)
{
	GIP_INDEX_ITEM_T *item;

	DEBUG("idx:%p  file:%s\n",idx,file);

	if ((item = new_index_item(idx,GIP_INDEX_VIDEO))) {
		item->type = GIP_INDEX_TYPE_SCROLL;
		if (file)
			item->it.scroll.file = strdup(file);
	}
	return item;
}

/*
**	add a new entry for overlay images
*/
GIP_INDEX_ITEM_T *GIP_index_add_overlay(GIP_INDEX_T *idx,const char *file)
{
	GIP_INDEX_ITEM_T *item;

	DEBUG("idx:%p  file:%s\n",idx,file);

	if ((item = new_index_item(idx,GIP_INDEX_OVERLAY))) {
		item->type = GIP_INDEX_TYPE_OVERLAY;
		if (file)
			item->it.overlay.file = strdup(file);
	}
	return item;
}

/*
**	computed an index (image durations will be set)
*/
int GIP_index_compute(GIP_INDEX_T *idx,int default_fpi,int default_fpt,double framerate,GIP_INDEX_INFO_T *info)
{
	int nr = 0,frames,pad,fpi;
	GIP_INDEX_ITEM_T *video = NULL;
	GIP_INDEX_ITEM_T *audio = NULL;
	GIP_INDEX_ITEM_T *overlay = NULL;
	int total_frames = 0;

	if (info)
		memset(info,0,sizeof(*info));

	/*
	**	audio pass
	*/
	for (audio = idx->audio;audio;audio = audio->next) {
		/*
		**	compute the audio duration
		*/
		audio->frames = GIP_audio_duration(audio->it.audio.file) * framerate;
		total_frames += audio->frames;
		if (info)
			info->num_audios++;
	}

	/*
	**	video pass
	*/
	for (video = idx->video, frames = 0;video;video = video->next) {
		if (info)
			switch (video->type) {
				case GIP_INDEX_TYPE_TRANSITION:
					info->num_transitions++;
					break;
				case GIP_INDEX_TYPE_SOLID:
					info->num_solids++;
					break;
				case GIP_INDEX_TYPE_IMAGE:
					info->num_images++;
					break;
				case GIP_INDEX_TYPE_SCROLL:
					info->num_scrolls++;
					break;
			}

		/*
		**	set the default for unspecified transitions
		*/
		if (video->type == GIP_INDEX_TYPE_TRANSITION) {
			if (video->frames < 0)
				video->frames = default_fpt;
			if (video->it.transition.type == GIP_TRANSITION_TYPE_RANDOM)
				video->it.transition.type = GIP_transition_random();
		}
		if (video->next) {
			/*
			**	check if transitions and non-transitions do alter
			**	in the video index
			*/
			if ((video->type == GIP_INDEX_TYPE_TRANSITION &&
					video->next->type == GIP_INDEX_TYPE_TRANSITION) ||
				(video->type != GIP_INDEX_TYPE_TRANSITION &&
					video->next->type != GIP_INDEX_TYPE_TRANSITION)) {
				CRITICAL("transition alternation error (entry %d and next)\n",video->nr);
			}
		}

		/*
		**	count the number of frames which are already set
		*/
		if (video->frames < 0)
			nr++;
		else
			frames += video->frames;
	}
	DEBUG("frames already set:%d  images to set frames:%d\n",frames,nr);

	/*
	**	overlay pass
	*/
	for (overlay = idx->overlay;overlay;overlay = overlay->next) {
		if (info)
			info->num_overlays++;
	}

	/*
	**	compute the duration for the images
	*/
	if (total_frames <= 0) {
		fpi = default_fpi;
		pad = 0;
		total_frames = 0;
	}
	else {
		if ((fpi = (nr) ? (total_frames - frames) / nr : 0) < 0)
			fpi = 0;
		pad = (total_frames - frames) - fpi * nr;
	}
	DEBUG("fpi:%d  remaining frames:%d  pad:%d\n",fpi,total_frames - frames,pad);

	/*
	**	distribute the frames
	*/
	for (video = idx->video, frames = 0;video;video = video->next) {
		if (video->frames < 0)
			video->frames = fpi + ((pad-- > 0) ? 1 : 0);
		frames += video->frames;
	}

	/*
	**	recheck the number of frames
	*/
	if (total_frames && frames != total_frames)
		CRITICAL("total_frames: %d != %d\n",frames,total_frames);

	if (info) {
		info->framerate = framerate;
		info->frames = frames;
	}

	return 0;
}

/*
**	print an index
*/
void GIP_index_print(const GIP_INDEX_T *idx)
{
	const GIP_INDEX_ITEM_T *audio;
	const GIP_INDEX_ITEM_T *video;
	const GIP_INDEX_ITEM_T *overlay;

	for (audio = idx->audio;audio;audio = audio->next) {
		INFORMATION("[%03d] %s: frames:%d"
				   "  prev/next:[%03d]/[%03d]"
				   "\n",
			   audio->nr,
			   GIP_index_type2str(audio->type),
			   audio->frames,
			   (audio->prev) ? audio->prev->nr : -1,
			   (audio->next) ? audio->next->nr : -1);
		switch (audio->type) {
			case GIP_INDEX_TYPE_AUDIO:
				INFORMATION("      "
					   "  file:%-10.10s  "
					   "\n",
					   audio->it.audio.file);
				break;
		}
	}
	for (video = idx->video;video;video = video->next) {
		INFORMATION("[%03d] %s: frames:%d"
				   "  prev/next:[%03d]/[%03d]"
				   "\n",
			   video->nr,
			   GIP_index_type2str(video->type),
			   video->frames,
			   (video->prev) ? video->prev->nr : -1,
			   (video->next) ? video->next->nr : -1);
		switch (video->type) {
			case GIP_INDEX_TYPE_TRANSITION:
				INFORMATION("      "
					   "  type:%s"
					   "\n",
					   GIP_transition_type2str(video->it.transition.type));
				break;
			case GIP_INDEX_TYPE_SOLID:
				INFORMATION("      "
					   "  rgb:%-10.10s  "
					   "\n",
					   video->it.solid.rgb);
				break;
			case GIP_INDEX_TYPE_IMAGE:
				INFORMATION("      "
					   "  file:%-10.10s  "
					   "  size:%dx%dpx  "
					   "  From Geo:%-15.15s"
					   "  To Geo:%-15.15s"
					   "\n",
					   video->it.image.file,
					   video->it.image.width,video->it.image.height,
					   (video->it.image.from_geo) ? video->it.image.from_geo : GIP_GEOMETRY_PERCENT_100,
					   (video->it.image.to_geo) ? video->it.image.to_geo : GIP_GEOMETRY_PERCENT_100);
				break;
			case GIP_INDEX_TYPE_SCROLL:
				INFORMATION("      "
					   "  file:%-10.10s  "
					   "  size:%dx%dpx  "
					   "\n",
					   video->it.scroll.file,
					   video->it.scroll.width,video->it.image.height);
				break;
		}
	}
	for (overlay = idx->overlay;overlay;overlay = overlay->next) {
		INFORMATION("[%03d] %s: frames:%d"
				   "  prev/next:[%03d]/[%03d]"
				   "\n",
			   overlay->nr,
			   GIP_index_type2str(video->type),
			   overlay->frames,
			   (overlay->prev) ? overlay->prev->nr : -1,
			   (overlay->next) ? overlay->next->nr : -1);
		switch (overlay->type) {
			case GIP_INDEX_TYPE_OVERLAY:
				INFORMATION("      "
					   "  file:%-10.10s  "
					   "\n",
					   overlay->it.overlay.file);
				break;
		}
	}
}

/*
**	write an index file
*/
int GIP_index_write(const GIP_INDEX_T *idx,const char *file,int force,const char *title,double framerate)
{
	const GIP_INDEX_ITEM_T *audio;
	const GIP_INDEX_ITEM_T *video;
	const GIP_INDEX_ITEM_T *overlay;
	FILE *fd;
	char buffer[20];

	/*
	**	dont allow overwriting if not in force mode
	*/
	if (!force && access(file,R_OK) != 0)
		CRITICAL("file %s exists -- force overwriting required!\n",file);

	/*
	**	open file for writing/truncate
	*/
	if (!(fd = fopen(file,"w+")))
		CRITICAL("fopen(%s) failed\n",file);
	fprintf(fd,"; %s\n",title);

	/*
	**	write the audio records
	*/
	for (audio = idx->audio;audio;audio = audio->next) {
		switch (audio->type) {
			case GIP_INDEX_TYPE_AUDIO:
				fprintf(fd,"type=%s"
							",file=%s"
							"\n",
					   GIP_index_type2str(audio->type),
					   audio->it.audio.file);
				break;
		}
	}

	/*
	**	write the overlay records
	*/
	for (overlay = idx->overlay;overlay;overlay = overlay->next) {
		switch (overlay->type) {
			case GIP_INDEX_TYPE_OVERLAY:
				fprintf(fd,"type=%s"
							",file=%s"
							"\n",
					   GIP_index_type2str(overlay->type),
					   overlay->it.overlay.file);
				break;
		}
	}

	/*
	**	write the video records
	*/
	for (video = idx->video;video;video = video->next) {
		switch (video->type) {
			case GIP_INDEX_TYPE_TRANSITION:
				fprintf(fd,"type=%s"
							",time=%s"
							",transition=%s"
							"\n",
					   GIP_index_type2str(video->type),
					   GIP_frames2time(video->frames, framerate, buffer),
					   GIP_transition_type2str(video->it.transition.type));
				break;
			case GIP_INDEX_TYPE_SOLID:
				fprintf(fd,"type=%s"
							",time=%s"
							",rgb=%s"
							"\n",
					   GIP_index_type2str(video->type),
					   GIP_frames2time(video->frames, framerate, buffer),
					   video->it.solid.rgb);
				break;
			case GIP_INDEX_TYPE_IMAGE:
				fprintf(fd,"type=%s"
							",time=%s"
							",file=%s"
							",from_geo=%s"
							",to_geo=%s"
							",aspectcorrection=%s"
							"\n",
					   GIP_index_type2str(video->type),
					   GIP_frames2time(video->frames, framerate, buffer),
					   video->it.image.file,
					   (video->it.image.from_geo) ? video->it.image.from_geo : GIP_GEOMETRY_PERCENT_100,
					   (video->it.image.to_geo) ? video->it.image.to_geo : GIP_GEOMETRY_PERCENT_100,
					   GIP_aspectcorrection_type2str(video->it.image.aspectcorrection));
				break;
			case GIP_INDEX_TYPE_SCROLL:
				fprintf(fd,"type=%s"
							",time=%s"
							",file=%s"
							"\n",
					   GIP_index_type2str(video->type),
					   GIP_frames2time(video->frames, framerate, buffer),
					   video->it.scroll.file);
				break;
		}
	}

	fclose(fd);
	return 0;
}

/*
**	read an index back
*/
GIP_INDEX_T *GIP_index_read(const char *file,int kenburns,int aspectcorrection,double framerate)
{
	GIP_INDEX_T *idx = NULL;
	FILE *fd;
	char buffer[1000];
	char *line;
	int line_nr = 0;

	/*
	**	open file for reading
	*/
	if (!(fd = fopen(file,"r")))
		CRITICAL("fopen(%s) failed\n",file);

	/*
	**	create a new index
	*/
	idx = GIP_index_create();

	/*
	**	read the image records
	*/
	while ((line = fgets(buffer,sizeof(buffer),fd))) {
		/*
		**	skip empty lines or comments
		*/
		char *attr,*key,*val;
		int type = GIP_INDEX_TYPE_UNKNOWN;
		GIP_INDEX_ITEM_T *video = NULL;
		GIP_INDEX_ITEM_T *audio = NULL;
		GIP_INDEX_ITEM_T *overlay = NULL;

		++line_nr;
		switch (*line) {
			case ';':
			case '#':
			case '\n':
			case '\r':
			case '\0':
				continue;
		}

		attr = line;
		while ((key = strsep(&attr,",\n\r")) && *key) {
			/*
			**	setup the key an value pair
			*/
			if (!(val = strchr(key,'=')))
				CRITICAL("syntax error in %s:%d\n",file,line_nr);
			*val++ = '\0';

			if (type == GIP_INDEX_TYPE_UNKNOWN) {
				if (strcasecmp(key,"type"))
					CRITICAL("expected 'type' not found in %s:%d\n",file,line_nr);
				switch ((type = GIP_index_str2type(val))) {
					case GIP_INDEX_TYPE_AUDIO:
						audio = GIP_index_add_audio(idx,NULL);
						break;
					case GIP_INDEX_TYPE_TRANSITION:
						video = GIP_index_add_transition(idx,GIP_TRANSITION_TYPE_RANDOM);
						break;
					case GIP_INDEX_TYPE_SOLID:
						video = GIP_index_add_solid(idx,NULL);
						break;
					case GIP_INDEX_TYPE_IMAGE:
						video = GIP_index_add_image(idx,NULL,kenburns,aspectcorrection);
						break;
					case GIP_INDEX_TYPE_SCROLL:
						video = GIP_index_add_scroll(idx,NULL);
						break;
					case GIP_INDEX_TYPE_OVERLAY:
						overlay = GIP_index_add_overlay(idx,NULL);
						break;
					default:
						CRITICAL("unknown type '%s' in %s:%d\n",val,file,line_nr);
				}
			}
			else if (!strcasecmp(key,"file") || !strcasecmp(key,"rgb")) {
				/*
				**	file/rgb
				*/
				switch (type) {
					case GIP_INDEX_TYPE_AUDIO:
						audio->it.audio.file = strdup(val);
						break;
					case GIP_INDEX_TYPE_SOLID:
						video->it.solid.rgb = strdup(val);
						break;
					case GIP_INDEX_TYPE_IMAGE:
						video->it.image.file = strdup(val);
						break;
					case GIP_INDEX_TYPE_SCROLL:
						video->it.scroll.file = strdup(val);
						break;
					case GIP_INDEX_TYPE_OVERLAY:
						overlay->it.overlay.file = strdup(val);
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else if (!strcasecmp(key,"frames")) {
				/*
				**	the image frames
				*/
				switch (type) {
					case GIP_INDEX_TYPE_AUDIO:
						audio->frames = atoi(val);
						break;
					case GIP_INDEX_TYPE_TRANSITION:
					case GIP_INDEX_TYPE_SOLID:
					case GIP_INDEX_TYPE_IMAGE:
					case GIP_INDEX_TYPE_SCROLL:
						video->frames = atoi(val);
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else if (!strcasecmp(key,"time")) {
				/*
				**	the image times
				*/
				switch (type) {
					case GIP_INDEX_TYPE_AUDIO:
						audio->frames = GIP_time2frames(val, framerate);
						break;
					case GIP_INDEX_TYPE_TRANSITION:
					case GIP_INDEX_TYPE_SOLID:
					case GIP_INDEX_TYPE_IMAGE:
					case GIP_INDEX_TYPE_SCROLL:
						video->frames = GIP_time2frames(val, framerate);
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else if (!strcasecmp(key,"transition")) {
				/*
				**	the transition type
				*/
				switch (type) {
					case GIP_INDEX_TYPE_TRANSITION:
						video->it.transition.type = GIP_transition_str2type(val);
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else if (!strcasecmp(key,"from_geo")) {
				/*
				**	from_geo
				*/
				if (kenburns)
					val = GIP_GEOMETRY_PERCENT_100;
				switch (type) {
					case GIP_INDEX_TYPE_IMAGE:
						video->it.image.from_geo = (strcmp(val,GIP_GEOMETRY_PERCENT_100)) ? strdup(val) : NULL;
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else if (!strcasecmp(key,"to_geo")) {
				/*
				**	the to geometry
				*/
				if (kenburns)
					val = GIP_GEOMETRY_PERCENT_KENBURNS;
				switch (type) {
					case GIP_INDEX_TYPE_IMAGE:
						video->it.image.to_geo = (strcmp(val,GIP_GEOMETRY_PERCENT_100)) ? strdup(val) : NULL;
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else if (!strcasecmp(key,"aspectcorrection")) {
				/*
				**	the aspectcorrection
				*/
				switch (type) {
					case GIP_INDEX_TYPE_IMAGE:
						video->it.image.aspectcorrection = GIP_aspectcorrection_str2type(val);
						break;
					default:
						CRITICAL("key '%s' not allowed for type '%s' in %s:%d\n",
							key,GIP_index_type2str(type),file,line_nr);
				}
			}
			else
				CRITICAL("unknown key '%s' in %s:%d\n",key,file,line_nr);
		}
	}
	return idx;
}

/*
**	free an audio index
*/
void GIP_index_items_free(GIP_INDEX_T *idx,int type)
{
	const GIP_INDEX_ITEM_T *item,*next;
	GIP_INDEX_ITEM_T **anchor = NULL;

	/*
	**	get the anchor
	*/
	switch (type) {
		case GIP_INDEX_AUDIO:
			anchor = &(idx->audio);
			break;
		case GIP_INDEX_VIDEO:
			anchor = &(idx->video);
			break;
		case GIP_INDEX_OVERLAY:
			anchor = &(idx->overlay);
			break;
		default:
			CRITICAL("unknown index type\n");
	}

	for (item = *anchor;item;item = next) {
		switch (item->type) {
			case GIP_INDEX_TYPE_AUDIO:
				if (item->it.audio.file)
					free((void *) item->it.audio.file);
				break;
			case GIP_INDEX_TYPE_SOLID:
				if (item->it.solid.rgb)
					free((void *) item->it.solid.rgb);
				break;
			case GIP_INDEX_TYPE_IMAGE:
				if (item->it.image.file)
					free((void *) item->it.image.file);
				if (item->it.image.from_geo)
					free((void *) item->it.image.from_geo);
				if (item->it.image.to_geo)
					free((void *) item->it.image.to_geo);
				break;
			case GIP_INDEX_TYPE_SCROLL:
				if (item->it.scroll.file)
					free((void *) item->it.scroll.file);
				break;
			case GIP_INDEX_TYPE_OVERLAY:
				if (item->it.overlay.file)
					free((void *) item->it.overlay.file);
				break;
		}
		next = item->next;
		free((void *) item);
	}
	*anchor = NULL;
}

/*
**	free a complete index
*/
void GIP_index_free(GIP_INDEX_T *idx)
{
	GIP_index_items_free(idx,GIP_INDEX_AUDIO);
	GIP_index_items_free(idx,GIP_INDEX_VIDEO);
	GIP_index_items_free(idx,GIP_INDEX_OVERLAY);
	free((void *) idx);
}

/*
**	load an image
*/
static GIP_IMAGE_T *load_image(GIP_INDEX_ITEM_T *item,int width,int height,double aspectratio)
{
	GIP_IMAGE_T *img = NULL;

	if (item) {
		DEBUG("type:%s  size:%dx%dpx\n",
				GIP_index_type2str(item->type),
				width,height);

		switch (item->type) {
			case GIP_INDEX_TYPE_SOLID:
				/*
				**	create the image from the given rgb color
				*/
				if (!(img = GIP_image_create(item->it.solid.rgb,width,height)))
					CRITICAL("couldn't create image from RGB color %s\n",item->it.solid.rgb);
				DEBUG("loaded rgb:%s  size:%dx%dpx\n",item->it.solid.rgb,img->width,img->height);
				break;
			case GIP_INDEX_TYPE_IMAGE:
				/*
				**	load the image from the given file
				*/
				if (!(img = GIP_image_load(item->it.image.file)))
					CRITICAL("couldn't load image from file %s\n",item->it.image.file);

				/*
				**	do some correction on the input image
				*/
				if (!(img = GIP_aspectcorrection_process(img,width,height,aspectratio,item->it.image.aspectcorrection)))
					CRITICAL("couldn't process images aspect correction: %s\n",item->it.image.file);

				/*
				**	do some prescaling
				*/
				if (!_gip_setup.quick_mode && (item->it.image.from_geo || item->it.image.to_geo)) {
					/*
					**	dynamic images will be prescaled for
					**	- better results, if these are too small, and
					**	- better performance, if these are too large
					*/
					int from_x = 0,from_y = 0,from_w = img->width,from_h = img->height;
					int to_x = 0,to_y = 0,to_w = img->width,to_h = img->height;
					double scale;

					/*
					**	compute smallest width/height used by the geometries
					*/
					if (item->it.image.from_geo)
						GIP_geometry_parse(item->it.image.from_geo,img->width,img->height,width,height,
							&from_x,&from_y,&from_w,&from_h);
					if (item->it.image.to_geo)
						GIP_geometry_parse(item->it.image.to_geo,img->width,img->height,width,height,
							&to_x,&to_y,&to_w,&to_h);
					scale = _prescale_factor / min((double) min(from_w,to_w) / width,(double) min(from_h,to_h) / height);
					DEBUG("img:%s  %dx%dpx  scale=%f\n",
							item->it.image.file,
							img->width,img->height,
							scale);

					/*
					**	compute the scale factor
					**
					**	NOTE: we don't only check the width or height to
					**	decide of the dimensions are to large. Instead,
					**	we compute the plane area (w * h) of the image.
					*/
					if (img->width * scale * img->height * scale >
							width * _maxupscale_factor * height * _maxupscale_factor) {
						/*
						**	the image upscale is too large
						*/
						scale = sqrt((double) width * _maxupscale_factor * height * _maxupscale_factor
							/ img->width / img->height);
					}

					DEBUG("img:%s  %dx%dpx  scale=%f  scaled img: %dx%dpx\n",
							item->it.image.file,
							img->width,img->height,
							scale,
							(int) (img->width * scale),(int) (img->height * scale));
					GIP_image_scale(img,img->width * scale,img->height * scale);
				}
				else if (_gip_setup.quick_mode) {
					/*
					**	to speed up in quick mode, we scale
					**	the loaded image down to the half of
					**	the target size
					*/
					GIP_image_scale(img,width / 2,height / 2);
				}
				else {
					/*
					**	fixed size images we will scaled on load
					**	to the target size
					*/
					GIP_image_scale(img,width,height);
				}
				DEBUG("loaded img:%s  size:%dx%dpx\n",item->it.image.file,img->width,img->height);
				break;
			case GIP_INDEX_TYPE_SCROLL:
				/*
				**	load the image from the given file
				*/
				if (!(img = GIP_image_load(item->it.scroll.file)))
					CRITICAL("couldn't load image from file %s\n",item->it.scroll.file);
				DEBUG("loaded img:%s  size:%dx%dpx\n",item->it.scroll.file,img->width,img->height);
				break;
			case GIP_INDEX_TYPE_OVERLAY:
				/*
				**	load the image from the given file
				*/
				if (!(img = GIP_image_load(item->it.overlay.file)))
					CRITICAL("couldn't load image from file %s\n",item->it.overlay.file);
				GIP_image_scale(img,width,height);
				DEBUG("loaded img:%s  size:%dx%dpx\n",item->it.overlay.file,img->width,img->height);
				break;
			default:
				CRITICAL("type %d(%s) is not implemented\n",
					item->type,GIP_index_type2str(item->type));
				break;
		}
	}
	return img;
}

/*
**	helper function to compute a geometry scaling if any
*/
static GIP_IMAGE_T *compute_dynamic(const GIP_INDEX_ITEM_T *item,const GIP_IMAGE_T *img,int step,int width,int height)
{
	GIP_IMAGE_T *imgout = NULL;

	if (item) {
		int steps = GIP_INDEX_IMAGE_FRAMES(item);
		int x,y,w,h;

		switch (item->type) {
			case GIP_INDEX_TYPE_IMAGE:
				if (item->it.image.from_geo || item->it.image.to_geo) {
					/*
					**	if geometry specifiers are given, we
					**	have to scale the image
					*/
					GIP_geometry_process(item->it.image.from_geo,item->it.image.to_geo,
							img->width,img->height,
							width,height,
							step,steps,
							&x,&y,&w,&h);
					imgout = GIP_image_geoscale(img,x,y,w,h,width,height);
				}
				break;
			case GIP_INDEX_TYPE_SCROLL:
				/*
				**	scroll thru the image
				*/
				if (img->width > img->height) {
					/*
					**	scroll from left to right
					*/
					w = width * img->height / height;
					h = img->height;
					x = (img->width - w) * step / steps;
					y = 0;
				}
				else {
					/*
					**	scroll from top to bottom
					*/
					w = img->width;
					h = height * img->width / width;
					x = 0;
					y = (img->height - h) * step / steps;
				}
				imgout = GIP_image_geoscale(img,x,y,w,h,width,height);
				break;
		}
	}
	return imgout;
}

/*
**	process the image index
*/
int GIP_index_check(const GIP_INDEX_T *idx)
{
	const GIP_INDEX_ITEM_T *item;
	GIP_PROGRESS_T *progress;
	GIP_IMAGE_T *img = NULL;
	int pic = 0,pics = 0;

	/*
	**	check if the index is already computed
	*/
	for (item = idx->video;item;item = item->next) {
		switch (item->type) {
			case GIP_INDEX_TYPE_IMAGE:
			case GIP_INDEX_TYPE_SCROLL:
				pics++;
				break;
			default:
				break;
		}
	}

	/*
	**	process the index
	*/
	progress = GIP_progress_start(pics,"Images",stderr);
	for (item = idx->video;item;item = item->next) {
		/*
		**	setup the images
		*/
		switch (item->type) {
			case GIP_INDEX_TYPE_IMAGE:
			case GIP_INDEX_TYPE_SCROLL:
				/*
				**	load the next image
				*/
				INFORMATION("loading %s\n",item->it.image.file);
				if ((img = GIP_image_load(item->it.image.file)))
					GIP_image_free(img);
				else {
					CRITICAL("error loading %s\n",item->it.image.file);
				}
				GIP_progress_update(progress,pic++);
				break;
			default:
				break;
		}
	}
	GIP_progress_complete(progress);
	return 1;
}

static GIP_FRAME_T *setup_frame(GIP_FRAME_T *frame,const GIP_INDEX_ITEM_T *item,const GIP_IMAGE_T *img,const GIP_IMAGE_T *imgnext,const GIP_IMAGE_T *overlayimg,int nr,int width,int height,FILE *stream)
{
	memset(frame,0,sizeof(GIP_FRAME_T));
	frame->item = item;
	frame->img = img;
	frame->imgnext = imgnext;
	frame->overlayimg = overlayimg;
	frame->nr = nr;
	frame->width = width;
	frame->height = height;
	frame->stream = stream;
	frame->used = 1;
	return frame;
}

/*
**	process a single frame as described in the frame struct
*/
static void process_frame(GIP_FRAME_T *frame)
{
	DEBUG("item:%p  img:%p  imgnext:%p  frame:%d  width:%d  height:%d\n",frame->item,frame->img,frame->imgnext,frame->nr,frame->width,frame->height);

	if (frame->item->type == GIP_INDEX_TYPE_TRANSITION) {
		/*
		**	transition
		*/
		GIP_IMAGE_T *transimg;
		GIP_IMAGE_T *imgout = NULL;
		GIP_IMAGE_T *imgnextout = NULL;

		/*
		**	we may have to geoscale the current *and*
		**	the next image
		*/
		imgout = compute_dynamic(frame->item->prev,frame->img,
					GIP_INDEX_IMAGE_PREV_TRANS(frame->item->prev) +
					GIP_INDEX_IMAGE_PREV_FRAMES(frame->item) + frame->nr,frame->width,frame->height);
		imgnextout = compute_dynamic(frame->item->next,frame->imgnext,
					frame->nr,frame->width,frame->height);

		/*
		**	now do the transistion
		*/
		transimg = GIP_transition_process((imgout) ? imgout : frame->img,
				(imgnextout) ? imgnextout : frame->imgnext,
				frame->item->it.transition.type,
				frame->nr,frame->item->frames);

		/*
		**	we make the transition image to the output image
		*/
		if (imgout) {
			GIP_image_free(imgout);
			imgout = NULL;
		}
		if (imgnextout) {
			GIP_image_free(imgnextout);
			imgnextout = NULL;
		}
		frame->imgout = transimg;
	}
	else {
		/*
		**	solid, image, scroll, ...
		**
		**	we may have to compute a geo scaled output
		*/
		frame->imgout = compute_dynamic(frame->item,frame->img,
					GIP_INDEX_IMAGE_PREV_TRANS(frame->item) + frame->nr,frame->width,frame->height);
	}

	/*
	**	handle a possible overlay
	*/
	if (frame->overlayimg) {
		if (!frame->imgout)
			frame->imgout = GIP_image_duplicate(frame->img);
		GIP_image_overlay(frame->imgout,0,0,frame->overlayimg);
	}

	/*
	**	book-keeping
	*/
	DEBUG("terminating thread for frame #%d\n",frame->nr);
	frame->complete = 1;
	pthread_mutex_unlock(&mutex);
}

/*
**	write a single frame to the output pipe
*/
static void write_frame(GIP_FRAME_T *frame)
{
	if (!frame->used || !frame->complete)
		CRITICAL("frame not complete!\n");

	/*
	**	write the image
	*/
	GIP_image_savefd((frame->imgout) ? frame->imgout : frame->img,frame->stream);

	/*
	**	free the output image
	*/
	if (frame->imgout)
		GIP_image_free(frame->imgout);

	/*
	**	book-keeping
	*/
	frame->used = 0;
	frame->complete = 0;
}

/*
**	search for the smallest frame number in use
**
**	if the frame is not completed yet, -1 is returned
**	if no frame could be found, -1 is returned
*/
static int get_completed_frame(int wait)
{
	int slot,found = -1;

	for (slot = 0;slot < _max_threads;slot++)
		if (_thread[slot].used) {
			if (found < 0 || _thread[slot].nr < _thread[found].nr)
				found = slot;
		}

	if (found >= 0) {
		/*
		**	we got a candidate
		*/
		if (_thread[found].complete || wait) {
			/*
			**	we have to wait upon completion
			*/
			if (pthread_join(_thread[found].thread,NULL))
				CRITICAL("couldn't join thread!\n");
		}
		else
			return -1;
	}
	return found;
}

/*
**	find a free slot
*/
static int get_free_frame(void)
{
	int slot;

	for (;;) {
		/*
		**	write all frames to the output which are completed
		*/
		while ((slot = get_completed_frame(0)) >= 0)
			write_frame(&_thread[slot]);

		/*
		**	find an unused slot
		*/
		for (slot = 0;slot < _max_threads;slot++)
			if (!_thread[slot].used)
				return slot;

		/*
		**	try to lock the mutex
		**	if execution continues, a thread completed
		*/
		pthread_mutex_lock(&mutex);
		pthread_mutex_unlock(&mutex);
	}
	return -1;
}

/*
**	process the image index
*/
int GIP_index_process(const GIP_INDEX_T *idx,int width,int height,double aspectratio,FILE *stream,int max_threads)
{
	const GIP_INDEX_ITEM_T *item;
	GIP_PROGRESS_T *progress;
	GIP_IMAGE_T *img = NULL,*imgnext = NULL;
	GIP_IMAGE_T *overlayimg = NULL;
	int n;
	int frames = 0,total_frames = 0;
	int slot;

	DEBUG("idx:%p  width:%d  height:%d  stream:%p\n",idx,width,height,stream);

	/*
	**	check if the index is already computed
	*/
	for (item = idx->video;item;item = item->next) {
		if (item->frames < 0)
			CRITICAL("image index is not yet computed\n");
		total_frames += item->frames;
	}

	/*
	**	allocate memory for the threading stuff
	*/
	if (!(_thread = calloc((_max_threads = max_threads) ? max_threads : 1,sizeof(GIP_FRAME_T))))
		CRITICAL("couldn't allocate memory for threads\n");

	/*
	**	if there is an overlay, load it
	*/
	if (idx->overlay)
		overlayimg = load_image(idx->overlay,width,height,aspectratio);

	/*
	**	initialize the fist image
	*/
	imgnext = load_image(idx->video,width,height,aspectratio);
	progress = GIP_progress_start(total_frames,"Frames",stderr);

	/*
	**	process the index
	*/
	for (item = idx->video;item && item->next;item = item->next) {
		/*
		**	setup the images
		*/
		DEBUG("loading next image ...\n");
		if (item->type != GIP_INDEX_TYPE_TRANSITION) {
			/*
			**	free the previuos image
			*/
			if (img)
				GIP_image_free(img);

			/*
			**	make the previous image the current one
			*/
			img = imgnext;

			/*
			**	load the next image
			*/
			imgnext = load_image((item->next) ? item->next->next : NULL,width,height,aspectratio);
		}

		DEBUG("processing item ...\n");
		for (n = 0;n < item->frames;n++) {
			/*
			**	loop over the frames
			*/

			/*
			**	update the progress meter
			*/
			GIP_progress_update(progress,frames++);

			if (_max_threads > 1) {
				/*
				**	this is the threaded version
				*/

				/*
				**	find a free thread slot
				*/
				if ((slot = get_free_frame()) < 0)
					CRITICAL("couldn't find a free slot\n");

				/*
				**	setup the frame to process
				*/
				setup_frame(&_thread[slot],item,img,imgnext,overlayimg,n,width,height,stream);

				/*
				**	fire up the thread
				*/
				DEBUG("starting thread for frame #%d\n",_thread[slot].nr);
				if (pthread_create(&_thread[slot].thread,NULL,(void*) process_frame,&_thread[slot]))
					CRITICAL("couldn't create slot\n");
			}
			else {
				/*
				**	non-threaded version
				*/

				/*
				**	setup the frame to process
				*/
				setup_frame(_thread,item,img,imgnext,overlayimg,n,width,height,stream);

				/*
				**	process this frame
				*/
				process_frame(_thread);

				/*
				**	write the frame
				*/
				write_frame(_thread);
			}
		}

		if (_max_threads > 1) {
			/*
			**	write all frames to the output
			*/
			while ((slot = get_completed_frame(1)) >= 0)
				write_frame(&_thread[slot]);
		}
	}
	GIP_progress_complete(progress);

	/*
	**	free the images
	*/
	if (img)
		GIP_image_free(img);
	if (imgnext)
		GIP_image_free(imgnext);
	if (overlayimg)
		GIP_image_free(overlayimg);

	/*
	**	free the threading memory
	*/
	if (_thread) {
		free(_thread);
		_thread = NULL;
	}
	_max_threads = 0;

	return 1;
}/**/
