/*
**	$Id: index.h 1077 2007-07-20 09:21:12Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	handle index files
*/

#ifndef __GIP_INDEX_H__
#define __GIP_INDEX_H__ 1

#include <stdio.h>

enum {
	GIP_INDEX_TYPE_UNKNOWN = 0,
	GIP_INDEX_TYPE_TRANSITION,
	GIP_INDEX_TYPE_AUDIO,
	GIP_INDEX_TYPE_IMAGE,
	GIP_INDEX_TYPE_SCROLL,
	GIP_INDEX_TYPE_SOLID,
	GIP_INDEX_TYPE_OVERLAY,
};

enum {
	GIP_INDEX_AUDIO = 0,
	GIP_INDEX_VIDEO,
	GIP_INDEX_OVERLAY,
};

typedef struct _GIP_index_item {
	int nr;							// the entry number
	int type;						// the type of the entry
	int frames;						// number of frames for this entry
	union {
		struct _audio {
			const char *file;		// file name
		} audio;
		struct _transition {
			int type;				// the transition type
		} transition;
		struct _solid {
			const char *rgb;		// the RGB color spec
		} solid;
		struct _image {
			const char *file;		// file name
			int width;				// width of the image
			int height;				// height of the image
			int aspectcorrection;	// aspect correction: none, fit-in, fill-in
			const char *from_geo;	// let us scroll & zoom from this geometry to ...
			const char *to_geo;		// this geometry (in <frames> steps)
		} image;
		struct _scroll {
			const char *file;		// file name
			int width;				// width of the image
			int height;				// height of the image
		} scroll;
		struct _overlay {
			const char *file;		// file name
		} overlay;
	} it;

	struct _GIP_index_item *next;
	struct _GIP_index_item *prev;
} GIP_INDEX_ITEM_T;

typedef struct _GIP_index {
	GIP_INDEX_ITEM_T *audio;		// the audio index
	GIP_INDEX_ITEM_T *video;		// the video index
	GIP_INDEX_ITEM_T *overlay;		// the overlay index
} GIP_INDEX_T;

typedef struct _GIP_index_info {
	int num_audios;					// number of audio streams
	int num_solids;					// number of solid images
	int num_images;					// number of images
	int num_scrolls;				// number of scrolls
	int num_overlays;				// number of overlays
	int num_transitions;			// number of transitions
	int frames;						// total number of frames
	double framerate;				// the frame rate
} GIP_INDEX_INFO_T;

typedef struct _GIP_frame {
	/*
	**	processing information
	*/
	const GIP_INDEX_ITEM_T *item;
	const GIP_IMAGE_T *img;
	const GIP_IMAGE_T *imgnext;
	const GIP_IMAGE_T *overlayimg;
	int nr;
	int width;
	int height;
	FILE *stream;

	/*
	**	the resulting frame
	*/
	GIP_IMAGE_T *imgout;

	/*
	**	thread informations
	*/
	int used;
	int complete;
	pthread_t thread;
} GIP_FRAME_T;

/*
**	how many frames is this idximg to show
*/
#if 0
#define GIP_INDEX_IMAGE_PREV_TRANS(idx)		(((idx)->prev) ? (idx)->prev->transition.frames : 0)
#define GIP_INDEX_IMAGE_FRAMES(idx)			(GIP_INDEX_IMAGE_PREV_TRANS(idx) + \
												(idx)->frames + (idx)->transition.frames)
#else
#define GIP_INDEX_IMAGE_TRANS(idx)			(((idx)->type == GIP_INDEX_TYPE_TRANSITION) \
												? (idx)->frames : 0)
#define GIP_INDEX_IMAGE_PREV_TRANS(idx)		(((idx)->prev) ? GIP_INDEX_IMAGE_TRANS((idx)->prev) : 0)
#define GIP_INDEX_IMAGE_NEXT_TRANS(idx)		(((idx)->next) ? GIP_INDEX_IMAGE_TRANS((idx)->next) : 0)
#define GIP_INDEX_IMAGE_FRAMES(idx)			(GIP_INDEX_IMAGE_PREV_TRANS(idx) \
											 + (((idx)) ? idx->frames : 0) \
											 + GIP_INDEX_IMAGE_NEXT_TRANS(idx))
#define GIP_INDEX_IMAGE_PREV_FRAMES(idx)	(((idx) && (idx)->prev) ? ((idx)->prev->frames) : 0)
#define GIP_INDEX_IMAGE_NEXT_FRAMES(idx)	(((idx) && (idx)->next) ? ((idx)->next->frames) : 0)
#endif

/*
**	init & shutdown the API
*/
int GIP_index_init(void);
int GIP_index_exit(void);

/*
**	convert between the plain text type and the type code
*/
int GIP_index_str2type(const char *s);
const char *GIP_index_type2str(int type);

/*
**	create a new index
*/
GIP_INDEX_T *GIP_index_create(void);

/*
**	add a new entry
*/
GIP_INDEX_ITEM_T *GIP_index_add_audio(GIP_INDEX_T *idx,const char *file);
GIP_INDEX_ITEM_T *GIP_index_add_transition(GIP_INDEX_T *idx,int type);
GIP_INDEX_ITEM_T *GIP_index_add_solid(GIP_INDEX_T *idx,const char *rgb);
GIP_INDEX_ITEM_T *GIP_index_add_image(GIP_INDEX_T *idx,const char *file,int kenburns,int aspectcorrection);
GIP_INDEX_ITEM_T *GIP_index_add_scroll(GIP_INDEX_T *idx,const char *file);
GIP_INDEX_ITEM_T *GIP_index_add_overlay(GIP_INDEX_T *idx,const char *file);

/*
**	print an index
*/
void GIP_index_print(const GIP_INDEX_T *idx);

/*
**	compute an index
*/
int GIP_index_compute(GIP_INDEX_T *idx,int fpi,int fpt,double framerate,GIP_INDEX_INFO_T *info);

/*
**	write the index to the given file
*/
int GIP_index_write(const GIP_INDEX_T *idx,const char *file,int force,const char *title,double framerate);

/*
**	read an index from a file
*/
GIP_INDEX_T *GIP_index_read(const char *file,int kenburns,int aspectcorrection,double framerate);

/*
**	free an index
*/
void GIP_index_items_free(GIP_INDEX_T *GIP_index,int type);
void GIP_index_free(GIP_INDEX_T *GIP_index);

/*
**	process an index
*/
int GIP_index_check(const GIP_INDEX_T *idx);
int GIP_index_process(const GIP_INDEX_T *GIP_index,int width,int height,double aspectratio,FILE *stream,int max_threads);

#endif

/**/
