/*
**	$Id: progress.c 937 2006-08-10 09:37:18Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	handle the progress bar
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <config.h>
#include "gip.h"

#define ETA_UPDATE	5	// update ETA only every n-th second

static const char *seconds2hhmmss(time_t seconds,char *buffer)
{
	int HH,MM,SS;

	SS = seconds % 60;
	MM = seconds / 60;
	HH = MM / 60;
	MM -= HH * 60;

	sprintf(buffer,"%d:%02d:%02d",HH,MM,SS);
	return buffer;
}

/*
**	init the API
*/
int GIP_progress_init(void)
{
	return 1;
}

/*
**	shutdown the API
*/
int GIP_progress_exit(void)
{
	return 1;
}

GIP_PROGRESS_T *GIP_progress_start(int max,const char *units,FILE *stream)
{
	GIP_PROGRESS_T *progress;

	/*
	**	allocate a new progress struct
	*/
	if (!(progress = malloc(sizeof(GIP_PROGRESS_T))))
		CRITICAL("couldn't allocate memory for progress struct\n");
	memset(progress,0,sizeof(GIP_PROGRESS_T));
 
 	/*
	**	initialize the progress meter
	*/
	progress->max = max;
	progress->percent = -1;
	progress->start = time(NULL);
	progress->stream = (stream) ? stream : stdout;
	progress->units = strdup(units);

	/*
	**	do the initial display
	*/
	GIP_progress_update(progress,0);

	return progress;
}

void GIP_progress_update(GIP_PROGRESS_T *progress,int n)
{
	static const char rotor[] = "-/|\\";
	time_t now = time(NULL);
	char buffer[20];
	float percent;

	/*
	**	check the range of the current value
	*/
	if (n >= progress->max)
		n = progress->max;
	else if (n < 0)
		n = 0;

	/*
	**	compute the percent
	*/
	if ((percent = (float) (n + 1) * 100 / progress->max) > 100)
		percent = 100;

	/*
	**	compute the UPS (Units Per Second)
	**	and the ETA (Estamte Time of Arrival)
	**	we will update only every X seconds
	**	to get more useful values
	*/
	if (n && now - progress->last > ETA_UPDATE) {
		progress->eta = progress->start + (now - progress->start) * progress->max / n - now;
		progress->ups = (double) (n - progress->last_n) / (now - progress->last);
		progress->last = now;
		progress->last_n = n;
	}

	fprintf(progress->stream,"Processing:%5.1f%%  ETA(hh:mm:ss):%s  %s Per Second:%6.2f  %c  \r",
			percent,seconds2hhmmss(progress->eta,buffer),
			progress->units,progress->ups,
			rotor[++(progress->tick) % (sizeof(rotor) - 1)]);
}

void GIP_progress_complete(GIP_PROGRESS_T *progress)
{
	int seconds = time(NULL) - progress->start;
	char buffer[20];
	
	progress->ups = (seconds > 0) ? (double) progress->max / seconds : 0;
	fprintf(progress->stream,"Processing complete after %s  %s Per Second:%6.2f     \n",
				seconds2hhmmss(seconds,buffer),
				progress->units,progress->ups);

	/*
	**	destroy the progress meter struct again
	*/
	if (progress->units)
		free(progress->units);
	free(progress);
}/**/
