/*
**	$Id: transition.c 937 2006-08-10 09:37:18Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	do a transition between two images
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <config.h>
#include "gip.h"

/*
**	init the API
*/
int GIP_transition_init(void)
{
	/*
	**	initialize the random generator
	*/
	srand((unsigned int) time(NULL));

	return 1;
}

/*
**	shutdown the API
*/
int GIP_transition_exit(void)
{
	return 1;
}

/*
**	convert a transition name into the corresponding type code
*/
int GIP_transition_str2type(const char *s)
{
#define GIP_TRANSITION_STR2TYPE(type)	if (!strcasecmp(s,# type) || !strcasecmp(s,"GIP_TRANSITION_TYPE_" # type)) \
										return GIP_TRANSITION_TYPE_ ## type;
	GIP_TRANSITION_STR2TYPE(RANDOM);
	GIP_TRANSITION_STR2TYPE(FADE);
	GIP_TRANSITION_STR2TYPE(FADE_1);
	GIP_TRANSITION_STR2TYPE(FADE_2);
	GIP_TRANSITION_STR2TYPE(FADE_3);
	GIP_TRANSITION_STR2TYPE(SLIDE_L2R);
	GIP_TRANSITION_STR2TYPE(SLIDE_R2L);
	GIP_TRANSITION_STR2TYPE(SLIDE_T2B);
	GIP_TRANSITION_STR2TYPE(SLIDE_B2T);
	GIP_TRANSITION_STR2TYPE(PUSH_L2R);
	GIP_TRANSITION_STR2TYPE(PUSH_R2L);
	GIP_TRANSITION_STR2TYPE(PUSH_T2B);
	GIP_TRANSITION_STR2TYPE(PUSH_B2T);
	GIP_TRANSITION_STR2TYPE(SWAP_L2R);
	GIP_TRANSITION_STR2TYPE(SWAP_R2L);
	GIP_TRANSITION_STR2TYPE(SWAP_T2B);
	GIP_TRANSITION_STR2TYPE(SWAP_B2T);
	GIP_TRANSITION_STR2TYPE(ROLL_L2R);
	GIP_TRANSITION_STR2TYPE(ROLL_R2L);
	GIP_TRANSITION_STR2TYPE(ROLL_T2B);
	GIP_TRANSITION_STR2TYPE(ROLL_B2T);
#if defined(GIP_TRANSITION_TYPE_SPLIT_HIN)
	GIP_TRANSITION_STR2TYPE(SPLIT_HIN);
	GIP_TRANSITION_STR2TYPE(SPLIT_VIN);
	GIP_TRANSITION_STR2TYPE(SPLIT_HOUT);
	GIP_TRANSITION_STR2TYPE(SPLIT_VOUT);
#endif
	return GIP_TRANSITION_TYPE_RANDOM;
#undef GIP_TRANSITION_STR2TYPE
}

/*
**	convert a transition type code into the corresponding name
*/
const char *GIP_transition_type2str(int type)
{
#define GIP_TRANSITION_TYPE2STR(type)	case GIP_TRANSITION_TYPE_ ## type:	return # type;
	switch (type) {
		GIP_TRANSITION_TYPE2STR(RANDOM);
		GIP_TRANSITION_TYPE2STR(FADE);
		GIP_TRANSITION_TYPE2STR(FADE_1);
		GIP_TRANSITION_TYPE2STR(FADE_2);
		GIP_TRANSITION_TYPE2STR(FADE_3);
		GIP_TRANSITION_TYPE2STR(SLIDE_L2R);
		GIP_TRANSITION_TYPE2STR(SLIDE_R2L);
		GIP_TRANSITION_TYPE2STR(SLIDE_T2B);
		GIP_TRANSITION_TYPE2STR(SLIDE_B2T);
		GIP_TRANSITION_TYPE2STR(PUSH_L2R);
		GIP_TRANSITION_TYPE2STR(PUSH_R2L);
		GIP_TRANSITION_TYPE2STR(PUSH_T2B);
		GIP_TRANSITION_TYPE2STR(PUSH_B2T);
		GIP_TRANSITION_TYPE2STR(SWAP_L2R);
		GIP_TRANSITION_TYPE2STR(SWAP_R2L);
		GIP_TRANSITION_TYPE2STR(SWAP_T2B);
		GIP_TRANSITION_TYPE2STR(SWAP_B2T);
		GIP_TRANSITION_TYPE2STR(ROLL_L2R);
		GIP_TRANSITION_TYPE2STR(ROLL_R2L);
		GIP_TRANSITION_TYPE2STR(ROLL_T2B);
		GIP_TRANSITION_TYPE2STR(ROLL_B2T);
#if defined(GIP_TRANSITION_TYPE_SPLIT_HIN)
		GIP_TRANSITION_TYPE2STR(SPLIT_HIN);
		GIP_TRANSITION_TYPE2STR(SPLIT_VIN);
		GIP_TRANSITION_TYPE2STR(SPLIT_HOUT);
		GIP_TRANSITION_TYPE2STR(SPLIT_VOUT);
#endif
	}
	return NULL;
#undef GIP_TRANSITION_TYPE2STR
}

/*
**	return a random transition
*/
int GIP_transition_random(void)
{
	return abs(rand()) % (GIP_TRANSITION_TYPE_MAX + 1);
}

/*
**	do a transition from from -> to and uses dst as a buffer
*/
GIP_IMAGE_T *GIP_transition_process(const GIP_IMAGE_T *from,const GIP_IMAGE_T *to,int type,int step,int steps)
{
	GIP_IMAGE_T *dst;
	int w,h;

	DEBUG("from:%p  to:%p  type:%d  step:%d/%d\n",from,to,type,step,steps);

// increase value with increasing step
#define INC(v)	((v) * (step + 1) / steps)
// descrease value with increasing step
#define DEC(v)	((v) * (steps - step - 1) / steps)

	if (step < 0 || step >= steps)
		CRITICAL("step:%d  is out of range [0..%d]\n",step,steps);

	/*
	**	create a new target image and
	**	copy the from image onto
	*/
	dst = GIP_image_create(GIP_COLOR_STR_BLACK,w = from->width,h = from->height);

	switch (type) {
		/*
		**	sliding
		*/
		case GIP_TRANSITION_TYPE_SLIDE_L2R:
			GIP_image_overlay(dst,0,0,from);
			GIP_image_overlay(dst,DEC(-w),0,to);
			break;
		case GIP_TRANSITION_TYPE_SLIDE_R2L:
			GIP_image_overlay(dst,0,0,from);
			GIP_image_overlay(dst,DEC(w),0,to);
			break;
		case GIP_TRANSITION_TYPE_SLIDE_T2B:
			GIP_image_overlay(dst,0,0,from);
			GIP_image_overlay(dst,0,DEC(-h),to);
			break;
		case GIP_TRANSITION_TYPE_SLIDE_B2T:
			GIP_image_overlay(dst,0,0,from);
			GIP_image_overlay(dst,0,DEC(h),to);
			break;
		/*
		**	pushing
		*/
		case GIP_TRANSITION_TYPE_PUSH_L2R:
			GIP_image_overlay(dst,INC(w),0,from);
			GIP_image_overlay(dst,DEC(-w),0,to);
			break;
		case GIP_TRANSITION_TYPE_PUSH_R2L:
			GIP_image_overlay(dst,INC(-w),0,from);
			GIP_image_overlay(dst,DEC(w),0,to);
			break;
		case GIP_TRANSITION_TYPE_PUSH_T2B:
			GIP_image_overlay(dst,0,INC(h),from);
			GIP_image_overlay(dst,0,DEC(-h),to);
			break;
		case GIP_TRANSITION_TYPE_PUSH_B2T:
			GIP_image_overlay(dst,0,INC(-h),from);
			GIP_image_overlay(dst,0,DEC(h),to);
			break;
		/*
		**	swapping
		*/
		case GIP_TRANSITION_TYPE_SWAP_L2R:
			if (step < steps / 2) {
				GIP_image_overlay(dst,INC(w),0,to);
				GIP_image_overlay(dst,INC(-w),0,from);
			}
			else {
				GIP_image_overlay(dst,DEC(-w),0,from);
				GIP_image_overlay(dst,DEC(w),0,to);
			}
			break;
		case GIP_TRANSITION_TYPE_SWAP_R2L:
			if (step < steps / 2) {
				GIP_image_overlay(dst,INC(-w),0,to);
				GIP_image_overlay(dst,INC(w),0,from);
			}
			else {
				GIP_image_overlay(dst,DEC(w),0,from);
				GIP_image_overlay(dst,DEC(-w),0,to);
			}
			break;
		case GIP_TRANSITION_TYPE_SWAP_T2B:
			if (step < steps / 2) {
				GIP_image_overlay(dst,0,INC(h),to);
				GIP_image_overlay(dst,0,INC(-h),from);
			}
			else {
				GIP_image_overlay(dst,0,DEC(-h),from);
				GIP_image_overlay(dst,0,DEC(h),to);
			}
			break;
		case GIP_TRANSITION_TYPE_SWAP_B2T:
			if (step < steps / 2) {
				GIP_image_overlay(dst,0,INC(-h),to);
				GIP_image_overlay(dst,0,INC(h),from);
			}
			else {
				GIP_image_overlay(dst,0,DEC(h),from);
				GIP_image_overlay(dst,0,DEC(-h),to);
			}
			break;
		/*
		**	rolling
		*/
		case GIP_TRANSITION_TYPE_ROLL_L2R:
			if (INC(w))
				GIP_image_scaleblit(dst,0,0,INC(w),h,to,0,0,w,h);
			if (DEC(w))
				GIP_image_scaleblit(dst,INC(w),0,DEC(w),h,from,0,0,w,h);
			break;
		case GIP_TRANSITION_TYPE_ROLL_R2L:
			if (DEC(w))
				GIP_image_scaleblit(dst,0,0,DEC(w),h,from,0,0,w,h);
			if (INC(w))
				GIP_image_scaleblit(dst,DEC(w),0,INC(w),h,to,0,0,w,h);
			break;
		case GIP_TRANSITION_TYPE_ROLL_T2B:
			if (INC(h))
				GIP_image_scaleblit(dst,0,0,w,INC(h),to,0,0,w,h);
			if (DEC(h))
				GIP_image_scaleblit(dst,0,INC(h),w,DEC(h),from,0,0,w,h);
			break;
		case GIP_TRANSITION_TYPE_ROLL_B2T:
			if (DEC(h))
				GIP_image_scaleblit(dst,0,0,w,DEC(h),from,0,0,w,h);
			if (INC(h))
				GIP_image_scaleblit(dst,0,DEC(h),w,INC(h),to,0,0,w,h);
			break;
#if defined(GIP_TRANSITION_TYPE_SPLIT_HIN)
		/*
		**	splitting
		*/
		case GIP_TRANSITION_TYPE_SPLIT_HIN:
			GIP_image_overlay(dst,0,0,from);
			GIP_image_bitblit(dst,DEC(-w / 2),0,to,0,0,w / 2,h);
			GIP_image_bitblit(dst,w / 2 + DEC(w / 2),0,to,w / 2,0,w / 2,h);
			break;
		case GIP_TRANSITION_TYPE_SPLIT_VIN:
			GIP_image_overlay(dst,0,0,from);
			GIP_image_bitblit(dst,0,DEC(-h / 2),to,0,0,w,h / 2);
			GIP_image_bitblit(dst,0,h / 2 + DEC(h / 2),to,0,h / 2,w,h / 2);
			break;
		case GIP_TRANSITION_TYPE_SPLIT_HOUT:
			GIP_image_overlay(dst,0,0,to);
			GIP_image_bitblit(dst,INC(-w / 2),0,from,0,0,w / 2,h);
			GIP_image_bitblit(dst,w / 2 + INC(w / 2),0,from,w / 2,0,w / 2,h);
			break;
		case GIP_TRANSITION_TYPE_SPLIT_VOUT:
			GIP_image_overlay(dst,0,0,to);
			GIP_image_bitblit(dst,0,INC(-h / 2),from,0,0,w,h / 2);
			GIP_image_bitblit(dst,0,h / 2 + INC(h / 2),from,0,h / 2,w,h / 2);
			break;
#endif

		/*
		**	fading
		*/
		case GIP_TRANSITION_TYPE_FADE:
		case GIP_TRANSITION_TYPE_FADE_1:
		case GIP_TRANSITION_TYPE_FADE_2:
		case GIP_TRANSITION_TYPE_FADE_3:
		default:
			GIP_image_blend(dst,from,to,1.0 / steps * step);
			break;
	}
	return dst;
}/**/
