/*
**	$Id: image2ppm.c 1090 2007-09-15 09:57:17Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <getopt.h>
#include "image2ppm.h"
#include "../gip/gip.h"

#define max(a,b)	(((a) > (b)) ? (a) : (b))

/*
**	files for the temporary streams
*/
static char _stream_ppm[1024];

/*
**	atexit handler
*/
static void cleanup(void)
{
	if (_stream_ppm[0])
		unlink(_stream_ppm);
}

/*
**  the shutdown handler sets <_shutdown> to non-zero
*/
static void abort_handler(int sig)
{
	CRITICAL("received signal %d (%s) -- aborting\n",
			sig,strsignal(sig));
}

/*
**	print the usage
*/
static void usage(const char *argv0)
{
	int n;

	fprintf(stderr,"Usage: %s [options] [image files ...] [audio files ...]\n",argv0);
	fprintf(stderr,"Options:\n");
	fprintf(stderr," -V\n --version\n"
					"          print version number an exit\n");
	fprintf(stderr," -N\n --nop\n"
					"          don't generate output stream but print the calcualtions\n");
	fprintf(stderr," -v <level>\n --verbose <level>\n"
					"          be verbose\n");
	fprintf(stderr," -j <threads>\n --threads <threads>\n"
					"          use a maximum of threads in parallel\n");
	fprintf(stderr," -q\n --quick\n"
					"          run in quick mode (= pore quality)\n");
	fprintf(stderr," -k\n --kenburns\n"
					"          force the Ken-Burns-Effect on all images -- even if\n"
					"          not or otherwise specified in an index file\n");
	fprintf(stderr," -w <pixel>\n --width <pixel>\n"
					"          output frame with\n");
	fprintf(stderr," -h <pixel>\n --height <pixel>\n"
					"          output frame height\n");
	fprintf(stderr," -s\n --scroll\n"
					"          enable scrolling of the images\n"
					"          this is used to generate credits\n");
	fprintf(stderr," -t <seconds>\n --time-per-image <seconds>\n --time-per-photo <seconds>\n"
					"          time for each photo in the resulting stream\n");
	fprintf(stderr," -T <seconds>\n --time-per-transition <seconds>\n"
					"          time per transition in the resulting stream\n");
	fprintf(stderr," -m <mode>\n --transition <mode>\n"
					"          this is the mode used for transitions between the images\n"
					"          by default RANDOM is used. The available transition modes are:");
	for (n = 0;n <= GIP_TRANSITION_TYPE_MAX;n++) {
		if (!(n % 4))
			fprintf(stderr,"\n            ");
		fprintf(stderr,"%s%s",GIP_transition_type2str(n),
				(n < GIP_TRANSITION_TYPE_MAX) ? ", " : "\n");
	}
	fprintf(stderr," -f <num>\n --format <num>\n"
					"          this is the format for the output stream (see MJPEG tools):\n"
					"            0,3: generic output, no presets\n"
					"            1,2: VCD format: 352x288px\n"
					"            4,5: SVCD format: 480x576px for PAL, 480x480px for NTSC\n"
					"            6,7: VCD/SVCD still images: 704x576px for PAL, 704x480px for NTSC\n"
					"            8,9: DVD format: 720x576px for PAL, 720x480px for NTSC\n");
	fprintf(stderr," -n <norm>\n --video-norm <norm>\n"
					"          define PAL or NTSC video norm; default is PAL\n");
	fprintf(stderr," -F <rate>\n --frame-rate <rate>\n"
					"          the framerate to use; the default is 25 for PAL, and,\n"
					"          30000/1001 for NTSC\n");
	fprintf(stderr," -i\n --write-frame-info\n"
					"          if set, the generated output stream will contain image frame\n"
					"          information as comments for a succeeding filter\n");
	fprintf(stderr," -o <file>\n --output <file>\n"
					"          specifies the name of the output stream\n");
	fprintf(stderr," -I <file>\n --index <file>\n"
					"          write an index file which may be edited and processed later\n");
	exit(-1);
}

int main(int argc,char *argv[])
{
	int c,n,m;
	char *stream_ppm = NULL;
	int width = 0;
	int height = 0;
	double tpi = TIME_PER_PHOTO;
	double tpt = TIME_PER_TRANSITION;
	int scroll = 0;
	double framerate = FRAMERATE;
	int transition_mode = GIP_TRANSITION_TYPE_RANDOM;
	int format = 0;
	int pal = 1;
	int nop = 0;
	int quick = 0;
	int kenburns = 0;
	double aspectratio = 4.0 / 3.0;
	int aspectcorrection = GIP_ASPECTCORRECTION_TYPE_AUTO;
	int check = 0;
	int verbose = 0;
	int threads = 0;
	int write_frame_info = 0;
	FILE *output = stdout;
	GIP_INDEX_T *idx;
	GIP_INDEX_INFO_T idxinfo;
	int jpg = 0;
	int index_file_loaded = 0;
	char *index_file = NULL;
	char *resize_filter = NULL;
	char *prescale_factor = NULL;
	char *maxupscale_factor = NULL;
	char version[1000];

	char short_options[1024];
	struct option long_options[] = {
		{ "version",				0,	0,	'V' },
		{ "help",					0,	0,	'?' },
		{ "nop",					0,	0,	'N' },
		{ "threads",				1,	0,	'j' },
		{ "verbose",				1,	0,	'v' },
		{ "quick",					0,	0,	'q' },
		{ "kenburns",				0,	0,	'k' },
		{ "aspect-ratio",			1,	0,	'a' },
		{ "aspect-correction",		1,	0,	'A' },
		{ "width",					1,	0,	'w' },
		{ "height",					1,	0,	'h' },
		{ "scroll",					0,	0,	's'	},
		{ "time-per-photo",			1,	0,	't' },
		{ "time-per-image",			1,	0,	't' },
		{ "time-per-transition",	1,	0,	'T' },
		{ "transition",				1,	0,	'm'	},
		{ "format",					1,	0,	'f' },
		{ "video-norm",				1,	0,	'n' },
		{ "frame-rate",				1,	0,	'F' },
		{ "write-frame-info",		0,	0,	'i'	},
		{ "output",					1,	0,	'o'	},
		{ "index",					1,	0,	'I'	},
		{ "check",					0,	0,	'c'	},
		{ NULL,						0,	0,	0	},
	};

	/*
	**	setup the short option string
	*/
	for (n = m = 0;long_options[n].name;n++) {
		short_options[m++] = long_options[n].val;
		if (long_options[n].has_arg)
			short_options[m++] = ':';
	}
	short_options[m++] = '\0';

	while ((c = getopt_long(argc,argv,short_options,long_options,NULL)) >= 0) {
		switch (c) {
			case 'V':	/*
						**	print version and exit
						*/
						printf("*** %s Version %s ***\n",__TITLE__,__VERSION_NR__);
						printf("(c) 2005 by Christian Lorenz\n");
						printf("Linked against %s\n",GIP_version(version));
						exit(0);
						break;
			case '?':	/*
						**	print the usage
						*/
						goto usage;
						break;
			case 'j':	/*
						**	threads
						*/
						threads = atoi(optarg);
						break;
			case 'N':	/*
						**	do nothing but print some infos
						*/
						nop = 1;
						break;
			case 'q':	/*
						**	quick mode
						*/
						quick = 1;
						break;
			case 'k':	/*
						**	ken burns effect
						*/
						kenburns = 1;
						break;
			case 'v':	/*
						**	verbose
						*/
						verbose = atoi(optarg);
						break;
			case 'w':	/*
						**	the target image width
						*/
						width = atoi(optarg);
						break;
			case 'h':	/*
						**	the target image height
						*/
						height = atoi(optarg);
						break;
			case 's':	/*
						**	scroll mode -- useful for credits and panorama pictures
						*/
						scroll = 1;
						break;
			case 't':	/*
						**	time per photo
						*/
						tpi = atof(optarg);
						break;
			case 'T':	/*
						**	time per transition
						*/
						tpt = atof(optarg);
						break;
			case 'm':	/*
						**	transition type
						*/
						transition_mode = GIP_transition_str2type(optarg);
						break;
			case 'a':	/*
						**	aspect ratio
						*/
						if (strchr(optarg,':')) {
							double width,height;

							sscanf(optarg,"%lf:%lf",&width,&height);
							if (height > 0)
								aspectratio = width / height;
						}
						else
							sscanf(optarg,"%lf",&aspectratio);
						break;
			case 'A':	/*
						**	aspect correction
						*/
						aspectcorrection = GIP_aspectcorrection_str2type(optarg);
						break;
			case 'f':	/*
						**	format of output
						*/
						format = atoi(optarg);
						break;
			case 'n':	/*
						**	video norm
						*/
						switch (tolower(*optarg)) {
							case 'n':	/*
										**	NTSC
										*/
										framerate = 30000.0 / 1001.0;
										pal = 0;
										break;
							case 'p':	/*
										**	PAL
										*/
										framerate = 25.0;
										pal = 1;
										break;
							default:	goto usage;
						}
						break;
			case 'F':	/*
						**	the framerate (as float or of the form n:m)
						*/
						{
							char *div;

							framerate = atoi(optarg);
							if ((div = strchr(optarg,':')))
								framerate /= atoi(++div);
						}
						break;
			case 'i':	/*
						**	write frame info
						*/
						write_frame_info = 1;
						break;
			case 'o':	/*
						**	name of the output file
						*/
						if (strcmp(optarg,"-"))
							stream_ppm = strdup(optarg);
						break;
			case 'I':	/*
						**	name of the index file
						*/
						index_file = strdup(optarg);
						nop = 1;
						break;
			case 'c':	/*
						**	check the image by loading them all
						*/
						check = 1;
						break;
			default:
			usage:		/*
						**	usage
						*/
						fprintf(stderr,"%s: unknown option %c\n",
							__TITLE__,c);
						usage(argv[0]);
						exit(-1);
						break;
		}
	}

	/*
	**	init the GIP API
	*/
	resize_filter = getenv(IMAGE2PPM_RESIZE_FILTER);
	prescale_factor = getenv(IMAGE2PPM_PRESCALE_FACTOR);
	maxupscale_factor = getenv(IMAGE2PPM_MAXUPSCALE_FACTOR);
	GIP_init(__TITLE__,(verbose) ? "-" : NULL,
		quick,
		resize_filter,
		(prescale_factor) ? atoi(prescale_factor) : 0,
		(maxupscale_factor) ? atoi(maxupscale_factor) : 0);

	/*
	**	depending on format, do some presets
	*/
	switch (format) {
		case 0:
		case 3:		/*
					**	generic formats, no presets available
					*/
					break;
		case 1:
		case 2:		/*
					**	standard VCD
					**	VCD
					*/
					width = 352;
					height = (pal) ? 288 : 288;
					break;
		case 4:
		case 5:		/*
					**	standard SVCD,
					**	user SVCD
					*/
					width = 480;
					height = (pal) ? 576 : 480;
					break;
		case 6:
		case 7:		/*
					**	VCD Stills sequences
					**	SVCD Stills sequences
					*/
					width = 704;
					height = (pal) ? 576 : 480;
					break;
		case 8:
		case 9:		/*
					**	DVD
					*/
					width = 720;
					height = (pal) ? 576 : 480;
					break;
		default:	fprintf(stderr,"%s: unknown or unsupported format identifier: %d\n",
						__TITLE__,format);
					usage(argv[0]);
					exit(-1);
					break;
	}

	/*
	**	scan the input files
	*/
	idx = GIP_index_create();
	GIP_index_add_solid(idx,GIP_COLOR_STR_BLACK);
	for (n = optind;n < argc;n++) {
		char *ext = NULL;

		if (argv[n][0] == '@') {
			/*
			**	load an index file
			*/
			if (n > optind)
				CRITICAL("mixing of input files and index files is not supported\n");
			index_file_loaded = 1;
			GIP_index_free(idx);
			idx = GIP_index_read(&argv[n][1],kenburns,aspectcorrection,framerate);
		}
		else {
			if ((ext = strrchr(argv[n],'.'))) {
				/*
				**	input file
				*/
				if (index_file_loaded)
					CRITICAL("mixing of input files and index files is not supported\n");
				if (access(argv[n],R_OK))
					CRITICAL("couldn't access input file '%s' -- aborting!\n",argv[n]);
				else if (!strcasecmp(ext,".jpg") || !strcasecmp(ext,".jpeg") || !strcasecmp(ext,".png")) {
					/*
					**	image file
					*/
					GIP_index_add_transition(idx,
						(jpg++) ? transition_mode : GIP_TRANSITION_TYPE_FADE);
					if (scroll)
						GIP_index_add_scroll(idx,argv[n]);
					else
						GIP_index_add_image(idx,argv[n],kenburns,aspectcorrection);
				}
				else if (!strcasecmp(ext,".mp3") || !strcasecmp(ext,".ogg") || !strcasecmp(ext,".wav")) {
					/*
					**	audio stream
					*/
					GIP_index_add_audio(idx,argv[n]);
				}
				else
					ext = NULL;
			}
			if (!ext)
				CRITICAL("unknown input file '%s' -- skipping!\n",argv[n]);
		}
	}
	if (!index_file_loaded) {
		/*
		**	add a transition into black if single images were given
		*/
		GIP_index_add_transition(idx,GIP_TRANSITION_TYPE_FADE);
		GIP_index_add_solid(idx,GIP_COLOR_STR_BLACK);
	}

	/*
	**	we may have to write the index file
	*/
	if (index_file)
		GIP_index_write(idx,index_file,1,"genereated my " __TITLE__ " Version " __VERSION__,framerate);

	/*
	**	compute the index
	*/
	GIP_index_compute(idx,tpi * framerate,tpt * framerate,framerate,&idxinfo);

	/*
	**	print the computed settings
	*/
	INFORMATION("      frame dimensions: %dx%dpx\n",width,height);
	INFORMATION("          aspect ratio: %.4f\n",aspectratio);
	if (!index_file_loaded) {
		INFORMATION("        time per image: %.2fs\n",tpi);
		INFORMATION("   time per transition: %.2fs\n",tpt);
		INFORMATION("       transision mode: %s\n",GIP_transition_type2str(transition_mode));
		INFORMATION("   scroll mode enabled: %s\n",(scroll) ? "yes" : "no");
	}
	INFORMATION("      write frame info: %d\n",write_frame_info);
	INFORMATION("           output file: %s\n",(stream_ppm) ? stream_ppm : "-");
	INFORMATION("            frame rate: %.3f\n",idxinfo.framerate);
	INFORMATION("      number of audios: %d\n",idxinfo.num_audios);
	INFORMATION("      number of solids: %d\n",idxinfo.num_solids);
	INFORMATION("      number of images: %d\n",idxinfo.num_images);
	INFORMATION("     number of scrolls: %d\n",idxinfo.num_scrolls);
	INFORMATION("    number of overlays: %d\n",idxinfo.num_overlays);
	INFORMATION(" number of transitions: %d\n",idxinfo.num_transitions);
	INFORMATION("          total frames: %d\n",idxinfo.frames);
	INFORMATION("        total duration: %.02fs\n",idxinfo.frames / idxinfo.framerate);
	INFORMATION("            quick mode: %d\n",quick);
	INFORMATION("         resize filter: %s\n",(resize_filter) ? resize_filter : "default");
	INFORMATION("       prescale factor: %d\n",(prescale_factor) ? atoi(prescale_factor) : 0);
	INFORMATION("     maxupscale factor: %d\n",(maxupscale_factor) ? atoi(maxupscale_factor) : 0);
	INFORMATION("force Ken-Burns-Effect: %d\n",kenburns);
//	GIP_index_print(idx);

	if (index_file) {
		free(index_file);
		index_file = NULL;
	}
	if (nop)
		exit(0);
	if (check) {
		GIP_index_check(idx);
		exit(0);
	}

	/*
	**	setup the output stream
	*/
	if (stream_ppm) {
		if (!(output = fopen(stream_ppm,"w+"))) {
			fprintf(stderr,"couldn't create output file '%s'!",stream_ppm);
			exit(-1);
		}
	}

	/*
	**	set signal handlers
	*/
	signal(SIGHUP,SIG_IGN);
	signal(SIGINT,abort_handler);
	signal(SIGQUIT,abort_handler);
	signal(SIGABRT,abort_handler);
	signal(SIGTERM,abort_handler);
	signal(SIGUSR1,SIG_IGN);
	signal(SIGUSR2,SIG_IGN);
	signal(SIGCHLD,SIG_IGN);
	signal(SIGTRAP,SIG_IGN);
	signal(SIGALRM,SIG_IGN);
	signal(SIGURG,SIG_IGN);
	signal(SIGPIPE,SIG_IGN);
	signal(SIGVTALRM,SIG_IGN);

	/*
	**	set the cleanup handler
	*/
	atexit(cleanup);

	/*
	**	write the number of expected frames into the output stream
	*/
	if (write_frame_info)
		fprintf(output,"# frames:%d\n",idxinfo.frames);

	/*
	**	process the index
	*/
	GIP_index_process(idx,width,height,aspectratio,output,threads);

	/*
	**	close the stream
	*/
	if (stream_ppm) {
		fclose(output);
		free(stream_ppm);
	}

	/*
	**	drop the index
	*/
	GIP_index_free(idx);

	/*
	**	shutdown the API
	*/
	GIP_exit();

	return 0;
}/**/
