/*
**	$Id: ppminppm.c 1071 2007-07-19 10:05:43Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <getopt.h>
#include "../gip/gip.h"
#include "ppminppm.h"

#define min(a,b)				(((a) < (b)) ? (a) : (b))
#define max(a,b)				(((a) > (b)) ? (a) : (b))

static char _stream_ppm[1024];

/*
**	atexit handler
*/
static void cleanup(void)
{
	if (_stream_ppm[0])
		unlink(_stream_ppm);
}

int main(int argc,char *argv[])
{
	int c,n,m;
	int frames = 0, iframes = 0, fno = 0;
	char *stream_ppm = NULL;
	int width = 0, height = 0;
	int iwidth = 0, iheight = 0;
	int xoff = 0, yoff = 0;
	int xrel = -1, yrel = -1;
	int nop = 0;
	int loop = 0;
	int verbose = 0;
	int border = 0;
	int slide_delay = 0;
	int slide_in_start = 0, slide_in_stop = 0;
	int slide_out_start = 0, slide_out_stop = 0;
	int write_frame_info = 0;
	double slide_speed = 1.0;
	int slide_in = SLIDE_NONE;
	int slide_out = SLIDE_NONE;
	char *insert_file = NULL;
	FILE *insert = NULL;
	FILE *output = stdout;
	GIP_IMAGE_T *mainimg;
	GIP_IMAGE_T *insertimg;
	GIP_IMAGE_T *borderimg;
	GIP_IMAGE_T *helperimg;
	GIP_PROGRESS_T *progress;
	char version[1000];

	char short_options[1024];
	struct option long_options[] = {
		{ "version",				0,	0,	'V' },
		{ "help",					0,	0,	'?' },
		{ "nop",					0,	0,	'N' },
		{ "verbose",				1,	0,	'v' },
		{ "insert-file",			1,	0,	'i' },
		{ "insert-loop",			1,	0,	'l' },
		{ "x-offset",				1,	0,	'x' },
		{ "y-offset",				1,	0,	'y' },
		{ "slide-delay",			1,	0,	'd' },
		{ "slide-speed",			1,	0,	'p' },
		{ "slide-in",				1,	0,	's' },
		{ "slide-out",				1,	0,	'S' },
		{ "output",					1,	0,	'o'	},
		{ NULL,						0,	0,	0	},
	};

	/*
	**	setup the short option string
	*/
	for (n = m = 0;long_options[n].name;n++) {
		short_options[m++] = long_options[n].val;
		if (long_options[n].has_arg)
			short_options[m++] = ':';
	}
	short_options[m++] = '\0';

	while ((c = getopt_long(argc,argv,short_options,long_options,NULL)) >= 0) {
		switch (c) {
			case 'V':	/*
						**	print version and exit
						*/
						printf("*** %s Version %s ***\n",__TITLE__,__VERSION_NR__);
						printf("(c) 2003 by Christian Lorenz\n");
						printf("Linked against %s\n",GIP_version(version));
						exit(0);
						break;
			case '?':	/*
						**	print the usage
						*/
						goto usage;
						break;
			case 'N':	/*
						**	do nothing but print some infos
						*/
						nop = 1;
						break;
			case 'v':	/*
						**	verbose
						*/
						verbose = atoi(optarg);
						break;
			case 'i':	/*
						**	insert file
						*/
						insert_file = strdup(optarg);
						break;
			case 'l':	/*
						**	insert loop
						*/
						loop = 1;
						break;
			case 'x':	/*
						**	the x offset
						*/
						xoff = atoi(optarg);
						xrel = (strchr(optarg,'%')) ? 1 : 0;
						break;
			case 'y':	/*
						**	the y offset
						*/
						yoff = atoi(optarg);
						yrel = (strchr(optarg,'%')) ? 1 : 0;
						break;
			case 'p':	/*
						**	the slide speed
						*/
						slide_speed = atof(optarg);
						break;
			case 'd':	/*
						**	the slide speed
						*/
						slide_delay = atoi(optarg);
						break;
			case 's':	/*
						**	slide in
						*/
						if (!strcasecmp(optarg,"none"))
							slide_in = SLIDE_NONE;
						else if (!strcasecmp(optarg,"left"))
							slide_in = SLIDE_LEFT;
						else if (!strcasecmp(optarg,"right"))
							slide_in = SLIDE_RIGHT;
						else if (!strcasecmp(optarg,"up"))
							slide_in = SLIDE_UP;
						else if (!strcasecmp(optarg,"down"))
							slide_in = SLIDE_DOWN;
						else
							goto usage;
						break;
			case 'S':	/*
						**	slide out
						*/
						if (!strcasecmp(optarg,"none"))
							slide_out = SLIDE_NONE;
						else if (!strcasecmp(optarg,"left"))
							slide_out = SLIDE_LEFT;
						else if (!strcasecmp(optarg,"right"))
							slide_out = SLIDE_RIGHT;
						else if (!strcasecmp(optarg,"up"))
							slide_out = SLIDE_UP;
						else if (!strcasecmp(optarg,"down"))
							slide_out = SLIDE_DOWN;
						else
							goto usage;
						break;
			case 'o':	/*
						**	name of the output file
						*/
						stream_ppm = strdup(optarg);
						break;
			default:
			usage:		/*
						**	usage
						*/
						fprintf(stderr,"Usage: %s [options]\n",argv[0]);
						fprintf(stderr,"Options:\n");
						fprintf(stderr," -V\n --version\n"
										"          print version number an exit\n");
						fprintf(stderr," -N\n --nop\n"
										"          don't generateg MPEG stream but print the calcualtions\n");
						fprintf(stderr," -v\n --verbose\n"
										"          be verbose\n");
						fprintf(stderr," -x <num[%%]>\n --x-offset <num[%%]>\n"
										"          output frame with\n");
						fprintf(stderr," -y <num>[%%]\n --y-offset <num[%%]>\n"
										"          output frame height\n");
						fprintf(stderr," -o <file>\n --output <file>\n"
										"          xxx\n");
						exit(-1);
						break;
		}
	}

	/*
	**	init GIP API
	*/
	GIP_init(__TITLE__,(verbose) ? "-" : NULL,0,NULL,0,0);

	/*
	**	read the first frame from the stdin stream to find out
	**	the frame dimensions
	*/
	fscanf(stdin,"# frames:%d\n,",&frames);
	if (verbose)
		INFORMATION("loading 1st frame from stdin\n");
	if (!(mainimg = GIP_image_loadfd(stdin)))
		CRITICAL("GIP_image_loadfd() failed\n");
	width = mainimg->width;
	height = mainimg->height;
	if (verbose)
		INFORMATION("stdin stream: %d frames; width x height: %dx%dpx\n",frames,width,height);

	/*
	**	read the first frame from the stream we have to insert
	**	to get the frame dimensions
	*/
	if (!insert_file || !(insert = fopen(insert_file,"r")))
		CRITICAL("couldn't open file '%s' for reading\n",(insert_file) ? insert_file : "");
	fscanf(insert,"# frames:%d\n,",&iframes);
	if (verbose)
		INFORMATION("loading 1st frame from insert stream\n");
	if (!(insertimg = GIP_image_loadfd(stdin)))
		CRITICAL("GIP_image_loadfd() failed\n");
	iwidth = insertimg->width;
	iheight = insertimg->height;
	if (verbose)
		INFORMATION("insert stream: %d frames; width x height: %dx%dpx\n",iframes,iwidth,iheight);

	/*
	**	create the frame containing the border and fill it up with black
	*/
	border = (width + height) / 2.0 * 0.005 + 0.5;
	border = 1;
	if (!(borderimg = GIP_image_create("#000000",iwidth + 2 * 2 * border,iheight + 2 * 2 * border)))
		CRITICAL("GIP_image_create() failed\n");

	/*
	**	create the inner frame containing the border and fill it up with white
	*/
	if (!(helperimg = GIP_image_create("#000000",iwidth + 2 * border,iheight + 2 * border)))
		CRITICAL("GIP_image_create() failed\n");

	/*
	**	blit the inner border image into the outer one
	*/
	GIP_image_overlay(borderimg,border,border,helperimg);

	/*
	**	compute the absolute offsets
	*/
	if (xrel < 0)
		xoff = (width - iwidth) / 2;
	else if (xrel)
		xoff = width * xoff / 100;
	if (xoff < 0)
		xoff += width - iwidth;
	if (yrel < 0)
		yoff = (height - iheight) / 2;
	else if (yrel)
		yoff = height * yoff / 100;
	if (yoff < 0)
		yoff += height - iheight;

	/*
	**	compue the sliding start/stop frames
	*/
	slide_in_start = slide_delay;
	slide_in_stop = slide_in_start;
	switch (slide_in) {
		case SLIDE_LEFT:
		case SLIDE_RIGHT:
			slide_in_stop += width / slide_speed;
			break;
		case SLIDE_UP:
		case SLIDE_DOWN:
			slide_in_stop += height / slide_speed;
			break;
	}
	slide_out_start = ((loop) ? frames : iframes) - slide_delay;
	slide_out_stop = slide_out_start;
	switch (slide_out) {
		case SLIDE_LEFT:
		case SLIDE_RIGHT:
			slide_out_start -= width / slide_speed;
			break;
		case SLIDE_UP:
		case SLIDE_DOWN:
			slide_out_start -= height / slide_speed;
			break;
	}

	/*
	**	print the computed settings
	*/
	fprintf(stderr,"      frame dimensions: %dx%dpx\n",width,height);
	fprintf(stderr,"     iframe dimensions: %dx%dpx\n",iwidth,iheight);
	fprintf(stderr,"              x offset: %dpx\n",xoff);
	fprintf(stderr,"              y offset: %dpx\n",yoff);
	fprintf(stderr,"                border: %dpx\n",border);
	fprintf(stderr,"        slide in start: %d\n",slide_in_start);
	fprintf(stderr,"         slide in stop: %d\n",slide_in_stop);
	fprintf(stderr,"       slide out start: %d\n",slide_out_start);
	fprintf(stderr,"        slide out stop: %d\n",slide_out_stop);
	fprintf(stderr,"          total frames: %d\n",frames);
	fprintf(stderr,"         total iframes: %d\n",iframes);
	fprintf(stderr,"      write frame info: %d\n",write_frame_info);
	fprintf(stderr,"           output file: %s\n",stream_ppm);

	if (nop)
		exit(0);

	/*
	**	setup the output stream
	*/
	if (strcmp(stream_ppm,"-")) {
		if (!(output = fopen(stream_ppm,"w+"))) {
			fprintf(stderr,"couldn't create output file '%s'!",stream_ppm);
			exit(-1);
		}
	}

	/*
	**	write the number of expected frames into the output stream
	*/
	if (write_frame_info)
		fprintf(output,"Frames: %d\n",frames);

	/*
	**	set the cleanup handler
	*/
	atexit(cleanup);

	progress = GIP_progress_start(frames,"Frames",stderr);
	for (fno = 0;fno < frames;fno++) {
		GIP_progress_update(progress,fno);

		if (fno) {
			/*
			**	read the next frames
			**	the first frames are already loaded
			*/
			if (!(mainimg = GIP_image_loadfd(stdin)))
				CRITICAL("GIP_image_loadfd() failed\n");
			if (width != mainimg->width || height != mainimg->height)
				CRITICAL("input frame %d has dimensions %dx%dpx which differs from the expected!\n",
					fno,mainimg->width,mainimg->height);
			if (insert) {
				/*
				**	there is input expected
				*/
				if (!(insertimg = GIP_image_loadfd(insert))) {
					if (loop) {
						/*
						**	try again
						*/
						fseek(insert,0,SEEK_SET);
						if (!(insertimg = GIP_image_loadfd(insert)))
							CRITICAL("GIP_image_loadfd() failed\n");
					}
					else {
						/*
						**	there is no more input expected
						*/
						fclose(insert);
						insert = NULL;
					}
				}
				if (insert) {
					if (iwidth != insertimg->width || iheight != insertimg->height)
						CRITICAL("insert frame %d has dimensions %dx%dpx which differs from the expected!\n",
							fno,insertimg->width,insertimg->height);
				}
			}
		}

		/*
		**	insert the image
		*/
		if (insert) {
			int x = xoff,y = yoff;

			if (fno <= slide_in_stop && slide_in_start != slide_in_stop) {
				/*
				**	slide in
				*/
				double scaled = (double) (fno - slide_in_start) / (slide_in_stop - slide_in_start);

				switch (slide_in) {
					case SLIDE_LEFT:
						x = width - (width - xoff) * scaled;
						break;
					case SLIDE_RIGHT:
						x = xoff * scaled;
						break;
					case SLIDE_UP:
						y = height - (height - yoff) * scaled;
						break;
					case SLIDE_DOWN:
						y = yoff * scaled;
						break;
				}
			}
			if (fno >= slide_out_start && slide_out_start != slide_out_stop) {
				/*
				**	slide out
				*/
				double scaled = (double) (fno - slide_out_start) / (slide_out_stop - slide_out_start);

				switch (slide_out) {
					case SLIDE_LEFT:
						x = xoff - (xoff + iwidth) * scaled;
						break;
					case SLIDE_RIGHT:
						x = xoff + (width - xoff) * scaled;
						break;
					case SLIDE_UP:
						y = yoff - (yoff + iheight) * scaled;
						break;
					case SLIDE_DOWN:
						y = yoff + (height - yoff) * scaled;
						break;
				}
			}
			GIP_image_overlay(mainimg,x - 2 * border,y - 2 * border,borderimg);
			GIP_image_overlay(mainimg,x,y,insertimg);
		}

		/*
		**	write the image
		*/
		if (!GIP_image_savefd(mainimg,output))
			CRITICAL("GIP_image_savefd() failed\n");

		/*
		**	delete the main & insert image
		*/
		GIP_image_free(mainimg);
		if (insertimg)
			GIP_image_free(insertimg);
	}
	GIP_progress_complete(progress);

	/*
	**	delete the border & helper images
	*/
	GIP_image_free(borderimg);
	GIP_image_free(helperimg);

	fprintf(stderr,"Complete.\n");
	return 0;
}/**/
