/*
**	$Id: webcam2ppm.c 1071 2007-07-19 10:05:43Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <getopt.h>
#include <dirent.h>
#include "webcam2ppm.h"
#include "../gip/gip.h"

#define max(a,b)	(((a) > (b)) ? (a) : (b))

/*
**	files for the temporary streams
*/
static char _stream_ppm[1024];

/*
**	atexit handler
*/
static void cleanup(void)
{
	if (_stream_ppm[0])
		unlink(_stream_ppm);
}

/*
**  the shutdown handler sets <_shutdown> to non-zero
*/
static void abort_handler(int sig)
{
	CRITICAL("received signal %d (%s) -- aborting\n",
			sig,strsignal(sig));
}

/*
**	print the usage
*/
static void usage(const char *argv0)
{
	int n;

	fprintf(stderr,"Usage: %s [options] [image files ...] [audio files ...]\n",argv0);
	fprintf(stderr,"Options:\n");
	fprintf(stderr," -V\n --version\n"
					"          print version number an exit\n");
	fprintf(stderr," -N\n --nop\n"
					"          don't generate output stream but print the calcualtions\n");
	fprintf(stderr," -v <level>\n --verbose <level>\n"
					"          be verbose\n");
	fprintf(stderr," -j <threads>\n --threads <threads>\n"
					"          use a maximum of threads in parallel\n");
	fprintf(stderr," -q\n --quick\n"
					"          run in quick mode (= pore quality)\n");
	fprintf(stderr," -w <pixel>\n --width <pixel>\n"
					"          output frame with\n");
	fprintf(stderr," -h <pixel>\n --height <pixel>\n"
					"          output frame height\n");
	fprintf(stderr," -s\n --scroll\n"
					"          enable scrolling of the images\n"
					"          this is used to generate credits\n");
	fprintf(stderr," -t <seconds>\n --time-per-image <seconds>\n --time-per-photo <seconds>\n"
					"          time for each photo in the resulting stream\n");
	fprintf(stderr," -T <seconds>\n --time-per-transition <seconds>\n"
					"          time per transition in the resulting stream\n");
	fprintf(stderr," -m <mode>\n --transition <mode>\n"
					"          this is the mode used for transitions between the images\n"
					"          by default RANDOM is used. The available transition modes are:");
	for (n = 0;n <= GIP_TRANSITION_TYPE_MAX;n++) {
		if (!(n % 4))
			fprintf(stderr,"\n            ");
		fprintf(stderr,"%s%s",GIP_transition_type2str(n),
				(n < GIP_TRANSITION_TYPE_MAX) ? ", " : "\n");
	}
	fprintf(stderr," -f <num>\n --format <num>\n"
					"          this is the format for the output stream (see MJPEG tools):\n"
					"            0,3: generic output, no presets\n"
					"            1,2: VCD format: 352x288px\n"
					"            4,5: SVCD format: 480x576px for PAL, 480x480px for NTSC\n"
					"            6,7: VCD/SVCD still images: 704x576px for PAL, 704x480px for NTSC\n"
					"            8,9: DVD format: 720x576px for PAL, 720x480px for NTSC\n");
	fprintf(stderr," -n <norm>\n --video-norm <norm>\n"
					"          define PAL or NTSC video norm; default is PAL\n");
	fprintf(stderr," -F <rate>\n --frame-rate <rate>\n"
					"          the framerate to use; the default is 25 for PAL, and,\n"
					"          30000/1001 for NTSC\n");
	fprintf(stderr," -i\n --write-frame-info\n"
					"          if set, the generated output stream will contain image frame\n"
					"          information as comments for a succeeding filter\n");
	fprintf(stderr," -o <file>\n --output <file>\n"
					"          specifies the name of the output stream\n");
	fprintf(stderr," -I <file>\n --index <file>\n"
					"          write an index file which may be edited and processed later\n");
	exit(-1);
}

/*
**	filterfunction for scandir()
*/
static int filterimages(const struct dirent *dirent)
{
	char *dot = strrchr(dirent->d_name,'.');

	return (dot && (!strcasecmp(dot + 1,"jpg") || !strcasecmp(dot + 1,"png")));
}

int main(int argc,char *argv[])
{
	int c,n,m;
	char *stream_ppm = "-";
	int width = 0;
	int height = 0;
	double framerate = FRAMERATE;
	int format = 0;
	int pal = 1;
	int nop = 0;
	int quick = 0;
	int nogeo = 0;
	int border_width = 0;
	int border_height = 0;
	int label_width = 0;
	int label_height = 0;
	int left_aligned = 0;
	int top_aligned = 0;
	int ipf = 1;
	int verbose = 0;
	int write_frame_info = 0;
	FILE *output = stdout;
	char version[1000];

	char short_options[1024];
	struct option long_options[] = {
		{ "version",				0,	0,	'V' },
		{ "help",					0,	0,	'?' },
		{ "nop",					0,	0,	'N' },
		{ "verbose",				1,	0,	'v' },
		{ "quick",					0,	0,	'q' },
		{ "kenburns",				0,	0,	'k' },
		{ "width",					1,	0,	'w' },
		{ "height",					1,	0,	'h' },
		{ "border",					1,	0,	'b' },
		{ "label",					1,	0,	'l' },
		{ "images-per-frame",		1,	0,	't' },
		{ "format",					1,	0,	'f' },
		{ "video-norm",				1,	0,	'n' },
		{ "frame-rate",				1,	0,	'F' },
		{ "write-frame-info",		0,	0,	'i'	},
		{ "output",					1,	0,	'o'	},
		{ "index",					1,	0,	'I'	},
		{ "check",					0,	0,	'c'	},
		{ NULL,						0,	0,	0	},
	};

	/*
	**	setup the short option string
	*/
	for (n = m = 0;long_options[n].name;n++) {
		short_options[m++] = long_options[n].val;
		if (long_options[n].has_arg)
			short_options[m++] = ':';
	}
	short_options[m++] = '\0';

	while ((c = getopt_long(argc,argv,short_options,long_options,NULL)) >= 0) {
		switch (c) {
			case 'V':	/*
						**	print version and exit
						*/
						printf("*** %s Version %s ***\n",__TITLE__,__VERSION_NR__);
						printf("(c) 2005 by Christian Lorenz\n");
						printf("Linked against %s\n",GIP_version(version));
						exit(0);
						break;
			case '?':	/*
						**	print the usage
						*/
						goto usage;
						break;
			case 'N':	/*
						**	do nothing but print some infos
						*/
						nop = 1;
						break;
			case 'q':	/*
						**	quick mode
						*/
						quick = 1;
						nogeo = 1;
						break;
			case 'v':	/*
						**	verbose
						*/
						verbose = atoi(optarg);
						break;
			case 'w':	/*
						**	the target image width
						*/
						width = atoi(optarg);
						break;
			case 'h':	/*
						**	the target image height
						*/
						height = atoi(optarg);
						break;
			case 'b':	/*
						**	border widthXheight
						*/
						{
							char *s;

							border_width = atoi(optarg);
							if ((s = strchr(optarg,'x')))
								border_height = atoi(s + 1);
						}
						break;
			case 'l':	/*
						**	label widthXheight
						*/
						{
							char *s;

							label_width = atoi(optarg);
							if ((s = strchr(optarg,'x')))
								label_height = atoi(s + 1);
						}
						break;
			case 't':	/*
						**	frames per image
						*/
						ipf = atof(optarg);
						break;
			case 'T':	/*
						**	timestamp box
						*/
						break;
			case 'f':	/*
						**	format of output
						*/
						format = atoi(optarg);
						break;
			case 'n':	/*
						**	video norm
						*/
						switch (tolower(*optarg)) {
							case 'n':	/*
										**	NTSC
										*/
										framerate = 30000.0 / 1001.0;
										pal = 0;
										break;
							case 'p':	/*
										**	PAL
										*/
										framerate = 25.0;
										pal = 1;
										break;
							default:	goto usage;
						}
						break;
			case 'F':	/*
						**	the framerate (as float or of the form n:m)
						*/
						{
							char *div;

							framerate = atoi(optarg);
							if ((div = strchr(optarg,':')))
								framerate /= atoi(++div);
						}
						break;
			case 'i':	/*
						**	write frame info
						*/
						write_frame_info = 1;
						break;
			case 'o':	/*
						**	name of the output file
						*/
						stream_ppm = strdup(optarg);
						break;
			default:
			usage:		/*
						**	usage
						*/
						fprintf(stderr,"%s: unknown option %c\n",
							__TITLE__,c);
						usage(argv[0]);
						exit(-1);
						break;
		}
	}

	/*
	**	init the GIP API
	*/
	GIP_init(__TITLE__,(verbose) ? "-" : NULL,quick,NULL,0,0);

	/*
	**	check some arguments
	*/
	if (ipf < 1) {
		fprintf(stderr,"images per frame has to be at least 1\n");
		exit(-1);
	}

	/*
	**	depending on format, do some presets
	*/
	switch (format) {
		case 0:
		case 3:		/*
					**	generic formats, no presets available
					*/
					break;
		case 1:
		case 2:		/*
					**	standard VCD
					**	VCD
					*/
					width = 352;
					height = (pal) ? 288 : 288;
					break;
		case 4:
		case 5:		/*
					**	standard SVCD,
					**	user SVCD
					*/
					width = 480;
					height = (pal) ? 576 : 480;
					break;
		case 6:
		case 7:		/*
					**	VCD Stills sequences
					**	SVCD Stills sequences
					*/
					width = 704;
					height = (pal) ? 576 : 480;
					break;
		case 8:
		case 9:		/*
					**	DVD
					*/
					width = 720;
					height = (pal) ? 576 : 480;
					break;
		default:	fprintf(stderr,"%s: unknown or unsupported format identifier: %d\n",
						__TITLE__,format);
					usage(argv[0]);
					exit(-1);
					break;
	}

	/*
	**	print the computed settings
	*/
	INFORMATION("      frame dimensions: %dx%dpx\n",width,height);
	INFORMATION("     border dimensions: %dx%dpx\n",border_width,border_height);
	INFORMATION("      label dimensions: %dx%dpx\n",label_width,label_height);
	INFORMATION("      write frame info: %d\n",write_frame_info);
	INFORMATION("           output file: %s\n",stream_ppm);
	INFORMATION("            quick mode: %d\n",quick);

	if (nop)
		exit(0);

	/*
	**	setup the output stream
	*/
	if (strcmp(stream_ppm,"-")) {
		if (!(output = fopen(stream_ppm,"w+"))) {
			fprintf(stderr,"couldn't create output file '%s'!",stream_ppm);
			exit(-1);
		}
	}

	/*
	**	set signal handlers
	*/
	signal(SIGHUP,SIG_IGN);
	signal(SIGINT,abort_handler);
	signal(SIGQUIT,abort_handler);
	signal(SIGABRT,abort_handler);
	signal(SIGTERM,abort_handler);
	signal(SIGUSR1,SIG_IGN);
	signal(SIGUSR2,SIG_IGN);
	signal(SIGCHLD,SIG_IGN);
	signal(SIGTRAP,SIG_IGN);
	signal(SIGALRM,SIG_IGN);
	signal(SIGSTOP,SIG_IGN);
	signal(SIGURG,SIG_IGN);
	signal(SIGPIPE,SIG_IGN);
	signal(SIGVTALRM,SIG_IGN);

	/*
	**	set the cleanup handler
	*/
	atexit(cleanup);

	/*
	**	read each image
	*/
	for (n = optind;n < argc;n++) {
		int num_images,i,f;
		struct dirent **images = NULL;
		char file[1024];
		char timestamp[100];
		char *dash;
		struct tm ts;
		GIP_IMAGE_T *img,*bimg;

		memset(&ts,0,sizeof(ts));

		/*
		**	scan the directory
		*/
		if ((num_images = scandir(argv[n],&images,filterimages,alphasort)) < 0)
			perror("scandir");

		for (i = 0;i < num_images;i++) {
			/*
			**	setup the filename
			*/
			sprintf(file,"%s/%s",argv[n],images[i]->d_name);

			/*
			**	scan the timestamp
			*/
			if ((dash = strrchr(images[i]->d_name,'-')))
				dash++;
			else
				dash = images[i]->d_name;
			sscanf(dash,"%4d%2d%2d%2d%2d%2d.",
				&ts.tm_year,&ts.tm_mon,&ts.tm_mday,
				&ts.tm_hour,&ts.tm_min,&ts.tm_sec);
			ts.tm_year -= 1900;
			ts.tm_mon -= 1;
			mktime(&ts);
			strftime(timestamp,sizeof(timestamp) - 1,"%H:%M %a\n%d.%m.%Y",&ts);
INFORMATION("[%d] %s => %s\n",i,file,timestamp);

			/*
			**	load the image
			*/
			img  = GIP_image_load(file);

			/*
			**	scale the image
			*/
			if (border_width && border_height) {
INFORMATION("[%d] scaling image from %dx%dpx to %dx%dpx\n",i,img->width,img->height,width - 2 * border_width,height - 2 * border_height);
				GIP_image_scale(img,width - 2 * border_width,height - 2 * border_height);
			}

			/*
			**	add the border
			*/
			bimg = GIP_image_border(img,GIP_COLOR_STR_RED,border_width,border_height);
			GIP_image_free(img);
			img = bimg;

			if (label_width || label_height) {
				/*
				**	create the timestamp box
				*/
				bimg = GIP_image_create(GIP_COLOR_STR_GREEN,label_width,label_height);

				/*
				**	put the timestamp box onto the scaled image
				*/
				GIP_image_bitblit(img,
					(left_aligned) ? border_width : width - border_width - label_width,
					(top_aligned) ? border_height : height - border_height - label_height,
					bimg,0,0,bimg->width,bimg->height);
			}

			/*
			**	write the image out
			*/
			for (f = 0;f < ipf;f++)
				GIP_image_savefd(img,output);

			/*
			**	drop the image
			*/
			GIP_image_free(img);
		}

		/*
		**	free the file list
		*/
		free(images);
	}

	/*
	**	close the stream
	*/
	if (strcmp(stream_ppm,"-")) {
		fclose(output);
	}

	/*
	**	shutdown the API
	*/
	GIP_exit();

	return 0;
}/**/
