#
# Copyright (c) 2005-2006 by Lee Braiden.  Released under the GNU General
# Public License, version 2 or later.  Please see the included LICENSE.txt
# file for details.  Legally, that file should have been included with this
# one.  If not, please contact Lee Braiden (email lee.b@digitalunleashed.com)
# for a copy.
#
import sys
import os
import urllib
import datetime

from khtml import KHTMLPart
from kdecore import KURL
from qt import SIGNAL, QObject

appRoot = os.path.dirname(os.path.abspath(os.path.realpath(sys.argv[0])))

sys.path.append(os.path.join(appRoot, 'code'))
import config
import utils
import Feed
import Media
import KTVBookmarksListView

class KTVHTMLPart(KHTMLPart):
	"""Subclasses KHTMLPart, to handle all the special features that
	our feed browser needs.  For instance, clicking on URIs with
	KTVHTMLPart will add the URI to a channel list, rather than
	treating it as a normal URI to go to.  Also, many custom pages
	are generated by this class, to show lists of downloading items,
	welcome pages, etc."""
	
	def __init__(self, parentWidget, window):
		"""Initialises a new KTVHTMLPart object.  Requires the same
		arguments as KHTMLPart, plus the parent window it will attach
		to."""
		KHTMLPart.__init__(self, parentWidget)
		self._window = window
		
		QObject.connect(self.browserExtension(), SIGNAL('openURLRequestDelayed(const KURL&, const KParts::URLArgs&)'), self._openURLRequest)
		
		self.__lastURL = None
		self.__currentURL = None
		self.setFormNotification(self.NoNotification)

	def _considerForHistory(self, url):
		"""set the last URL (basic history support) unless the history
		hasn't changed (ie, page is just being refreshed)"""
		if self.__lastURL == url:
			# just a reload of the page; ignore it
			return
	
		# make sure it's not an action url
		# all action urls urls which just
		# perform an action, but don't
		# show their own content) are
		# non-historical, in that they don't
		# get stored in the browse history.
		actionURLs = [
			u'katchtv:delete:',
			u'katchtv:add_download:',
			u'katchtv:downloads:',
		]
		historical = True
		for aURL in actionURLs:
			if url.startswith(aURL):
				historical = False
		if historical:
			self.__lastURL = self.__currentURL
			self.__currentURL = url

		if self.__lastURL:
			self._window.enableBackButton()
		else:
			self._window.disableBackButton()

	def _reloadCurrentPage(self):
		assert self.__currentURL is not None
		self.goToURL(self.__currentURL)
	
	def _previousPage(self):
		return self.__lastURL
	
	def loadPreviousPage(self):
		if self.__lastURL:
			self.goToURL(self.__lastURL)
	
	def _writePage(self, title, bodyText, otherStyle=None, refreshURL=None, refreshSecs=1, bodyTags=u"", script=u""):
		"""Writes a generic page into the KHTMLPart, supplying
		stylesheet information etc."""
		
		page = u'<html><head>'
		
		# make doubly sure of the encoding
		page += u'<meta http-equiv="Content-Type" content="text/html; charset=utf-8">'
		
		if refreshURL is not None:
			self.setMetaRefreshEnabled(True)
			page += u'<meta http-equiv="refresh" content="%d; URL=%s">' % (refreshSecs, refreshURL)
		
		if title is not None:
			page += u'<title>KatchTV - ' + title + u'</title>'
		else:
			page += u'<title>KatchTV</title>'
		
		if otherStyle is None:
			page += u"""<style type="text/css">
				BODY {
					background: #B6D0F5;
				}
				H1 { color: black; }
				H2 { font-size: -1; }
				DIV.Lowlight {
					border: 2px solid #6985e4;
					background: #eaf5ff;
					display: block;
					clear: right;
					margin-left: 1em;
					margin-right: 1.5em;
					margin-bottom: 0.75em;
				}
				DIV.Highlight {
					border: 2px solid #6985e4;
					background: #F6F6B3;
					display: block;
					clear: right;
					margin-left: 1em;
					margin-right: 1.5em;
					margin-bottom: 0.75em;
				}
				DIV.Notice {
					border: 2px solid #6985e4;
					background: #ffffff;
					display: block;
					clear: right;
					margin-left: 1em;
					margin-right: 1.5em;
					margin-bottom: 0.75em;
				}
				H2.BoxOut {
					width: 100%;
					display: block;
					background: #6985e4;
					color: white;
					margin: 0px;
				}
				UL.BoxOutFooter {
					width: 100%;
					display: block;
					background: #6985e4;
					border: #b5dafd;
					border-top: 1px;
					padding-top: 0px;
					padding-bottom: 2px;
					padding-left: 0px;
					padding-right: 0px;
					margin: 0px;
					whitespace: nowrap;
				}
				UL.BoxOutFooter LI {
					display: inline;
					list-style: none;
					background: #b5dafd;
					position: relative;
					margin-right: 2px;
					padding-left: 1em;
					padding-right: 1em;
					padding-top: 1px;
					padding-bottom: 1px;
				}
				IMG.Title {
					float: right;
					position: relative;
					height: 8em;
					min-height: 8em;
					max-height: 8em;
					max-width: 30%;
					top: -1.3em;
					right: -3px;
					border-bottom: 0px;
					margin-bottom: 0px;
				}
				TD.Description {
					margin-top: 0.5cm;
					margin-bottom: 1cm;
					border-style: solid;
					border-top-width: 0px;
					border-left-width: 0px;
					border-right-width: 0px;
					border-bottom-width: 1px;
					border-color: gray;
				}
			</style>"""
		else:
			page += u'<style type="text/css">' + otherStyle + u'</style>'
		
		if script != u"":
			page += u'<script language="javascript">' + script + u'</script>'
		
		page += u'</head><body'
		if bodyTags != u'':
			page += u' ' + bodyTags
		page += u'>'
		
		if title is not None:
			page += u'<h1>' + utils.enableWrapping(title) + u'</h1>'
			
		
		page += bodyText + u"</body><html>"
		
		self.begin()
		try:
			self.write(unicode(page).encode('utf-8'))
		except AttributeError:
			print "page is type:", page.__class__
			print "value: ", page
		self.end()
	
	def _writeUnknown(self):
		"""Simply generates an "unknown page" page, which directs the
		user back to the welcome page.  This should never be called,
		but could be called if, for example, a katchtv: url is
		incorrect, or not supported by an old version of KatchTV which
		the user still hasn't upgraded."""
		self._writePage(u'Unknown Page Requested', u"""<p>An
			unrecognised page was requested.  You may need to
			upgrade KatchTV to support this feature.  For now,
			you can return to the <a href="%s">
			page you were at previously</a>.</p>""" % self._previousURL())
	
	def _writeWelcome(self):
		"""Writes a welcome page, which intelligently
		directs the user to new content, or informs of
		upgrades available, etc."""
		
		self._statusReport(u'Opening Welcome page...')
		
		logoPath = os.path.join(appRoot, "Docs/Manual/images/Logo.png")
		body = u"""<div class="Notice">
		<img align="right" src="%s" width="320" height="200">
		
		<h2>Welcome to KatchTV!</h2>
		
		<p>""" % logoPath
		
		if self._window.getMediaManager().numDownloaded() > 0:
			body += u"""There is media <a href="katchtv:ready:">
				Ready to Watch</a>!  Or, you"""
		else:
			body += u"You"
		
		body += u""" might want to explore the Channel Directories to
		the side. Just double-click on your chosen directory to load
		it.  From there, you can subscribe to your favourite shows,
		just by clicking on their feed links.</p>
		
		<p>Don't miss the Help page if you're unsure of anything --
		you can read it by clicking the help button in the bottom-left
		corner!</p>
		</div>"""
		
		self._writePage(u'KatchTV', body)
	
	def _writeReady(self):
		"""Generates a "Ready to Watch" page, which lists
		all complete downloads."""
		
		self._statusReport(u'Opening Ready To Watch page...')
		
		# mark the ready list's items as viewed
		self._window.bmarksList.readyToWatchBMark.setText(KTVBookmarksListView.COL_NUMNEW, u'')
		
		count = 0
		body = u""
		for mediaItem in self._window.getMediaManager().eachDownloaded():
			mediaItem.acquireLock()
			miTitle = mediaItem.getStatisticNoLock(Media.Item.TITLE)
			
			miEncodedForURIFind = mediaItem.encodeForURIFindNoLock()
			
			mediaItem.releaseLock()
			
			statNames = [
				Media.Item.DESCRIPTION,
				Media.Item.TOTAL_BYTES,
				Media.Item.SOURCE_FEED_URI,

#				# this is informative, but potentially messy
#				Media.Item.LOCAL_PATH,
			]
			
			# title and description
			body += u"""
				<div class="Highlight">
					<h2 class="BoxOut">%s</h2>""" % miTitle
			
			# general statistics
			body += """<table width="100%%">"""
			
			availStats = mediaItem.getChosenStatistics(statNames)
			
			for statK in statNames:
				if statK not in availStats.keys():
					continue
				
				if statK == Media.Item.DESCRIPTION:
					body += """<tr><td colspan="2" class="Description">%s</td></tr>""" % availStats[statK]
				elif statK == Media.Item.TOTAL_BYTES:
					body += """<tr><th width="160" align="left">Total Size</th><td>%s</td></tr>""" % utils.prettyPrintBytes(int(availStats[statK]))
				elif statK == Media.Item.SOURCE_FEED_URI:
					# not shown here; we show it as a
					# button/command later instead.
					pass
				else:
					# default formatter
					body += """<tr><th width="160" align="left">%s</th><td>%s</td></tr>""" % (statK, utils.enableWrapping(str(availStats[statK])))
			
			body += """</table>"""
			
			miSourceFeedURI = availStats[Media.Item.SOURCE_FEED_URI]
			
			# commands available
			commandDict = {
				u'Play': u'katchtv:media:%s' % miEncodedForURIFind,
				u'Delete': u'katchtv:delete:%s' % miEncodedForURIFind,
				u'View Feed': u'katchtv:feed:%s' % miSourceFeedURI,
			}
			
			body += """<ul class="BoxOutFooter">"""
			
			for cmdKey in commandDict.keys():
				body += """
							<li><a href="%s">%s</a></li>""" % (commandDict[cmdKey], cmdKey)
			
			body += """
						</ul>
					</div>"""
			
			count += 1
		if count == 0:
			body += u"""<p>Nothing ready to watch so far.  Please
			subscribe to a channel and download an episode.</p>"""
		
		self._writePage(u'Ready to Watch', body, refreshURL=u'katchtv:ready:', refreshSecs=1)
	
	def _writeCurrentlyDownloading(self):
		"""Generates a "Current Downloads" page, which shows downloads in progress."""
		
		self._statusReport(u'Opening Current Downloads page...')
		
		count = 0
		body = u""
		for mediaItem in self._window.getMediaManager().eachInProgress():
			mediaItem.acquireLock()
			miTitle = mediaItem.getStatisticNoLock(Media.Item.TITLE)
			
			miEncodedForURIFind = mediaItem.encodeForURIFindNoLock()
			
			mediaItem.releaseLock()
			
			statNames = [
				Media.Item.DESCRIPTION,
				Media.Item.DOWNLOAD_STATUS,
				Media.Item.DOWNLOAD_METHOD,
				Media.Item.ERRORMSG,
				Media.Item.PERCENT_COMPLETE,
				Media.Item.UPLOAD_BPS_AVERAGE,
				Media.Item.DOWNLOAD_BPS_AVERAGE,
				Media.Item.TOTAL_BYTES,
				Media.Item.TOTAL_BYTES_DOWNLOADED,
				Media.Item.TOTAL_BYTES_UPLOADED,
				Media.Item.TIME_ELAPSED,
				Media.Item.TIME_REMAINING,
				Media.Item.SOURCE_FEED_URI,
#				Media.Item.URI,
#				Media.Item.LOCAL_PATH,
			]
			
			# title and description
			body += u"""
				<div class="Highlight">
					<h2 class="BoxOut">%s</h2>""" % miTitle
			
			# general statistics
			body += """<table width="100%%">"""
			
			availStats = mediaItem.getChosenStatistics(statNames)
			
			for statK in statNames:
				if statK not in availStats.keys():
					continue
				
				if statK == Media.Item.DESCRIPTION:
					body += """<tr><td colspan="2" class="Description">%s</td></tr>""" % availStats[statK]
				elif statK == Media.Item.PERCENT_COMPLETE:
					try:
						percentComplete = int(availStats[statK])
					except ValueError:
						percentComplete = 0
					body += """<tr>
										<th width="160" align="left">Progress</th>
										<td align="left" bgcolor="black">
											<img src="%s" width="%d%%" height="22" />
										</td>
									</tr>""" % (config.progressBarImageURI, percentComplete)
					body += """<tr><th width="160" align="left">Complete</th><td>%3.1f %%</td></tr>""" % percentComplete
				elif statK == Media.Item.TOTAL_BYTES:
					body += """<tr><th width="160" align="left">Total Size</th><td>%s</td></tr>""" % utils.prettyPrintBytes(availStats[statK])
				elif statK == Media.Item.TOTAL_BYTES_DOWNLOADED:
					body += """<tr><th width="160" align="left">Downloaded</th><td>%s</td></tr>""" % unicode(utils.prettyPrintBytes(availStats[statK]))
				elif statK == Media.Item.TOTAL_BYTES_UPLOADED:
					body += """<tr><th width="160" align="left">Uploaded</th><td>%s</td></tr>""" % unicode(utils.prettyPrintBytes(availStats[statK]))
				elif statK == Media.Item.DOWNLOAD_BPS_AVERAGE or statK == Media.Item.UPLOAD_BPS_AVERAGE:
					body += """<tr><th width="160" align="left">""" + statK + """</th><td>%s</td></tr>""" % unicode(utils.prettyPrintBitsPerSec(availStats[statK]))
				elif statK == Media.Item.TIME_ELAPSED:
					body += """<tr><th width="160" align="left">Time Elapsed</th><td>%s</td></tr>""" % utils.prettyPrintSeconds(availStats[statK])
				elif statK == Media.Item.TIME_REMAINING:
					body += """<tr><th width="160" align="left">Time Remaining</th><td>%s</td></tr>""" % utils.prettyPrintSeconds(availStats[statK])
				elif statK == Media.Item.SOURCE_FEED_URI:
					# not shown here; we show it as a
					# button/command later instead.
					pass
				else:
					# default formatter
					body += """<tr><th width="160" align="left">%s</th><td>%s</td></tr>""" % (statK, utils.enableWrapping(str(availStats[statK])))
			
			body += """</table>"""
			
			miSourceFeedURI = availStats[Media.Item.SOURCE_FEED_URI]
			
			# commands available
			commandDict = {
				u'Delete': u'katchtv:delete:%s' % miEncodedForURIFind,
				u'View Feed': u'katchtv:feed:%s' % miSourceFeedURI,
			}
			
			body += """<ul class="BoxOutFooter">"""
			
			for cmdKey in commandDict.keys():
				body += """
							<li><a href="%s">%s</a></li>""" % (commandDict[cmdKey], cmdKey)
			
			body += """
						</ul>
					</div>"""
			
			count += 1
		if count == 0:
			body += u"""<p>Nothing downloading so far.  Please
			subscribe to a channel from the channel guides, and
			download an episode from the subscribed feed.  Just
			press help, if you're unsure of anything.</p>"""
		
		self._writePage(u'Currently Downloading', body, refreshURL=u'katchtv:downloads:', refreshSecs=1)
	
	def _writeFeed(self, feedURI):
		"""Displays a feed (subscribed channel), highlighting individual
		episodes, converting media links into special katchtv: links,
		etc."""
		
		self._statusReport(u'Opening Feed page for %s...' % feedURI)
		
		Feed.FeedCache.acquireLock()
		try:
			feed = Feed.FeedCache.getFeedFromURINoLock(feedURI)
		finally:
			Feed.FeedCache.releaseLock()
		
		if feed is None:
			self._writePage(u'Feed at URI "%s" failed to parse' % feedURI, u"""<p>Sorry, this feed failed to parse.  The server may have been down, or it may be very slow.  If problems persist with this feed, you may want to remove it your subscribed channels list.""")
			return
		
		feed.acquireLock()
		try:
			feedTitle = feed.title()
			
			# begin an empty html body
			body = u""
			
			logoURI = feed.logoURI()
			body += """<img class="Title" alt="%s" src="%s" />""" % (feedTitle, logoURI)
			
			body += u'<p>' + feed.summary() + u'</p>'
			
			episodeCount = 0
			
			# reverse the list of episodes
			revList = []
			for epi in feed.episodes():
				newRevList = [ epi ]
				newRevList.extend(revList)
				revList = newRevList

			for episode in revList:
				# make sure we don't parse huge/slow feeds for
				# too long
				if episodeCount > config.maxFeedEntries:
					break
				
				enclosures = episode.enclosures()
				
				# default to highlighting entries based
				# on new-ness, and change that if the
				# episode has already been downloaded
				downloading = False
				if len(enclosures) > 0:
					for enc in enclosures:
						if enc.hasMediaItem():
							downloading = True
	
				if downloading:
					body += u"""<div class="Notice">"""
				else:
					if episode.isSeen():
						body += u"""<div class="LowLight">"""
					else:
						body += u"""<div class="Highlight">"""
				
				body += u"""<h2 class="BoxOut">""" + utils.enableWrapping(episode.title()) + u"""</h2>"""
				
				body += u"""<p>""" + episode.body() + """</p>"""
				
				if len(enclosures) > 0:
					body += u"""<h3>Enclosures</h3>
					<ul>"""
					enclosureCount = 1
					for enc in enclosures:
						state = 'Not marked for download'
						if enc.hasMediaItem():
							# mark this enclosure
							# as downloaded if it's
							# already stored locally
							state = u'Downloading'
							mediaItem = self._window.getMediaManager().itemForURI(enc._downloadURI())
							if mediaItem.isDownloaded():
								state = u'Ready to Watch'
						
						enclosureTitle = utils.enableWrapping(enc.title())
						# if there's more than one
						# enclosure, then add an
						# incrementing number to the
						# title for each one
						if len(episode.enclosures()) > 1:
							enclosureTitle += u' (enclosure ' + unicode(enclosureCount) + u')'
						
						
						# finally, write a listitem
						# for this enclosure
						if state == u'Downloading':
							body += u"""
							<li>
								<a href="katchtv:downloads:">%s (downloading; click to see progress)</a>
							</li>""" % (enclosureTitle)
						elif state == u'Ready to Watch':
							vals = enc.getMediaItemKWArgs()
							valsURI = utils.argsToURI(vals)
							body += u"""
							<li>
								<a href="katchtv:media:%s">%s (downloaded; click to open)</a>
							</li>""" % (valsURI, enclosureTitle)
						else:
							# state == 'Not Marked for Download'
							
							vals = enc.getMediaItemKWArgs()
							valsURI = utils.argsToURI(vals)
							
							realURI = vals['URI']
							if utils.isExternalURL(realURI):
								body += u"""<a
href="%s">%s (%s)""" % (realURI, enclosureTitle, enc.prettyType())
							else:
								body += u"""
							<li>
								<a href="katchtv:add_download:%s">%s (%s)</a>
							</li>""" % (valsURI, enclosureTitle, enc.prettyType())
						
						if enc.sizeInBytes() > 0:
							body += u' [%s]' % utils.prettyPrintBytes(enc.sizeInBytes())
						
						enclosureCount += 1
					
					# finish the list
					body += u"""</ul>"""
				
				if episode.hasLink():
					link = episode.link()
					if utils.isValidPageLink(link):
						body += u"""<ul class="BoxOutFooter">
							<li>
								<a href=""" + link + """>
									Visit webpage for this episode
								</a>
							</li>
						</ul>"""
				
				body += u"""</div>"""
				
				episodeCount += 1
		finally:
			feed.releaseLock()
		
		self._writePage(feedTitle, body)
	
	def _statusReport(self, msg):
		"""Updates the main window's status bar with a new
		message."""
		self._window.statusBar().message(msg)

	def _writeMedia(self, mediaItem):
		"""Shows or plays a given download, if it has completed,
		or shows explanatory messages, if the download is still
		in progress, or hasn't started yet."""

		assert mediaItem.isDownloaded()
		
		mediaItem.acquireLock()
		
		miTitle = mediaItem.getStatistic(Media.Item.TITLE)
		
		try:
			# fixme: only support playing one bittorrent media file
			# for now; the most likely (largest) one
			miMediaFiles = mediaItem.getStatistic(Media.Item.MEDIA_FILES)
			if isinstance(miMediaFiles, list):
				miLocalPath = utils.mostLikelyMedia(miMediaFiles)
			else:
				miLocalPath = miMediaFiles
		except KeyError:
			miLocalPath = mediaItem.getStatistic(Media.Item.LOCAL_PATH)

		self._statusReport(u"Playing media %s, please wait a moment while the player loads." % miTitle)
		
		assert miLocalPath is not None and miLocalPath != u""
#		path = u"file://localhost" + urllib.quote(os.path.abspath(miLocalPath))
		path = os.path.abspath(miLocalPath)
		assert not '[u' in path
		
		winWidth = self.widget().width() - 40
		winHeight = self.widget().height() - 40
		self._writePage(None, u"""
			<table width="%d" height="%d">
				<tr>
					<td width="%d" height="%d">
						<embed width="%d" height="%d" autostart="true" loop="true" nojava="true" uiMode="full" src="%s" ShowControls="true" ShowDisplay="false" ShowStatusBar="false" id="player"></embed>
					</td>
				</tr>
			</table>""" % (winWidth, winHeight, winWidth, winHeight, winWidth, winHeight, path), u"BODY { background: rgb(0,0,0) }", script=u"""
				function focusPlayer() {
					player.focus();
				}
			""",
			bodyTags=u'onLoad="player.focus();"')
		
		mediaItem.releaseLock()

	def _writeDownload(self, mediaItem):
		"""Starts a new download, and informs the user."""
		
		mediaItem.acquireLock()
		miTitle = mediaItem.getStatisticNoLock(Media.Item.TITLE)
		
		mediaItem.releaseLock()
		
		self._statusReport(u'Beginning download of %s...' % miTitle)
		self._reloadCurrentPage()

	def _writeDelete(self, mediaItem):
		"""Purges a downloaded file from the local media
		store, and informs the user."""
		mediaItem.acquireLock()
		try:
			miTitle = mediaItem.getStatisticNoLock(Media.Item.TITLE)
		except KeyError:
			miTitle = mediaItem.getStatisticNoLock(Media.Item.URI)
		mediaItem.releaseLock()
		
		wrappedTitle = utils.enableWrapping(miTitle)
		
		self._window.getMediaManager().purgeItem(mediaItem)
		
		self._statusReport(u'%s deleted.' % miTitle)
		self._reloadCurrentPage()

	def handleCustomURL(self, url):
		"""Interprets the secondary part of katchtv: urls -- the
		command, in other words, and calls the appropriate
		methods to handle it."""
		page = url[8:].lower()
		args = url[8 + 1 + url[8:].index(u':'):]
		
		if page == u'welcome:':
			self._writeWelcome()
			return
		elif page == u'ready:':
			self._writeReady()
			return
		elif page == u'downloads:':
			self._writeCurrentlyDownloading()
			return
		elif page.startswith(u'feed:'):
			self._writeFeed(args)
			return
		elif page.startswith(u'add_download:'):
			# building a mediaitem for the first time
			mediaItem = self._window.getMediaManager().constructFromURI(args)
			self._writeDownload(mediaItem)
			return
		else:
			# it's one of the commands that operates on a
			# pre-existing mediaitem, which is passed as an
			# encoded arg
			mediaItem = self._window.getMediaManager().findFromURI(args)
			if page.startswith(u'delete:'):
				if mediaItem: self._writeDelete(mediaItem)
				return
			elif page.startswith(u'media:'):
				if mediaItem:
					self._writeMedia(mediaItem)
				else:
					self._statusReport("Cannot play this item.  The file has not been downloaded, or has been deleted, or moved.  Try downloading again.")
				return
		
		self._writeUnknown()
	
	def urlSelected(self, url, button_unused, state_unused, target_unused, args_unused):
		"""A hook (Qt signal slot, actually) which is called when the
		user selects a new URL in the browser."""
		self.goToURL(unicode(url))

	def launchURLExternally(self, url):
		"""Launches URLs through KDE.  Used for filetypes that
		the internal browser cannot display, and that we do
		not specially handle."""
		self._window.statusBar().message(u"Unknown data; launching via KDE: %s" % url)
		os.system(u'kfmclient exec "%s"' % url)

	def goToURL(self, url):
		"""A hook (Qt signal slot actually) which is called to make
		make the browser navigate to a new URL.  By overriding
		this, we can do special things that normal browsers don't
		do, but that our application needs so that the browser is
		integrated with the rest of the app."""
	
		fullKURL = self.completeURL(url)
		url = unicode(fullKURL.url())

		if utils.isExternalURL(url):
			self.launchURLExternally(url)
			return

		if url[:4] == u'mms:':
			# just treat as http
			url = 'http:' + url[4:]
		
		# check if this url should go into the
		# browser history (for the back button)
		# and if so, store it
		self._considerForHistory(url)
		
		lowerurl = url.lower()
		
		if lowerurl == u'':
			return
		
		if lowerurl[:7] == u'mailto:':
			# special case; handle mailto urls, since they're used
			# in the manual for contacting the author with bug
			# reports etc.
			# FIXME: this probably isn't very secure
			self.launchURLExternally(lowerurl)
		elif lowerurl == u'katchtv:help:':
			self.goToURL(config.helpContents)
		elif lowerurl[:8] == u'katchtv:':
			self.handleCustomURL(url)
		else:
			isFeed, newUrl = utils.isAFeed(url)
			if isFeed:
				if self._window.bmarksList.haveChannel(newUrl):
					self._window.statusBar().message(u'Viewing Channel Feed' + url)
					self._writeFeed(newUrl)
				else:
					self._window.statusBar().message(u'Adding Channel Feed' + url)
					Feed.FeedCache.acquireLock()
					try:
						feed = Feed.FeedCache.getFeedFromURINoLock(newUrl)
					finally:
						Feed.FeedCache.releaseLock()
					
					feed.acquireLock()
					try:
						self._window.bmarksList.addChannel(feed)
					finally:
						feed.releaseLock()
			elif lowerurl[-5:] == '.html':
				# it's probably the manual, and should be
				# viewable anyway, so just show it
				self._window.statusBar().message(u'Loading URI ' + url)
				KHTMLPart.openURL(self, fullKURL)
			else:
				# before we load this, make sure it's going to
				# be html-like, since khtmlpart won't load
				# other kparts for us
				mimetype = utils.mimetypeForHTTPURL(url)
				
				for ext in config.knownPlayableExtens:
					if lowerurl[-len(ext):] == ext:
						title = u"%s (direct download)" % url
						args = {
							Media.Item.URI: url,
							Media.Item.TITLE: url,
							Media.Item.DESCRIPTION: "This item was directly downloaded from a website, rather than through a subscribed channel's enclosed items.  No description is available; sorry."
						}
						mediaItem = Media.Item(args)
						self._writeDownload(mediaItem)
						return
				
				if not mimetype in config.knownHTMLMimetypes:
					self.launchURLExternally(url)
				else:
					self._window.statusBar().message(u'Loading URI ' + url)
					KHTMLPart.openURL(self, fullKURL)
	
	def _openURLRequest(self, kurl, urlargs):
		"""Another hook (Qt signal slot) which catches requests for URLs, and
		passes them to through the goToURL filter."""
		self.goToURL(kurl)

