/*
 * KDE Radio Station
 * Copyright (C) 2003 Josef Spillner <spillner@kde.org>
 * Published under GNU GPL conditions.
 */

#include <kpushbutton.h>
#include <klocale.h>
#include <kstdguiitem.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <ksimpleconfig.h>

#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qcheckbox.h>
#include <qdir.h>

#include "options.h"
#include "view.h"

OptionsDialog::OptionsDialog(QWidget *parent, const char *name)
: KDialogBase(Plain, i18n("Settings"), Ok | Cancel, Ok, parent, name, true, true)
{
	QWidget *page = plainPage();

	QVBoxLayout *layout = new QVBoxLayout(page, marginHint(), spacingHint());

	QGroupBox *box = new QGroupBox(0, Qt::Vertical, i18n("Meta Server"), page);
	layout->addWidget(box);
	QGridLayout *boxLayout = new QGridLayout(box->layout(), 2, 1, spacingHint());

	KPushButton *viewServers = new KPushButton(KGuiItem(i18n("View"), "filenew"), box);
	boxLayout->addWidget(viewServers, 0, 0);

	updateServers = new KPushButton(KGuiItem(i18n("Update now"), "reload"), box);
	boxLayout->addWidget(updateServers, 1, 0);

	box = new QGroupBox(0, Qt::Vertical, i18n("Radio Stations"), page);
	layout->addWidget(box);
	boxLayout = new QGridLayout(box->layout(), 3, 2, spacingHint());

	QLabel *label = new QLabel(i18n("Automatic update:"), box);
	boxLayout->addWidget(label, 0, 0);

	stationUpdateCombo = new QComboBox( box );
	stationUpdateCombo->insertItem(i18n("On startup"));
	stationUpdateCombo->insertItem(i18n("Weekly"));
	stationUpdateCombo->insertItem(i18n("Never"));
	boxLayout->addWidget(stationUpdateCombo, 0, 1);

	useCache = new QCheckBox(i18n("Use cache file"), box);
	boxLayout->addMultiCellWidget(useCache, 1, 1, 0, 1);

	updateStations = new KPushButton(KGuiItem(i18n("Update now"), "reload"), box);
	boxLayout->addMultiCellWidget(updateStations, 2, 2, 0, 1);

	QHBoxLayout *hboxLayout = new QHBoxLayout(layout, spacingHint());

	label = new QLabel(i18n("Output Plugin"), page);
	hboxLayout->addWidget(label);

	outputPlugin = new QComboBox(page);

	KStandardDirs d;
	QString path = d.findResource("data", "kderadiostation/specs/");
	kdDebug() << "specpath is " << path << endl;
	QDir dir(path);
	QStringList list = dir.entryList("*");
	QStringList::iterator it;

	for(it = list.begin(); it != list.end(); it++)
	{
		kdDebug() << "spec: " << (*it) << endl;
		KSimpleConfig conf(path + "/" + (*it));
		conf.setGroup("desktopspec");
		QString name = conf.readEntry("name");

		if(name)
		{
			outputPlugin->insertItem(name);
			specmap[name] = (*it);
		}
	}

	hboxLayout->addWidget(outputPlugin);

	hboxLayout = new QHBoxLayout(layout, spacingHint());

	label = new QLabel(i18n("Station list"),page);
	hboxLayout->addWidget(label);

	listMode = new QComboBox(page);
	listMode->insertItem(i18n("flat"));
	listMode->insertItem(i18n("bandwidth grouping"));
	listMode->insertItem(i18n("style grouping"));
	hboxLayout->addWidget(listMode);

	connect(updateServers, SIGNAL(clicked()), SLOT(slotUpdateMetaservers()));
	connect(updateStations, SIGNAL(clicked()), SLOT(slotUpdateStations()));
	connect(viewServers, SIGNAL(clicked()), SLOT(slotViewMetaservers()));

	load();
}

void OptionsDialog::slotUpdateMetaservers()
{
	updateServers->setEnabled(false);
	emit signalUpdateMetaservers();
}

void OptionsDialog::slotUpdateStations()
{
	updateStations->setEnabled(false);
	emit signalUpdateStations();
}

void OptionsDialog::slotNewMetaserver(QString uri)
{
	if(uri.isNull())
	{
		updateServers->setEnabled(true);
	}
	else
	{
		if(!metaservers.contains(uri)) metaservers << uri;
	}
}

void OptionsDialog::slotStations()
{
	updateStations->setEnabled(true);
}

void OptionsDialog::slotViewMetaservers()
{
	QStringList::iterator it;
	ViewDialog v(this);

	for(it = metaservers.begin(); it != metaservers.end(); it++)
		v.add((*it));

	v.exec();
}

void OptionsDialog::load()
{
	KConfig *config;
	QString output, sync, grouping, cache;

	config = kapp->config();
	config->setGroup("settings");
	output = config->readEntry("output", "xmmsspec");
	sync = config->readEntry("synchronization", "weekly");
	grouping = config->readEntry("grouping", "flat");
	cache = config->readEntry("cache", "yes");

	QMap<QString, QString>::Iterator it;
	for(it = specmap.begin(); it != specmap.end(); it++)
	{
		if(it.data() == output) outputPlugin->setCurrentText(it.key());
	}

	if(sync == "startup") stationUpdateCombo->setCurrentItem(0);
	else if(sync == "weekly") stationUpdateCombo->setCurrentItem(1);
	else if(sync == "never") stationUpdateCombo->setCurrentItem(2);

	if(grouping == "flat") listMode->setCurrentItem(0);
	else if(grouping == "bandwidth") listMode->setCurrentItem(1);
	else if(grouping == "style") listMode->setCurrentItem(2);

	if(cache == "yes") useCache->setChecked(true);

	oldgrouping = grouping;
}

void OptionsDialog::slotOk()
{
	KConfig *config;
	QString output, sync, grouping, cache;
	int item;
	QString text;

	text = outputPlugin->currentText();
	output = specmap[text];

	item = stationUpdateCombo->currentItem();
	if(item == 0) sync = "startup";
	else if(item == 1) sync = "weekly";
	else if(item == 2) sync = "never";

	item = listMode->currentItem();
	if(item == 0) grouping = "flat";
	else if(item == 1) grouping = "bandwidth";
	else if(item == 2) grouping = "style";

	if(useCache->isChecked()) cache = "yes";
	else cache = "no";

	config = kapp->config();
	config->setGroup("settings");
	config->writeEntry("output", output);
	config->writeEntry("synchronization", sync);
	config->writeEntry("grouping", grouping);
	config->writeEntry("cache", cache);
	config->sync();

	if(oldgrouping != grouping)
		emit signalGrouping();

	accept();
}

