/*
 * KDE Radio Station
 * Copyright (C) 2003, 2004 Josef Spillner <spillner@kde.org>
 * Published under GNU GPL conditions.
 */

#include "stationlist.h"
#include "options.h"
#include "share.h"
#include "station.h"

#include <kpushbutton.h>
#include <kstdguiitem.h>
#include <klocale.h>
#include <klistview.h>
#include <kio/netaccess.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kconfig.h>
#include <ksimpleconfig.h>
#include <kapplication.h>
#include <kdeversion.h>
#include <kurldrag.h>
#include <kmediaplayer/player.h>
#include <dcopref.h>
#include <dcopclient.h>

#include <qlayout.h>
#include <qdom.h>
#include <qpixmap.h>
#include <qlistview.h>
#include <qfile.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qtextstream.h>
#include <qpopupmenu.h>

#include <ctime>

StationList::StationList(QWidget *parent, const char *name)
: QWidget(parent, name)
{
	QVBoxLayout *vbox;
	KConfig *config;
	QString cache;

	m_popup = NULL;
	m_station = NULL;

	view = new KListView(this);
	view->setAllColumnsShowFocus(true);
	view->addColumn(i18n("Stream"));
	view->addColumn(i18n("Bandwidth"));
	view->addColumn(i18n("Style"));
	view->addColumn(i18n("Location"));
	view->addColumn(i18n("URI"));
	view->addColumn(i18n("Type"));

	vbox = new QVBoxLayout(this, 5);
	vbox->add(view);

	m_query = "<?xml version=\"1.0\"><query class=\"metasound\" type=\"connection\" delta=\"%1\">0.1</query>\n";
	m_metaquery = "<?xml version=\"1.0\"><query class=\"metasound\" type=\"meta\">0.1</query>\n";

	m_update = "<?xml version=\"1.0\"?>";
	m_update += "<update class=\"metasound\" type=\"connection\" username=\"%1\" password=\"%2\">";
	m_update += "<option name=\"mode\">add</option>";
	m_update += "<option name=\"version\">0.1</option>";
	m_update += "<option name=\"stream\">%3</option>";
	m_update += "<option name=\"uri\">%4</option>";
	m_update += "<option name=\"location\">%5</option>";
	m_update += "<option name=\"speed\">%6</option>";
	m_update += "<option name=\"style\">%7</option>";
	m_update += "<option name=\"type\">%8</option>";
	m_update += "</update>\n";

	m_broken = "<?xml version=\"1.0\"?>";
	m_broken += "<update class=\"metasound\" type=\"connection\" username=\"%1\" password=\"%2\">";
	m_broken += "<option name=\"mode\">delete</option>";
	m_broken += "<option name=\"uri\">%4</option>";
	m_broken += "</update>\n";

	connect(view, SIGNAL(executed(QListViewItem *)), SLOT(slotActivate(QListViewItem *)));

	config = kapp->config();
	config->setGroup("meta");
	m_metaservers = config->readListEntry("metaservers");

	config->setGroup("settings");
	cache = config->readEntry("cache", "yes");

	show();

	if(cache == "yes")
		loadcache();

	view->setDragEnabled(true);

	connect(view, SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int)),
		SLOT(slotMenu(QListViewItem*, const QPoint&, int)));

	QTimer::singleShot(500, this, SLOT(slotDelayedUpdateStations()));
}

StationList::~StationList()
{
}

void StationList::slotDelayedUpdateStations()
{
	KConfig *config;
	QString sync;
	int lastupdate;
	time_t time;

	config = kapp->config();
	config->setGroup("settings");
	sync = config->readEntry("synchronization", "weekly");

	if(sync == "startup")
		slotUpdateStations();
	else if(sync == "weekly")
	{
		lastupdate = config->readNumEntry("lastupdate");
		time = ::time(NULL);
		if(time - lastupdate > 7 * 86400)
		{
			slotUpdateStations();
			config->setGroup("settings");
			config->writeEntry("lastupdate", time);
			config->sync();
		}
	}
}

void StationList::startDrag()
{
	QListViewItem *item = view->currentItem();

	if(item)
	{
		KURLDrag *d = new KURLDrag(KURL::List(KURL(item->text(4))), this, "DragObject");
		d->dragCopy();
	}
}

QStringList StationList::metaservers(bool writeable)
{
	QStringList tmp;
	QStringList::iterator it;
	KConfig *config;
	bool w;

	if(writeable)
	{
		config = kapp->config();
		config->setGroup("contributions");
		for(it = m_metaservers.begin(); it != m_metaservers.end(); it++)
		{
			w = config->readBoolEntry((*it));
			if(w) tmp << (*it);
		}
		return tmp;
	}
	else return m_metaservers;
}

void StationList::removeStation(QString uri)
{
	QString username, password;
	QString metauri;
	QString broken;
	QStringList::iterator it;
	KConfig *config;
	QString cleanurl;

	username = "";
	password = "";

	broken = m_broken.arg(username
		).arg(password
		).arg(uri);
	broken.replace("&", "&amp;");
	kdDebug() << "REMOVE: " << broken << endl;

	config = kapp->config();
	config->setGroup("contributions");

	for(it = m_metaservers.begin(); it != m_metaservers.end(); it++)
	{
		cleanurl = (*it);
		if(config->readEntry(cleanurl.replace("=", "%3d"))) metauri = (*it);
	}

	doupdate(broken, metauri);
}

void StationList::addStation(QString metaserver, QString stream, QString uri, QString location,
	QString speed, QString style, QString type)
{
	QString username, password;
	QString metauri;
	QString update;
	QStringList::iterator it;
	KConfig *config;
	QString cleanurl;

	username = "";
	password = "";

	update = m_update.arg(username
		).arg(password
		).arg(stream
		).arg(uri
		).arg(location
		).arg(speed
		).arg(style
		).arg(type);
	update.replace("&", "&amp;");
	kdDebug() << "UPDATE: " << update << endl;

	Station *s = new Station();
	s->setProperty("stream", stream);
	s->setProperty("speed", speed);
	s->setProperty("style", style);
	s->setProperty("location", location);
	s->setProperty("type", type);
	s->setProperty("uri", uri);
	s->setProperty("::status", "new");
	m_station = s;

	config = kapp->config();
	config->setGroup("identifiers");

	metauri = metaserver;
	for(it = m_metaservers.begin(); it != m_metaservers.end(); it++)
	{
		cleanurl = (*it);
		if(config->readEntry(cleanurl.replace("=", "%3d")) == metaserver) metauri = (*it);
	}

	m_transaction = trans_update;
	doupdate(update, metauri);
}

void StationList::slotGrouping()
{
	for(Station *s = m_stations.first(); s; s = m_stations.next())
		s->setProperty("::gui", "false");
	view->clear();
	guiupdate();
}

void StationList::slotUpdateMetaservers()
{
	m_transaction = trans_meta;
	doconnection(m_metaquery);
}

void StationList::slotUpdateStations()
{
	m_transaction = trans_query;
	doconnection(m_query);
}

void StationList::doconnection(QString query)
{
	QStringList::iterator it;
	bool ret;
	int size;
	QString tmpbuf, tmpname;
	QStringList tmpservers;
	KConfig *config;
	QString alias, cleanurl;
	QTimer timer;
	time_t time;
	int oldtime;
	int count;
	int writeable;

	connect(&timer, SIGNAL(timeout()), SLOT(slotTimeout()));

	config = kapp->config();

	for(QStringList::iterator it = m_metaservers.begin(); it != m_metaservers.end(); it++)
	{
		config->setGroup("identifiers");
		cleanurl = (*it);
		alias = config->readEntry(cleanurl.replace("=", "%3d"));
		if(alias.isNull()) alias = (*it);
		config->setGroup("meta");
		if(config->readNumEntry(alias) != 1) tmpservers += (*it);
	}

	count = 1;
	for(it = tmpservers.begin(); it != tmpservers.end(); it++)
	{
		QString buf;
		QString tmp = (*it);
		kdDebug() << "Synchronize resource: " << tmp << endl;

		emit signalProgress(tmp, 100 * count / tmpservers.count());
		count += 1;

		config->setGroup("contributions");
		writeable = config->readNumEntry(tmp);

		if(tmp.startsWith("ggzmeta://"))
		{
			KURL url(tmp);
			m_source = tmp;
			m_feedback = "false";
			if(writeable) m_feedback = "true";
			m_host = url.host();
			m_port = url.port();
			if(!m_port) m_port = 15689;

			config->setGroup("deltas");
			oldtime = config->readNumEntry(tmp);
			if(query == m_query)
				m_curquery = query.arg(oldtime);
			else
				m_curquery = query;
			time = ::time(NULL);
			config->writeEntry(tmp, time);
			config->sync();

			m_synchronized = 0;

			sock = new QSocket();
			connect(sock, SIGNAL(connected()), SLOT(slotConnected()));
			connect(sock, SIGNAL(readyRead()), SLOT(slotRead()));
			connect(sock, SIGNAL(error(int)), SLOT(slotError(int)));
			sock->connectToHost(m_host, m_port);

			timer.start(10000, true);
			while(!m_synchronized)
				kapp->processEvents();
			timer.stop();
		}
		else if(query != m_metaquery)
		{
			KURL uri(tmp);
			ret = KIO::NetAccess::download(uri, tmpname, this);
			if(ret)
			{
				QFile f(tmpname);
				f.open(IO_ReadOnly);
				while((size = f.readLine(tmpbuf, 1024)) != -1)
				{
					buf += tmpbuf;
				}
				f.close();
				KIO::NetAccess::removeTempFile(tmpname);

				QRegExp exp("\\<\\!\\-\\-([^>])*\\>");
				buf.replace(exp, "");
				buf = QString::fromUtf8(buf);

				QRegExp icecast("\\<entry_list\\>");
				QRegExp ggzmeta("\\<resultset referer=\"([^\"])*\"\\>");
				QRegExp metaservlocal("\\<metaserver\\>");
				QRegExp wiki("TipiWiki");

				m_source = tmp;
				m_feedback = "false";

				if(icecast.search(buf, 0, QRegExp::CaretAtZero) >= 0)
				{
					kdDebug() << "## detected format: icascast" << endl;
					processicecast(buf);
				}
				else if(ggzmeta.search(buf, 0, QRegExp::CaretAtZero) >= 0)
				{
					kdDebug() << "## detected format: ggzmeta" << endl;
					kdDebug() << "## shouldn't happen, not handled yet" << endl;
				}
				else if(metaservlocal.search(buf, 0, QRegExp::CaretAtZero) >= 0)
				{
					kdDebug() << "## detected format: metaservlocal" << endl;
					processlocal(buf);
				}
				else if(wiki.search(buf, 0, QRegExp::CaretAtZero) >= 0)
				{
					kdDebug() << "## detected format: wiki" << endl;
					processwiki(buf);
				}
				else kdDebug() << "## format not recognized" << endl;
			}
		}
	}

	if(m_transaction == trans_meta) emit signalNewMetaserver(QString::null);
	else if(m_transaction == trans_query) emit signalStations();
	emit signalProgress("", 0);
}

void StationList::doupdate(QString update, QString uri)
{
	m_curquery = update;

	KURL url(uri);

	emit signalProgress(i18n("Share..."), 50);

	sock = new QSocket();
	connect(sock, SIGNAL(connected()), SLOT(slotConnected()));
	connect(sock, SIGNAL(readyRead()), SLOT(slotRead()));
//	connect(sock, SIGNAL(error(int)), SLOT(slotError(int)));
	sock->connectToHost(url.host(), url.port());
}

void StationList::slotConnected()
{
kdDebug() << "QUERY:" << m_curquery << endl;
	sock->writeBlock(m_curquery, m_curquery.length());
	sock->flush();
}

void StationList::slotError(int error)
{
	kdDebug() << "error!" << endl;
	m_synchronized = 1;

	//emit signalProgress("", 0);
}

void StationList::processicecast(QString content)
{
	QDomDocument dom;
	QDomNode node;
	QDomElement element, child, streamchild;
	QString buf, tmp, value, key;

	dom.setContent(content);
	node = dom.documentElement().firstChild();

	if(!node.isNull()) element = dom.documentElement();

	buf = "<resultset referer=\"\">";

	while(!node.isNull())
	{
		element = node.toElement();
		if((element.tagName() == "entry") && (!element.attribute("name").isNull()))
		{
			child = element.firstChild().toElement();
			while(!child.isNull())
			{
				if(child.tagName() == "stream")
				{
					streamchild = child.firstChild().toElement();

					tmp = "<result preference=\"0\">";
					while(!streamchild.isNull())
					{
						key = streamchild.tagName();
						value = streamchild.text();
						if(key == "listen_url")
						{
							tmp += "<uri>" + value + "</uri>";
						}
						else if(key == "stream_type")
						{
							if(value == "Ogg Vorbis")
								tmp += "<type>stream</type>";
							else if(value == "MP3 audio")
								tmp += "<type>stream</type>";
							else
								tmp += "<type>unknown</type>";
						}
						else if(key == "stream_description")
						{
							tmp += "<stream>" + value + "</stream>";
						}
						else if(key == "current_song")
						{
							// ignore
						}
						else if(key == "genre")
						{
							tmp += "<style>" + value + "</style>";
						}
						else if(key == "audio_info")
						{
							tmp += "<speed>" + value + "</speed>";
						}
						streamchild = streamchild.nextSibling().toElement();
					}
					tmp += "</result>";

					buf += tmp;
				}

				child = child.nextSibling().toElement();
			}
		}
		node = node.nextSibling();
	}

	buf += "</resultset>";

kdDebug() << "processicecast delegates to process(" << buf << ")" << endl;
	process(buf);
}

void StationList::processwiki(QString content)
{
	QString buf, tmp;
	QDomDocument dom;
	QDomNode node;
	QDomElement element, child;

	content.replace("&", "&amp;");
	//content.replace(QRegExp("\\<\\!\\-\\-.*\\-\\-\\>"), ""); // howto reliably eliminate comments ??
	content.replace(QRegExp("\\<\\!DOCTYPE[^\\>]*\\>"), "");
kdDebug() << "CONTENT-replaced: " << content << endl;

	buf = "<resultset referer=\"\">";

	dom.setContent(content);
	node = dom.documentElement().firstChild();

	if(!node.isNull()) element = dom.documentElement();

	if(element.tagName() == "html")
	{
		element = element.firstChild().toElement();
		while(!element.isNull())
		{
			if(element.tagName() == "body")
			{
				element = element.firstChild().toElement();
				while(!element.isNull())
				{
					if((element.tagName() == "div") && (element.attribute("id") == "wikicontent"))
					{
kdDebug() << "WIKI: found wikicontent" << endl;
						element = element.firstChild().toElement();
						element = element.firstChild().toElement();
						while(!element.isNull())
						{
kdDebug() << "WIKI: found xxx " << element.tagName() << endl;
							if(element.tagName() == "pre")
							{
kdDebug() << "WIKI: found pre" << endl;
								child = element.firstChild().toElement();
								tmp = "<result preference=\"0\">";
								while(!child.isNull())
								{
									if(child.tagName() == "span")
									{
kdDebug() << "WIKI: found span" << endl;
										tmp += "<stream>" + child.text() + "</stream>";
									}
									else if(child.tagName() == "em")
									{
kdDebug() << "WIKI: found em" << endl;
										QRegExp speed("Speed: ");
										QRegExp location("Location: ");
										QRegExp uri("Uri: ");
										QRegExp bandwidth("Bandwidth: ");
										QRegExp style("Style: ");
										QRegExp type("Type: ");
										QString text = child.text();
										if(speed.search(text, 0, QRegExp::CaretAtZero) >= 0)
										{
											text.replace(speed, "");
											tmp += "<speed>" + text + "</speed>";
										}
										else if(location.search(text, 0, QRegExp::CaretAtZero) >= 0)
										{
											text.replace(location, "");
											tmp += "<location>" + text + "</location>";
										}
										else if(uri.search(text, 0, QRegExp::CaretAtZero) >= 0)
										{
											text.replace(uri, "");
											tmp += "<uri>" + text + "</uri>";
										}
										else if(bandwidth.search(text, 0, QRegExp::CaretAtZero) >= 0)
										{
											text.replace(bandwidth, "");
											tmp += "<speed>" + text + "</speed>";
										}
										else if(style.search(text, 0, QRegExp::CaretAtZero) >= 0)
										{
											text.replace(style, "");
											tmp += "<style>" + text + "</style>";
										}
										else if(type.search(text, 0, QRegExp::CaretAtZero) >= 0)
										{
											text.replace(type, "");
											tmp += "<type>" + text + "</type>";
										}
									}
									child = child.nextSibling().toElement();
								}
								tmp += "</result>";
								buf += tmp;
							}
							element = element.nextSibling().toElement();
						}
					}
					element = element.nextSibling().toElement();
				}
			}
			element = element.nextSibling().toElement();
		}
	}

	buf += "</resultset>";

kdDebug() << "processwiki delegates to process(" << buf << ")" << endl;
	process(buf);
}

void StationList::processlocal(QString content)
{
	QDomDocument dom;
	QDomNode node;
	QDomElement element, child;
	QString buf, tmp, value;
	QDomNamedNodeMap map;

	dom.setContent(content);
	node = dom.documentElement().firstChild();

	if(!node.isNull()) element = dom.documentElement();

	buf = "<resultset referer=\"\">";

	while(!node.isNull())
	{
		element = node.toElement();
		if((element.tagName() == "entrylist") && (element.attribute("class") == "metasound"))
		{
			child = element.firstChild().toElement();
			while(!child.isNull())
			{
				if(child.tagName() == "connection")
				{
					tmp = "<result preference=\"0\">";
					value = child.text();
					value.replace("&amp;", "&");
					value.replace("&", "&amp;");
					tmp += "<uri>" + value + "</uri>";
					map = child.attributes();
					for(unsigned int i = 0; i < map.count(); i++)
					{
						if(map.item(i).nodeName() != "version")
						{
							tmp += "<" + map.item(i).nodeName() + ">";
							tmp += map.item(i).nodeValue();
							tmp += "</" + map.item(i).nodeName() + ">";
						}
					}
					tmp += "</result>";

					buf += tmp;
				}

				child = child.nextSibling().toElement();
			}
		}
		node = node.nextSibling();
	}

	buf += "</resultset>";

kdDebug() << "processlocal delegates to process(" << buf << ")" << endl;
	process(buf);
}

void StationList::process(QString content)
{
	QDomDocument dom;
	QDomNode node;
	QDomElement element, child;
	QString name;
	QString cache;
	KConfig *config;
	QString response;
	QStringList::iterator it;
	//bool has_metaserver = false;
	//bool has_stations = false;
	bool consideration;
	bool writeable;
	QString cleanurl;
	Station *station;
	QString uri;

	// Initialization of data handling

	config = kapp->config();

	dom.setContent(content);
	node = dom.documentElement().firstChild();

	// Check whether data is resource or metaserver response

	if(!node.isNull())
	{
		element = dom.documentElement();
		kdDebug() << "we're at " << element.tagName() << endl;
		if(element.attribute("referer") == "update")
		{
			emit signalProgress(i18n("Shared!"), 100);

			child = element.firstChild().toElement();
			response = child.text();
			KMessageBox::information(this, i18n("Server response was: %1").arg(response), i18n("Update"));
			if(response == "accepted")
			{
				if(m_popup)
				{
					if(m_station)
					{
						m_station->setProperty("::status", "deleted");
						m_station->setProperty("::gui", "false");
						if(view->currentItem()) delete view->currentItem();
					}
					guiupdate();
				}
				else
				{
					m_stations.append(m_station);
					guiupdate();
				}
			}
			return;
		}
	}

	// Extract all information from the resource content

	while(!node.isNull())
	{
		name = QString::null;
		writeable = false;

		element = node.toElement();
		child = element.firstChild().toElement();
		station = new Station();
		//station->setProperty("metaserver", "???");
		while(!child.isNull())
		{
			station->setProperty(child.tagName(), child.text());
			child = child.nextSibling().toElement();
		}
		if(m_transaction == trans_meta)
		{
			for(it = m_metaservers.begin(); it != m_metaservers.end(); it++)
				//if((*it) == uri) uri = QString::null;
				if((*it) == uri) it = m_metaservers.remove(it);
			if(!uri.isNull())
			{
				m_metaservers << uri;
				emit signalNewMetaserver(uri);

				config->setGroup("meta");
				config->writeEntry("metaservers", m_metaservers);
				config->setGroup("identifiers");
				cleanurl = uri;
				config->writeEntry(cleanurl.replace("=", "%3d"), name);
				config->setGroup("contributions");
				config->writeEntry(uri, writeable);
				config->sync();
			}
			//has_metaserver = true;
		}
		else
		{
			//if(!has_stations)
			//{
			//	/*view->clear();*/
			//	has_stations = true;
			//}

			consideration = true;

			for(Station *s = m_stations.first(); s; s = m_stations.next())
				if(s->property("stream") == station->property("stream")
				&& s->property("speed") == station->property("speed")
				&& s->property("style") == station->property("style")
				&& s->property("location") == station->property("location")
				&& s->property("uri") == station->property("uri")
				&& s->property("type") == station->property("type"))
				consideration = false;

			if(consideration)
			{
				station->setProperty("::status", "new");
				station->setProperty("::source", m_source);
				station->setProperty("::feedback", m_feedback);
				m_stations.append(station);
			}
		}
		node = node.nextSibling();
	}

	//if(has_metaserver)
	//{
		//emit signalNewMetaserver(QString::null);
	//}
	//if(has_stations)
	if(m_transaction == trans_query)
	{
		//emit signalStations();

		config->setGroup("settings");
		cache = config->readEntry("cache", "yes");

		if(cache == "yes")
			savecache();
	}

	kdDebug() << "process is ready" << endl;

	guiupdate();
}

void StationList::guiupdate()
{
	KListViewItem *tmp;
	QListViewItem *qtmp;
	QString grouping;
	KStandardDirs d;
	KConfig *config;

	// Read GUI settings

	config = kapp->config();

	config->setGroup("settings");
	grouping = config->readEntry("grouping", "flat");

	if(grouping == "flat")
		view->setRootIsDecorated(false);
	else
		view->setRootIsDecorated(true);

	// Update the GUI

	for(Station *s = m_stations.first(); s; s = m_stations.next())
	{
		if(s->property("::gui") == "true") continue;
		else s->setProperty("::gui", "true");

		if(grouping == "bandwidth")
		{
			qtmp = view->findItem(s->property("speed"), 0);
			if(!qtmp) qtmp = new KListViewItem(view, s->property("speed"));
			tmp = new KListViewItem(qtmp, s->property("stream"), s->property("speed"),
				s->property("style"), s->property("location"), s->property("uri"), s->property("type"));
		}
		else if(grouping == "style")
		{
			qtmp = view->findItem(s->property("style"), 0);
			if(!qtmp) qtmp = new KListViewItem(view, s->property("style"));
			tmp = new KListViewItem(qtmp, s->property("stream"), s->property("speed"),
				s->property("style"), s->property("location"), s->property("uri"), s->property("type"));
		}
		else
			tmp = new KListViewItem(view, s->property("stream"), s->property("speed"),
				s->property("style"), s->property("location"), s->property("uri"), s->property("type"));
		QPixmap pix(d.findResource("data", "kderadiostation/kderadio16.png"));
		if(s->property("::status") == "new")
			pix = QPixmap(d.findResource("data", "kderadiostation/kderadio16_new.png"));
		else if(s->property("::status") == "deleted")
			pix = QPixmap(d.findResource("data", "kderadiostation/kderadio16_obsolete.png"));
		tmp->setPixmap(0, pix);
	}
}

void StationList::slotRead()
{
	QString rdata, rtmp;
	QCString cs;

	while(sock->bytesAvailable())
	{
		kdDebug() << "bytesAvailable " << sock->bytesAvailable() << endl;
/*		rtmp += sock->readLine();*/
		cs.resize(sock->bytesAvailable());
		sock->readBlock(cs.data(), sock->bytesAvailable());
		rtmp += cs;
		sock->waitForMore(500);
	}
	rdata = QString::fromUtf8(rtmp);
	//kdDebug() << "got rawdata(" << rdata << ")" << endl;

	rdata.truncate(rdata.length() - 1);

	process(rdata);

	if(m_transaction == trans_update)
		emit signalProgress("", 0);

	m_synchronized = 1;

	sock->close();
	//delete sock;
}

void StationList::slotActivate(QListViewItem *item)
{
	bool success;
	bool cache = true;
	QString tmp, normalized;
	KProcess *proc;
	KConfig *config = kapp->config();
	QString output;
	QTimer timer;

	if(item->text(1).isNull()) return;

	success = true;
	if(item->text(5) == "playlist")
	{
		if(cache)
		{
			//KMD5 md5(item->text(4));
			//normalized = QString("kderadiostation-") + QString(md5.hexDigest());
			normalized = QString("kderadiostation-") + item->text(4);
			normalized.replace("/", "_");
			tmp = locateLocal("tmp", normalized);
		}
		else
		{
			KTempFile t(QString::null, ".pls");
			tmp = t.name();
		}
		if(!cache || !QFile::exists(tmp))
		{
			if(KIO::NetAccess::download(item->text(4), tmp, this))
				success = true;
			else
			{
				KMessageBox::error(this, i18n("The playlist could not be downloaded."), i18n("Failure"));
				success = false;
			}
		}
	}
	else if(item->text(5) == "stream")
	{
		tmp = item->text(4).latin1();
		success = true;
	}
	else
	{
		KMessageBox::error(this, i18n("Unknown type of audio stream."), i18n("Failure"));
		success = false;
	}

	if(success)
	{
		config->setGroup("settings");
		output = config->readEntry("output", "xmmsspec");

		KStandardDirs d;
		QString path = d.findResource("data", "kderadiostation/specs/");
		KSimpleConfig conf(path + "/" + output);
		conf.setGroup("desktopspec");
		QString exec = conf.readEntry("exec");
		conf.setGroup("techspec");
		QString interface = conf.readEntry("interfaces");

		if(interface == "kmediaplayer")
		{
			bool found;
			QString app;
			DCOPClient *client = kapp->dcopClient();
			client->attach();

			QCStringList list = client->registeredApplications();
			found = false;
			for(QCStringList::iterator it = list.begin(); it != list.end(); it++)
			{
				QString s((*it));
				if(s.startsWith(QString("%1-").arg(exec))) found = 1;
			}

			if(!found)
			{
				proc = new KProcess();
				*proc << exec.latin1();
				proc->start();

				m_synchronized = 0;
				connect(&timer, SIGNAL(timeout()), SLOT(slotTimeout()));
				timer.start(5000, true);

				app = QString("%1-%2").arg(exec).arg(proc->pid());
				while((!client->isApplicationRegistered(app.latin1())) && (!m_synchronized))
					kapp->processEvents();
				timer.stop();
			}

			QByteArray data;
			QDataStream ds(data, IO_WriteOnly);
			ds << KURL(tmp);
			found = client->send(QString("%1-*").arg(exec).latin1(),
				"KMediaPlayer", "openURL(KURL)", data);
			if(!found) kdDebug() << "broadcast for " << exec << "failed" << endl;
		}
		else
		{
			proc = new KProcess();
			bool found_url = false;
			if(exec.find("%U") >= 0)
			{
				found_url = true;
				exec.replace("%U", tmp);
			}

			if(exec.find(" ") >= 0)
			{
				QStringList sl;
				sl = sl.split(" ", exec);
				*proc << sl;
			}
			else
			{
				*proc << exec.latin1();
			}

			if(!found_url)
			{
				*proc << tmp.latin1();
			}
			proc->start();
		}
	}
}

void StationList::slotTimeout()
{
	kdDebug() << "timeout!" << endl;

	//emit signalProgress("", 0);
	m_synchronized = 1;
	sock->close();
	//delete sock;
}

void StationList::savecache()
{
	QString file;

	file = locateLocal("data", "kderadiostation/cache_format2");
	QFile f(file);
	if(!f.open(IO_WriteOnly)) return;
	QTextStream stream(&f);

	for(Station *s = m_stations.first(); s; s = m_stations.next())
	{
		QStringList prop = s->properties();
		for(QStringList::iterator it = prop.begin(); it != prop.end(); it++)
			stream << s->property((*it)) << ":::";
		stream << "\n";
	}

	f.close();
}

void StationList::loadcache()
{
	QString file, line;
	QStringList list;
	KStandardDirs d;

	file = locateLocal("data", "kderadiostation/cache_format2");
	QFile f(file);
	if(!f.open(IO_ReadOnly)) return;
	QTextStream stream(&f);

	while(!stream.eof())
	{
		line = stream.readLine();
		list = QStringList::split(":::", line);

		Station *s = new Station();
		QStringList prop = s->properties();
		QStringList::iterator it2 = prop.begin();
		for(QStringList::iterator it = list.begin(); it != list.end(); it++)
		{
			s->setProperty((*it2), (*it));
			it2++;
		}
		m_stations.append(s);
	}

	f.close();

	guiupdate();
}

void StationList::slotMenu(QListViewItem *item, const QPoint& point, int c)
{
	if(!item) return;

	if(m_popup) delete m_popup;
	m_popup = new QPopupMenu(this);

	m_popup->insertItem(i18n("Report as broken"), menu_broken);
	m_popup->insertItem(i18n("Modify..."), menu_modify);
	m_popup->insertItem(i18n("More information..."), menu_information);

	for(Station *s = m_stations.first(); s; s = m_stations.next())
		if(s->property("stream") == item->text(0)
		&& s->property("speed") == item->text(1)
		&& s->property("style") == item->text(2)
		&& s->property("location") == item->text(3)
		&& s->property("uri") == item->text(4)
		&& s->property("type") == item->text(5))
		{
			if(s->property("::feedback") != "true")
			{
				m_popup->setItemEnabled(0, false);
				m_popup->setItemEnabled(1, false);
			}
			m_station = s;
		}

	m_popup->popup(point);

	connect(m_popup, SIGNAL(activated(int)), SLOT(slotMenuAction(int)));
}

void StationList::slotMenuAction(int id)
{
	QListViewItem *item;

	item = view->currentItem();

	switch(id)
	{
		case menu_broken:
			removeStation(item->text(4));
			break;
		case menu_modify:
			break;
		case menu_information:
			break;
	}
}

