// -*- c++ -*-

/*
 * Copyright (C) 2002 Richard Moore <rich@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify 
 *   it under the terms of the GNU General Public License as published by 
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <qptrlist.h>
#include <qpair.h>
#include <qpoint.h>
#include <qsize.h>
#include <qpixmap.h>
#include <kparts/dockmainwindow.h>

#include "kdetvviewiface.h"
#include "settingsdialogpage.h"

class KAction;
class KToggleAction;
class KRadioAction;
class KPopupMenu;
class KConfig;
class KDockWidget;

class KdetvWidget;
class LibKdetvActions;
class SysTray;
class QEvent;

class SettingsDialogPage;
class KdetvSettingsWidgetImpl;

class Lirc;

class KdetvPart;

/**
 * MainWindow shell for Kdetv.
 *
 */
class MainWindow : public KParts::DockMainWindow,
                   virtual public KdetvViewIface,
                   public SettingsDialogPageFactory
{
    Q_OBJECT

public:
    MainWindow(QWidget* parent = 0, const char* name = 0, WFlags f = 0);
    ~MainWindow();

    KPopupMenu *createMenu( const QString &menuname );
    KPopupMenu *insertContextMenu( const QString &menuname );

public slots:
    void toggleMenubar();
    void toggleStatus();
    void disableScreenSaver( bool );

    void showWebSite();
    void showTipOfTheDay();

    /**
     * Activates/deactivates sleep timer.
     */
    virtual void sleepTimer();

    /**
     * Activates the specified view mode.
     */
    void setViewMode( int mode );

    /**
     * Toggle full screen.
     * Change to fullscreen, and store the old mode.
     * If already in fullscreen change back to Normal mode, or to the stored old mode.
     */
    void toggleFullScreen();

    /**
     * Switches to the next view mode.
     */
    void nextViewMode();

    /**
     * Display toolbar config dialog
     */
    void configureToolbars();

    /**
     * If active is true, this method enables the view mode
     * 'None'. This is a special mode that disables the video feed
     * allowing another application (eg. the Kdetv screensaver to
     * take over the video source.
     */
    void setViewModeNone()         { setViewMode( ViewNone );         };

    /**
     * Normal window with tool-, menu- etc. bars.
     */
    void setViewModeNormal()       { setViewMode( ViewNormal );       };

    /**
     * Window  without tool-, menu- etc. bars.
     */
    void setViewModeTV()           { setViewMode( ViewTV );           };

    /**
     * Fullscreen ("TV Mode")
     */
    void setViewModeFullScreen()   { setViewMode( ViewFullScreen );   };

    /**
     * Display video on desktop background
     */
    void setViewModeVideoDesktop() { setViewMode( ViewVideoDesktop ); };

signals:
    /// Emits if a lirc number key was pressed
    void numberKeyPressed(int);

    /**
     * Emitted when the "time to sleep before turning off" has been changed.
     */
    void sleepTimerDelay(const QString &);
    
    /**
     * Emitted if view mode is changed
     */
    void viewModeChanged( int mode );

    /**
     * Emitted if widget visibility is changed
     */
    void visibilityChanged( bool visible );

protected slots:
    void setShowTrayIcon( bool show );
    void lircCommand(const QString &, unsigned);
    void keyBindings();
    void sbProgramName(const QString& name);
    void statusMessage(const QString& name);
    void shortMessage(const QString& name);
    void clearStatusBar();
    void restoreViewMode();
    void bringToFront();

    void restoreWindowSettings();

    void devicePluginsChanged();
    void audioModesChanged();

    // for use with toggle actions (avoids flicker, they are called if action is disabled, too)
    void actSetViewModeNormal(bool enable)       { if (enable) setViewMode( ViewNormal );       };
    void actSetViewModeTV(bool enable)           { if (enable) setViewMode( ViewTV );           };
    void actSetViewModeFullScreen(bool enable)   { if (enable) setViewMode( ViewFullScreen );   };
    void actSetViewModeVideoDesktop(bool enable) { if (enable) setViewMode( ViewVideoDesktop ); };

    // Sidebar dock window
    void setSidebarVisibility(bool);
    void sideBarVisibilityChanged();

    void checkScreenSaver();

protected:
    void createActions();
    void addDynamicActions();

    void setupInfraRed();

    void saveWindowSettings();

    void hideGUIElements();
    void restoreGUIElements();

    void moveEvent( QMoveEvent* e );
    void resizeEvent( QResizeEvent* e );
    void showEvent( QEvent* e );
    void hideEvent( QEvent* e );

    void doSetViewMode( int mode, bool enable );

    QPtrList<SettingsDialogPage>* getPages();

    /*
     * If the following functions are called with active = true
     * they perform the transition form viewNormal to the desired mode.
     * If they are called with active = false, they undo all actions
     * until back to normal mode.
     *
     * These must only be called via setViewMode() to prevent errors.
     */ 
    void setShowNone( bool active );
    void setShowNormal( bool active );
    void setShowTV( bool active );
    void setShowFullScreen( bool active );
    void setShowVideoDesktop( bool active );

private slots:
    void slotSleepTimerTimeout();
    void updateTaskbarButton();
    void updateWindowAttr();

private:
    friend class KdetvSettingsWidgetImpl;

    /**
     * Identifiers for the screensaver modes.
     */
    enum SaverMode {
        SaverRunning=0, SaverVisible, SaverFullScreen
    };

    /**
     * Identifiers for the various view modes.
     */
    enum ViewMode { 
        ViewNormal=0, ViewFullScreen=1, ViewTV=2, ViewVideoDesktop=3,
        ViewMaxIterable=ViewVideoDesktop,
        ViewNone=128
    };

    KdetvPart* _part;
    
    bool _viewNormalEnabled;
    bool _viewFullEnabled;
    bool _viewTVEnabled;
    bool _viewDesktopEnabled;

    KdetvWidget*     _screen;
    LibKdetvActions* _actions;
    SysTray*         _systray;
    
    QTimer* _sleeptimer;

    KDockWidget* _sideDock;
    KDockWidget* _mainDock;
    
    KAction* _actChangeLog;
    KAction* _actCVSLog;
    KAction* _actWebSite;
    KAction* _actConfigToolbar;
    KAction* _actShowTips;
    
    KToggleAction*    _sleepTime;
    KToggleAction*    _actShowSidebar;
    KToggleAction*    _actShowMenubar;
    KRadioAction*     _actShowNorm;
    KRadioAction*     _actShowFull;
    KRadioAction*     _actShowTV;
    KRadioAction*     _actShowDesk;
    KAction*          _actNextView;
    KToggleAction*    _actToggleFull;
    QPtrList<KAction> _importActionList;
    QPtrList<KAction> _exportActionList;

    KConfig* _cfg;

    Lirc* _lirc;

    bool   _changingViewMode;
    QPoint _topLevelPos;
    QSize  _topLevelSize;
    QPoint _pos;
    QSize  _size;

    int  _viewMode;
    int  _oldViewMode;
    bool _staysOnTop;
    bool _sticky;
    bool _noStickyFull;
    bool _wasSticky;

    bool _hideTBNormal;
    bool _hideTBTV;
    bool _hideTBFull;
    
    bool _screenSaverWasEnabled;
    bool _dpmsWasEnabled;
    bool _disableSaver;
    int  _screenSaverMode;
        
    int _sleepdelay;

    bool _haveKWin;

    bool _savedGuiStateValid;
    bool _menubarVisible;
    bool _sidebarVisible;
    bool _statusbarVisible;
    typedef QPair<KToolBar*, bool> toolBarVisibilityPair;
    QPtrList<toolBarVisibilityPair> _toolbarVisibilityList;
    QPixmap _miniIcon;
    QPixmap _icon;
    QColor _bgcol;
};

#endif // MAINWINDOW_H

