/* encode.c
 * Handlers for encoding a wav stream into a kexis stream.
 * Copyright (C) 2000 Wayde Milas (wmilas@rarcoa.com)
 *
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
  
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "types.h"
#include "encode.h"
#include "header.h"
#include "kexis.h"
#include "predictor.h"
#include "rice.h"
#include "bits.h"

void compress(WAVHEADER *wavHead, OPTIONSTRUCT *options)
{
  int length;
	unsigned long pcmLength=0;
  KEXISBLOCKSTRUCT kexisBlock;
	PCMBLOCKSTRUCT pcmBlock;

  options->inFileStream = fopen(options->inFileName, "rb");
  if(options->inFileStream == NULL) {
    sprintf(options->errorString, "Can't open file: %s!",
			options->inFileName);
    handle_error(options);
  }

	if(options->progress)
		printf("\nEncoding file: %s\n", options->inFileName);

  // Suck in the WaveHeader
  parse_wave_header(wavHead, options);
  pcmLength = wavHead->data_ckSize;

	//Save the original size right now as a quick hack. Remove this later.
	pcmBlock.pcmStreamLength = wavHead->data_ckSize;

  // Since the Wav Header data length is in bytes, we need to convert it to
  // "units". Each "unit" is 2 bytes, or a short int. so we devide by 2.
  pcmLength = pcmLength / 2;

	// If the frame size is LARGER than the pcmLength, make it the pcmLength
	if(options->frameSize > wavHead->data_ckSize)
		options->frameSize = wavHead->data_ckSize;
	
  // Here we malloc 200,000 units, or 400,000 bytes.
  pcmBlock.data = malloc(sizeof(short)*options->frameSize);
	kexisBlock.data = malloc(sizeof(long)*(options->frameSize/2));

	// Clear the Predictor Table
	zero_predictor_table(&kexisBlock.predictor, options);
	kexisBlock.bitCounter=0;
	kexisBlock.bitBucket=0;
	kexisBlock.dataPosition=0;
	kexisBlock.sumTotalSize=0;
	kexisBlock.debugBlockCount=0;

  while(pcmLength > 0) {

    if(options->progress)
			print_progress(&kexisBlock, pcmLength, options, wavHead->data_ckSize,
				COMPRESS);

    if(pcmLength < options->frameSize)
      pcmBlock.size = pcmLength;
    else
      pcmBlock.size = options->frameSize;

    length = fread(pcmBlock.data, sizeof(short), pcmBlock.size,
			options->inFileStream);
    if(length < pcmBlock.size) {
      sprintf(options->errorString, "Error: read %d units out of %ld units.\n Error Reading PCM stream. Short Block count. Your WAV file is\n corrupt and is not as long as it should be.", length, pcmBlock.size);
      handle_error(options);
    }

    encode_pcmblock(&pcmBlock, &kexisBlock, options);
    pcmLength = pcmLength - pcmBlock.size;
  }

	cleanup_bits(&kexisBlock, options, &pcmBlock);
	handle_verbose(options, &kexisBlock, &pcmBlock);
	free_allocated(&pcmBlock, &kexisBlock, options);
}

void encode_pcmblock(PCMBLOCKSTRUCT *pcmBlock, KEXISBLOCKSTRUCT *kexisBlock,
	OPTIONSTRUCT *options)
{
  long   left, right, mid, ave;
  int     loop=0, holdM, holdA;

	if(options->predictorVersion == FRAME_PREDICTOR) {
		precompute_predictor(kexisBlock, pcmBlock, &holdA, &holdM);
		push_bits(kexisBlock, holdA, 3, options, pcmBlock);
		push_bits(kexisBlock, holdM, 3, options, pcmBlock);
	}
	
  while(loop < pcmBlock->size) {
    left = pcmBlock->data[loop];
    right = pcmBlock->data[loop+1];

    // X = average of the 2 channels
    // Y = distance from mid to right channel.
    // Thats all we need to extract L and R later, plus hopefully
    // Y is now a smaller number.
		if(options->joint == 1) {
    	mid = left - right;
    	ave = right + (mid/2);
		}
		else {
			mid = left;
			ave = right;
		}

		// Now use our predictor to tranform X and Y into differences.
		if(options->predictorVersion == FRAME_PREDICTOR) 
			frame_difference_calc(&kexisBlock->predictor, holdA, holdM,
				mid, ave, COMPRESS);
		else
			difference_calc(&kexisBlock->predictor, options->predictorVersion,
				mid, ave, COMPRESS);

		// If we are gathering statistics, gather them :P
		if(options->verbose)
			collect_prediction_stats(kexisBlock, kexisBlock->predictor.diffMid,
				kexisBlock->predictor.diffAve);

		// now we need to do our bit encoding
		rice_encode(kexisBlock, kexisBlock->predictor.diffMid, options,
			pcmBlock, MID);
		//kexisBlock->debugBlockCount++;
		rice_encode(kexisBlock, kexisBlock->predictor.diffAve, options,
		  pcmBlock, AVE);
		//kexisBlock->debugBlockCount++;

    loop = loop +2;
  }
}

void write_kexisblock(OPTIONSTRUCT *options, KEXISBLOCKSTRUCT *kexisBlock,
	PCMBLOCKSTRUCT *pcmBlock)
{
  int length;
	KEXISHEADER kexisHeader;

	// Check and see if its the beginning of the stream.
  if(options->outFileStream == NULL) {
		// Make the filename
    options->outFileName = calloc(1, strlen(options->inFileName)+1);
    strncpy(options->outFileName, options->inFileName,
      strlen(options->inFileName)-3);
    strcat(options->outFileName, "kxs\0");

		// Open the file
    options->outFileStream = fopen(options->outFileName, "wb");
    if(options->outFileStream == NULL) {
      sprintf(options->errorString, "Can't open file: %s!",
				options->outFileName);
			handle_error(options);
		}

		// Create the header
		create_kexis_header(&kexisHeader, options, pcmBlock->pcmStreamLength);
		length = fwrite(&kexisHeader, sizeof(kexisHeader), 1,
			options->outFileStream);
		if(length != 1) {
			sprintf(options->errorString, "Error writing Header to %s!",
				options->outFileName);
			handle_error(options);
    }
  }

  length = fwrite(kexisBlock->data, sizeof(long), kexisBlock->dataPosition,
    options->outFileStream);

	if(options->progress)
		kexisBlock->sumTotalSize += length;
	
  if(length != kexisBlock->dataPosition) {
		sprintf(options->errorString, "Error writing to %s!",
			options->outFileName);
		handle_error(options);
	}
}
