/* header.c
 * Functions to parse Wav and Kexis file headers.
 * Copyright (C) 2000 Wayde Milas (wmilas@rarcoa.com)
 * 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "types.h"
#include "header.h"
#include "kexis.h"

void parse_kexis_header(KEXISHEADER *kexisHead, OPTIONSTRUCT *options)
{
	int length;

	length = fread(kexisHead, sizeof(KEXISHEADER), 1, options->inFileStream);
	if(length!= 1) {
    sprintf(options->errorString,
			"This file is not long enough to contain a Kexis header!");
    handle_error(options);
  }

	if(strncmp("KEXS", kexisHead->kexisID, 4) != 0) {
    sprintf(options->errorString, "File does not contain a KEXS header!");
		handle_error(options);
  }
	else if(kexisHead->encoderVersion <= 0 ||
		kexisHead->encoderVersion > MAX_STANDARD_ENCODER) {
		sprintf(options->errorString, "Invalid Encoder Version: %hd.",
			kexisHead->encoderVersion);
		handle_error(options);	
	}
	else if(kexisHead->predictorVersion <= 0 || 
    kexisHead->predictorVersion > MAX_STANDARD_PREDICTOR) {
    sprintf(options->errorString, "Invalid Predictor Version: %hd.",
			kexisHead->predictorVersion);
		handle_error(options);
  }
	else if(kexisHead->kHistorySize < 1 || 
		kexisHead->kHistorySize > MAX_KHISTORY) {
		sprintf(options->errorString, "Invalid K History Size: %hd.",
			kexisHead->kHistorySize);
		handle_error(options);
	}
	else if(kexisHead->frameSize < 64) {
		if(kexisHead->predictorVersion == FRAME_PREDICTOR) {
    	sprintf(options->errorString, "Invalid Frame Size: %ld.",
    		kexisHead->frameSize);
    	handle_error(options);
		}
		kexisHead->frameSize = FRAME_UNITS;
  }
	else if(kexisHead->options > 1) {
		sprintf(options->errorString, "Invalid Stereo Option: %d.",
			kexisHead->options);
		handle_error(options);
	}

	options->encoderVersion = kexisHead->encoderVersion;
	options->predictorVersion = kexisHead->predictorVersion;
	options->kHistorySize = kexisHead->kHistorySize;
	options->frameSize = kexisHead->frameSize;
	options->joint = kexisHead->options;

	if(options->displayHeader && !options->dec_stdout) {
    printf("Kexis File Header:\n");
    printf("   KEXS Header                 : %c%c%c%c\n",
      kexisHead->kexisID[0], kexisHead->kexisID[1], kexisHead->kexisID[2],
				kexisHead->kexisID[3]);
    printf("   Kexis Header Version        : %hd\n", kexisHead->headerVersion);
		printf("   Encoder Version             : %hd\n",
			kexisHead->encoderVersion);
		printf("   Predictor Version           : %hd\n",
      kexisHead->predictorVersion-1);
    printf("   Length of Data Block        : %lu\n", kexisHead->dataLength);
		printf("   K History Size              : %hd\n",
				kexisHead->kHistorySize);
		printf("   Use Stereo Data for Comp    : %d\n", kexisHead->options);
		printf("   Frame Size                  : %ld\n", kexisHead->frameSize);
		printf("\n");
  }
}

void create_kexis_header(KEXISHEADER *kexisHeader, OPTIONSTRUCT *options,
	unsigned long length)
{
	strncpy(kexisHeader->kexisID, "KEXS", 4);
	kexisHeader->headerVersion = CURRENT_KEXIS_HEADER;
	kexisHeader->encoderVersion = options->encoderVersion;
	kexisHeader->predictorVersion = options->predictorVersion;
	kexisHeader->dataLength = length;
	kexisHeader->kHistorySize = options->kHistorySize;
	kexisHeader->frameSize = options->frameSize;
	kexisHeader->options = options->joint;
	kexisHeader->pad3 = 0;
}

void parse_wave_header(WAVHEADER *wavHead, OPTIONSTRUCT *options)
{
  int length;
	unsigned long hold;

	length = fread(wavHead->ckID, sizeof(char)*4, 1, options->inFileStream);
  if(strncmp("RIFF", wavHead->ckID, 4) != 0 || length != 1) {
    sprintf(options->errorString, "File does not contain a RIFF header!");
    handle_error(options);
  }

	length = fread(&wavHead->ckSize, sizeof(unsigned long), 1,
		options->inFileStream);
	length = fread(wavHead->wave_ckID, sizeof(char)*4, 1, options->inFileStream);
	if(strncmp("WAVE", wavHead->wave_ckID, 4) != 0 || length != 1) {
    sprintf(options->errorString, "File does not contain a WAVE Header!");
    handle_error(options);
  }

	length = fread(wavHead->fmt_ckID, sizeof(char)*4, 1, options->inFileStream);
  // Suck bits till we hit the fmt chunk
	while ((strncmp("fmt ", wavHead->fmt_ckID, 4) != 0) && length == 1) {
		length = fread(&hold, sizeof(unsigned long), 1, options->inFileStream);
		fseek(options->inFileStream, hold, SEEK_CUR);	
		length = fread(wavHead->fmt_ckID, sizeof(char)*4, 1, options->inFileStream);
	}
	if(length <= 0) {
		sprintf(options->errorString,
			"Could not find the fmt chunk in the WAV header!");
		handle_error(options);
  }
		
	length = fread(&wavHead->fmt_ckSize, sizeof(unsigned long), 1,
	 	options->inFileStream);
	length = fread(&wavHead->formatTag, sizeof(unsigned short), 1,
	  options->inFileStream);
	if(wavHead->formatTag != 1 || length != 1) {
    sprintf(options->errorString, "WAV file contains non PCM data!");
    handle_error(options);
  }

	length = fread(&wavHead->nChannels, sizeof(unsigned short), 1,
	  options->inFileStream);
  if(wavHead->nChannels != 2 || length != 1) {
    sprintf(options->errorString,
			"This is a Mono WAV file. Hexis can only compress stereo files.");
    handle_error(options);
  }
	
	length = fread(&wavHead->nSamplesPerSec, sizeof(unsigned long), 1,
	  options->inFileStream);
  if(wavHead->nSamplesPerSec != 44100 || length != 1) {
    sprintf(options->errorString,
			"Kexis can only compress 44100 hz WAV Files. This one is %lu hz.",
			wavHead->nSamplesPerSec);
    handle_error(options);
  }

	length = fread(&wavHead->nAvgBytesPerSec, sizeof(unsigned long), 1,
	  options->inFileStream);
	length = fread(&wavHead->nBlockAlign, sizeof(unsigned short), 1,
	  options->inFileStream);

	length = fread(&wavHead->nBitsPerSample, sizeof(unsigned short), 1,
	  options->inFileStream);
  if(wavHead->nBitsPerSample != 16 || length != 1) {
    sprintf(options->errorString,
			"Kexis can only compress 16 bit data files. This one is %ld bits.",
			wavHead->data_ckSize);
    handle_error(options);
  }

	length = fread(wavHead->data_ckID, sizeof(char)*4, 1, options->inFileStream);
	// suck bits till we hit the data chunk
	while (strncmp("data", wavHead->data_ckID, 4) != 0 && length == 1) {
    length = fread(&hold, sizeof(unsigned long), 1, options->inFileStream);
    fseek(options->inFileStream, hold, SEEK_CUR);
		length = fread(wavHead->data_ckID, sizeof(char)*4, 1,
			options->inFileStream);
  }
  if(length <= 0) {
    sprintf(options->errorString,
      "Could not find the data chunk in the WAV header!");
    handle_error(options);
  }

	length = fread(&wavHead->data_ckSize, sizeof(unsigned long), 1,
	  options->inFileStream);
	if(wavHead->data_ckSize <= 0) {
		sprintf(options->errorString, "WAV file contains NO PCM data!");
	  handle_error(options);
	}

	if(options->displayHeader) {
    printf("Wave File Header:\n");
    printf("   RIFF Header                 : %c%c%c%c\n",
      wavHead->ckID[0], wavHead->ckID[1], wavHead->ckID[2], wavHead->ckID[3]);
    printf("   File length (minus 8 bytes) : %lu bytes\n", wavHead->ckSize);
    printf("   RIFF Type                   : %c%c%c%c\n",
      wavHead->wave_ckID[0], wavHead->wave_ckID[1], wavHead->wave_ckID[2],
      wavHead->wave_ckID[3]);
    printf("   Next chunk (should be fmt)  : %c%c%c%c\n",
       wavHead->fmt_ckID[0], wavHead->fmt_ckID[1], wavHead->fmt_ckID[2],
       wavHead->fmt_ckID[3]);
    printf("   Length of fmt data          : %lu bytes\n", wavHead->fmt_ckSize);
    printf("   Format tag (1 = PCM)        : %hd\n", wavHead->formatTag);
    printf("   Number of Channels          : %hd\n", wavHead->nChannels);
    printf("   Sapmpes per second (hz)     : %lu\n", wavHead->nSamplesPerSec);
    printf("   Average Bytes per Second\n");
    printf("    Sample Rate * Block Align  : %lu\n", wavHead->nAvgBytesPerSec);
    printf("   Block Align\n");
    printf("    Channels * Bits/Sample / 8 : %hd\n", wavHead->nBlockAlign);
    printf("   Bits Per Sample             : %hd\n", wavHead->nBitsPerSample);
    printf("   Next chunk (should be data) : %c%c%c%c\n",
      wavHead->data_ckID[0], wavHead->data_ckID[1], wavHead->data_ckID[2],
      wavHead->data_ckID[3]);
    printf("   Length of Data Block        : %lu\n\n", wavHead->data_ckSize);
  }
}

void create_wave_header(WAVHEADER *wavHeader, unsigned long length)
{
	 strncpy(wavHeader->ckID, "RIFF", 4);
	 wavHeader->ckSize = length+36;
	 strncpy(wavHeader->wave_ckID, "WAVE", 4);
	 strncpy(wavHeader->fmt_ckID, "fmt ", 4);
	 wavHeader->fmt_ckSize = 16;
	 wavHeader->formatTag = 1;
	 wavHeader->nChannels = 2;
	 wavHeader->nSamplesPerSec = 44100;
	 wavHeader->nAvgBytesPerSec = 176400;
	 wavHeader->nBlockAlign = 4;
	 wavHeader->nBitsPerSample = 16;
	 strncpy(wavHeader->data_ckID, "data", 4);
	 wavHeader->data_ckSize = length;
}
