/* kexis.c
 * Kexis is a lossless wav (PCM) sound compressor.
 * Copyright (C) 2000 Wayde Milas (wmilas@rarcoa.com)
 *
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/resource.h>
#include <unistd.h>
#include "types.h"
#include "kexis.h"
#include "encode.h"
#include "decode.h"
#include "rice.h"

void useage(void)
{
	printf("Useage:\n");
	printf("kexis [MODE] [OPTION]... FILE\n\n");
	printf("Mode:\n");
	printf(" -c   Compress a WAV file. This is the default if no mode is given.\n");
	printf("      A WAV file is the input and a KEX file is the output.\n");
	printf(" -x   Extract a KEX file to a WAV.\n");
	printf(" -s   Extract a KEX file to stdout.\n\n");
	printf("Options:\n");
	printf(" -p   Display a progress bar and ending statistics during\n");
	printf("      compression/extraction.\n");
	printf("      Default is to display NO data whatsoever.\n");
	printf(" -h   Display full WAV header if in Compress Mode.\n");
	printf("      Display full KEXIS header if in Uncompress (Extract) Mode.\n");
	printf(" -v   Display Verbose performance/debugging data during\n");
	printf("      procesing. Enabling this also assumes and enables -p.\n");
	printf(" -d   When encoding, delete the WAV file after the encode is\n");
	printf("      finished.\n");
	printf(" -ns  Do NOT use similarities between stereo channels when\n");
	printf("      encoding. Note: This almost always results in worse\n");
	printf("      compression, so only use it if you know you need it.\n");
	printf("\n");
	printf("Predictors when encoding:\n");
	printf(" -pr # Where # is:\n");
	printf("      0 Use Zero order predictors.\n");
	printf("      1 Use First order predictors.\n");
	printf("      2 Use Second order predictors.\n");
	printf("      3 Use Third order predictors.\n");
	printf("      4 Use predictive Predictors. The predictor uses the best\n");
	printf("      order predictors based on the LAST values.\n");
	printf("      5 Break the data stream down into frames, preprocess, and\n");
	printf("      use the best order predictors for that frame. Default.\n");
	printf(" -fr # Where # is the frame size you want to use. Default 1024.\n");
	printf("      There is no \"right\" number. Optimum values differ per\n");
	printf("      individual file.\n");
	printf("\n");
	printf("Encoders used to compress:\n");
	printf(" -en # Where # is:\n");
	printf("      1 Single K Pseudo Rice encoding.\n");
	printf("      2 Dual K Pseudo Rice encoding.\n");
	printf("      3 Dual K Pseudo Rice encoding with Variable History.\n");
	printf("      Defaiult. The Variable history size is setable via -kh\n");
	printf(" -kh # Where # is the K history size you want to use. Default 32.\n");
	printf("      There is no \"right\" number. Optimum values differ per\n");
	printf("      individual file.\n");
	printf("\n");
	return;
}

void display_copyright(void)
{
	printf("Kexis version %d.%d.%d, Copyright (C) 2000 Wayde Milas ",
		VERSION_MAJOR, VERSION_MINOR, VERSION_SUB_MINOR);
	printf("(wmilas@rarcoa.com)\n");
	printf("Kexis comes with ABSOLUTELY NO WARRANTY; for details see included ");
	printf("GPL license.\n");
	printf("This is free software, and you are welcome to redistribute it ");
	printf("under certain\nconditions; for details see included GPL license.");
	printf("\n");
}

int parse_arg(OPTIONSTRUCT *options, int argc, char *argv[])
{
	int loop =1;
	
	if (argc == 1) {
		display_copyright();
		useage();
		return 1;
	}

	// The default Mode
	options->inFileStream=NULL;
	options->outFileStream=NULL;
	options->inFileName=NULL;
	options->outFileName=NULL;
	options->mode = COMPRESS;
	options->progress = 0;
	options->displayHeader = 0;
	options->dec_stdout = 0;
	options->verbose = 0;
	options->del_file = 0;
	options->argPosition = 1;
	options->predictorVersion = STANDARD_PREDICTOR;
	options->encoderVersion = STANDARD_ENCODER;
	options->kHistorySize = DEFAULT_KHISTORY;
	options->frameSize = FRAME_UNITS;
	options->joint = 1;
	
	for(loop=1;loop < argc;loop++) {

		if(strcmp("-c", argv[loop]) == 0)
			options->mode = COMPRESS;
		else if(strcmp("-d", argv[loop]) == 0)
			options->del_file = 1;
		else if(strcmp("-x", argv[loop]) == 0)
			options->mode = DECOMPRESS;
		else if(strcmp("-s", argv[loop]) == 0) {
			options->mode = DECOMPRESS; 
			options->dec_stdout = 1;
		}
		else if(strcmp("-pr", argv[loop]) == 0) {
			loop++;
			sscanf(argv[loop], "%hd", &options->predictorVersion);
			options->predictorVersion++;
			if(options->predictorVersion < MIN_STANDARD_PREDICTOR ||
				options->predictorVersion > MAX_STANDARD_PREDICTOR) {
				printf("Predictors must be between %d and %d!\n",
					MIN_STANDARD_PREDICTOR-1, MAX_STANDARD_PREDICTOR-1);
				display_copyright();
				useage();
				return -1;
			}
		}
		else if(strcmp("-fr", argv[loop]) == 0) {
      loop++;
      sscanf(argv[loop], "%ld", &options->frameSize);
      if(options->frameSize < 64){
				printf("Frames must be at least 64 long!\n");
        display_copyright();
        useage();
        return -1;
      }
    }
		else if(strcmp("-en", argv[loop]) == 0) {
      loop++;
      sscanf(argv[loop], "%hd", &options->encoderVersion);
      if(options->encoderVersion < MIN_STANDARD_ENCODER ||
        options->encoderVersion > MAX_STANDARD_ENCODER) {
        printf("Encoders must be between %d and %d!\n",
          MIN_STANDARD_ENCODER, MAX_STANDARD_ENCODER);
        display_copyright();
        useage();
        return -1;
      }
    }	
		else if(strcmp("-kh", argv[loop]) == 0) {
			loop++;
			sscanf(argv[loop], "%d", &options->kHistorySize);
			if(options->kHistorySize < 1 || options->kHistorySize > MAX_KHISTORY){
				printf("K history Size must be between 1 and %d\n",MAX_KHISTORY);
				display_copyright();
				useage();
				return -1;
			}
		}
		else if(strcmp("-ns", argv[loop]) == 0)
			options->joint = 0;
		else if(strcmp("-p", argv[loop]) == 0)
			options->progress = 1;
		else if(strcmp("-h", argv[loop]) == 0)
      options->displayHeader = 1;
		else if(strcmp("-v", argv[loop]) == 0) {
			options->progress = 1;
			options->verbose = 1;
		}
		else
			break;
	}

	if(loop == argc) {
		display_copyright();
		useage();
		return -1;
	}

	options->argPosition = loop;
	return 0;
}

int parse_files(OPTIONSTRUCT *options, int argc, char *argv[])
{
	char *ext=NULL;

	// Check and see if we are in compress mode that the file ends in .wav
	// and .kex is decompress mode
	ext = strrchr(argv[options->argPosition], '.');
	if(ext != NULL) {
		if(options->mode == COMPRESS) {
			if(strcmp(ext, ".wav") != 0) {
				printf("Kexis can only compress files that end in a .wav!\n");
				return -1;
			}
		}
		else {
			if(strcmp(ext, ".kxs") != 0) {
				printf("Kexis can only uncompress files that end in a .kxs!\n");
				return -1;
			}
		}
	}
	else {
		if(options->mode == COMPRESS)
			printf("Kexis can only compress files that end in a .wav!\n");
		else
			printf("Kexis can only uncompress files that end in a .kxs!\n");
		return -1;
	}

	options->inFileName = calloc(1, strlen(argv[options->argPosition])+2);
	strcpy(options->inFileName, argv[options->argPosition]);

	return 0;
}

void print_progress(KEXISBLOCKSTRUCT *kexisBlock, unsigned long pcmLength,
  OPTIONSTRUCT *options, unsigned long totalLength, int mode)
{
  struct timeval currentTime;
  PREDICTORTABLE *pred;

	gettimeofday(&currentTime,0);
	if((currentTime.tv_sec - options->progTime.tv_sec > 0) ||
		(currentTime.tv_usec - options->progTime.tv_usec > 250000)) {

		options->progTime.tv_sec = currentTime.tv_sec;
		options->progTime.tv_usec = currentTime.tv_usec;

  	pred = &kexisBlock->predictor;
		if(mode == COMPRESS)
  		printf("Progress:%6.2f%%   ",
    		100.0-(((float)pcmLength*2.0)/(float)totalLength)*100);
 		else
			printf("Progress:%6.2f%%   ",
				(((float)pcmLength*2.0)/(float)totalLength)*100);

  	printf("Time: %ld:%.2ld   ", (currentTime.tv_sec - options->time.tv_sec)/60,
    	(currentTime.tv_sec - options->time.tv_sec)%60);
  	if(options->encoderVersion == RICE_SINGLE_K)
			if(mode == COMPRESS)
    		printf("K:%3d   Pd(mid,ave):%6d,%6d\r", pred->mid_K[0],
      		pred->diffMid, pred->diffAve);
			else
				printf("K:%3d   mid,ave:%6d,%6d\r", pred->mid_K[0],
					pred->diffMid, pred->diffAve);
  	else if(options->encoderVersion == RICE_DUAL_K) {
    	printf("K(m,a):%3d,%3d   ", pred->mid_K[0], pred->ave_K[0]);
			if(mode == COMPRESS)
    		printf("Pd(m,a):%6d,%6d\r", pred->diffMid, pred->diffAve);
			else
				printf("mid,ave:%6d,%6d\r", pred->diffMid, pred->diffAve);
  	}
  	else if(options->encoderVersion == RICE_DUAL_K_BLOCK) {
    	printf("K(m,a):%3d,%3d   ",
      	block_ave_K(pred->mid_K, options->kHistorySize),
      	block_ave_K(pred->ave_K, options->kHistorySize));
			if(mode == COMPRESS)
  			printf("Pd(m,a):%6d,%6d\r", pred->diffMid, pred->diffAve);
			else
				printf("mid,ave:%6d,%6d\r", pred->diffMid, pred->diffAve);
  	}
  	fflush(stdout);
	}
}

int main(int argc, char *argv[])
{
	WAVHEADER 				wavHead;
	KEXISHEADER				kexisHead;
	OPTIONSTRUCT			options;
	struct timeval 		currentTime;

	gettimeofday(&options.time,0);
	gettimeofday(&options.progTime,0);
	if(parse_arg(&options, argc, argv))
		exit(0);

	if((options.progress || options.displayHeader) && !options.dec_stdout)
		display_copyright();

	while(options.argPosition < argc) {
		if(parse_files(&options, argc, argv))
	  	exit(0);

		if(options.mode == COMPRESS)
			compress(&wavHead, &options);
	
		if(options.mode == DECOMPRESS)
			decompress(&kexisHead, &options);

		if(options.progress && !options.dec_stdout) {
			gettimeofday(&currentTime,0);
			printf("\nTotal elapsed time: %ld:%.2ld\n", 
				(currentTime.tv_sec - options.time.tv_sec)/60,
				(currentTime.tv_sec - options.time.tv_sec)%60);
		}

		options.argPosition++;
	}

	exit_nicely(&options);
	exit(0);
}

void handle_verbose(OPTIONSTRUCT *options, KEXISBLOCKSTRUCT *kexisBlock,
	PCMBLOCKSTRUCT *pcmBlock)
{
	struct rusage usage;

	if(options->progress && options->mode==COMPRESS && !options->dec_stdout)
		printf("\nCompression:%6.2f%% (of original file size)",
			(float) 100.0 * 
			((float)(kexisBlock->sumTotalSize*2)/
			 	((float)pcmBlock->pcmStreamLength/2)));

	if(options->verbose && !options->dec_stdout) {
		getrusage(RUSAGE_SELF,&usage);
		printf("\nTotal User Time  : %ld seconds, %ld ms.\n",
	  	usage.ru_utime.tv_sec, usage.ru_utime.tv_usec/1000);
		printf("Total System Time: %ld seconds, %ld ms.\n",
			usage.ru_stime.tv_sec, usage.ru_stime.tv_usec/1000);
		printf("Low Pd mid: %d\tHigh Pd mid: %d\tAverage Pd mid: %9.3f\n",
			kexisBlock->predictor.lowDiffMid,
			kexisBlock->predictor.highDiffMid,
			kexisBlock->predictor.sumDiffMid/kexisBlock->predictor.countDiffMid);
		printf("Low Pd ave: %d\tHigh Pd ave: %d\tAverage Pd ave: %9.3f\n",
		  kexisBlock->predictor.lowDiffAve,
			kexisBlock->predictor.highDiffAve,
			kexisBlock->predictor.sumDiffAve/
			kexisBlock->predictor.countDiffAve);
		if(options->encoderVersion == RICE_SINGLE_K)
			printf("Low K     : %d\t\tHigh K     : %d\t\tAverage K     : %9.3f",
				kexisBlock->predictor.midLowK, kexisBlock->predictor.midHighK,
				kexisBlock->predictor.midSumK/kexisBlock->predictor.midCountK);
		else if(options->encoderVersion == RICE_DUAL_K ||
			options->encoderVersion == RICE_DUAL_K_BLOCK) {
			printf("Low K mid : %d\t\tHigh K mid : %d\t\tAverage K mid : %9.3f\n",
				kexisBlock->predictor.midLowK, kexisBlock->predictor.midHighK,
				kexisBlock->predictor.midSumK/kexisBlock->predictor.midCountK);
			printf("Low K ave : %d\t\tHigh K ave : %d\t\tAverage K ave : %9.3f",
			  kexisBlock->predictor.aveLowK, kexisBlock->predictor.aveHighK,
				kexisBlock->predictor.aveSumK/kexisBlock->predictor.aveCountK);
	 	}
	}
}

void handle_error(OPTIONSTRUCT *options)
{
	printf("\n%s\n", options->errorString);
	printf("Was not able to finish. Exiting...\n");
	exit_nicely(options);
}

void exit_nicely(OPTIONSTRUCT *options)
{
	exit(0);
}

void free_allocated(PCMBLOCKSTRUCT *pcmBlock, KEXISBLOCKSTRUCT *kexisBlock,
	OPTIONSTRUCT *options)
{
	int hold;

	// check and see if we need to delete an encoded file
	if(options->mode == COMPRESS && options->del_file &&
		options->inFileName != NULL) {
		hold = unlink(options->inFileName);
		if(hold == -1)
			printf("Was not able to delete file %s.\n", options->inFileName);
	}
	
	if(options->inFileStream != NULL)
    fclose(options->inFileStream);
  if(options->outFileStream != NULL && !options->dec_stdout)
    fclose(options->outFileStream);

  if(options->inFileName != NULL)
    free(options->inFileName);
  if(options->outFileName != NULL && !options->dec_stdout)
    free(options->outFileName);

	if(kexisBlock->predictor.mid_K != NULL)
  	free(kexisBlock->predictor.mid_K);
	if(kexisBlock->predictor.ave_K != NULL)	
		free(kexisBlock->predictor.ave_K);
	if(kexisBlock->predictor.newKLookUP != NULL)	
		free(kexisBlock->predictor.newKLookUP);

	if(kexisBlock->data != NULL)
		free(kexisBlock->data);
	if(pcmBlock->data != NULL)
		free(pcmBlock->data);

	options->inFileStream = NULL;
	if(options->dec_stdout == 0)
		options->outFileStream = NULL;
	options->inFileName = NULL;
	if(options->dec_stdout == 0)
		options->outFileName = NULL;
	kexisBlock->predictor.mid_K = NULL;
	kexisBlock->predictor.ave_K = NULL;
	kexisBlock->predictor.newKLookUP = NULL;
	kexisBlock->data = NULL;
	pcmBlock->data = NULL;
}
