/* rice.c
 * Pseudo Rice encoding of an integer stream
 * Copyright (C) 2000 Wayde Milas (wmilas@rarcoa.com)
 *
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "types.h"
#include "rice.h"
#include "header.h"
#include "bits.h"

void rice_decode(KEXISBLOCKSTRUCT *kexisBlock, long *data,
	OPTIONSTRUCT *options, PCMBLOCKSTRUCT *pcmBlock, int kType)
{
  long sign, loop, zeroes=0, newK;
	int	K;

	K = get_K(&kexisBlock->predictor, kType, options->encoderVersion,
		options->kHistorySize);
  sign = bit_suck(kexisBlock, 1, pcmBlock, options);

  loop = bit_suck(kexisBlock, 1, pcmBlock, options);
  while(loop != 1) {
    zeroes++;
    loop = bit_suck(kexisBlock, 1, pcmBlock, options);
  }

	if(K != 0)
  	*data = bit_suck(kexisBlock, K, pcmBlock, options);
	else
		*data = 0;
	
	// move zeroes, which is an actual number now, into the prefix location,
	// ie, at the high end of the word.
  zeroes <<= K;
  *data = *data | zeroes;

	newK = calc_newK(*data, &kexisBlock->predictor);
	update_K(&kexisBlock->predictor, newK, kType, options->encoderVersion,
		options->kHistorySize);

	// collect K statistics if we need to
	if(options->verbose && !options->dec_stdout)
		collect_K_stats(&kexisBlock->predictor, newK, kType,
			options->encoderVersion);

 	if(sign == 0)
    *data = -*data;

	//debug_prints(0,0, kexisBlock->sumTotalSize,-1,-1,1);
	//debug_prints(*data,32, kexisBlock->sumTotalSize,0,0,0);
	//debug_prints(0,0, kexisBlock->sumTotalSize,-1,-1,1);
}

void rice_encode(KEXISBLOCKSTRUCT *kexisBlock, long data,
	OPTIONSTRUCT *options, PCMBLOCKSTRUCT *pcmBlock, int kType)
{
  long  holdData, loop, newK, remainder=0;
	int		K;

  K = get_K(&kexisBlock->predictor, kType, options->encoderVersion,
		options->kHistorySize);

	// Push the first sign bit
  if(data < 0)
    push_bits(kexisBlock, 0, 1, options, pcmBlock);
  else
    push_bits(kexisBlock, 1, 1, options, pcmBlock);

  // Take the absolute value
  holdData = abs(data);
	newK = calc_newK(holdData, &kexisBlock->predictor);

  // Figure out how many zeroes we need to pop in.
  loop = holdData >> K;
  while(loop != 0) {
    push_bits(kexisBlock, 0, 1, options, pcmBlock);
    loop--;
  }

  //push a seperatior 1
  push_bits(kexisBlock, 1, 1, options, pcmBlock);

	// Only push bits if K is not 0.
	if(K != 0) {
  // next put in the last K bits.
  remainder = remainder | (holdData & ((1<<K) -1));
  push_bits(kexisBlock, remainder, K, options, pcmBlock);
	}

	//debug_prints(0,0, kexisBlock->debugBlockCount, -1, -1, 1);
	update_K(&kexisBlock->predictor, newK, kType, options->encoderVersion,
		options->kHistorySize);

	// collect K statistics if we need to
	if(options->verbose)
		collect_K_stats(&kexisBlock->predictor, newK, kType,
			options->encoderVersion);

	//if(DEBUG == 1 || PRED_DEBUG == 1)
	//	kexisBlock->debugBlockCount++;
}

void collect_K_stats(PREDICTORTABLE *pred, long newK, int kType,
	unsigned short encoderVersion)
{
	if((encoderVersion == RICE_DUAL_K || encoderVersion == RICE_DUAL_K_BLOCK)
		&& kType == AVE) {
		if(newK < pred->aveLowK)
			pred->aveLowK = newK;
	  if(newK > pred->aveHighK)
	    pred->aveHighK = newK;

		pred->aveCountK++;
		pred->aveSumK += newK;
	}
	else {
		if(newK < pred->midLowK)
			pred->midLowK = newK;
		if(newK > pred->midHighK)
			pred->midHighK = newK;
	
		pred->midCountK++;
		pred->midSumK += newK;
	}
}

long get_K(PREDICTORTABLE *pred, int kType, unsigned short encoderVersion,
	int kHist)
{
	if(kType == MID) {
		if(encoderVersion == RICE_SINGLE_K || encoderVersion == RICE_DUAL_K)
			return pred->mid_K[0];
		else if(encoderVersion == RICE_DUAL_K_BLOCK) {
			return block_ave_K(pred->mid_K, kHist);
		}
	}
	else {
		if(encoderVersion == RICE_SINGLE_K)
			return pred->mid_K[0];
		else if(encoderVersion == RICE_DUAL_K)
			return pred->ave_K[0];
		else if(encoderVersion == RICE_DUAL_K_BLOCK) {
			return block_ave_K(pred->ave_K, kHist);
		}
	}
	return 0; // to stop compiler warnings
}

int block_ave_K(int table[], int kHist)
{
	int sum =0;
	int	loop;

	for(loop=0;loop<kHist;loop++) sum+= table[loop];
	//printf("%d %f\n", sum, ceil(sum/kHist));
	return ceil((float)sum/kHist);
}

long calc_newK(long holdData, PREDICTORTABLE *pred)
{
	long	newK;

	if(holdData == 0)
		newK = 0;
	else
		//newK = log10(holdData)/pred->preLog10_2;
		newK = lookup_newK(holdData, pred);
	
	return newK;
}

void update_K(PREDICTORTABLE *pred, long newK, int kType,
	unsigned short encoderVersion, int kHist)
{
	if(kType == MID) {
		if(encoderVersion == RICE_SINGLE_K || encoderVersion == RICE_DUAL_K)
			pred->mid_K[0] = ceil((pred->mid_K[0] + newK)/2.0);
		else if(encoderVersion == RICE_DUAL_K_BLOCK)
			pred->mid_K[pred->kBlockPointer] = newK;
	}
	else {
		if(encoderVersion == RICE_SINGLE_K)
			pred->mid_K[0] = ceil((pred->mid_K[0] + newK)/2.0);
		else if(encoderVersion == RICE_DUAL_K)
			pred->ave_K[0] = ceil((pred->ave_K[0] + newK)/2.0);
		else if(encoderVersion == RICE_DUAL_K_BLOCK) {
			pred->ave_K[pred->kBlockPointer] = newK;
			pred->kBlockPointer++;
			//pred->kBlockPointer = pred->kBlockPointer % kHist;
			if(pred->kBlockPointer == kHist)
				pred->kBlockPointer = 0;
		}
	}
}

long lookup_newK(long holdData, PREDICTORTABLE *pred)
{
	if(holdData <= K_LOOKUP_SIZE) {
		if(pred->newKLookUP[holdData] != -1)
			return pred->newKLookUP[holdData];
		else {
			pred->newKLookUP[holdData] = log10(holdData)/pred->preLog10_2;
			return pred->newKLookUP[holdData];
		}
	}
	else
		return log10(holdData)/pred->preLog10_2;
}
