// $Id: pspfile.cpp,v 1.2 2006/01/27 18:48:33 robin_d Exp $

/***************************************************************************
 *   Copyright (C) 2006 by Robin Doer                                      *
 *   robin@robind.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <kurl.h>
#include <klocale.h>
#include <kmimetype.h>
#include <kiconloader.h>
#include <kdebug.h>

#include <qfileinfo.h>
#include <qdir.h>

#include "pspfile.h"
#include "pspprotocol.h"

/**
 * Directories of different mediatypes. Might be empty (virtual folder).
 */
QString mediatype2dir[] = {
  "",
  "",
  "photo",
  "video", // !!! Unsure !!!
  "music",
  "savedata"
};

/**
 * Returns stat-struct of the given file.
 * NOTE: The file must exist!!
 */
inline struct stat filestat(const char* file) {
  struct stat fstat;
  stat(file, &fstat);
  return fstat;
}

/* PSPFile::PSPFile (1) ****************************************************/
PSPFile::PSPFile(PSPProtocol* prot, const KURL& url) {
  KURL tempUrl = url;
  tempUrl.cleanPath(true);

  parseURL(tempUrl.path(), m_type, m_pspPath);

  if (m_type == Unsupported)
    return;

  m_protocol = prot;

  initMediaPath();

  setName(m_mediaPath + m_pspPath);

  kdDebug(7002) << "PSPFile::PSPFile(1): " << name() << endl;
}

/* PSPFile::PSPFile (2) ****************************************************/
PSPFile::PSPFile(const PSPFile& parent, const QString& path) {
  m_protocol = parent.m_protocol;
  m_type = parent.m_type;
  m_mediaPath = parent.m_mediaPath;

  m_pspPath = parent.m_pspPath;
  if (!m_pspPath.endsWith("/"))
    m_pspPath += "/";

  m_pspPath += path;

  setName(m_mediaPath + m_pspPath);
  kdDebug(7002) << "PSPFile::PSPFile(2): " << name() << endl;
}

/* PSPFile::PSPFile (3) ****************************************************/
PSPFile::PSPFile(PSPProtocol* prot, MediaType type, const QString& path) {
  m_protocol = prot;
  m_type = type;
  m_pspPath = path;

  initMediaPath();

  setName(m_mediaPath + m_pspPath);

  kdDebug(7002) << "PSPFile::PSPFile(3): " << name() << endl;
}

/* PSPFile::PSPFile (4) ****************************************************/
PSPFile::PSPFile(const PSPFile& other)
  : QFile() {

  m_protocol = other.m_protocol;
  m_type = other.m_type;
  m_mediaPath = other.m_mediaPath;
  m_pspPath = other.m_pspPath;

  setName(m_mediaPath + m_pspPath);
}

/* PSPFile::type ***********************************************************/
PSPFile::MediaType PSPFile::type() const {
  return m_type;
}

/* PSPFile::pspPath ********************************************************/
QString PSPFile::pspPath() const {
  return m_pspPath;
}

/* PSPFile::pspExists ******************************************************/
bool PSPFile::pspExists() const {
  return (m_type == Empty) || // Root
         (m_type != Unsupported && m_pspPath.isEmpty()) || // Media-basedirs
         (m_type != Unsupported && !m_pspPath.isEmpty() && exists());
}

/* PSPFile::udsEntry *******************************************************/
KIO::UDSEntry PSPFile::udsEntry() const {
  KIO::UDSEntry entry;

  if (m_pspPath.isEmpty())
    return virtualEntry(m_type);

  QFileInfo fi(*this);
  if (fi.exists()) {
    KMimeType::Ptr mimetype = KMimeType::findByURL(name());
    struct stat fstat = filestat(name().data());

    addAtom(entry, KIO::UDS_NAME, 0, fi.fileName());
    addAtom(entry, KIO::UDS_SIZE, fstat.st_size, "");
    addAtom(entry, KIO::UDS_USER, 0, fi.owner());
    addAtom(entry, KIO::UDS_GROUP, 0, fi.group());
    addAtom(entry, KIO::UDS_ACCESS, fstat.st_mode, "");
    addAtom(entry, KIO::UDS_FILE_TYPE, fstat.st_mode, "");

    QString url = "psp:/" + type2str(m_type);
    if (m_pspPath.startsWith("/"))
      url += m_pspPath;
    else
      url += "/" + m_pspPath;

    addAtom(entry, KIO::UDS_URL, 0, url);
    addAtom(entry, KIO::UDS_MIME_TYPE, 0, mimetype->name());
  }

  return entry;
}

KIO::UDSEntryList PSPFile::udsChildren() const {
  KIO::UDSEntryList udsEntries;

  if (m_type == Empty) {
    PSPFile picture(m_protocol, Picture, "");
    PSPFile music(m_protocol, Music, "");
    PSPFile video(m_protocol, Video, "");
    PSPFile savegame(m_protocol, SaveGame, "");

    udsEntries << picture.udsEntry()
               << music.udsEntry()
               << video.udsEntry()
               << savegame.udsEntry();
  }
  else {
    QDir dir(name());
    QStringList children = dir.entryList();

    for (QStringList::const_iterator it = children.begin();
         it != children.end(); ++it) {

      QString fileName = (*it);
      if (fileName != "." && fileName != "..") {
        PSPFile child(*this, fileName);
        udsEntries << child.udsEntry();
      }
    }
  }

  return udsEntries;
}

void PSPFile::initMediaPath() {
  // Filename begins with mountpoint
  m_mediaPath = m_protocol->mountPoint();
  if (!m_mediaPath.endsWith("/"))
    m_mediaPath += "/";

  // Media-data are stored in a "psp"-subfolder
  m_mediaPath += "psp/";

  // Followed by dir specified by mediatype
  m_mediaPath += mediatype2dir[m_type];
  if (!m_mediaPath.endsWith("/"))
    m_mediaPath += "/";
}

/* PSPFile::type2str *******************************************************/
QString PSPFile::type2str(MediaType type) {
  static QString mediatype2str[] = {
    i18n("Unsupported"),
    i18n("Overview"),
    i18n("Pictures"),
    i18n("Videos"),
    i18n("Music"),
    i18n("SaveGames")
  };

  return mediatype2str[type];
}

/* PSPFile::parseURL *******************************************************/
void PSPFile::parseURL(const QString& urlPath, MediaType& type,
                       QString& path) {

  int firstSep = urlPath.find('/', 1); // Cut at first slash

  QString typestr = urlPath.mid(1, firstSep - 1);
  if (typestr == type2str(Picture))
    type = Picture;
  else if (typestr == type2str(Video))
    type = Video;
  else if (typestr == type2str(Music))
    type = Music;
  else if (typestr == type2str(SaveGame))
    type = SaveGame;
  else if (typestr.isEmpty())
    type = Empty;
  else
    type = Unsupported;

  path = urlPath.mid(firstSep);
  if (path.startsWith("/"))
    path = path.mid(1);
}

/* PSPFile::virtualEntry ***************************************************/
KIO::UDSEntry PSPFile::virtualEntry(MediaType type) {
  KIO::UDSEntry entry;

  int access = S_IRWXU|S_IRGRP|S_IXGRP; // rwxr-x---

  addAtom(entry, KIO::UDS_NAME, 0, type2str(type));
  addAtom(entry, KIO::UDS_FILE_TYPE, S_IFDIR, "");
  addAtom(entry, KIO::UDS_URL, 0, "psp:/" + type2str(type));
  addAtom(entry, KIO::UDS_ACCESS, access, "");

  QString pic;
  switch (type) {
    case Picture : pic = "folder_image"; break;
    case Video   : pic = "folder_video"; break;
    case Music   : pic = "folder_sound"; break;
    case SaveGame: pic = "joystick"; break;
    default      : pic = ""; break;
  }

  if (!pic.isEmpty())
    addAtom(entry, KIO::UDS_ICON_NAME, 0, pic);

  return entry;
}

/* PSPFile::addAtom ********************************************************/
void PSPFile::addAtom(KIO::UDSEntry& entry, KIO::UDSAtomTypes m_uds,
                      long long m_long, const QString& m_str) {

  KIO::UDSAtom atom;
  atom.m_uds = m_uds;
  atom.m_long = m_long;
  atom.m_str = m_str;
  entry.append(atom);
}
