// $Id: pspfile.h,v 1.2 2006/01/27 18:47:40 robin_d Exp $

/***************************************************************************
 *   Copyright (C) 2006 by Robin Doer                                      *
 *   robin@robind.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PSPFILE_H
#define PSPFILE_H

#include <qfile.h>

#include <kio/global.h>

class KURL;
class PSPProtocol;

/**
 * PSP-QFile-extension. Is responsibe for translation from virtual filename
 * from the URL to the real location. Provides an UDSEntry for each file
 * and children.
 *
 * @author Robin Doer
 */
class PSPFile : public QFile {
public:
  /**
   * MediaTypes supported by PSP.
   */
  enum MediaType {
    Unsupported = 0, /**< Indicates an unsupported type */
    Empty       = 1, /**< Overview-level */
    Picture     = 2, /**< Pictures */
    Video       = 3, /**< Videos */
    Music       = 4, /**< Music */
    SaveGame    = 5  /**< SameGames */
  };

  /**
   * Constructs a PSPFile from the given URL.
   */
  PSPFile(PSPProtocol* prot, const KURL& url);

  /**
   * Constructs a PSPFile with the given name and parent file.
   */
  PSPFile(const PSPFile& parent, const QString& path);

  PSPFile(PSPProtocol* prot, MediaType type, const QString& path);

  PSPFile(const PSPFile& other);

  /**
   * Returns the mediatype of the file. Might be also MediaType::Unsupported
   * in case of the user chose the wrong mediatype.
   */
  MediaType type() const;

  /**
   * Returns the path of the file relative to the PSP-mountpoint.
   */
  QString pspPath() const;

  /**
   * Tests weather the file exists (from PSP point-of-view). The file
   * needn't exist physically.
   */
  bool pspExists() const;

  /**
   * Returns the UDSEntry of the file.
   */
  KIO::UDSEntry udsEntry() const;

  /**
   * Returns the UDSEntryList of the file's children.
   */
  KIO::UDSEntryList udsChildren() const;

private:
  void initMediaPath();

  /**
   * Converts the given mediatype to a human-readable string.
   */
  static QString type2str(MediaType type);

  /**
   * Parses the given URL. The first part of the URL contains the mediatype.
   * The rest of the path is interpreted as the path to the resource.
   * @param url URL to parse
   * @param type Reference to a MediaType. The method writes into the
   *             reference.
   * @param path Reference to the path. The method writes into the reference.
   */
  static void parseURL(const QString& urlPath, MediaType& type, QString& path);

  /**
   * Creates an UDSEntry of a virtual folder. This folder might not really
   * exist.
   */
  static KIO::UDSEntry virtualEntry(MediaType type);

  /**
   * Appends an UDSAtom to the UDSEntry.
   */
  static void addAtom(KIO::UDSEntry& entry, KIO::UDSAtomTypes m_uds,
                      long long m_long, const QString& m_str);

private:
  /**
   * Association to the protocol-implementation
   */
  PSPProtocol* m_protocol;

  /**
   * Mediatype of file
   */
  MediaType m_type;

  /**
   * Basepath of media-type.
   */
  QString m_mediaPath;

  /**
   * Path-information
   */
  QString m_pspPath;
};

#endif  // PSPFILE_H
