// $Id: pspprotocol.cpp,v 1.2 2006/02/01 17:10:43 robin_d Exp $

/***************************************************************************
 *   Copyright (C) 2006 by Robin Doer                                      *
 *   robin@robind.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>

#include <kmimetype.h>
#include <klocale.h>
#include <kinstance.h>
#include <kglobal.h>
#include <kdirwatch.h>
#include <kdebug.h>

#include <qfile.h>
#include <qdir.h>

#include "pspprotocol.h"
#include "pspfile.h"
#include "mountpoint.h"
#include "mountpointlist.h"

inline bool rename_file(const char* src, const char* dest) {
  return (rename(src, dest) == 0);
}

inline bool mkdirs(const QDir& dir) {
  QStringList path = QStringList::split('/', dir.absPath(), false);
  QDir d = QDir::root();

  for (QStringList::const_iterator it = path.begin();
       it != path.end(); ++it) {

    if (!d.cd(*it)) { // Not exists or isn't readable, try to create
      if (d.mkdir(*it)) { // Well done!
        if (!d.cd(*it))
          return false;
      }
      else // Could not create
        return false;
    }
  }

  return true;
}

/* PSPProtocol::PSPProtocol ************************************************/
PSPProtocol::PSPProtocol(const QCString& pool, const QCString& app)
  : SlaveBase("psp", pool, app) {

  m_isMounted = false;
  m_mountPointList = new MountPointList(this);
  connect(m_mountPointList, SIGNAL(changed()),
          this, SLOT(scanForMountPoint()));
  scanForMountPoint();
}

/* PSPProtocol::~PSPProtocol ***********************************************/
PSPProtocol::~PSPProtocol() {
  kdDebug(7002) << "PSPProtocol destroyed" << endl;
}

/* PSPProtocol::get ********************************************************/
void PSPProtocol::get(const KURL& url) {
  PSPFile file(this, url);

  if (file.type() == PSPFile::Unsupported) {
    error(KIO::ERR_SLAVE_DEFINED, i18n("Unsupported mediatype!"));
    return;
  }

  kdDebug(7002) << "PSPProtocol::get: " << file.name() << endl;

  // Can only read really existing files, therefore check with exists()
  if (file.exists()) {
    // Mimetype of file
    KMimeType::Ptr type = KMimeType::findByURL(file.name());
    mimeType(type->name());

    if (!file.open(IO_ReadOnly)) {
      error(KIO::ERR_CANNOT_OPEN_FOR_READING, "");
      return;
    }

    // TODO: Reads chunks of data
    data(file.readAll());
    file.close();

    finished();
  }
  else
    error(KIO::ERR_DOES_NOT_EXIST, "");
}

/* PSPProtocol::put ********************************************************/
void PSPProtocol::put(const KURL& url, int permissions,
                      bool overwrite, bool resume) {
  PSPFile file(this, url);

  if (file.exists()) {
    if (!overwrite) {
      error(KIO::ERR_FILE_ALREADY_EXIST, "");
      return;
    }

    if (resume && !canResume(file.size())) {
      error(KIO::ERR_CANNOT_RESUME, "");
      return;
    }
  }
  else {
    // Make sure, the parent directory exists
    QFileInfo fi(file);
    QDir dir = fi.dir();

    if (!dir.exists() && !mkdirs(dir)) {
      error(KIO::ERR_COULD_NOT_MKDIR, fi.fileName());
      return;
    }
  }

  int openmode = (resume) ? (IO_WriteOnly | IO_Append)
                          : (IO_WriteOnly | IO_Truncate);

  if (file.open(openmode)) {
    dataReq(); // Ask for data

    QByteArray buffer;
    int bytes;
    while ((bytes = readData(buffer)) != 0) {
      if (bytes < 0) {
        error(KIO::ERR_COULD_NOT_WRITE, "");
        file.close();
        return;
      }

      file.writeBlock(buffer.data(), bytes);
      dataReq(); // Ask for more data
    }

    file.close();
    finished();
  }
  else
    error(KIO::ERR_CANNOT_OPEN_FOR_WRITING, "");
}

/* PSPProtocol::del ********************************************************/
void PSPProtocol::del(const KURL& url, bool isFile) {
  PSPFile file(this, url);

  if (file.pspPath().isEmpty()) {
    error(KIO::ERR_SLAVE_DEFINED, i18n("Cannot remove this folder!"));
    return;
  }

  QFileInfo fi(file.name());

  if (fi.isFile() && isFile) {
    if (file.remove())
      finished();
    else
      error(KIO::ERR_CANNOT_DELETE, "");
  }
  else if (fi.isDir() && !isFile) {
    QDir dir(file.name());
    if (dir.rmdir(file.name()))
      finished();
    else
      error(KIO::ERR_COULD_NOT_RMDIR, "");
  }
  else {
    error(KIO::ERR_DOES_NOT_EXIST, "");
  }
}

/* PSPProtocol::rename *****************************************************/
void PSPProtocol::rename(const KURL& src, const KURL& dest, bool overwrite) {
  PSPFile srcFile(this, src);
  PSPFile destFile(this, dest);

  if (destFile.exists() && !overwrite) {
    error(KIO::ERR_FILE_ALREADY_EXIST, "");
    return;
  }

  if (rename_file(srcFile.name().data(), destFile.name().data()))
    finished();
  else
    error(KIO::ERR_CANNOT_RENAME, "");
}

/* PSPProtocol::mkdir ******************************************************/
void PSPProtocol::mkdir(const KURL& url, int permissions) {
  PSPFile file(this, url);

  if (file.exists()) {
    error(KIO::ERR_FILE_ALREADY_EXIST, "");
    return;
  }

  QDir dir(file.name());
  if (mkdirs(dir))
    finished();
  else {
    QFileInfo fi(dir.absPath());
    error(KIO::ERR_COULD_NOT_MKDIR, fi.fileName());
  }
}

/* PSPProtocol::listDir ****************************************************/
void PSPProtocol::listDir(const KURL& url) {
  PSPFile file(this, url);

  if (file.type() == PSPFile::Unsupported) {
    error(KIO::ERR_SLAVE_DEFINED, i18n("Unsupported mediatype!"));
    return;
  }

  kdDebug(7002) << "PSPProtocol::listDir: " << file.name() << endl;

  // Check existence with pspExists(), it might be a virtual folder
  if (file.pspExists()) {
    listEntries(file.udsChildren());
    finished();
  }
  else
    error(KIO::ERR_DOES_NOT_EXIST, "");
}

/* PSPProtocol::stat *******************************************************/
void PSPProtocol::stat(const KURL& url) {
  if (!m_isMounted)
    error(KIO::ERR_SLAVE_DEFINED, i18n("Your PSP is not mounted!"));

  PSPFile file(this, url);
  kdDebug(7002) << "PSPProtocol::stat: " << file.name() << endl;

  if (file.type() == PSPFile::Unsupported) {
    error(KIO::ERR_SLAVE_DEFINED, i18n("Unsupported mediatype!"));
    return;
  }

  // Check existence with pspExists(), it might be a virtual folder
  if (file.pspExists()) {
    statEntry(file.udsEntry());
    finished();
  }
  else
    error(KIO::ERR_DOES_NOT_EXIST, "");
}

/* PSPProtocol::mountPoint *************************************************/
QString PSPProtocol::mountPoint() const {
  return m_mountPoint;
}

void PSPProtocol::scanForMountPoint() {
  // Scan for PSP-mount
  for (MountPointList::const_iterator it = m_mountPointList->begin();
       it != m_mountPointList->end(); ++it) {

    MountPoint mp = (*it);
    // Search for folder "psp", it's on the top-level
    QString pspFolder = mp.mountPoint() + "/psp";
    QFileInfo fi(pspFolder);

    if (fi.isDir()) {
      // This is the PSP!!!
      m_mountPoint = mp.mountPoint();
      m_isMounted = true;
      kdDebug(7002) << "PSPProtocol::scanForMountPoint: PSP mounted on"
                    << m_mountPoint << endl;
      return;
    }
  }

  m_isMounted = false;
  m_mountPoint = "";
  error(KIO::ERR_SLAVE_DEFINED, i18n("Your PSP is not mounted!"));
}

/* kdemain *****************************************************************/
extern "C" int kdemain(int argc, char* argv[]) {
  KLocale::setMainCatalogue("kdelibs");
  KInstance instance("kio_psp");
  KGlobal::locale()->insertCatalogue("kio_psp");;

  if (argc != 4) {
    fprintf(stderr, "Usage: kio_psp protocol "
                    "domain-socket1 domain-socket2\n");
    exit(-1);
  }

  PSPProtocol slave(argv[2], argv[3]);
  slave.dispatchLoop();
  return 0;
}
