/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kdeversion.h> // Detect KDE 3.5

#include "coverdisplay.h"
#include "progressbar.h"
#include "amarokapi.h"
#include "dcopinterface.h"
#include "playerinformation.h"
#include "taskbarwatcher.h"
#include "tools.h"
#include "thememanager.h"

#include <kapplication.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qimage.h>
#include <kimageeffect.h>
#include <qtooltip.h>
#include <qcursor.h>
#include <qdatetime.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <kiconloader.h>
#include <qdir.h>
#include <krun.h>
#include <kurl.h>
#include <qstylesheet.h>
#include <kglobal.h>
#include <klocale.h>
#include <kglobalsettings.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kwin.h>
#include <ksimpleconfig.h>
#include <qtextbrowser.h>

#include "pixmapcache.h"

#include "applet.h"
#include "settings.h"

#include <iostream>

const int FULL_SCREEN_TIMOUT = 3000;

const int CONTROL_BUTTONS_MAX_SIZE = 64;
const int STAR_IMAGE_SIZE          = 128;

const int TASKBAR_MARGIN = 4;

const QRect HIDDEN_RECT = QRect(-100, -100, 1, 1);

const bool ANIMATE_BUTTONS = true;
const bool ANIMATE_TRACKS  = true;

CoverDisplay::CoverDisplay(QWidget *parent, Applet *applet/* = 0*/)
 : QWidget(parent, 0, Qt::WNoAutoErase),
   m_underMouse(false),
   m_progressBarOnLeft(false), m_withReflexion(true),
   m_hoveredStar(-1), m_hoveredButton(-1), m_clickedButton(-1), m_hoveredTaskBarEntry(-1),
   m_orientation(Vertical), m_margin(0), m_progressMargin(0), m_buttonPadding(0), m_reflectionDistance(0),
   m_buttonsAnimator(500/*ms*/), m_playPauseAnimator(500/*ms*/),
   m_lyrics(0), m_ignoreNextLyricsMove(false),
   m_trackAnimator(300/*ms*/), m_reverseTrackAnimation(false), m_oldRating(0), m_oldDuration(0), m_oldPosition(0),
   m_applet(applet),
   m_nextPlayingRichText(0),
   m_taskBarWatcher(0),
   m_clockRichText(0), m_clockInitialized(false)
{
	// Make sure the next heightForWidth or widthForHeight will change the size:
	resize(1, 1);

	m_progressBar = new ProgressBar(this);
	m_progressBar->hide();
	connect( m_progressBar, SIGNAL(changePosition(int)), this, SLOT(setPosition(int)) );
	m_progressBar->installEventFilter(this);

	connect( &m_fullScreenTimer, SIGNAL(timeout()), this, SLOT(hideControls()) );

	m_infos = PlayerInformation::instance();
	connect( m_infos, SIGNAL(newInformation()),              this, SLOT(newInformation())              );
	connect( m_infos, SIGNAL(newRating()),                   this, SLOT(newRating())                   );
	connect( m_infos, SIGNAL(newProgress()),                 this, SLOT(newProgress())                 );
	connect( m_infos, SIGNAL(newLyrics()),                   this, SLOT(newLyrics())                   );
	connect( m_infos, SIGNAL(aboutToChangeTrack(int, bool)), this, SLOT(beforeAnimateTrack(int, bool)) );

	m_buttonsAnimator.setRange(0, 100);
	m_playPauseAnimator.setRange(0, 100);
	connect( &m_buttonsAnimator,   SIGNAL(newValue(int)), this, SLOT(updateStars())   );
	connect( &m_buttonsAnimator,   SIGNAL(newValue(int)), this, SLOT(updateButtons()) );
	connect( &m_playPauseAnimator, SIGNAL(newValue(int)), this, SLOT(updateButtons()) );

	m_trackAnimator.setRange(0, 100);
	connect( &m_trackAnimator, SIGNAL(beforeNewValue()), this, SLOT(updateCover())        );
	connect( &m_trackAnimator, SIGNAL(beforeNewValue()), this, SLOT(updateStars()  )      );
	connect( &m_trackAnimator, SIGNAL(beforeNewValue()), this, SLOT(updateText())         );
	connect( &m_trackAnimator, SIGNAL(beforeNewValue()), this, SLOT(updateProgressDraw()) ); // updateProgresBar() can only called to refresh animation (non-widget)
	connect( &m_trackAnimator, SIGNAL(newValue(int)),    this, SLOT(updateCover())        );
	connect( &m_trackAnimator, SIGNAL(newValue(int)),    this, SLOT(updateStars())        );
	connect( &m_trackAnimator, SIGNAL(newValue(int)),    this, SLOT(updateText())         );
	connect( &m_trackAnimator, SIGNAL(newValue(int)),    this, SLOT(updateProgressDraw()) ); // updateProgresBar() can only called to refresh animation (non-widget)

	connect( &m_taskBarFlickerTimer, SIGNAL(timeout()), this, SLOT(flickerTaskBar()) );

#if KDE_IS_VERSION( 3, 5, 0 )
	installEventFilter(KickerTip::the());
#endif
}

CoverDisplay::~CoverDisplay()
{
}

bool CoverDisplay::eventFilter(QObject *object, QEvent *event)
{
	if (object == m_progressBar) {
		if (event->type() == QEvent::MouseMove) {
			QMouseEvent *thisEvent = (QMouseEvent *) event;
			QMouseEvent *newEvent = new QMouseEvent(thisEvent->type(), thisEvent->pos() + m_progressBar->pos(), thisEvent->button(), thisEvent->state());
			mouseMoveEvent(newEvent);
			delete newEvent;
		}
	} else if ( m_lyrics && (object == m_lyrics ||
	                         object == m_lyrics->viewport() ||
	                         object == m_lyrics->verticalScrollBar() ||
	                         object == m_lyrics->horizontalScrollBar()) ) {
		if (event->type() == QEvent::MouseMove) {
			QWidget *widget = (QWidget *) object;
			QMouseEvent *thisEvent = (QMouseEvent *) event;
			QMouseEvent *newEvent = new QMouseEvent(thisEvent->type(), thisEvent->pos() + widget->pos(), thisEvent->button(), thisEvent->state());
			mouseMoveEvent(newEvent);
			delete newEvent;
		} else if (event->type() == QEvent::Wheel) {
			QWheelEvent *thisEvent = (QWheelEvent *) event;
			if (thisEvent->state() & Qt::ControlButton) {
				int delta = (thisEvent->delta() > 0 ? 1 : -1);
				if (m_lyricsZoom + delta >= 1) {
					m_lyricsZoom += delta;
					m_lyrics->zoomTo(this->font().pointSize() + m_lyricsZoom);
					Settings::setLyricsZoom(m_lyricsZoom);
					Settings::writeConfig();
				}
				return true;
			}
		}
	}

	// Not processed event, even if we effectively processed it (process it twice here and in the progressbar):
	return false;
}

void CoverDisplay::mouseMoveEvent(QMouseEvent *event)
{
	if (m_orientation == FullScreen && isVisible()) {
		// Update the "last moved mouse" time:
		m_fullScreenTimer.start(FULL_SCREEN_TIMOUT, /*singleShot=*/true);
		// Show controls if needed:
		if (m_underMouse == false) {
			m_underMouse = true;
			unsetCursor(); // Restore / show again the cursor
			if (m_lyrics)
				m_lyrics->viewport()->unsetCursor();
			if (ANIMATE_BUTTONS) {
				m_buttonsAnimator.toEnd();
				m_playPauseAnimator.toEnd();
			} else {
				updateButtons();
				updateStars();
			}
		}
	}

	// Get the number of the star that is under the mouse pointer:
	int hoveredStar = (areControlsShown() && m_underMouse && m_infos->status() != PlayerInformation::Stopped ? 0 : -1);
	if (hoveredStar != -1 && m_clickedButton == -1) {
		for (int i = 0; i < 5; i++) {
			QRect starRect(m_starsRect.x() + i * m_starWidth, m_starsRect.y(), m_starWidth, m_starWidth);
			if (starRect.contains(event->pos())) {
				hoveredStar = i + 1;
				break;
			}
		}
	}

	if (hoveredStar != m_hoveredStar) {
		m_hoveredStar = hoveredStar;
		updateStars();
		if (m_hoveredStar > 0) {
			setCursor(QCursor(Qt::PointingHandCursor));
			if (m_lyrics)
				m_lyrics->viewport()->setCursor(QCursor(Qt::PointingHandCursor));
		} else {
			unsetCursor();
			if (m_lyrics)
				m_lyrics->viewport()->unsetCursor();
		}
	}

	int hoveredButton = -1;
	if (m_underMouse) {
		if (areControlsShown() && m_prevRect.contains(event->pos()))
			hoveredButton = 0;
		else if (m_playRect.contains(event->pos()))
			hoveredButton = 1;
		else if (areControlsShown() && m_nextRect.contains(event->pos()))
			hoveredButton = 2;
		else if (m_infos->isPlaying() && m_fullRect.contains(event->pos()))
			hoveredButton = 3;
		else if (m_closeRect.contains(event->pos()))
			hoveredButton = 4;
		else if (m_themeRect.contains(event->pos()))
			hoveredButton = 5;
		else if (m_lyricsRect.contains(event->pos()))
			hoveredButton = 6;
	}

	if (m_hoveredButton != hoveredButton) {
		m_hoveredButton = hoveredButton;
		updateButtons();
		if (m_hoveredButton != -1) {
			setCursor(QCursor(Qt::PointingHandCursor));
			if (m_lyrics)
				m_lyrics->viewport()->setCursor(QCursor(Qt::PointingHandCursor));
		} else {
			unsetCursor();
			if (m_lyrics)
				m_lyrics->viewport()->unsetCursor();
		}
	}

	if (m_taskBarWatcher) {
		int hoveredTaskBarEntry = -1;
		const QValueList<TaskBarEntry> &entries = m_taskBarWatcher->entries();
		for (uint i = 0; i < entries.count(); ++i) {
			int entryX = m_taskBarRect.x() + (m_taskBarEntryWidth + TASKBAR_MARGIN) * i;
			if (QRect(entryX, m_taskBarRect.y(), m_taskBarEntryWidth, m_taskBarRect.height()).contains(event->pos()))
				hoveredTaskBarEntry = i;
		}

		if (m_hoveredTaskBarEntry != hoveredTaskBarEntry) {
			m_hoveredTaskBarEntry = hoveredTaskBarEntry;
			updateTaskBar();
			if (m_hoveredTaskBarEntry != -1) {
				setCursor(QCursor(Qt::PointingHandCursor));
				if (m_lyrics)
					m_lyrics->viewport()->setCursor(QCursor(Qt::PointingHandCursor));
			} else {
				unsetCursor();
				if (m_lyrics)
					m_lyrics->viewport()->unsetCursor();
			}
		}
	}

	QWidget::mouseMoveEvent(event);
}

void CoverDisplay::enterEvent(QEvent *event)
{
	setMouseTracking(true);

	QWidget::enterEvent(event);

	if (m_orientation != FullScreen) {
		m_underMouse = true;
		if (ANIMATE_BUTTONS) {
			if (m_infos->isPlaying())
				m_buttonsAnimator.toEnd();
			m_playPauseAnimator.toEnd();
		} else {
			updateButtons();
			updateStars();
		}

		// To be able to show stars even if the mouse enter one pixel and do not move anymore:
		QMouseEvent *mouseEvent = new QMouseEvent(QEvent::MouseMove, mapFromGlobal(QCursor::pos()), 0, 0);
		mouseMoveEvent(mouseEvent);
		delete mouseEvent;
	}
}

void CoverDisplay::leaveEvent(QEvent *event)
{
	setMouseTracking(false);

	if (m_orientation != FullScreen) { // TODO: void setMouseHovered(bool)
		if (m_hoveredStar != -1) {
			m_hoveredStar = -1;
			updateStars();
			unsetCursor();
			if (m_lyrics)
				m_lyrics->viewport()->unsetCursor();
		}
		m_hoveredButton = -1;
		m_underMouse = false;
		if (ANIMATE_BUTTONS) {
			m_buttonsAnimator.toBegin();
			m_playPauseAnimator.toBegin();
		} else {
			updateButtons();
			updateStars();
		}
	}

	QWidget::leaveEvent(event);
}

void CoverDisplay::mousePressEvent(QMouseEvent *event)
{
	// Typically in fullscreen display, the mouse & buttons just hidden and the user click:
	// the interface SHOULD responds, and not ask the user to move the mouse before:
	if (!m_underMouse) {
		QMouseEvent *mouseEvent = new QMouseEvent(QEvent::MouseMove, mapFromGlobal(QCursor::pos()), 0, 0);
		mouseMoveEvent(mouseEvent);
		delete mouseEvent;
	}

	if (m_hoveredStar > 0) {
		int newRating = m_hoveredStar * 2;
		if (m_infos->rating() == 1)
			newRating = 0;
		else if (m_infos->rating() == newRating)
			newRating -= 1;
		m_infos->changeRating(newRating);
	}

	if (m_hoveredTaskBarEntry != -1) {
		emit closeAsked();
		const QValueList<TaskBarEntry> &entries = m_taskBarWatcher->entries();
		KWin::forceActiveWindow(entries[m_hoveredTaskBarEntry].wid);
		m_hoveredTaskBarEntry = -1;
	}

	int clickedButton = -1;
	if (m_underMouse) {
		if (areControlsShown() && m_prevRect.contains(event->pos()))
			clickedButton = 0;
		else if (m_playRect.contains(event->pos()))
			clickedButton = 1;
		else if (areControlsShown() && m_nextRect.contains(event->pos()))
			clickedButton = 2;
		else if (m_infos->isPlaying() && m_fullRect.contains(event->pos()))
			clickedButton = 3;
		else if (m_closeRect.contains(event->pos()))
			clickedButton = 4;
		else if (m_themeRect.contains(event->pos()))
			clickedButton = 5;
		else if (m_lyricsRect.contains(event->pos()))
			clickedButton = 6;
	}


	if (m_clickedButton != clickedButton) {
		m_clickedButton = clickedButton;
		updateButtons();
	}

	QWidget::mousePressEvent(event);
}

void CoverDisplay::wheelEvent(QWheelEvent *event)
{
	if (areControlsShown()) {
		if (event->orientation() == Qt::Vertical) {
			PlayerInformation *infos = PlayerInformation::instance();
			if (infos->canSeek()) {
				int deltaSeconds = 10 * (event->delta() > 0 ? 1 : -1);
				m_infos->seekRelative(deltaSeconds);
			}
		} else {
			if (event->delta() > 0)
				AmarokApi::volumeUp();
			else
				AmarokApi::volumeDown();
		}
	}
}

void CoverDisplay::resizeEvent(QResizeEvent *event)
{
	if (m_orientation == Horizontal) {
		m_textRect.setWidth(width() - m_textRect.x());
		if (m_progressBarOnLeft) {
			m_progressRect.setWidth(width() - m_progressRect.x() - m_progressMargin);
			m_progressBar->resize(m_progressRect.size());
		}
	}

	QWidget::resizeEvent(event);
}

int CoverDisplay::progressHeightForSize(int size)
{
	return (size < 110 ? (size < 70 ? (size < 58 ? (size < 30 ? 3 : 4) : 5) : 6) : (6 + (size - 110 + 1) * 10 / (256 - 110)));
}

int CoverDisplay::heightForWidth(int width)
{
//	QTimer::singleShot( 100, this, SLOT(update()) );

//	if (!m_infos->isPlaying())
//		return 0;

	// Recompute only if there are changes to do:
	if (this->width() == width && m_orientation == Vertical)
		return minimumHeight();

	// Compute general values:
	m_orientation        = Vertical;
	m_margin             = (width < 58 ? (width < 30 ? 0 : 1) : 2);
	m_progressMargin     = (width < 58 ? (width < 30 ? 1 : 2) : 3);
	m_buttonPadding      = m_margin;
	m_reflectionDistance = m_margin;

	// Compute progress rectangle:
	m_progressRect.setX(      m_margin + m_progressMargin    );
	m_progressRect.setY(      m_margin                       );
	m_progressRect.setWidth(  width - 2 * m_progressRect.x() );
	m_progressRect.setHeight( progressHeightForSize(width)   );

	m_withReflexion = true;

	// Compute cover rectangle:
	m_coverRect.setX(      m_margin                               );
	m_coverRect.setY(      m_progressRect.bottom() + 1 + m_margin );
	m_coverRect.setWidth(  width - 2 * m_margin                   );
	m_coverRect.setHeight( m_coverRect.width()                    );

	// Compute the stars rectangle:
	m_starWidth = m_coverRect.width() / 5;
	m_starsRect.setX(      (width - 5 * m_starWidth) / 2                    );
	m_starsRect.setY(      m_coverRect.bottom() + 1 - (m_starWidth * 2 / 3) );
	m_starsRect.setWidth(  5 * m_starWidth                                  );
	m_starsRect.setHeight( m_starWidth                                      );

	// Compute text rectangle:
	QPainter painter(this);
	m_textFont = font();
	m_textFont.setPointSize(m_textFont.pointSize() - 2);
	painter.setFont(m_textFont);
	QRect textBoundingRect = painter.boundingRect(0, 0, 10000, 10000, 0, "A\nA\nA");
	m_textRect.setX(      m_margin                            );
	m_textRect.setY(      m_starsRect.bottom() + 1 + m_margin );
	m_textRect.setWidth(  width - m_margin                    );
	m_textRect.setHeight( textBoundingRect.height()           );

	computeTextColor();

	int appletHeight = m_textRect.bottom() + 1 + m_margin;
	setMinimumSize(width, appletHeight);
	resize(minimumSize());

	computeButtonRects(width);
	initPixmaps();

	// Recompute cover size:
	newInformation();

	return appletHeight;
}

int CoverDisplay::widthForHeight(int height)
{
//	QTimer::singleShot( 100, this, SLOT(update()) );

//	if (!m_infos->isPlaying())
//		return 0;

	// Recompute only if there are changes to do:
	if (this->height() == height && m_orientation == Horizontal)
		return minimumWidth();

	// Compute general values:
	m_orientation        = Horizontal;
	m_margin             = (height < 58 ? (height < 30 ? 0 : 1) : 2);
	m_progressMargin     = (height < 58 ? (height < 30 ? 1 : 1) : 3);
	m_buttonPadding      = m_margin;
	m_reflectionDistance = m_margin;

	// Compute the text size hint:
	QPainter painter(this);
	m_textFont = font();
	m_textFont.setPointSize(m_textFont.pointSize() - 2);
	painter.setFont(m_textFont);
	QString newLine = (m_orientation == FullScreen && !Settings::showLyrics() ? "\n\n" : "\n");
	QRect textBoundingRect = painter.boundingRect(0, 0, 10000, 10000, 0, "The avgerage width" + newLine + "of an album" + newLine + "name");

	// Compute progress position and height if positionned on top of the cover:
	m_progressBarOnLeft = (textBoundingRect.width() > height);
	if (!m_progressBarOnLeft) {
		m_progressRect.setX(      m_margin + m_progressMargin   );
		m_progressRect.setY(      m_margin                      );
		m_progressRect.setHeight( progressHeightForSize(height) );
	}

	m_withReflexion = (height > 110);

	// Compute cover rectangle:
	int coverTop = (m_progressBarOnLeft ? m_margin : m_progressRect.bottom() + 1 + m_margin);
//	m_starWidth = (height - coverTop - m_margin) * 2 / (5 * 2 + 1);
//	if (!m_withReflexion)
//		m_starWidth = (height - coverTop - m_margin) / 5;
	if (m_withReflexion)
		m_starWidth = (height - coverTop - m_margin) * 2 / (5 * 2 + 1);
	else
		m_starWidth = (height - coverTop - m_margin) / 5;
	m_coverRect.setX(      m_margin                                                         );
	m_coverRect.setY(      coverTop                                                         );
	m_coverRect.setWidth(  m_withReflexion ? 5 * m_starWidth : height - coverTop - m_margin );
	m_coverRect.setHeight( m_coverRect.width()                                              );

	// Compute progress width, determined by the cover width:
	if (!m_progressBarOnLeft)
		m_progressRect.setWidth( m_coverRect.width() - 2 * m_progressMargin );

	// Compute progress position and height if positionned on top of the text:
	if (m_progressBarOnLeft) {
		m_progressRect.setX(      m_coverRect.right() + 1 + m_progressMargin   );
		m_progressRect.setY(      m_margin                                     );
		m_progressRect.setHeight( progressHeightForSize(height)                );
	}

	int textTop = (m_progressBarOnLeft ? m_progressRect.bottom() + 1 + m_margin : m_margin);

	int availableHeightUnderText = height - textTop - textBoundingRect.height() - m_margin - m_margin;
	int possibleStarWidth = availableHeightUnderText;
	if (possibleStarWidth > textBoundingRect.width() / 5)
		possibleStarWidth = textBoundingRect.width() / 5;
	bool starsOnLeft = (!m_withReflexion && availableHeightUnderText > m_starWidth && possibleStarWidth > m_starWidth);

	// Compute the stars rectangle:
	if (starsOnLeft) {
		m_starWidth = possibleStarWidth;
		m_starsRect.setX( m_coverRect.right() + 1 + m_margin );
	} else {
		m_starsRect.setX( (m_coverRect.right() + 1 + m_margin - 5 * m_starWidth) / 2 );
	}
		m_starsRect.setY( m_withReflexion ? m_coverRect.bottom() + 1 - (m_starWidth * 2 / 3) : height - m_margin - m_starWidth );
	m_starsRect.setWidth(  5 * m_starWidth );
	m_starsRect.setHeight( m_starWidth     );

	// Compute text rectangle:
	m_textRect.setX(      m_coverRect.right() + (m_margin == 0 ? 1 : m_margin)         );
	m_textRect.setY(      textTop + (height - textTop - textBoundingRect.height() - (starsOnLeft ? m_starWidth + m_margin : 0)) / 2 - 1 );
	m_textRect.setWidth(  textBoundingRect.width()                                     );
	m_textRect.setHeight( textBoundingRect.height() + (m_withReflexion ? 0 : m_margin) ); // Add margin for Fitt's law to work with those small Kickers!

	computeTextColor();

	int appletWidth = m_textRect.right() + m_margin;
	setMinimumSize(appletWidth, height);
	resize(minimumSize());

	computeButtonRects(height);
	initPixmaps();

	if (m_progressBarOnLeft) {
		m_progressRect.setWidth(width() - m_progressRect.x() - m_progressMargin);
		m_progressBar->resize(m_progressRect.size());
	}

	// Recompute cover size:
	newInformation();

	return appletWidth;
}

void CoverDisplay::setFullScreen()
{
	// Recompute only if there are changes to do:
	if (m_orientation == FullScreen)
		return;

	setBackgroundMode(Qt::NoBackground);

	m_progressBar->setColors(
		Theme::current()->progressBackgroundColor(),
		Theme::current()->progressBarColor(),
		Theme::current()->progressBackgroundTextColor(),
		Theme::current()->progressBarTextColor()
	);


	PixmapCache::insert("", QPixmap()); // setCacheLimit() only works after something has been inserted (see the source code of PixmapCache, and when pm_cache is created)!
	PixmapCache::setCacheLimit(height() * width() * 4 * 2 / 1024); // We allow to store a total image surface of two screens, that is arround 8 MB.


	double ratio = height() / (double) width();
	// Compute general values:
	m_orientation        = FullScreen;
	m_margin             = (ratio > 0.7 && ratio < 0.8 ? 50 : 75); // less margin if ratio is roughly 3/4 (0.75); more margin for wide screens
	m_progressMargin     = (m_margin / 2);
	m_buttonPadding      = 2;
	m_reflectionDistance = 6;

	// Compute cover rectangle:
	int coverSize = 300;
	m_coverRect.setX(      m_margin                   );
//	m_coverRect.setY(      (height() - coverSize) / 2 );
	m_coverRect.setY(      (height() - coverSize) * 3 / 8 );
	m_coverRect.setWidth(  coverSize                  );
	m_coverRect.setHeight( coverSize                  );

	// Compute progress position and height:
	int progressHeight = 30;
	m_progressRect.setX(      m_margin + m_progressMargin                   );
	m_progressRect.setY(      (m_coverRect.y() - progressHeight) * 2 / 3    );
	m_progressRect.setWidth(  width() - 2 * m_margin - 2 * m_progressMargin );
	m_progressRect.setHeight( progressHeight                                );

	if (Settings::showLyrics()) {
		m_coverRect.moveTop(progressHeight * 2/*3*/ + m_margin);
		m_progressRect.moveTop(/*progressHeight*/m_margin);
	}

	// Compute the stars rectangle:
	m_starWidth = m_coverRect.width() / 5;
	m_starsRect.setX(      m_coverRect.x() + (m_coverRect.width() - 5 * m_starWidth) / 2 );
	m_starsRect.setY(      m_coverRect.bottom() + 1 - (m_starWidth * 2 / 3) );
	m_starsRect.setWidth(  5 * m_starWidth                                  );
	m_starsRect.setHeight( m_starWidth                                      );

	// Compute text rectangle:
	QPainter painter(this);
	m_textFont = font();
	m_textFont.setBold(true);
	m_textFont.setPointSize(m_textFont.pointSize() * 4);
	painter.setFont(m_textFont);
	QString newLine = (m_orientation == FullScreen && !Settings::showLyrics() ? "\n\n" : "\n");
	QRect textBoundingRect = painter.boundingRect(0, 0, 10000, 10000, 0, "The avgerage width" + newLine + "of an album" + newLine + "name");
	m_textRect.setX(      m_coverRect.right() + m_margin / 3                                       );
	m_textRect.setY(      m_coverRect.y() + (m_coverRect.height() - textBoundingRect.height()) / 2 );
	m_textRect.setWidth(  width() - m_textRect.x() - m_margin / 3                                  );
	m_textRect.setHeight( textBoundingRect.height()                                                );

	if (Settings::showLyrics()) {
		m_textRect.moveTop(m_coverRect.y());
	}

	// Computing button positions:
	int buttonWidth = CONTROL_BUTTONS_MAX_SIZE + 2 * m_buttonPadding;
/*
// Valable if there is no "Next Playing:" text:

	int x = (width() - 5 * buttonWidth) / 2;
	int y = m_coverRect.bottom() + 1 + (height() - m_coverRect.bottom() - 1 - buttonWidth) / 3;

	m_prevRect.setX(      x           );
	m_prevRect.setY(      y           );
	m_prevRect.setWidth(  buttonWidth );
	m_prevRect.setHeight( buttonWidth );

	m_playRect.setX(      x + 2 * buttonWidth );
	m_playRect.setY(      y                   );
	m_playRect.setWidth(  buttonWidth         );
	m_playRect.setHeight( buttonWidth         );

	m_nextRect.setX(      x + 4 * buttonWidth );
	m_nextRect.setY(      y                   );
	m_nextRect.setWidth(  buttonWidth         );
	m_nextRect.setHeight( buttonWidth         );
*/
	int y = m_starsRect.bottom() + 3 * m_starsRect.height() / 4;
	int buttonsSpacing = (m_coverRect.width() - 3 * buttonWidth) / (4 * 2);

	m_prevRect.setX(      m_coverRect.x() + buttonsSpacing );
	m_prevRect.setY(      y           );
	m_prevRect.setWidth(  buttonWidth );
	m_prevRect.setHeight( buttonWidth );

	m_playRect.setX(      m_coverRect.x() + (m_coverRect.width() - buttonWidth) / 2);
	m_playRect.setY(      y                   );
	m_playRect.setWidth(  buttonWidth         );
	m_playRect.setHeight( buttonWidth         );

	m_nextRect.setX(      m_coverRect.right() - 1 - buttonWidth - buttonsSpacing);
	m_nextRect.setY(      y                   );
	m_nextRect.setWidth(  buttonWidth         );
	m_nextRect.setHeight( buttonWidth         );


	m_fullRect = HIDDEN_RECT;

	int closeSize = 22 + 2 * m_buttonPadding;
	m_closeRect.setX(      width() - closeSize );
	m_closeRect.setY(      0                   );
	m_closeRect.setWidth(  closeSize           );
	m_closeRect.setHeight( closeSize           );

	m_themeRect.setX(      width() - closeSize * 5/2);
	m_themeRect.setY(      0                      );
	m_themeRect.setWidth(  closeSize              );
	m_themeRect.setHeight( closeSize              );

	m_lyricsRect.setX(      width() - closeSize * 7/2);
	m_lyricsRect.setY(      0                      );
	m_lyricsRect.setWidth(  closeSize              );
	m_lyricsRect.setHeight( closeSize              );

	initPixmaps();


	// We will need it below to compute bottomBar rect:
	newClockInformation();


	// Compute Next Playing Rect:
	m_nextPlayingRect.setX(m_textRect.x());
//	m_nextPlayingRect.setY(m_coverRect.y() + coverSize + (coverSize - m_textRect.height()) / 2 + m_reflectionDistance);

	// Get the maximum height of the Next Playing rich text
	QFont font = this->font();
	font.setPointSize(font.pointSize() + 3);
	QSimpleRichText richText("<nobr><font size=4><i>Line 1<br><b>Line 2<br>Line 3</b></i></font></nobr>", font);
	richText.setWidth(width());
	//int nextPlayingHeight = richText.height();

	// Get the taskbar Y:
	int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
	int taskbarHeight = TASKBAR_MARGIN + 16 + TASKBAR_MARGIN;
	int taskbarY      = this->height() - taskbarHeight - clockMargin - 1;

	int textYMin = m_coverRect.y() + coverSize + m_reflectionDistance;
	int textYMax = taskbarY;
	m_nextPlayingRect.setY(textYMin + (textYMax - textYMin - richText.height()) / 2);
	m_nextPlayingRect.setHeight(richText.height());


	if (Settings::showLyrics()) {
		m_nextPlayingRect.moveTop(taskbarY - 3 * clockMargin - richText.height());
		m_nextPlayingRect.setX(clockMargin/*m_coverRect.x()*/);
	}

	if (m_lyrics == 0) {
		m_lyrics = new QTextBrowser(this);
//		m_lyrics->setShown(Settings::showLyrics());
		QColor lyricsBackground = Theme::current()->lyricsBackgroundColor();//QColor("#262626");//Qt::black;
		QColor lyricsText       = Theme::current()->lyricsTextColor();//Qt::white;
		QColor scrollBackground = Theme::current()->lyricsScrollBackgroundColor();//QColor("#323232");//background;
		QColor scrollButtons    = Theme::current()->lyricsScrollButtonsColor();//QColor("#676767");//Tools::mixColors(background, text, 0.75);
		m_lyrics->setFrameShape(QFrame::NoFrame);

		m_lyrics->setFocusPolicy(QWidget::NoFocus);
		m_lyrics->setPaletteBackgroundColor(lyricsBackground);
		m_lyrics->setPaletteForegroundColor(lyricsText);

		m_lyrics->horizontalScrollBar()->setPalette(QPalette(scrollButtons, scrollBackground));
		m_lyrics->verticalScrollBar()->setPalette(QPalette(scrollButtons, scrollBackground));

// 		m_lyrics->setFrameShape(QFrame::Box);
// 		m_lyrics->setFrameShadow(QFrame::Plain);
// 		m_lyrics->setLineWidth(3);
// 		m_lyrics->setMidLineWidth(0);

		m_lyricsZoom = Settings::lyricsZoom();
		m_lyrics->zoomTo(this->font().pointSize() + m_lyricsZoom);
		m_lyrics->installEventFilter(this);
		m_lyrics->viewport()->installEventFilter(this);
		m_lyrics->horizontalScrollBar()->installEventFilter(this);
		m_lyrics->verticalScrollBar()->installEventFilter(this);

		connect( m_lyrics, SIGNAL(contentsMoving(int, int)), this, SLOT(lyricsMoving(int, int)) );
	}

	m_lyrics->setShown(Settings::showLyrics());
	if (Settings::showLyrics()) {
		//m_lyrics->move(m_textRect.x()/* + m_margin / 2*/, m_textRect.bottom() + clockMargin);
		//m_lyrics->resize(width() - m_lyrics->x() - m_margin/*m_textRect.width() - m_margin*/, m_nextPlayingRect.y() - clockMargin - m_lyrics->y());
		m_lyrics->move(m_coverRect.right() + m_margin, m_textRect.bottom() + m_margin / 3);
		m_lyrics->resize(width() - m_lyrics->x() - m_margin, m_nextPlayingRect.bottom() - 1 - m_lyrics->y());
	}

	if (Settings::showLyrics())
		m_nextPlayingRect.setWidth(width() - m_lyrics->x() - 10);
	else
		m_nextPlayingRect.setWidth(width() - m_nextPlayingRect.x() - m_margin / 3);



	// Draw the bottom frame:
//	int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
//	int taskbarHeight = TASKBAR_MARGIN + 16 + TASKBAR_MARGIN;
	int clockHeight   = m_clockRect.height();
	int bottomBarY    = this->height() - QMAX(taskbarHeight, clockHeight) - 2 * clockMargin - 1;
	QRect bottomRect = QRect(0, bottomBarY, width(), height() - bottomBarY);



	QRect coverRect = QRect(m_coverRect.x() - m_reflectionDistance / 2, m_coverRect.y() - m_reflectionDistance / 2,
	                        m_coverRect.width() + m_reflectionDistance, m_coverRect.height() + m_reflectionDistance);

	m_frameRects[Frame::Screen]            = QRect(0, 0, width(), height());
	m_frameRects[Frame::PluginIconBar]     = QRect(); // Not implemented yet!
	m_frameRects[Frame::ProgressBar]       = m_progressRect;
	m_frameRects[Frame::MiddleBar]         = QRect(0, coverRect.y(), width(), coverRect.height()); //
	m_frameRects[Frame::CoverAndTextInfos] = QRect(coverRect.x(), coverRect.y(), m_textRect.right() - coverRect.x(), coverRect.height());
	m_frameRects[Frame::Cover]             = m_coverRect;
	m_frameRects[Frame::CoverOverlay]      = m_coverRect;
	m_frameRects[Frame::TextInfos]         = m_textRect;
	m_frameRects[Frame::BelowCover]        = QRect(0, coverRect.bottom() + 1, width(), bottomRect.y() - coverRect.bottom() - 1); //
	m_frameRects[Frame::NextPlaying]       = m_nextPlayingRect;
	m_frameRects[Frame::BottomBar]         = bottomRect;
	m_frameRects[Frame::TaskBar]           = m_taskBarRect;
	m_frameRects[Frame::DateHour]          = m_clockRect;
	m_frameRects[Frame::Lyrics]            = (Settings::showLyrics() ? QRect(m_lyrics->pos(), m_lyrics->size()) : QRect());
	m_frameRects[Frame::Stars]             = m_starsRect; // Not implemented yet!

	// Recompute cover size:
	newInformation();
	newLyrics();

	if (m_taskBarWatcher == 0) {
		m_taskBarWatcher = new TaskBarWatcher();
		connect( m_taskBarWatcher, SIGNAL(changed()), this, SLOT(taskBarChanged()) );
	}

	connect( ThemeManager::instance(), SIGNAL(themeChanged()), this, SLOT(themeChanged()) );
}

void CoverDisplay::lyricsMoving(int /*nextX*/, int nextY)
{
// 	if (m_ignoreNextLyricsMove) {
// 		m_ignoreNextLyricsMove = false;
// 		return;
// 	}

	// Compute tehorical top:
	int tehoricalY = m_lyrics->contentsHeight() * m_infos->position() / m_infos->duration();
	int topY = tehoricalY - m_lyrics->visibleHeight() / 2 - (m_lyrics->visibleHeight() % 2 ? 1 : 0);
//	if (topY > m_lyrics->contentsHeight() - m_lyrics->visibleHeight())
//		topY = m_lyrics->contentsHeight() - m_lyrics->visibleHeight();
//	if (topY < 0) // After the previous test, because if contentsHeight < visibleHeight, it would end up with a negative Y
//		topY = 0;

	// Now, if the lyrics view has been moved by the user,
	// compute the offset between the user-wanted position and the tehorical one:
	m_lyricsOffset = nextY - topY;

	if (topY < 0 && nextY == 0)
		m_lyricsOffset = 0;

//	std::cout << "lyricsMoving to y=" << nextY << " (in theory: " << topY << "), offset=" << m_lyricsOffset << std::endl;
}

void CoverDisplay::themeChanged()
{
	update();
	newClockInformation(); // Colors are hard-coded in the clock rich text
	m_progressBar->setColors(
		Theme::current()->progressBackgroundColor(),
		Theme::current()->progressBarColor(),
		Theme::current()->progressBackgroundTextColor(),
		Theme::current()->progressBarTextColor()
	);

	if (m_lyrics) {
		QColor lyricsBackground = Theme::current()->lyricsBackgroundColor();
		QColor lyricsText       = Theme::current()->lyricsTextColor();
		QColor scrollBackground = Theme::current()->lyricsScrollBackgroundColor();
		QColor scrollButtons    = Theme::current()->lyricsScrollButtonsColor();
		m_lyrics->setPaletteBackgroundColor(lyricsBackground);
		m_lyrics->setPaletteForegroundColor(lyricsText);
		m_lyrics->horizontalScrollBar()->setPalette(QPalette(scrollButtons, scrollBackground));
		m_lyrics->verticalScrollBar()->setPalette(QPalette(scrollButtons, scrollBackground));
	}
}

void CoverDisplay::toggleLyrics()
{
	Settings::setShowLyrics(!Settings::showLyrics());
	m_orientation = Horizontal; // To trigger reflow below
	setFullScreen();
	update();
	Settings::writeConfig(); // Write after things have gone well without crashing, in case of...
}

void CoverDisplay::scrollLyricsUp()
{
	if (Settings::showLyrics() && m_lyrics)
		kapp->postEvent(m_lyrics, new QKeyEvent(QEvent::KeyPress, Qt::Key_Up, 0, 0));
}

void CoverDisplay::scrollLyricsDown()
{
	if (Settings::showLyrics() && m_lyrics)
		kapp->postEvent(m_lyrics, new QKeyEvent(QEvent::KeyPress, Qt::Key_Down, 0, 0));
}

void CoverDisplay::taskBarChanged()
{
	const QValueList<TaskBarEntry> &entries = m_taskBarWatcher->entries();
	// Add new windows:
	for (uint i = 0; i < entries.count(); ++i) {
		if (!m_taskBarFlickerings.contains(entries[i].wid))
			m_taskBarFlickerings[entries[i].wid] = 0;
	}
	// Remove old windows:
	QMap<WId, int>::Iterator it;
	for (it = m_taskBarFlickerings.begin(); it != m_taskBarFlickerings.end(); ++it) {
		if (!entries.contains(it.key()))
			m_taskBarFlickerings.remove(it);
	}

	// Example:
	// Flicker 5 times and keep blue after the 5° flickering:
	// 0 1 2 3 4 5 6 7 8 9 10
	// t   t   t   t   t t t t t t
	int flickerCount = taskBarFlickerCount();
	int tickCount = (flickerCount - 1) * 2;
	// bool on = (m_taskBarFlickerings[entries[i].wid] < tickCount ? ! (m_taskBarFlickerings[entries[i].wid] % 2) : true);

	// Compute if we need to start the timer:
	bool needTimer = false;
	for (it = m_taskBarFlickerings.begin(); it != m_taskBarFlickerings.end(); ++it) {
		if (it.data() < tickCount) {
			needTimer = true;
			break;
		}
	}

	if (needTimer)
		m_taskBarFlickerTimer.start(500, /*singleShot=*/false);

	updateTaskBar();
}

int CoverDisplay::taskBarFlickerCount()
{
	KSimpleConfig config("ktaskbarrc", /*readOnly=*/true);
	config.setGroup("Appearance");
	return config.readNumEntry("AttentionBlinkIterations", 4) + 1;
}

void CoverDisplay::flickerTaskBar()
{
	// Example:
	// Flicker 5 times and keep blue after the 5° flickering:
	// 0 1 2 3 4 5 6 7 8 9 10
	// t   t   t   t   t t t t t t
	int flickerCount = taskBarFlickerCount();
	int tickCount = (flickerCount - 1) * 2;
	// bool on = (m_taskBarFlickerings[entries[i].wid] < tickCount ? ! (m_taskBarFlickerings[entries[i].wid] % 2) : true);

	// Compute if we need to start the timer:
	bool needTimer = false;
	QMap<WId, int>::Iterator it;
	for (it = m_taskBarFlickerings.begin(); it != m_taskBarFlickerings.end(); ++it) {
		m_taskBarFlickerings[it.key()]++; // Make the button to flicker
		if (it.data() < tickCount)
			needTimer = true; // Continue
	}

	if (!needTimer)
		m_taskBarFlickerTimer.stop();

	updateTaskBar();
}

void CoverDisplay::computeButtonRects(int size)
{
	// One horizontal line:
	if (m_orientation == Horizontal && size < 46/*30*/) {
		int x = m_coverRect.x();
		int y = m_progressRect.bottom() + 1 + m_margin;
		int buttonWidth = (/*m_coverRect.*/height() - y - (m_coverRect.bottom() + 1 - m_starsRect.y()));
		buttonWidth = QMIN(buttonWidth, CONTROL_BUTTONS_MAX_SIZE + 2 * m_buttonPadding);
		if (buttonWidth > (width() - 2 * m_margin - 3) / 4) // Do not put buttons outside (on the right) of the applet
			buttonWidth = (width() - 2 * m_margin - 3) / 4;

		m_prevRect.setX(      x           );
		m_prevRect.setY(      y           );
		m_prevRect.setWidth(  buttonWidth );
		m_prevRect.setHeight( buttonWidth );

		m_playRect.setX(      x + buttonWidth + 1 );
		m_playRect.setY(      y                   );
		m_playRect.setWidth(  buttonWidth         );
		m_playRect.setHeight( buttonWidth         );

		m_nextRect.setX(      x + 2 * buttonWidth + 2 );
		m_nextRect.setY(      y                       );
		m_nextRect.setWidth(  buttonWidth             );
		m_nextRect.setHeight( buttonWidth             );

		m_fullRect.setX(      x + 3 * buttonWidth + 3 );
		m_fullRect.setY(      y                       );
		m_fullRect.setWidth(  buttonWidth             );
		m_fullRect.setHeight( buttonWidth             );

		m_closeRect  = HIDDEN_RECT;
		m_themeRect  = HIDDEN_RECT;
		m_lyricsRect = HIDDEN_RECT;
	// A square:
	} else if (size < 76) {
		int availableHeight = m_coverRect.height() - (m_coverRect.bottom() + 1 - m_starsRect.y());
		if (m_starsRect.x() >= m_coverRect.right()) // Stars are on left, buttons can span the whole cover:
			availableHeight = m_coverRect.height();
		int buttonWidth = availableHeight / 2 - 1;
		buttonWidth = QMIN(buttonWidth, CONTROL_BUTTONS_MAX_SIZE + 2 * m_buttonPadding);
		int x = m_coverRect.x() + (m_coverRect.width() - 2 * buttonWidth - 1) / 2; // Have the same button spacing (1 pixel) horizontally and vertically
		int y = m_coverRect.y();

		m_prevRect.setX(      x           );
		m_prevRect.setY(      y           );
		m_prevRect.setWidth(  buttonWidth );
		m_prevRect.setHeight( buttonWidth );

		m_playRect.setX(      x               );
		m_playRect.setY(      y + 1 + buttonWidth );
		m_playRect.setWidth(  buttonWidth     );
		m_playRect.setHeight( buttonWidth     );

		m_nextRect.setX(      x + 1 + buttonWidth );
		m_nextRect.setY(      y                   );
		m_nextRect.setWidth(  buttonWidth         );
		m_nextRect.setHeight( buttonWidth         );

		m_fullRect.setX(      x + 1 + buttonWidth );
		m_fullRect.setY(      y + 1 + buttonWidth );
		m_fullRect.setWidth(  buttonWidth         );
		m_fullRect.setHeight( buttonWidth         );

		m_closeRect  = HIDDEN_RECT;
		m_themeRect  = HIDDEN_RECT;
		m_lyricsRect = HIDDEN_RECT;
	// A "arrow keys"-like arrangement with 3 top buttons:
	} else {
		int buttonWidth = QMIN(m_coverRect.width() / 3, CONTROL_BUTTONS_MAX_SIZE + 2 * m_buttonPadding);
		int x = m_coverRect.x();
		int y = m_coverRect.y();

		m_prevRect.setX(      x           );
		m_prevRect.setY(      y           );
		m_prevRect.setWidth(  buttonWidth );
		m_prevRect.setHeight( buttonWidth );

		m_playRect.setX(      x + buttonWidth + (m_coverRect.width() - 3 * buttonWidth) / 2 );
		m_playRect.setY(      y                                                             );
		m_playRect.setWidth(  buttonWidth                                                   );
		m_playRect.setHeight( buttonWidth                                                   );

		m_nextRect.setX(      x + m_coverRect.width() - buttonWidth );
		m_nextRect.setY(      y                                     );
		m_nextRect.setWidth(  buttonWidth                           );
		m_nextRect.setHeight( buttonWidth                           );

		int fullSize = QMIN(128, m_coverRect.height() - buttonWidth - (m_coverRect.bottom() + 1 - m_starsRect.y()));
		m_fullRect.setX(      m_coverRect.x() + (m_coverRect.width() - fullSize) / 2 );
		m_fullRect.setY(      m_coverRect.y() + buttonWidth                          );
		m_fullRect.setWidth(  fullSize                                               );
		m_fullRect.setHeight( fullSize                                               );

		m_closeRect  = HIDDEN_RECT;
		m_themeRect  = HIDDEN_RECT;
		m_lyricsRect = HIDDEN_RECT;
	}
}

QPixmap CoverDisplay::getAmarokImage(const QString &imageName)
{

	// Not found in the Amarok folder, use a standard available star icon:
	return kapp->iconLoader()->loadIcon(imageName/*"services"*//*"favorites"*/, KIcon::Desktop, STAR_IMAGE_SIZE/*CONTROL_BUTTONS_MAX_SIZE*/, KIcon::DefaultState);
}

void CoverDisplay::initPixmaps()
{
	m_progressBar->move(m_progressRect.topLeft());
	m_progressBar->resize(m_progressRect.size());

	// Load normal and mini stars at the display size:
	m_starIcon     = getAmarokImage("star");
	m_miniStarIcon = getAmarokImage("smallstar");
	m_starIcon     = m_starIcon.smoothScale(m_starWidth, m_starWidth);
	m_miniStarIcon = m_miniStarIcon.smoothScale(m_starWidth, m_starWidth);

	// Compute images for other states of the stars:
	m_hoveredEmptyStarIcon = m_starIcon;
	m_hoveredEmptyStarIcon.detach();
	m_hoveredMiniStarIcon = m_miniStarIcon;
	m_hoveredMiniStarIcon.detach();
	m_hoveredStarIcon = m_starIcon;
	m_hoveredStarIcon.detach();

	KImageEffect::intensity(m_hoveredStarIcon, 0.5);
	KImageEffect::intensity(m_hoveredMiniStarIcon, 0.5);
	KImageEffect::toGray(m_hoveredEmptyStarIcon, /*fast=*/false);

	m_hoveredHoveredEmptyStarIcon = m_hoveredEmptyStarIcon;
	m_hoveredHoveredEmptyStarIcon.detach();
	KImageEffect::intensity(m_hoveredHoveredEmptyStarIcon, 0.5);

	// Compute reflexions once and for all:
	m_starReflexionIcon                    = Tools::reflexionImage(m_starIcon);
	m_miniStarReflexionIcon                = Tools::reflexionImage(m_miniStarIcon);
	m_hoveredStarReflexionIcon             = Tools::reflexionImage(m_hoveredStarIcon);
	m_hoveredMiniStarReflexionIcon         = Tools::reflexionImage(m_hoveredMiniStarIcon);
	m_hoveredEmptyStarReflexionIcon        = Tools::reflexionImage(m_hoveredEmptyStarIcon);
	m_hoveredHoveredEmptyStarReflexionIcon = Tools::reflexionImage(m_hoveredHoveredEmptyStarIcon);

	// Load the button icons:
	loadButtonImages(m_prevIcon,   "amarok_back",  "player_start",           m_prevRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_nextIcon,   "amarok_next",  "player_end",             m_nextRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_playIcon,   "amarok_play",  "player_play",            m_playRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_pauseIcon,  "amarok_pause", "player_pause",           m_playRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_stopIcon,   "amarok_stop",  "player_stop",            m_playRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_fullIcon,   "NON_FINDABLE_ICON", "window_fullscreen", m_fullRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_closeIcon,  "NON_FINDABLE_ICON", "fileclose",         m_closeRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_themeIcon,  "NON_FINDABLE_ICON", "background",        m_themeRect.width() - 2 * m_buttonPadding);
	loadButtonImages(m_lyricsIcon, "amarok_lyrics", "txt",                   m_lyricsRect.width() - 2 * m_buttonPadding);
}

void CoverDisplay::loadButtonImages(QImage icons[3], const QString &iconName, const QString &fallbackIconName, int width)
{
	QPixmap pixmap;
	QDir dir;
	QStringList folders = kapp->dirs()->resourceDirs("data");
	for (QStringList::Iterator it = folders.begin(); it != folders.end(); ++it) {
		QString imagePath = *it + "/amarok/icons/hicolor/" + QString::number(CONTROL_BUTTONS_MAX_SIZE) + "x" + QString::number(CONTROL_BUTTONS_MAX_SIZE) + "/actions/" + iconName + ".png";
		if (dir.exists(imagePath))
			pixmap = QPixmap(imagePath);
	}

	if (fallbackIconName == "window_fullscreen")
		pixmap = getAmarokImage("fullscreen");

	if (pixmap.isNull())
		pixmap = kapp->iconLoader()->loadIcon(fallbackIconName, KIcon::Desktop, 32, KIcon::DefaultState);

	QImage icon = pixmap.convertToImage();
	icon = icon.smoothScale(width, width);

	// We take take rect, but it can be any other: they are the same size (by now):
	QImage hoverImage = icon;
	hoverImage.detach();
	QImage clickedImage = icon;
	clickedImage.detach();

	KImageEffect::intensity(hoverImage, 0.5);
	KImageEffect::blend(Qt::black, clickedImage, 0.5);

	icons[0] = icon;
	icons[1] = hoverImage;
	icons[2] = clickedImage;
}

QString CoverDisplay::informationText(int numberOfLines)
{
	QString newLine = (m_orientation == FullScreen && !Settings::showLyrics() ? "\n\n" : "\n");

	QString title = m_infos->title();
	if (title.isEmpty()) {
		title = m_infos->url().fileName();
		int index = title.findRev(".");
		if (index >= 0)
			title = title.left(index);
	}

	QString line1 = (m_infos->track().isEmpty() ? title : (title.isEmpty() ? m_infos->track() : i18n("%1. %2").arg(m_infos->track(), title)));
	QString line2 = m_infos->artist();
	QString line3 = (m_infos->album().isEmpty() || m_infos->year().isEmpty() || m_infos->year() == "0" ? m_infos->album() : i18n("%1 (%2)").arg(m_infos->album(), m_infos->year()));

	if (numberOfLines == 3)
		return line1 + newLine + line2 + newLine + line3;
	else // 2 lines:
		return (line1.isEmpty() ? line2 : (line2.isEmpty() ? line1 : i18n("%1 - %2").arg(line1, line2))) + newLine + line3;
}

void CoverDisplay::computeTextColor()
{
	QColor backgroundColor = paletteBackgroundColor();

	if (backgroundPixmap()) {
		QRect textRect = rect().intersect(m_textRect);
		if (textRect.isValid() && !textRect.isNull()) {
			// Get the background image of the text:
			QPixmap pixmap(textRect.size());
			QPainter painter(&pixmap);
			int backgroundTextX = textRect.x() - ((QWidget *) parent())->x();
			int backgroundTextY = textRect.y() - ((QWidget *) parent())->y();
			painter.drawTiledPixmap(0, 0, textRect.width(), textRect.height(), *backgroundPixmap(), backgroundTextX, backgroundTextY);
			painter.end();
			// Get the average of the color of every pixels:
			QImage image = pixmap.convertToImage();
			image = image.smoothScale(1, 1);
			uint *p = (uint *)image.scanLine(0) + 0;
			backgroundColor = QColor((QRgb)*p);
		}
	}
	if (qGray(backgroundColor.rgb()) < 100)
		m_textColor = QColor(255, 255, 255);
	else
		m_textColor = QColor(0, 0, 0);
}

void CoverDisplay::setPalette(const QPalette &palette)
{
	QWidget::setPalette(palette);

	if (m_orientation != FullScreen)
		computeTextColor();
}



//QPixmap *cachedPixmap = 0;


// Default values in Kicker:
// OFFSET_X              0
// OFFSET_Y              0
// MULTIPLICATION_FACTOR 4.0
// MAX_OPACITY           120.0
// THICKNESS             2
// ALGORITHM             1
// SELECTION_TYPE        1
// ?,?,?                 0,0,0

// Multiplication factor for pixels directly above, under, or next to the text
#define AXIS_FACTOR 2.0
// Multiplication factor for pixels diagonal to the text
#define DIAGONAL_FACTOR 1.0


#define BLURED_SHADOW_THICKNESS 2
#define SOLID_SHADOW_THICKNESS  1


/// This code comes from kshadowengine.cpp of Kicker library code base:
void CoverDisplay::makeShadow(const QPixmap &textPixmap, QPixmap &resultPixmap, const QColor &shadowColor, bool blured)
{
	int thickness = (blured ? BLURED_SHADOW_THICKNESS : SOLID_SHADOW_THICKNESS);

	QImage result;

	// Create a new image for for the shaddow:
	int w = textPixmap.width();
	int h = textPixmap.height();

	// Avoid calling these methods for every pixel:
	int bgRed   = shadowColor.red();
	int bgGreen = shadowColor.green();
	int bgBlue  = shadowColor.blue();

	double alphaShadow;

	// The source pixmap:
	QImage img = textPixmap.convertToImage().convertDepth(32);

	// Resize the image if necessary
	if ((result.width() != w) || (result.height() != h))
		result.create(w, h, 32);

	result.fill(0); // all black
	result.setAlphaBuffer(true);

	if (!blured) {
		// Solid border (default decay):
		for (int i = thickness; i < w - thickness; i++) {
			for (int j = thickness; j < h - thickness; j++) {
				alphaShadow = (
					qGray(img.pixel(i-1,j-1)) * DIAGONAL_FACTOR +
					qGray(img.pixel(i-1,j  )) * AXIS_FACTOR +
					qGray(img.pixel(i-1,j+1)) * DIAGONAL_FACTOR +
					qGray(img.pixel(i  ,j-1)) * AXIS_FACTOR +
					0                         +
					qGray(img.pixel(i  ,j+1)) * AXIS_FACTOR +
					qGray(img.pixel(i+1,j-1)) * DIAGONAL_FACTOR +
					qGray(img.pixel(i+1,j  )) * AXIS_FACTOR +
					qGray(img.pixel(i+1,j+1)) * DIAGONAL_FACTOR) / 4;
				// update the shadow's i,j pixel.
				alphaShadow = (alphaShadow > 255 ? 255 : alphaShadow);
				result.setPixel(i,j, qRgba(bgRed, bgGreen , bgBlue, (int) alphaShadow));
			}
		}
	} else {
		// Blured shadow (no decay):
		int sx, sy;
		for (int i = thickness; i < w - thickness; i++) {
			for (int j = thickness; j < h - thickness; j++) {
				// create a new image for for the shaddow
//				int w = img.width();
//				int h = img.height();
//				int sx, sy;

				alphaShadow = 0;
				double opacity = 0;
				for (int k = 1; k <= thickness; k++) {
					/* Generate a shadow THICKNESS pixels thicker
					* on either side than the text image. Ensure
					* that i +/- k and j +/- k are within the
					* bounds of the text pixmap.
					*/
					opacity = 0;
					for (int l = -k; l <= k; l++) {
						if (i < k)
							sx = 0;
						else if (i >= w - k)
							sx = w - 1;
						else
							sx = i + l;

						for (int m = -k; m <= k; m++) {
							if (j < k)
								sy = 0;
							else if (j >= h - k)
								sy = h - 1;
							else
								sy = j + m;

							opacity += qGray(img.pixel(sx, sy));
						}
					}
					alphaShadow += opacity / 16;
				}
				//char min = qGray(img.pixel(i, j));
				//alphaShadow = (alphaShadow > min ? min : alphaShadow);
				alphaShadow = (alphaShadow > 255 ? 255 : alphaShadow);
				alphaShadow = (qGray(img.pixel(i, j)) > 0 ? 255 : alphaShadow);
				// update the shadow's i,j pixel.
				result.setPixel(i,j, qRgba(bgRed, bgGreen , bgBlue, (int) alphaShadow));
			}
		}
	}

	//QPixmap resultPixmap;
	resultPixmap.convertFromImage(result);
}


void CoverDisplay::drawShadowedText(QPainter *painter, const QRect &rect, int flags, const QString &text, const QColor &textColor, const QColor &shadowColor, bool blured)
{
	// No need to do complex QPixmap<->QImage manipulations and computings if nothing has to be drawn:
	if (text.stripWhiteSpace().isEmpty())
		return;

	QString key = "org.kde.kirocker.textShadow[" + text + "," + painter->font().key() + "," + textColor.name() + "," + shadowColor.name() + "]";

	int thickness = (blured ? BLURED_SHADOW_THICKNESS : SOLID_SHADOW_THICKNESS);

	QPixmap resultPixmap;
	if (!PixmapCache::find(key, resultPixmap) ) {
//		std::cout << PixmapCache::cacheLimit() << "***CREATE*** " << QString(key).replace("\n", "") << std::endl;

		QSize textSize = painter->boundingRect(0, 0, 32000, 32000, flags, text).size();

		QPixmap textPixmap(textSize.width() + 2 * 2 * thickness, textSize.height() + 2 * 2 * thickness);
		textPixmap.fill(Qt::black); // Black is "transparent"
		QPainter textPainter(&textPixmap);
		textPainter.setPen(Qt::white); // White is "opaque"
		textPainter.setFont(painter->font());
		textPainter.drawText(/*FIXME*/QRect(2 * thickness, 2 * thickness, textSize.width(), textSize.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
		textPainter.end();

		makeShadow(textPixmap, resultPixmap, shadowColor, blured);

		QPainter resultPainter(&resultPixmap);
		resultPainter.setPen(textColor);
		resultPainter.setFont(painter->font());
		resultPainter.drawText(QRect(2 * thickness, 2 * thickness, textSize.width(), textSize.height()), flags, text);
		resultPainter.end();

		PixmapCache::insert(key, resultPixmap);
	}
//	else
//		std::cout << "             " << QString(key).replace("\n", "") << std::endl;

	painter->drawPixmap(rect.x() - 2 * thickness, rect.y() + (rect.height() - resultPixmap.height()) / 2, resultPixmap);
}

void CoverDisplay::drawShadowedRichText(QPainter *painter, const QRect &rect, QSimpleRichText *richText, const QString &source, const QColor &textColor, const QColor &shadowColor, bool blured)
{
	// No need to do complex QPixmap<->QImage manipulations and computings if nothing has to be drawn:
//	if (text.stripWhiteSpace().isEmpty())
//		return;

	QString key = "org.kde.kirocker.richTextShadow[" + source + "," +
	              painter->font().key() + "," + textColor.name() + "," + shadowColor.name() + "]";

	int thickness = (blured ? BLURED_SHADOW_THICKNESS : SOLID_SHADOW_THICKNESS);

	QPixmap resultPixmap;
	if (!PixmapCache::find(key, resultPixmap) ) {
//		std::cout << PixmapCache::cacheLimit() << "***CREATE*** " << QString(key).replace("\n", "") << std::endl;

		QSize textSize = QSize(richText->widthUsed(), richText->height());

		QPixmap textPixmap(textSize.width() + 2 * 2 * thickness, textSize.height() + 2 * 2 * thickness);
		textPixmap.fill(Qt::black); // Black is "transparent"
		QPainter textPainter(&textPixmap);
//		textPainter.setPen(Qt::white); // White is "opaque"
		textPainter.setFont(painter->font());
		QColorGroup colorGroup = this->colorGroup();
		colorGroup.setColor(QColorGroup::Text, Qt::white); // White is "opaque"
		richText->draw(&textPainter, 2 * thickness, 2 * thickness, QRect(/*2 * thickness, 2 * thickness, textSize.width(), textSize.height()*/), colorGroup);
		//textPainter.drawText(/*FIXME*/QRect(thickness, thickness, textSize.width(), textSize.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
		textPainter.end();

		makeShadow(textPixmap, resultPixmap, shadowColor, blured);

		QPainter resultPainter(&resultPixmap);
		resultPainter.setFont(painter->font());
//		resultPainter.setPen(textColor);
//		resultPainter.drawText(QRect(thickness, thickness, textSize.width(), textSize.height()), flags, text);
// 		QColorGroup colorGroup = this->colorGroup();
		colorGroup.setColor(QColorGroup::Text, textColor);
		richText->draw(&resultPainter, 2 * thickness, 2 * thickness, QRect(/*2 * thickness, 2 * thickness, textSize.width(), textSize.height()*/), colorGroup);
		resultPainter.end();

		PixmapCache::insert(key, resultPixmap);
	}
//	else
//		std::cout << "             " << QString(key).replace("\n", "") << std::endl;

	painter->drawPixmap(rect.x() - 2 * thickness, rect.y() + (rect.height() - resultPixmap.height()) / 2, resultPixmap);
}

void CoverDisplay::drawFrames(QPainter *painter, Frame::ShapeDrawingPhase phase, const QRect &clipRect)
{
	if (m_orientation == FullScreen) {
		for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
			Frame::Shape shape = (Frame::Shape) i;
			Frame *frame = Theme::current()->frame(shape);
			if (frame && Frame::shapeDrawingPhase(shape) == phase && (shape != Frame::Lyrics || Settings::showLyrics())) {
//				std::cout << "Painting frame " << Frame::shapeName(shape) << " " << m_frameRects[i].x() << " " << m_frameRects[i].y() << " "
//				          << m_frameRects[i].width() << " " << m_frameRects[i].height() << std::endl;
	/*			int paddingTop;
				int paddingLeft;
				int paddingRight;
				int paddingBottom;
				Theme::current()->framePaddings(shape, &paddingTop, &paddingLeft, &paddingRight, &paddingBottom);
				int x = m_frameRects[i].x() - frame->outsideLeft() - (0) - paddingLeft;
				int y = m_frameRects[i].y() - frame->outsideTop()  - (0) - paddingTop;
				int rectWidth  = frame->outsideLeft() + paddingLeft + m_frameRects[i].width()  + paddingRight  + frame->outsideRight();
				int rectHeight = frame->outsideTop()  + paddingTop  + m_frameRects[i].height() + paddingBottom + frame->outsideBottom();
				if (clipRect.intersects(QRect(x, y, rectWidth, rectHeight))) {
					QPixmap framePixmap = frame->pixmap(rectWidth, rectHeight, 0, 0, 0, 0);//m_frameRects[i].width(), m_frameRects[i].height(), paddingTop, paddingLeft, paddingRight, paddingBottom);
					painter->drawPixmap(x, y, framePixmap);
				}
	*/
				int paddingTop;
				int paddingLeft;
				int paddingRight;
				int paddingBottom;
				Theme::current()->framePaddings(shape, &paddingTop, &paddingLeft, &paddingRight, &paddingBottom);
				QRect frameRect = frame->realRect(shape, m_frameRects[i], paddingTop, paddingLeft, paddingRight, paddingBottom);
				if (clipRect.intersects(frameRect)) {
					QPixmap framePixmap = frame->pixmap(frameRect.width(), frameRect.height(), 0, 0, 0, 0);
					painter->drawPixmap(frameRect.x(), frameRect.y(), framePixmap);
				}
			}
		}
	}
}

void CoverDisplay::updateContentFrames()
{
	if (m_orientation == FullScreen) {
		for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
			Frame::Shape shape = (Frame::Shape) i;
			Frame *frame = Theme::current()->frame(shape);
			if (frame && Frame::shapeDrawingPhase(shape) == Frame::ContentPhase) {
				int paddingTop;
				int paddingLeft;
				int paddingRight;
				int paddingBottom;
				Theme::current()->framePaddings(shape, &paddingTop, &paddingLeft, &paddingRight, &paddingBottom);
	/*			int x = m_frameRects[i].x() - frame->outsideLeft() - (0) - paddingLeft;
				int y = m_frameRects[i].y() - frame->outsideTop()  - (0) - paddingTop;
				int rectWidth  = frame->outsideLeft() + paddingLeft + m_frameRects[i].width()  + paddingRight  + frame->outsideRight();
				int rectHeight = frame->outsideTop()  + paddingTop  + m_frameRects[i].height() + paddingBottom + frame->outsideBottom();
				QRect rect(x, y, rectWidth, rectHeight);//m_frameRects[i].x() - paddingLeft, m_frameRects[i].y() - paddingTop,
				           //m_frameRects[i].width() + paddingLeft + paddingRight, m_frameRects[i].height() + paddingTop + paddingBottom);
	*/			QRect frameRect = frame->realRect(shape, m_frameRects[i], paddingTop, paddingLeft, paddingRight, paddingBottom);
				update(frameRect);
				if (m_trackAnimator.isInIntermediateState()) {
					int left = frameRect.left();
					frameRect.moveLeft(left + coverDelta());
					update(frameRect);
					frameRect.moveLeft(left + oldCoverDelta());
					update(frameRect);
				}
			}
		}
	}
}

void CoverDisplay::paintEvent(QPaintEvent *event)
{
// 	QTime begin = QTime::currentTime();

	// The progressbar is a QWidget, but for the sake of code simplicity, do like if it were drawn with other elements:
	bool showProgressBar = (m_infos->isPlaying() || (m_orientation == FullScreen && !m_infos->url().isEmpty())) && m_infos->duration() > 0;
	if (m_trackAnimator.isInIntermediateState())
		showProgressBar = false;
	bool progressBarWillBeShown = !m_progressBar->isShown() && showProgressBar && m_infos->duration() > 0;
	m_progressBar->setShown(showProgressBar);

	// Initialize the drawing:
	QRect clipRect = event->rect();
	QMemArray<QRect> regionRects = event->region().rects();
	QPixmap buffer(clipRect.width(), clipRect.height());
	QPainter painterBuffer(&buffer);
	painterBuffer.translate(-clipRect.x(), -clipRect.y());

//std::cout << clipRect.x() << " " << clipRect.y() << " " << clipRect.width() << " " << clipRect.height() << std::endl;

	// Draw background:
	if (backgroundPixmap()) {
		for (uint i = 0; i < regionRects.size(); i++) {
			QRect rect = regionRects[i];
			painterBuffer.drawTiledPixmap(rect, *backgroundPixmap(), rect.topLeft());
		}
	} else
		painterBuffer.fillRect(0, 0, width(), height(), paletteBackgroundColor());

	// Draw frames:
	drawFrames(&painterBuffer, Frame::BackgroundPhase, clipRect);

	int buttonsOpacity = (ANIMATE_BUTTONS ? m_buttonsAnimator.value() : 100);

	// stage == 0: Draw old cover & information
	// stage == 1: Draw new cover & information
	for (int stage = 0; stage < 2; stage++) {
		// Figure out if it is necessary to draw the old information (when not animated or animation finished):
		bool oldTrack = (stage == 0);
		if (oldTrack && !m_trackAnimator.isInIntermediateState())
			continue;

		// Translate the painter, so that the drawing code draw both at the same place:
		int xTranslation = (oldTrack ? oldCoverDelta() : coverDelta());
		painterBuffer.translate(xTranslation, 0);

		// Draw frames:
		if ((oldTrack ? m_oldStatus : m_infos->status()) != PlayerInformation::Stopped) {
			// The rect should prefectly fit the cover size:
			QRect initialCoverRect = m_frameRects[Frame::Cover];
			QPixmap coverPixmap = (oldTrack ? m_oldCoverScalledPixmap : (m_infos->isPlaying() || m_orientation == FullScreen ? m_coverScalledPixmap : QPixmap()));
			int x = m_coverRect.left() + (m_coverRect.width() - coverPixmap.width()) / 2;
			int y = m_coverRect.bottom() + 1 - coverPixmap.height();
			m_frameRects[Frame::Cover]        = QRect(x, y, coverPixmap.width(), coverPixmap.height());
			m_frameRects[Frame::CoverOverlay] = QRect(x, y, coverPixmap.width(), coverPixmap.height());
			// Draw the frames, including the cover frame:
			drawFrames(&painterBuffer, Frame::ContentPhase, clipRect);
			// Back to previous state:
			m_frameRects[Frame::Cover]        = initialCoverRect;
			m_frameRects[Frame::CoverOverlay] = initialCoverRect;
		}

		// Draw the progressbar intermediate position during track animations (not a widget to not flicker):
		if (m_trackAnimator.isInIntermediateState() || progressBarWillBeShown) {
			QPixmap progressPixmap;
			if (oldTrack && m_oldDuration > 0) {
				if (m_oldDuration > 0)
					progressPixmap = m_progressBar->progressPixmap(/*underMouse*/false, m_oldDuration, m_oldPosition, m_oldLeftText, m_oldRightText);
			} else if (!oldTrack && m_infos->duration() > 0) {
				if (m_infos->isPlaying())
					progressPixmap = m_progressBar->progressPixmap(m_progressBar->underMouse(), m_infos->duration(), m_infos->position(), m_progressBar->leftText(), m_progressBar->rightText());
			}
			if (!progressPixmap.isNull())
				painterBuffer.drawPixmap(m_progressRect.x(), m_progressRect.y(), progressPixmap);
		}

		// Draw the cover:
		QRect effectiveCoverRect = m_coverRect;
		effectiveCoverRect.setHeight(m_coverRect.height() * 2 + m_reflectionDistance);
		if ((areControlsShown() || m_trackAnimator.isInIntermediateState()) && clipRect.intersects(effectiveCoverRect)) {
			// Draw cover:
			QPixmap coverPixmap = (oldTrack ? m_oldCoverScalledPixmap : (m_infos->isPlaying() || m_orientation == FullScreen ? m_coverScalledPixmap : QPixmap()));
			QRect r = m_coverRect; // Do not draw the cover if out of screen. And, more importantly, do not compute the reflexion in this case.
			r.moveLeft(r.left() + xTranslation);
			int x = m_coverRect.left() + (m_coverRect.width() - coverPixmap.width()) / 2;
			int y = m_coverRect.bottom() + 1 - coverPixmap.height();
			if (r.intersects(clipRect)) {
				painterBuffer.drawPixmap(x, y, coverPixmap);
			}
			// Draw reflexion:
			r = m_coverRect;
			r.moveTop(r.bottom() + 1 + m_reflectionDistance);
			r.moveLeft(r.left() + xTranslation);
			if (m_withReflexion && !coverPixmap.isNull() && r.intersects(clipRect)) {
				QPixmap coverReflexion = (oldTrack ? m_oldCoverReflexionPixmap : m_coverScalledReflexionPixmap);
				painterBuffer.drawPixmap(x, m_coverRect.bottom() + 1 + m_reflectionDistance, coverReflexion);
			}
		}

		drawFrames(&painterBuffer, Frame::OverlayPhase, clipRect);

//QTime begin = QTime::currentTime();

		// Draw stars:
		QRect effectiveStarsRect = m_starsRect;
		effectiveStarsRect.setHeight(m_starsRect.height() * 2 + m_reflectionDistance / 2);
		if ((areControlsShown() || m_trackAnimator.isInIntermediateState()) && clipRect.intersects(effectiveStarsRect)) {
			int rating = (oldTrack ? m_oldRating : ((m_orientation != FullScreen && m_infos->isPlaying()) || (m_orientation == FullScreen && m_infos->status() != PlayerInformation::Stopped) ? m_infos->rating() : -1));
			int hoveredStar = (oldTrack ? -1 : (m_infos->isPlaying() || m_orientation == FullScreen ? m_hoveredStar * 2 : -1));
			for (int i = 1; i <= 10 && rating >= 0; i += 2) { // Rating is -1 when drawing the old track, and the old track was "paused" or "stopped"
				int opacity = 100;
				// Compute the star icon to paint:
				QImage icon;
				QImage reflexionIcon;
				if (i == rating) { // Mini star
					if (hoveredStar > i) {
						icon          = m_hoveredMiniStarIcon;
						reflexionIcon = m_hoveredMiniStarReflexionIcon;
					} else {
						icon          = m_miniStarIcon;
						reflexionIcon = m_miniStarReflexionIcon;
					}
				} else if (i < rating) { // Big star
					if (hoveredStar > i) {
						icon          = m_hoveredStarIcon;
						reflexionIcon = m_hoveredStarReflexionIcon;
					} else {
						icon          = m_starIcon;
						reflexionIcon = m_starReflexionIcon;
					}
				} else if (hoveredStar > -1 || !m_buttonsAnimator.atBegin()) { // Gray star
					opacity = buttonsOpacity;
					if (i + 1 <= hoveredStar) {
						icon          = m_hoveredHoveredEmptyStarIcon;
						reflexionIcon = m_hoveredHoveredEmptyStarReflexionIcon;
					} else {
						icon          = m_hoveredEmptyStarIcon;
						reflexionIcon = m_hoveredEmptyStarReflexionIcon;
					}
				}
				// Paint the star icon:
				int thisStarX = m_starsRect.x() + (i - i % 2) * m_starWidth / 2;
				//painterBuffer.drawImage(thisStarX, m_starsRect.y(), icon);

				QRect r = QRect(thisStarX + xTranslation, m_starsRect.y(), m_starWidth, m_starWidth); // Do not draw the star if out of screen. And, more importantly, do not compute the reflexion in this case.
				if (r.intersects(clipRect)) {
					drawIcon(painterBuffer, icon, thisStarX, m_starsRect.y(), buffer, clipRect.x() - xTranslation, clipRect.y(), opacity);
				}

				r.moveTop(r.bottom() + 1 + m_reflectionDistance / 2);

				if (m_withReflexion && r.intersects(clipRect)) {
					// Create star reflexion:
					int starReflexionY = m_starsRect.bottom() + 1 + m_reflectionDistance / 2;
					drawIcon(painterBuffer, reflexionIcon, thisStarX, starReflexionY, buffer, clipRect.x() - xTranslation, clipRect.y(), opacity);
				}
			}
		}

//QTime endStars = QTime::currentTime();
//std::cout << "STARS PAINTED IN " << beginStars.msecsTo(endStars) << std::endl;


		// Draw texts (before buttons because buttons can overlap text with very small horizontal Kicker):
		if ((areControlsShown() || m_trackAnimator.isInIntermediateState()) && clipRect.intersects(m_textRect)) {
			// Get the text to display:
			int numberOfLines = (m_orientation == Horizontal && height() - 2 * m_margin < m_textRect.height() ? 2 : 3);
			QString text = (oldTrack ? m_oldText : (m_infos->isPlaying() || m_orientation == FullScreen ? informationText(numberOfLines) : ""));
			// Get the font in which to display the text:
			QFont font = m_textFont;
			painterBuffer.setFont(font);
			if (m_orientation == FullScreen) {
				while (font.pointSize() > this->font().pointSize() && painterBuffer.boundingRect(0, 0, 10000, 10000, 0, text).width() > m_textRect.width() - 16) {
					font.setPointSize(font.pointSize() - 1);
					painterBuffer.setFont(font);
				}
			}
			// Get the text color:
			QColor textColor = m_textColor;
			if (m_orientation == FullScreen)
				textColor = Theme::current()->textColor();

			//if (m_orientation == FullScreen && m_infos->status() != PlayerInformation::Stopped)
				//drawFrame(&painterBuffer, m_textRect, clipRect);

			// Draw the text shadow:
			if (m_orientation != FullScreen) { // Too time consuming and too little useless in full screen
				QColor shadowColor;
				if (qGray(m_textColor.rgb()) < 100)
					shadowColor = QColor(255, 255, 255);
				else
					shadowColor = QColor(0,0,0);
				painterBuffer.setPen(shadowColor);
				painterBuffer.drawText(QRect(m_textRect.x() - 1, m_textRect.y() - 1, m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x() - 1, m_textRect.y(),     m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x() - 1, m_textRect.y() + 1, m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x() + 1, m_textRect.y() - 1, m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x() + 1, m_textRect.y(),     m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x() + 1, m_textRect.y() + 1, m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x(),     m_textRect.y() - 1, m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x(),     m_textRect.y(),     m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				painterBuffer.drawText(QRect(m_textRect.x(),     m_textRect.y() + 1, m_textRect.width(), m_textRect.height()), Qt::AlignLeft | Qt::AlignVCenter, text);
				// And draw that information text:
				painterBuffer.setPen(textColor);
				painterBuffer.drawText(m_textRect, Qt::AlignLeft | Qt::AlignVCenter, text);
			} else if (!Theme::current()->textShadowEnabled()) {
				painterBuffer.setPen(textColor);
				painterBuffer.drawText(m_textRect, Qt::AlignLeft | Qt::AlignVCenter, text);
			} else {
				QColor shadowColor = Theme::current()->textShadowColor();//QColor(qGray(textColor.rgb()) < 100 ? Qt::white : Qt::black);
				drawShadowedText(&painterBuffer, m_textRect, Qt::AlignLeft | Qt::AlignVCenter, text, textColor, shadowColor, /*blured=*/true);
			}
		}

		// Remove the translation for other draws to be correct:
		painterBuffer.translate(-xTranslation, 0);
	}

//QTime begin = QTime::currentTime();

	// Draw the buttons:
	bool showButtons = ((m_underMouse && areControlsShown()) || m_buttonsAnimator.isInIntermediateState());
	if (showButtons && clipRect.intersects(m_prevRect)) {
		// Prev button:
		int index = (m_clickedButton == 0 && m_hoveredButton == 0 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 0 ? 1 : 0));
		//painterBuffer.drawPixmap(m_prevRect.x() + m_buttonPadding, m_prevRect.y() + m_buttonPadding, m_prevIcon[index]);
		drawIcon(painterBuffer, m_prevIcon[index], m_prevRect.x() + m_buttonPadding, m_prevRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), buttonsOpacity);
	}
	if ((showButtons || !m_infos->isPlaying()) && clipRect.intersects(m_playRect)) {
		// Play/pause button:
		int index = (m_clickedButton == 1 && m_hoveredButton == 1 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 1 ? 1 : 0));
		QImage playPauseIcon = (m_infos->isPlaying() ? (m_infos->canPause() ? m_pauseIcon[index] : m_stopIcon[index]) : m_playIcon[index]);
		//painterBuffer.drawPixmap(m_playRect.x() + m_buttonPadding, m_playRect.y() + m_buttonPadding, playPauseIcon);
		int playOpacity = (ANIMATE_BUTTONS ? m_playPauseAnimator.value() : 100);
		drawIcon(painterBuffer, playPauseIcon, m_playRect.x() + m_buttonPadding, m_playRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), playOpacity);
	}
	if (showButtons && clipRect.intersects(m_nextRect)) {
		// Next button:
		int index = (m_clickedButton == 2 && m_hoveredButton == 2 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 2 ? 1 : 0));
		//painterBuffer.drawPixmap(m_nextRect.x() + m_buttonPadding, m_nextRect.y() + m_buttonPadding, m_nextIcon[index]);
		drawIcon(painterBuffer, m_nextIcon[index], m_nextRect.x() + m_buttonPadding, m_nextRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), buttonsOpacity);
	}
	if (showButtons && clipRect.intersects(m_fullRect)) {
		// Fullscreen button:
		int index = (m_clickedButton == 3 && m_hoveredButton == 3 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 3 ? 1 : 0));
		//painterBuffer.drawPixmap(m_fullRect.x() + m_buttonPadding, m_fullRect.y() + m_buttonPadding, m_fullIcon[index]);
		drawIcon(painterBuffer, m_fullIcon[index], m_fullRect.x() + m_buttonPadding, m_fullRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), buttonsOpacity);
	}
	if (showButtons && clipRect.intersects(m_closeRect)) {
		// Close button:
		int index = (m_clickedButton == 4 && m_hoveredButton == 4 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 4 ? 1 : 0));
		//painterBuffer.drawPixmap(m_closeRect.x() + m_buttonPadding, m_closeRect.y() + m_buttonPadding, m_closeIcon[index]);
		drawIcon(painterBuffer, m_closeIcon[index], m_closeRect.x() + m_buttonPadding, m_closeRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), buttonsOpacity);
	}
	if (showButtons && clipRect.intersects(m_themeRect)) {
		// Theme button:
		int index = (m_clickedButton == 5 && m_hoveredButton == 5 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 5 ? 1 : 0));
		//painterBuffer.drawPixmap(m_themeRect.x() + m_buttonPadding, m_themeRect.y() + m_buttonPadding, m_themeRect[index]);
		drawIcon(painterBuffer, m_themeIcon[index], m_themeRect.x() + m_buttonPadding, m_themeRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), buttonsOpacity);
	}
	if (showButtons && clipRect.intersects(m_lyricsRect)) {
		// Show Lyrics button:
		int index = (m_clickedButton == 6 && m_hoveredButton == 6 ? 2 : (m_clickedButton == -1 && m_hoveredButton == 6 ? 1 : 0));
		//painterBuffer.drawPixmap(m_themeRect.x() + m_buttonPadding, m_themeRect.y() + m_buttonPadding, m_themeRect[index]);
		drawIcon(painterBuffer, m_lyricsIcon[index], m_lyricsRect.x() + m_buttonPadding, m_lyricsRect.y() + m_buttonPadding, buffer, clipRect.x(), clipRect.y(), buttonsOpacity);
	}

	// Draw the bottom frame:
//	int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
//	int taskbarHeight = TASKBAR_MARGIN + 16 + TASKBAR_MARGIN;
//	int clockHeight   = m_clockRect.height();
//	int bottomBarY    = this->height() - QMAX(taskbarHeight, clockHeight) - 2 * clockMargin - 1;
//	QRect bottomRect = QRect(0, bottomBarY, width(), height() - bottomBarY);
//	if (m_orientation == FullScreen && clipRect.intersects(bottomRect))
//		drawFrame(&painterBuffer, bottomRect, clipRect);

	// Draw the taskbar:
	if (m_taskBarWatcher && clipRect.intersects(m_taskBarRect)) {
		const QValueList<TaskBarEntry> &entries = m_taskBarWatcher->entries();

		QFont font = this->font();
		font.setPixelSize(16);
		painterBuffer.setFont(font);

		for (int i = 0; i < (int) entries.count(); ++i) {
			int entryX = m_taskBarRect.x() + (m_taskBarEntryWidth + TASKBAR_MARGIN) * i;

			// Example:
			// Flicker 5 times and keep blue after the 5° flickering:
			// 0 1 2 3 4 5 6 7 8 9 10
			// t   t   t   t   t t t t t t
			int flickerCount = taskBarFlickerCount();
			int tickCount = (flickerCount - 1) * 2;
			bool on = (m_taskBarFlickerings[entries[i].wid] < tickCount ? ! (m_taskBarFlickerings[entries[i].wid] % 2) : true);

			// Draw background highlight:
			painterBuffer.setPen(on ? KGlobalSettings::highlightedTextColor() : Qt::white);
			painterBuffer.setBrush(QBrush());
			if (on) {
				KPixmap gradient;
				gradient.resize(m_taskBarEntryWidth - 2, m_taskBarRect.height() - 2);
				QColor color = KGlobalSettings::highlightColor();
				if (m_hoveredTaskBarEntry == i)
					color = color.light(115);
				KPixmapEffect::gradient(gradient, color, color.dark(), KPixmapEffect::VerticalGradient);
				painterBuffer.drawPixmap(entryX + 1, m_taskBarRect.y() + 1, gradient);
			}
			painterBuffer.drawRect(entryX, m_taskBarRect.y(), m_taskBarEntryWidth, m_taskBarRect.height());
			// Draw text:
			painterBuffer.drawPixmap(entryX + TASKBAR_MARGIN, m_taskBarRect.y() + (m_taskBarRect.height() - 16) / 2, entries[i].icon);
			painterBuffer.drawText(entryX + TASKBAR_MARGIN + 16 + TASKBAR_MARGIN / 2,
			                       m_taskBarRect.y(),
			                       m_taskBarEntryWidth - TASKBAR_MARGIN - 16 - TASKBAR_MARGIN / 2 - TASKBAR_MARGIN,
			                       m_taskBarRect.height(),
			                       Qt::AlignLeft | Qt::AlignVCenter, entries[i].title);
		}
	}

	// Draw the Next Playing text:
	if (m_nextPlayingRichText && clipRect.intersects(m_nextPlayingRect)) {
//		int nextPlayingMargin = width() / 40;
//		int x           = m_nextPlayingRect.x();//width()  - m_nextPlayingRichText->widthUsed() - nextPlayingMargin;
//		int y           = m_nextPlayingRect.y();//height() - m_nextPlayingRichText->height()    - nextPlayingMargin;
//		int width       = m_nextPlayingRichText->widthUsed();
//		int height      = m_nextPlayingRichText->height();

		if (!Theme::current()->nextPlayingShadowEnabled()) {
			QColorGroup colorGroup = this->colorGroup();
			colorGroup.setColor(QColorGroup::Text, Theme::current()->nextPlayingColor());
			m_nextPlayingRichText->draw(&painterBuffer, m_nextPlayingRect.x(), m_nextPlayingRect.y(), QRect()/*m_nextPlayingRect*/  /*QRect(x, y, width, height)*/, colorGroup);
		} else {
			QColor shadowColor = Theme::current()->nextPlayingShadowColor();//QColor(qGray(Theme::current()->nextPlayingColor().rgb()) < 100 ? Qt::white : Qt::black);
			drawShadowedRichText(&painterBuffer, m_nextPlayingRect, m_nextPlayingRichText, m_nextPlayingRichTextSource, Theme::current()->nextPlayingColor(), shadowColor, /*blured=*/true/*false*/);
		}

	}

	// Draw Lyrics frame on top of CoverAndText frame and Nextplaying text:
	drawFrames(&painterBuffer, Frame::OnTopPhase, clipRect);

	// Draw the clock:
	if (m_clockRichText && clipRect.intersects(m_clockRect)) {
//		int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
//		int x           = width()  - m_clockRichText->widthUsed() - clockMargin;
//		int y           = height() - m_clockRichText->height()    - clockMargin;
//		int width       = m_clockRichText->widthUsed();
//		int height      = m_clockRichText->height();
	//	QColorGroup colorGroup = this->colorGroup();
		//colorGroup.setColor(QColorGroup::Text, Qt::white);

//		if (m_orientation == FullScreen) {
			//drawFrame(&painterBuffer, m_clockRect/*QRect(x, y, width, height)*/, clipRect);
			//painterBuffer.setPen(Qt::red);
			//painterBuffer.drawRect(m_clockRect/*QRect(x, y, width, height)*/, clipRect);
//		}


		if (!Theme::current()->dateHourShadowEnabled()) {
			QColorGroup colorGroup = this->colorGroup();
			colorGroup.setColor(QColorGroup::Text, Theme::current()->dateHourColor());
			m_clockRichText->draw(&painterBuffer, m_clockRect.x(), m_clockRect.y(), QRect()/*m_clockRect*/  /*QRect(x, y, width, height)*/, colorGroup);
		} else {
			QColor shadowColor = Theme::current()->dateHourShadowColor();//QColor(qGray(Theme::current()->dateHourColor().rgb()) < 100 ? Qt::white : Qt::black);
			drawShadowedRichText(&painterBuffer, m_clockRect, m_clockRichText, m_clockRichTextSource, Theme::current()->dateHourColor(), shadowColor, /*blured=*/true);
		}

	}


//QTime end = QTime::currentTime();
//std::cout << "BUTTONS PAINTED IN " << begin.msecsTo(end) << std::endl;

// 	// Test debug:
// 	int length = 17;
// 	QColor colors[] = { Qt::white, Qt::black, Qt::red, Qt::darkRed, Qt::green,
// 	                    Qt::darkGreen, Qt::blue, Qt::darkBlue, Qt::cyan, Qt::darkCyan,
// 	                    Qt::magenta, Qt::darkMagenta, Qt::yellow, Qt::darkYellow, Qt::gray,
// 	                    Qt::darkGray, Qt::lightGray };
// 	static int i = -1;
// 	i++;
// 	painterBuffer.setPen(colors[i % length]);
// 	painterBuffer.drawRect(clipRect);
// 	QRect r = clipRect;
// 	r = QRect(r.x() + 1, r.y() + 1, r.width() - 2, r.height() - 2);
// 	painterBuffer.drawRect(r);
	//painterBuffer.fillRect(0, 0, width(), height(), colors[i]);

// 	for (int i = 0; i < regionRects.size(); i++) {
// 		QRect r = regionRects[i];
// 		painterBuffer.drawRect(r);
// 	}

	// End:
	painterBuffer.end();
	QPainter painter(this);
//	painter.drawPixmap(clipRect.x(), clipRect.y(), buffer);
	for (uint i = 0; i < regionRects.size(); i++) {
		QRect r = regionRects[i];
		painter.drawPixmap(r.x(), r.y(), buffer, r.x() - clipRect.x(), r.y() - clipRect.y(), r.width(), r.height());
	}
	painter.end();

// 	QTime end = QTime::currentTime();
// 	std::cout << m_orientation << " : Painted in " << begin.msecsTo(end) << "  ";
// 	if (m_underMouse && areControlsShown() && clipRect.intersects(m_prevRect))   std::cout << " Previous";
// 	if (m_underMouse && clipRect.intersects(m_playRect))                         std::cout << " Play";
// 	if (m_underMouse && areControlsShown() && clipRect.intersects(m_nextRect))   std::cout << " Next";
// 	if (m_underMouse && m_infos->isPlaying() && clipRect.intersects(m_fullRect)) std::cout << " FullScreen";
// 	if (m_underMouse && clipRect.intersects(m_closeRect))                        std::cout << " Close";
// 	if (m_underMouse && clipRect.intersects(m_themeRect))                        std::cout << " Theme";
// 	if (areControlsShown() && clipRect.intersects(m_progressRect))               std::cout << " Progress";
// 	if (areControlsShown() && clipRect.intersects(effectiveCoverRect))           std::cout << " Cover";
// 	if (areControlsShown() && clipRect.intersects(m_starsRect))                  std::cout << " Stars";
// 	if (areControlsShown() && clipRect.intersects(m_textRect))                   std::cout << " Text";
// 	std::cout << std::endl;
}


void CoverDisplay::drawIcon(QPainter &painter, const QImage &iconImage, int x, int y, const QPixmap &buffer, int bufferX, int bufferY, int percentVisible)
{
	if (percentVisible >= 100)
		painter.drawImage(x, y, iconImage);
	else if (percentVisible > 0) {
		// Create a pixmap that copies the background of the icon:
		QPixmap backgroundPixmap(iconImage.size());
		QPainter backgroundPainter(&backgroundPixmap);
		backgroundPainter.drawPixmap(0, 0, buffer, x - bufferX, y - bufferY);
		backgroundPainter.end();
		// Create images in order to do the composition:
		QImage backgroundImage = backgroundPixmap.convertToImage();
		KImageEffect::blendOnLower(iconImage, QPoint(0, 0), backgroundImage, QRect(0, 0, backgroundImage.width(), backgroundImage.height()), percentVisible / 100.0);
		painter.drawImage(x, y, backgroundImage); // FIXME: Directly blend on buffer pixmap?
	}
}


void CoverDisplay::mouseReleaseEvent(QMouseEvent *event)
{
	bool justClosed = false;

	if (m_clickedButton != -1) {
		// If the button has been clicked, and released, execute it:
		if (m_clickedButton == m_hoveredButton) {
			switch (m_clickedButton) {
				case 0:
					AmarokApi::previousTrack();
					break;
				case 1:
					// Stop is Pause if not supported (for Last.fm for instance):
					if (!m_infos->canPause())
						AmarokApi::stop();
					// Pause if it is playing:
					else if (m_infos->isPlaying())
						AmarokApi::playPause();
					// Try to play:
					else if (!AmarokApi::playPause())
						// Not successful, amarok is not started yet. Start it and play:
						KRun::runCommand("amarok --play", "amarok", "amarok"); // Application name and icon name is for startup feedback mechanism
					break;
				case 2:
					AmarokApi::nextTrack();
					break;
				case 3:
					emit fullScreenAsked();
					break;
				case 4:
					emit closeAsked();
					justClosed = true;
					break;
				case 5:
					dialogAppeared();
					ThemeManager::chooseTheme(width(), height(), this/*topLevelWidget()*/);
					dialogDisappeared();
					break;
				case 6:
					toggleLyrics();
					break;
				default: ;
			}
		}
		// Visually unpress the button:
		m_clickedButton = -1;
		updateButtons();
	}

	if (m_orientation == FullScreen && !justClosed) {
		// The buttons were still shown during click, considere that mouse release as another mouse activity, to not hide the buttons immediatly:
		m_fullScreenTimer.start(FULL_SCREEN_TIMOUT, /*singleShot=*/true);
	}

	QWidget::mouseReleaseEvent(event);
}

void CoverDisplay::updateProgressBar()
{
	QString time = i18n("%1:%2").arg(m_infos->position() / 60).arg(QString::number(m_infos->position() % 60).rightJustify(2, '0'));
	QString left = i18n("%1:%2").arg((m_infos->duration() - m_infos->position()) / 60).arg(QString::number((m_infos->duration() - m_infos->position()) % 60).rightJustify(2, '0'));

	m_progressBar->setCurrent(m_infos->position());
	m_progressBar->setLeftText(m_infos->duration() == 0 ? "" : time);
	m_progressBar->setRightText(m_infos->duration() == 0 ? "" : "-" + left);

	if (m_infos->isPlaying() && m_orientation != FullScreen) {
		m_timesString = (m_infos->duration() > 0 ? i18n("%1, %2 left").arg(time, left) : time);
		QString newLine = (m_orientation == FullScreen && !Settings::showLyrics() ? "\n\n" : "\n");
		QString text = m_timesString + "\n\n" + informationText(3);
		if (!m_infos->nextPlaying().isEmpty())
			text += "\n\n" + m_infos->nextPlaying();
		QToolTip::add(this, text);
	} else {
		m_timesString = "";
		QToolTip::add(this, "");
	}
#if KDE_IS_VERSION( 3, 5, 0 )
	Client::updateKickerTip();
#endif
}

void CoverDisplay::updateNextPlaying()
{
	if (m_nextPlayingRichText) {
//		int nextPlayingMargin = width() / 40;
//		int x           = width()  - m_nextPlayingRichText->widthUsed() - nextPlayingMargin;
//		int y           = height() - m_nextPlayingRichText->height()    - nextPlayingMargin;
		int width       = m_nextPlayingRichText->widthUsed();
		int height      = m_nextPlayingRichText->height();
		int shadow = (Theme::current()->nextPlayingShadowEnabled() ? 2 * BLURED_SHADOW_THICKNESS/*SOLID_SHADOW_THICKNESS*/ : 0);
		QRect rect = QRect(m_nextPlayingRect.x() - shadow, m_nextPlayingRect.y() - shadow, width + 2 * shadow, height + 2 * shadow);
		update(rect);
	}
}

void CoverDisplay::updateClock()
{
	if (m_clockRichText) {
//		int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
//		int x           = width()  - m_clockRichText->widthUsed() - clockMargin;
//		int y           = height() - m_clockRichText->height()    - clockMargin;
//		int width       = m_clockRichText->widthUsed();
//		int height      = m_clockRichText->height();
//		m_clockRect = QRect(x, y, width, height);
		update(m_clockRect);
	}
}

void CoverDisplay::updateTaskBar()
{
	if (m_taskBarWatcher) {
		int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
		int x           = clockMargin;
		int width       = this->width() - m_clockRect.width() - 3 * clockMargin;
		int height      = TASKBAR_MARGIN + 16 + TASKBAR_MARGIN;
		int y           = this->height() - height - clockMargin - 1;
		m_taskBarRect = QRect(x, y, width, height);

		const QValueList<TaskBarEntry> &entries = m_taskBarWatcher->entries();
		m_taskBarEntryWidth = 0;
		if (entries.count() > 0)
			m_taskBarEntryWidth = QMIN(/*maxWidth=*/200, (width - (entries.count() - 1) * TASKBAR_MARGIN) / entries.count());

		update(x, y, width + clockMargin, height); // Update more width, in case the clock changed date
	}
}

void CoverDisplay::newClockInformation()
{
	updateClock();

	delete m_clockRichText;
// 	QString clockText = QString("<center><font size=4 color=%1>%2</font>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<font size=6 color=%3><b>%4</b></font></center>").arg(
// 		Theme::current()->dateHourColor().name(),
// 		KGlobal::locale()->formatDate(QDate::currentDate(), /*shortFormat=*/false)
// 	).arg(
// 		Theme::current()->dateHourColor().name(),
// 		KGlobal::locale()->formatTime(QTime::currentTime(), /*includeSeconds=*/false)
// 	);
	QString clockText = QString("<center><font size=4>%1</font>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<font size=6><b>%2</b></font></center>").arg(
		KGlobal::locale()->formatDate(QDate::currentDate(), /*shortFormat=*/false),
		KGlobal::locale()->formatTime(QTime::currentTime(), /*includeSeconds=*/false)
	);
	QFont font = this->font();
	font.setPointSize(font.pointSize() + 3);
	m_clockRichText = new QSimpleRichText(clockText, font);
	m_clockRichText->setWidth(width());
	m_clockRichText->setWidth(m_clockRichText->widthUsed());
	m_clockRichTextSource = clockText;

	int clockMargin = 10/*width() / 40*/; // TODO: Duplicate
	int x           = width()  - m_clockRichText->widthUsed() - clockMargin;
	int y           = height() - m_clockRichText->height()    - clockMargin;
	int width       = m_clockRichText->widthUsed();
	int height      = m_clockRichText->height();
	m_clockRect = QRect(x, y, width, height);

	if (!m_clockInitialized) {
		QTimer::singleShot( 60 * 1000 - 1000 * QTime::currentTime().second() - QTime::currentTime().msec(), this, SLOT(initClock()) );
		m_clockInitialized = true;
	}

	updateClock();
}

void CoverDisplay::initClock()
{
	connect( &m_clockTimer, SIGNAL(timeout()), this, SLOT(newClockInformation()) );
	m_clockTimer.start(60 * 1000, /*singleShot=*/false);
	newClockInformation();
}

void CoverDisplay::setPosition(int position)
{
	if (areControlsShown())
		m_infos->changePosition(position);
		//AmarokApi::seek(position);
}

// bool CoverDisplay::close(bool alsoDelete)
// {
// 	// Same as in hideEvent() but for times where is called by code (close()). There should be no updateButtons()!
// 	// This "event handler" does not take into account minimization of the window, in contrary to hideEvent().
// 	hideControls(/*update=*/false);
//
// 	return QWidget::close(alsoDelete);
// }

void CoverDisplay::hideEvent(QHideEvent *event)
{
	if (ANIMATE_BUTTONS) {
		m_buttonsAnimator.setValueAndState(0, ShowAnimator::NotRunning);
		m_playPauseAnimator.setValueAndState(0, ShowAnimator::NotRunning);
	}

	// Do not hide the mouse & buttons a few seconds later
	// if the user hide the window and show it again less than a few seconds later:
	hideControls(/*update=false*/);

	QWidget::hideEvent(event);
}

void CoverDisplay::hideControls(/*bool update = true*/)
{
	//if (m_clickedButton == -1) {
		m_fullScreenTimer.stop(); // In case it is called outside of a timeout (eg. when hiding window)
		m_underMouse = false;
		m_hoveredStar = -1;
		//QTimer::singleShot( 1000, this, SLOT(setBlankCursor()) ); // After a small delay for the VERY FIRST time the full-screen is shown when clicking the button in the applet: the cursor was not hidden
		// Does not solve the problem :-/
		setBlankCursor();
		//setCursor(Qt::BlankCursor);
		//if (m_lyrics)
		//	m_lyrics->viewport()->setCursor(Qt::BlankCursor);
		m_progressBar->removeHighlighting();
		m_hoveredButton = -1;
//		if (false/*update*/) {
			if (ANIMATE_BUTTONS) {
				m_buttonsAnimator.toBegin();
				m_playPauseAnimator.toBegin();
			} else if (isVisible()) {
				updateButtons();
				updateStars();
			}
//		}
	//}
}

void CoverDisplay::setBlankCursor()
{
	setCursor(Qt::BlankCursor);
	if (m_lyrics)
		m_lyrics->viewport()->setCursor(Qt::BlankCursor);
}

void CoverDisplay::dialogAppeared()
{
// From ???:
	if (m_hoveredButton != -1) {
		m_hoveredButton = -1;
		updateButtons();
	}

// From mouseMoveEvent:   // TODO add a showControls() method
	if (m_orientation == FullScreen && isVisible()) {
		// Update the "last moved mouse" time:
		m_fullScreenTimer.start(FULL_SCREEN_TIMOUT, /*singleShot=*/true);
		// Show controls if needed:
		if (m_underMouse == false) {
			m_underMouse = true;
			unsetCursor(); // Restore / show again the cursor
			if (m_lyrics)
				m_lyrics->viewport()->unsetCursor();
			if (ANIMATE_BUTTONS) {
				m_buttonsAnimator.toEnd();
				m_playPauseAnimator.toEnd();
			} else {
				updateButtons();
				updateStars();
			}
		}
	}

// From leaveEvent:
	if (m_hoveredStar != -1) {
		m_hoveredStar = -1;
		updateStars();
//		unsetCursor();
//		if (m_lyrics)
//			m_lyrics->viewport()->unsetCursor();
	}

	unsetCursor();
	if (m_lyrics)
		m_lyrics->viewport()->unsetCursor();
	m_fullScreenTimer.stop();

	focusCoverOnDialogDisappearing = true;
}

bool CoverDisplay::focusCoverOnDialogDisappearing = true;

void CoverDisplay::dialogDisappeared()
{
//	m_fullScreenTimer.start(FULL_SCREEN_TIMOUT, /*singleShot=*/true);
	if (m_orientation == FullScreen && isVisible()) {
		QMouseEvent *mouseEvent = new QMouseEvent(QEvent::MouseMove, mapFromGlobal(QCursor::pos()), 0, 0);
		mouseMoveEvent(mouseEvent);
		if (focusCoverOnDialogDisappearing) {
			QWidget *parentWidget = dynamic_cast<QWidget *>(parent());
			parentWidget->setFocus();
			KWin::forceActiveWindow(parentWidget->winId());
		}
	}
}

int CoverDisplay::coverDelta()
{
	int value = m_trackAnimator.value();

	if (value == 100 || value == 0)
		return 0;

	if (m_reverseTrackAnimation)
		return (width() * value / 100) - width();
	else
		return width() - (width() * value / 100);
}

int CoverDisplay::oldCoverDelta()
{
	int value = m_trackAnimator.value();

	if (value == 100 || value == 0)
		return -1000;

	if (m_reverseTrackAnimation)
		return coverDelta() + width();
	else
		return coverDelta() - width();
}

void CoverDisplay::updateCover()
{
	QRect rect = m_coverRect;
	rect.setHeight(rect.height() + m_reflectionDistance + rect.height() * 2 / 3);
//	rect.moveLeft(rect.left() + coverDelta());
	update(rect);
	if (m_trackAnimator.isInIntermediateState()) {
		//rect.moveLeft(rect.left() - coverDelta() + oldCoverDelta());
		//update(rect);
		int left = rect.left();
		rect.moveLeft(left + coverDelta());
		update(rect);
		rect.moveLeft(left + oldCoverDelta());
		update(rect);
	}
	updateContentFrames();
}

void CoverDisplay::updateStars()
{
	if (!isVisible())
		return;

	QRect rect = m_starsRect;
	rect.setHeight(rect.height() + m_reflectionDistance + rect.height() * 2 / 3);
//	rect.moveLeft(rect.left() + coverDelta());
	update(rect);
	if (m_trackAnimator.isInIntermediateState()) {
		int left = rect.left();
		rect.moveLeft(left + coverDelta());
		update(rect);
		rect.moveLeft(left + oldCoverDelta());
		update(rect);
	}
}

void CoverDisplay::updateButtons()
{
	if (!isVisible())
		return;

	int length = 7;
	QRect rects[] = { m_prevRect, m_nextRect, m_playRect, m_fullRect, m_closeRect, m_themeRect, m_lyricsRect }; // Important: change length when changing the list!

	for (int i = 0; i < length; i++)
		update(rects[i]);
}

void CoverDisplay::updateText()
{
	int shadow = (m_orientation == FullScreen ? (Theme::current()->textShadowEnabled() ? 2 * BLURED_SHADOW_THICKNESS : 0) : 1);

	QRect rect(m_textRect.x() - shadow, m_textRect.y() - shadow, m_textRect.width() + 2 * shadow, m_textRect.height() + 2 * shadow);
	if (m_orientation == FullScreen)
		rect = QRect(m_textRect.x() - 1 - 5, m_textRect.y() - 1 - 5, m_textRect.width() + 2 + 10, m_textRect.height() + 2 + 10);

//	rect.moveLeft(rect.left() + coverDelta());
	update(rect);
	if (m_trackAnimator.isInIntermediateState()) {
		int left = rect.left();
		rect.moveLeft(left + coverDelta());
		update(rect);
		rect.moveLeft(left + oldCoverDelta());
		update(rect);
	}

	updateContentFrames();

//	if (m_orientation == FullScreen && !m_trackAnimator.isInIntermediateState()) // Optimisation, to not loss 100 ms in every timeout of Amarok DCOP calls
//		m_infos->resumePolling(); // Animation finished
}

void CoverDisplay::updateProgressDraw()
{
	QRect rect = m_progressRect;
//	rect.moveLeft(rect.left() + coverDelta());
	update(rect);
	if (m_trackAnimator.isInIntermediateState()) {
		int left = rect.left();
		rect.moveLeft(left + coverDelta());
		update(rect);
		rect.moveLeft(left + oldCoverDelta());
		update(rect);
	}
}

bool CoverDisplay::areControlsShown() // FIXME: bool canShowControls()
{
	return (m_infos->isPlaying() || m_orientation == FullScreen);
}

void CoverDisplay::newInformation()
{
	// Update mouse hover effect:
	if (m_underMouse) {
		m_hoveredStar   = -1;
		m_hoveredButton = -1;
		m_clickedButton = -1;
		QMouseEvent *mouseEvent = new QMouseEvent(QEvent::MouseMove, mapFromGlobal(QCursor::pos()), 0, 0);
		kapp->postEvent(this, mouseEvent);
	}

	// Get the cover pixmap:
	QPixmap coverPixmap = m_infos->coverPixmap();
	// Scale it at its display size:
	QImage coverScalledImage = coverPixmap.convertToImage();
	if (!coverPixmap.isNull())
		coverScalledImage = coverScalledImage.smoothScale(m_coverRect.width(), m_coverRect.height(), QImage::ScaleMin);
	// And transform the cover and its reflexion to QPixmap, for very fast drawing:
	m_coverScalledPixmap.convertFromImage(coverScalledImage);
	m_coverScalledReflexionPixmap = Tools::reflexionPixmap(coverScalledImage);

	// Update Next Playing area:
	if (m_orientation == FullScreen) {
		updateNextPlaying();

		delete m_nextPlayingRichText;
		QString text = m_infos->nextPlaying();
		int position = text.find('\n');
		QString line1      = (position >= 0 ? text.left(position)                    : text).replace('<', "&lt;").replace('\n', "<br>");
		QString otherLines = (position >= 0 ? text.right(text.length() - position - 1) : "").replace('<', "&lt;").replace('\n', "<br>");
		QString nextPlayingText = QString("<nobr><font size=4><i>%1%2</i></font></nobr>").arg(line1, (otherLines.isEmpty() ? "" : QString("<br><b>%2</b>").arg(otherLines)));
		QFont font = this->font();
		font.setPointSize(font.pointSize() + 3);
		m_nextPlayingRichText = new QSimpleRichText(nextPlayingText, font);
		m_nextPlayingRichText->setWidth(width());
		m_nextPlayingRichTextSource = nextPlayingText;

		updateNextPlaying();
	}


	// Update animations:
	if (ANIMATE_BUTTONS && m_underMouse && m_orientation != FullScreen) {
		if (m_infos->isPlaying())
			m_buttonsAnimator.toEnd();
		else
			m_buttonsAnimator.toBegin();
	}

//	// Reset the user-moved lyrics offset:
//	m_lyricsOffset = 0;

	// Update the progressbar data (it does not take its information from m_infos);
	newProgress();

	m_progressBar->setTextFlickering(m_infos->status() == PlayerInformation::Paused);

	// Update the display:
	updateCover();
	updateStars();
	updateText();
}

void CoverDisplay::newRating()
{
	updateStars();
}

void CoverDisplay::newProgress()
{
	m_progressBar->setTotal(m_infos->duration());
	m_progressBar->setCurrent(m_infos->position());
	updateProgressBar();

	if (m_lyrics && m_infos->duration() > 0) {
		int x = m_lyrics->contentsX() + m_lyrics->visibleWidth() / 2; // Keep same position
		int y = m_lyrics->contentsHeight() * m_infos->position() / m_infos->duration();

// 		int topY = y - m_lyrics->visibleHeight() / 2 - (m_lyrics->visibleHeight() % 2 ? 1 : 0);
// 		if (topY > m_lyrics->contentsHeight() - m_lyrics->visibleHeight())
// 			topY = m_lyrics->contentsHeight() - m_lyrics->visibleHeight();
// 		if (topY < 0) // After the previous test, because if contentsHeight < visibleHeight, it would end up with a negative Y
// 			topY = 0;
// 		std::cout << "about to move to y=" << topY << std::endl;

		m_lyrics->center(x, y + m_lyricsOffset);
	}
}

void CoverDisplay::newLyrics()
{
	if (m_lyrics) {
		if (m_infos->lyrics().isEmpty()) {
			m_lyrics->setTextFormat(Qt::RichText);
			m_lyrics->setText(
				"<p><strong>" "No lyrics available for this song." "</strong></p><ul>"
				"<li>" "<strong>Wait</strong> a few seconds;" "</li>"
				"<li>" "Always <strong>open the Lyrics contextual tab</strong> in Amarok to enable automatic retrieving;" "</li>"
				"<li>" "If Amarok asks you, <strong>choose between several lyrics possibilities</strong> in the contextual tab;" "</li>"
				"<li>" "<strong>Install the script Wiki-Lyrics</strong> available at http://kde-apps.org/content/show.php?content=35151 to retreive lyrics from more websites;" "</li>"
				"<li>" "<strong>Manually assign lyrics</strong> to a song in its Track Information window." "</li>"
				"</ul>"
			);
		} else {
			//m_lyrics->setTextFormat(Qt::PlainText);
			m_lyrics->setTextFormat(Qt::RichText);
			QString lyrics = m_infos->lyrics();
			lyrics = lyrics.replace("\n", "<br>");
			m_lyrics->setText(lyrics);
		}
	}
}

void CoverDisplay::beforeAnimateTrack(int nextStatus, bool isPrevious)
{
	if (!ANIMATE_TRACKS)
		return;

	// Full-screen always show information. No transition for playing / not playing:
	if (m_orientation == FullScreen && (m_infos->status() == PlayerInformation::Paused || nextStatus == PlayerInformation::Paused))//m_infos->isPlaying() != (nextStatus != PlayerInformation::Stopped))
		return;

	m_trackAnimator.setValueAndState(0, ShowAnimator::MoveForward);
	m_reverseTrackAnimation = isPrevious/* || (!m_infos->isPlaying() && willPlay)*/;

	// If full screen, we are moving to another song, reset the user-moved lyrics offset:
	m_lyricsOffset = 0;

	int numberOfLines = (m_orientation == Horizontal && height() - 2 * m_margin < m_textRect.height() ? 2 : 3);

	m_oldStatus               = ( m_infos->status() );
	m_oldCoverScalledPixmap   = ( m_infos->isPlaying() ? m_coverScalledPixmap           : QPixmap() );
	m_oldCoverReflexionPixmap = ( m_infos->isPlaying() ? m_coverScalledReflexionPixmap  : QPixmap() );
	m_oldText                 = ( m_infos->isPlaying() ? informationText(numberOfLines) : ""        );
	m_oldRating               = ( m_infos->isPlaying() ? m_infos->rating()              : -1        ); // -1 to not draw any star, because it relates to no track
	m_oldDuration             = ( m_infos->isPlaying() ? m_infos->duration()            : 0         );
	m_oldPosition             = ( m_infos->isPlaying() ? m_infos->position()            : 0         );
	m_oldLeftText             = ( m_infos->isPlaying() ? m_progressBar->leftText()      : ""        );
	m_oldRightText            = ( m_infos->isPlaying() ? m_progressBar->rightText()     : ""        );

//	if (m_orientation == FullScreen) // Optimisation, to not loss 100 ms in every timeout of Amarok DCOP calls
//		m_infos->suspendPolling();
}

const QPixmap* CoverDisplay::backgroundPixmap()
{
	if (m_orientation == FullScreen)
		return & Theme::current()->backgroundPixmap(width(), height());
	else
		return paletteBackgroundPixmap();
}

#if KDE_IS_VERSION( 3, 5, 0 )
void CoverDisplay::updateKickerTip(KickerTip::Data &data)
{
	if (!m_infos->isPlaying() || m_orientation == FullScreen)
		return;

	QString text = informationText(3);
	if (!m_infos->nextPlaying().isEmpty())
		text += "\n\n" + m_infos->nextPlaying();

	data.message   = m_timesString;
	data.subtext   = QStyleSheet::escape(text).replace('\n', "<br>");
	data.icon      = kapp->iconLoader()->loadIcon("kirocker", KIcon::Desktop, 48, KIcon::DefaultState);
	data.direction = m_applet->popupDirection();
}
#endif

void CoverDisplay::fontChange(const QFont &oldFont)
{
	QWidget::fontChange(oldFont);

	if (m_orientation == Horizontal || m_orientation == Vertical) {
		m_orientation = FullScreen; // This is temporar. In a few milliseconds, widthForHeight or heightForWidth will be called
		emit updateLayout();        //  We set to a different orientation to be sure the whole layout is recomputed (do not return the cached geometry)
	}
}

#include "coverdisplay.moc"
