/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "framemanager.h"

#include "settings.h"
//#include "framedialogs.h"
#include "tools.h"

#include <qpixmap.h>
#include <qimage.h>
#include <qdir.h>
#include <qstringlist.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kstandarddirs.h>
#include <ksimpleconfig.h>
#include "pixmapcache.h"

#include <iostream>

/** Class Frame: */

QString Frame::shapeName(Shape shape)
{
	switch (shape) {
		case Screen:            return "Screen";

		case PluginIconBar:     return "PluginIconBar";
		case ProgressBar:       return "ProgressBar";

		case MiddleBar:         return "MiddleBar";
		case CoverAndTextInfos: return "CoverAndTextInfos";
		case Cover:             return "Cover";
		case CoverOverlay:      return "CoverOverlay";
		case TextInfos:         return "TextInfos";

		case BelowCover:        return "BelowCover";
		case NextPlaying:       return "NextPlaying";

		case BottomBar:         return "BottomBar";
		case TaskBar:           return "TaskBar";
		case DateHour:          return "DateHour";

		case Lyrics:            return "Lyrics";
		case Stars:             return "Stars";
		default:                return "ERROR_IN_PROGRAM: shape is " + QString::number(shape);
	}
}

QString Frame::shapeLabel(Shape shape)
{
	switch (shape) {
		case Screen:            return "Screen:";

		case PluginIconBar:     return "Plugin icon bar:";
		case ProgressBar:       return "Progress bar:";

		case MiddleBar:         return "Middle bar:";
		case CoverAndTextInfos: return "Cover && text:";
		case Cover:             return "Cover:";
		case CoverOverlay:      return "Cover overlay:";
		case TextInfos:         return "Text:";

		case BelowCover:        return "Below cover:";
		case NextPlaying:       return "Next playing:";

		case BottomBar:         return "Bottom bar:";
		case TaskBar:           return "Task bar:";
		case DateHour:          return "Date && hour:";

		case Lyrics:            return "Lyrics:";
		case Stars:             return "Stars:";
		default:                return "ERROR_IN_PROGRAM: shape is " + QString::number(shape);
	}
}

bool Frame::lineAfterShape(Shape shape)
{
	return (shape == ProgressBar || shape == TextInfos || shape == NextPlaying || shape == DateHour);
}

bool Frame::haveDefaultPaddings(Shape shape)
{
	return (shape != PluginIconBar && shape != MiddleBar && shape != CoverAndTextInfos && shape != BelowCover && shape != BottomBar);
}

Frame *Frame::forName(const QString name)
{
	QPtrListIterator<Frame> it(Frame::list());
	Frame *frame;
	while ((frame = it.current()) != 0) {
		++it;
		if (frame->folderName() == name)
			return frame;
	}
	return 0;
}

Frame::ShapeDrawingPhase Frame::shapeDrawingPhase(Frame::Shape shape)
{
	switch (shape) {
		case Screen:            return BackgroundPhase;

		case PluginIconBar:     return BackgroundPhase;
		case ProgressBar:       return ContentPhase;

		case MiddleBar:         return BackgroundPhase;
		case CoverAndTextInfos: return ContentPhase;
		case Cover:             return ContentPhase;
		case CoverOverlay:      return OverlayPhase;
		case TextInfos:         return ContentPhase;

		case BelowCover:        return BackgroundPhase;
		case NextPlaying:       return BackgroundPhase;

		case BottomBar:         return BackgroundPhase;
		case TaskBar:           return BackgroundPhase;
		case DateHour:          return BackgroundPhase;

		case Lyrics:            return OnTopPhase;
		case Stars:             return BeforeStarsPhase;
		default:                return BackgroundPhase;
	}
}

Frame::Frame(const QString &location, const QString &folderName)
 : m_location(location), m_folderName(folderName), m_pixmap(0), m_frameComputed(false)
{
	if (!m_location.endsWith("/"))
		m_location += "/";

	load();
}

Frame::~Frame()
{
	setUsed(false);
}

// Frame information:
QString Frame::location()   const { return m_location;      }
QString Frame::folderName() const { return m_folderName;    }
// Fixed parts:
int Frame::fixedTop()       const { return m_fixedTop;      }
int Frame::fixedLeft()      const { return m_fixedLeft;     }
int Frame::fixedRight()     const { return m_fixedRight;    }
int Frame::fixedBottom()    const { return m_fixedBottom;   }
// Outside:
int Frame::outsideTop()     const { return m_outsideTop;    }
int Frame::outsideLeft()    const { return m_outsideLeft;   }
int Frame::outsideRight()   const { return m_outsideRight;  }
int Frame::outsideBottom()  const { return m_outsideBottom; }
// Borders:
int Frame::borderTop()      const { return m_borderTop;     }
int Frame::borderLeft()     const { return m_borderLeft;    }
int Frame::borderRight()    const { return m_borderRight;   }
int Frame::borderBottom()   const { return m_borderBottom;  }
// Default padding:
bool Frame::noDefaultPaddingForBlocks() const { return m_noDefaultPaddingForBlocks; }

void Frame::computeFrame()
{
	if (m_frameComputed)
		return;

	QStringList locations;
	locations << "frame.png" << "frame.jpg" << "frame.jpeg" << "frame.gif";

	// Load the first adequate image:
	QImage image;
	uint i = 0;
	while (image.isNull() && i < locations.count()) {
		image.load(m_location + locations[i]);
		i++;
	}

	// If nothing to load, load nothing:
	if (image.isNull()) {
		m_corners[0] = image;
		m_corners[1] = image;
		m_corners[2] = image;
		m_corners[3] = image;
		m_borders[0] = image;
		m_borders[1] = image;
		m_borders[2] = image;
		m_borders[3] = image;
		m_content    = image;
		m_frameComputed = true;
		return;
	}

	// Load the frame:
	QPixmap pixmap(image);
	int width  = image.width();
	int height = image.height();

	// CORNERS :
	// Top-left:
	m_corners[0].resize(m_fixedLeft, m_fixedTop);
	copyBlt(&m_corners[0], 0, 0, &pixmap, 0,                    0,                      m_fixedLeft,  m_fixedTop);
	// Top-right:
	m_corners[1].resize(m_fixedRight, m_fixedTop);
	copyBlt(&m_corners[1], 0, 0, &pixmap, width - m_fixedRight, 0,                      m_fixedRight, m_fixedTop);
	// Bottom-left:
	m_corners[2].resize(m_fixedLeft, m_fixedBottom);
	copyBlt(&m_corners[2], 0, 0, &pixmap, 0,                    height - m_fixedBottom, m_fixedLeft,  m_fixedBottom);
	// Bottom-right:
	m_corners[3].resize(m_fixedRight, m_fixedBottom);
	copyBlt(&m_corners[3], 0, 0, &pixmap, width - m_fixedRight, height - m_fixedBottom, m_fixedRight, m_fixedBottom);

	// BORDERS:
	m_borders[0] = image.copy(m_fixedLeft,           0,                        width - m_fixedLeft - m_fixedRight, m_fixedTop);                          // Top
	m_borders[1] = image.copy(0,                     m_fixedTop,               m_fixedLeft,                        height - m_fixedTop - m_fixedBottom); // Left
	m_borders[2] = image.copy(width - m_fixedRight,  m_fixedTop,               m_fixedRight,                       height - m_fixedTop - m_fixedBottom); // Right
	m_borders[3] = image.copy(m_fixedLeft,           height - m_fixedBottom,   width - m_fixedLeft - m_fixedRight, m_fixedBottom);                       // Bottom

	// CONTENT:
	m_content = image.copy(m_fixedLeft,  m_fixedTop, width - m_fixedLeft - m_fixedRight, height - m_fixedTop - m_fixedBottom);

	m_frameComputed = true;
}


QRect Frame::realRect(Shape s, const QRect &rect, int paddingTop, int paddingLeft, int paddingRight, int paddingBottom)
{
	// Note: Screen & Stars is not processed here:
	bool internBorderTop    = (                      s == MiddleBar || s == CoverAndTextInfos || s == Cover || s == CoverOverlay || s == BelowCover || s == BottomBar);
	bool internBorderLeft   = (                                        s == CoverAndTextInfos || s == Cover || s == CoverOverlay);
	bool internBorderRight  = (                                        s == CoverAndTextInfos || s == Cover || s == CoverOverlay);
	bool internBorderBottom = (s == PluginIconBar || s == MiddleBar || s == CoverAndTextInfos || s == Cover || s == CoverOverlay || s == BelowCover);

	int x = rect.x() - outsideLeft() - (internBorderLeft ? 0 : borderLeft()) - paddingLeft;
	int y = rect.y() - outsideTop()  - (internBorderTop  ? 0 : borderTop())  - paddingTop;
	int width  = outsideLeft() + paddingLeft + (internBorderLeft ? 0 : borderLeft()) + rect.width()  + (internBorderRight  ? 0 : borderRight())  + paddingRight  + outsideRight();
	int height = outsideTop()  + paddingTop  + (internBorderTop  ? 0 : borderTop())  + rect.height() + (internBorderBottom ? 0 : borderBottom()) + paddingBottom + outsideBottom();
	return QRect(x, y, width, height);
	//			m_frameRects[i].x() - paddingLeft, m_frameRects[i].y() - paddingTop,
	//			m_frameRects[i].width() + paddingLeft + paddingRight, m_frameRects[i].height() + paddingTop + paddingBottom);
}

// Pixmap:
const QPixmap Frame::pixmap(int width, int height, int paddingTop, int paddingLeft, int paddingRight, int paddingBottom)
{
	computeFrame();

	// Compute the real rectangle:
//	int rectX      = rect.x() - paddingLeft;
//	int rectY      = rect.y() - paddingTop;
	int rectWidth  = /*rect.*/width/*()*/ + paddingLeft + paddingRight;
	int rectHeight = /*rect.*/height/*()*/ + paddingTop + paddingBottom;

	QString key = "org.kde.kirocker.frames[" + m_location + "," + QString::number(rectWidth) + "," + QString::number(rectHeight) + "]";

	QPixmap *pixmap = PixmapCache::find(key);
	if (pixmap) {
//		std::cout << "             " << QString(key).replace("\n", "") << std::endl;
		return *pixmap;
	}
//	std::cout << PixmapCache::cacheLimit() << "***CREATE*** " << QString(key).replace("\n", "") << std::endl;

	// BORDERS:
	QImage top    = Tools::smoothScale(m_borders[0], rectWidth - m_fixedLeft - m_fixedRight, m_fixedTop);    // Top
	QImage left   = Tools::smoothScale(m_borders[1], m_fixedLeft,  rectHeight - m_fixedTop - m_fixedBottom); // Left
	QImage right  = Tools::smoothScale(m_borders[2], m_fixedRight, rectHeight - m_fixedTop - m_fixedBottom); // Right
	QImage bottom = Tools::smoothScale(m_borders[3], rectWidth - m_fixedLeft - m_fixedRight, m_fixedBottom); // Bottom

	// CONTENT:
	QImage contentScaled = Tools::smoothScale(m_content, rectWidth - m_fixedLeft - m_fixedRight, rectHeight - m_fixedTop - m_fixedBottom);

	QPixmap result(rectWidth, rectHeight);
	QPixmap temp;

	// CORNERS:
	copyBlt(&result, 0,                        0,                          &m_corners[0], 0, 0, m_corners[0].width(), m_corners[0].height()); // Top-left
	copyBlt(&result, rectWidth - m_fixedRight, 0,                          &m_corners[1], 0, 0, m_corners[1].width(), m_corners[1].height()); // Top-right
	copyBlt(&result, 0,                        rectHeight - m_fixedBottom, &m_corners[2], 0, 0, m_corners[2].width(), m_corners[2].height()); // Bottom-left
	copyBlt(&result, rectWidth - m_fixedRight, rectHeight - m_fixedBottom, &m_corners[3], 0, 0, m_corners[3].width(), m_corners[3].height()); // Bottom-right

	// BORDERS:
	temp.convertFromImage(top);
	copyBlt(&result, m_fixedLeft, 0,                          &temp, 0, 0, temp.width(), temp.height());
	temp.convertFromImage(left);
	copyBlt(&result, 0,           m_fixedTop,                 &temp, 0, 0, temp.width(), temp.height());
	temp.convertFromImage(right);
	copyBlt(&result, rectWidth - m_fixedRight, m_fixedTop,    &temp, 0, 0, temp.width(), temp.height());
	temp.convertFromImage(bottom);
	copyBlt(&result, m_fixedLeft, rectHeight - m_fixedBottom, &temp, 0, 0, temp.width(), temp.height());

	// CONTENT:
	temp.convertFromImage(contentScaled);
	copyBlt(&result, m_fixedLeft, m_fixedTop, &temp, 0, 0, temp.width(), temp.height());

	PixmapCache::insert(key, result);

	return QPixmap(result);
}

void Frame::setUsed(bool used) // TODO Rename reset()
{
	if (!used /*&& m_pixmap*/) {
		//delete m_pixmap;
		//m_pixmap = 0;
	}
}

void Frame::copyTo(Frame *frame) const
{
	// Frame information:
	frame->m_location      = m_location;
	frame->m_folderName    = m_folderName;
	// Fixed parts:
	frame->m_fixedTop      = m_fixedTop;
	frame->m_fixedLeft     = m_fixedLeft;
	frame->m_fixedRight    = m_fixedRight;
	frame->m_fixedBottom   = m_fixedBottom;
	// Outside:
	frame->m_outsideTop    = m_outsideTop;
	frame->m_outsideLeft   = m_outsideLeft;
	frame->m_outsideRight  = m_outsideRight;
	frame->m_outsideBottom = m_outsideBottom;
	// Border:
	frame->m_borderTop     = m_borderTop;
	frame->m_borderLeft    = m_borderLeft;
	frame->m_borderRight   = m_borderRight;
	frame->m_borderBottom  = m_borderBottom;
	// Default padding:
	frame->m_noDefaultPaddingForBlocks = m_noDefaultPaddingForBlocks;
}

void Frame::load()
{
	KSimpleConfig config(m_location + "kirocker-frame.config", /*readOnly=*/true);

	config.setGroup("FixedParts");
	m_fixedTop    = config.readNumEntry("Top",    0);
	m_fixedLeft   = config.readNumEntry("Left",   0);
	m_fixedRight  = config.readNumEntry("Right",  0);
	m_fixedBottom = config.readNumEntry("Bottom", 0);

	config.setGroup("Outside");
	m_outsideTop    = config.readNumEntry("Top",    m_fixedTop);
	m_outsideLeft   = config.readNumEntry("Left",   m_fixedLeft);
	m_outsideRight  = config.readNumEntry("Right",  m_fixedRight);
	m_outsideBottom = config.readNumEntry("Bottom", m_fixedBottom);

	config.setGroup("Border");
	m_borderTop    = config.readNumEntry("Top",    0);
	m_borderLeft   = config.readNumEntry("Left",   0);
	m_borderRight  = config.readNumEntry("Right",  0);
	m_borderBottom = config.readNumEntry("Bottom", 0);

	config.setGroup("General");
	m_noDefaultPaddingForBlocks = config.readBoolEntry("NoDefaultPaddingForBlocks", false);
}

void Frame::save()
{
	KSimpleConfig config(m_location + "kirocker-frame.config", /*readOnly=*/false);

	config.setGroup("FixedParts");
	config.writeEntry("Top",     m_fixedTop);
	config.writeEntry("Left",    m_fixedLeft);
	config.writeEntry("Right",   m_fixedRight);
	config.writeEntry("Bottom",  m_fixedBottom);

	config.setGroup("Outside");
	config.writeEntry("Top",     m_outsideTop);
	config.writeEntry("Left",    m_outsideLeft);
	config.writeEntry("Right",   m_outsideRight);
	config.writeEntry("Bottom",  m_outsideBottom);

	config.setGroup("Border");
	config.writeEntry("Top",     m_borderTop);
	config.writeEntry("Left",    m_borderLeft);
	config.writeEntry("Right",   m_borderRight);
	config.writeEntry("Bottom",  m_borderBottom);

	config.setGroup("General");
	config.writeEntry("NoDefaultPaddingForBlocks",  m_noDefaultPaddingForBlocks);

	setUsed(false);
}

bool      Frame::s_loaded = false;
FrameList Frame::s_frames = FrameList();

FrameList &Frame::list()
{
	if (!s_loaded) {
		// Load Frame List:
		QStringList directories = KGlobal::dirs()->resourceDirs("data"); // eg. { "/home/seb/.kde/share/apps/", "/usr/share/apps/" }
		// For each data folder:
		for (QStringList::Iterator it = directories.begin(); it != directories.end(); ++it) {
			// For each frame folder in this data folder:
			QDir dir(*it + "kirocker/frames/", /*nameFilder=*/"", /*sortSpec=*/QDir::Name | QDir::IgnoreCase, /*filterSpec=*/QDir::Dirs | QDir::NoSymLinks);
			QStringList files = dir.entryList();
			for (QStringList::Iterator it2 = files.begin(); it2 != files.end(); ++it2) {
				if (*it2 != "." && *it2 != "..") {
					Frame *frame = new Frame(*it + "kirocker/frames/" + *it2, *it2);
					s_frames.inSort(frame);
				}
			}
		}
		s_loaded = true;
	}

	return s_frames;
}

/** Class FrameList: */

FrameList::FrameList()
 : QPtrList<Frame>()
{
}

int FrameList::compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2)
{
	Frame *frame1 = (Frame *) item1;
	Frame *frame2 = (Frame *) item2;
	return frame1->folderName().lower().compare(frame2->folderName().lower());
}

//# include "framemanager. moc"
