/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef FRAMEMANAGER_H
#define FRAMEMANAGER_H

#include <qstring.h>
#include <qptrlist.h>
#include <qimage.h>

class QPixmap;

class FrameList;

/**
 * @author Sébastien Laoût <slaout@linux62.org>
 */
class Frame
{
  public:
	// Constructor:
	Frame(const QString &location, const QString &folderName);
	~Frame();

	enum Shape {
		Screen = 0,
		PluginIconBar, ProgressBar,
		MiddleBar, CoverAndTextInfos, Cover, CoverOverlay, TextInfos,
		BelowCover, NextPlaying,
		BottomBar, TaskBar, DateHour,
		Lyrics, Stars,
		SHAPE_COUNT = Stars + 1
	};
	static QString shapeName(Shape shape);
	static QString shapeLabel(Shape shape);
	static bool lineAfterShape(Shape shape);
	static bool haveDefaultPaddings(Shape shape);
	static Frame *forName(const QString name);
	enum ShapeDrawingPhase {
		BackgroundPhase = 0,
		ContentPhase,
		BeforeStarsPhase,
		OnTopPhase,
		OverlayPhase
	};
	static ShapeDrawingPhase shapeDrawingPhase(Shape shape);

	// GET:
	QString location() const;
	QString folderName() const;

	int fixedTop() const;
	int fixedLeft() const;
	int fixedRight() const;
	int fixedBottom() const;

	int outsideTop() const;
	int outsideLeft() const;
	int outsideRight() const;
	int outsideBottom() const;

	int borderTop() const;
	int borderLeft() const;
	int borderRight() const;
	int borderBottom() const;

	bool noDefaultPaddingForBlocks() const;

	QRect realRect(Shape shape, const QRect &rect, int paddingTop, int paddingLeft, int paddingRight, int paddingBottom);
	// Pixmap:
	const QPixmap pixmap(int width, int height, int paddingTop, int paddingLeft, int paddingRight, int paddingBottom);
	void setUsed(bool used);

	// For the editor to store a temporar version:
	void copyTo(Frame *frame) const;

	int countOfThemesUsingIt();

	// SET:
	void setFixedTop(int fixed);
	void setFixedLeft(int fixed);
	void setFixedRight(int fixed);
	void setFixedBottom(int fixed);

	void setOutsideTop(int outside);
	void setOutsideLeft(int outside);
	void setOutsideRight(int outside);
	void setOutsideBottom(int outside);

	void setBorderTop(int border);
	void setBorderLeft(int border);
	void setBorderRight(int border);
	void setBorderBottom(int border);

	void setNoDefaultPaddingForBlocks(bool noDefaultPadding);

	// Disk Management:
	void load();
	void save();

	static FrameList &list();

  protected:
	void computeFrame();

	static bool s_loaded;
	static FrameList s_frames;

  private:
	int m_fixedTop;
	int m_fixedLeft;
	int m_fixedRight;
	int m_fixedBottom;

	int m_outsideTop;
	int m_outsideLeft;
	int m_outsideRight;
	int m_outsideBottom;

	int m_borderTop;
	int m_borderLeft;
	int m_borderRight;
	int m_borderBottom;

	bool m_noDefaultPaddingForBlocks;

	// Pixmaps & images:
	QString  m_location;
	QString  m_folderName;
	QPixmap *m_pixmap;

	bool    m_frameComputed;
	QPixmap m_corners[4];
	QImage  m_borders[4];
	QImage  m_content;
};

class FrameList : public QPtrList<Frame>
{
  public:
	FrameList();
  protected:
	int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2);
};

#endif // FRAMEMANAGER_H
