/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "fullscreen.h"
#include "coverdisplay.h"
#include "amarokapi.h"
#include "playerinformation.h"
#include "thememanager.h"

#include <qdesktopwidget.h>
#include <klocale.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <kglobal.h>
#include <qimage.h>
#include <kimageeffect.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <dcopclient.h>


#include <X11/Xlib.h>
#ifdef HAVE_XTEST
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>
#endif


FullScreen::FullScreen(QWidget *parent, const char *name)
 : QWidget(parent, name, Qt::WNoAutoErase | Qt::WGroupLeader)
{
	setWFlags(Qt::WGroupLeader);

	QDesktopWidget desktop;
	resize(desktop.screenGeometry(this).size());

	setCaption(i18n("Kirocker Music Display"));
	setIcon(kapp->iconLoader()->loadIcon("kirocker", KIcon::Desktop, 48, KIcon::DefaultState));
//	setFocusPolicy(QWidget::StrongFocus);

	m_coverDisplay = new CoverDisplay(this);
	m_coverDisplay->setGeometry(rect());

	m_coverDisplay->hideControls();

	m_coverDisplay->setFullScreen();
	m_coverDisplay->show();

	connect( m_coverDisplay, SIGNAL(closeAsked()), this, SLOT(close()) );



	//BEGIN This code is inspired form the Kaffeine project, in file src/kaffeine.cpp :
	// KWin are you there?
	m_haveKWin = KApplication::dcopClient()->isApplicationRegistered("kwin");
	connect( &m_screensaverTimer, SIGNAL(timeout()), this, SLOT(slotFakeKeyEvent()) );
	// XTest are you there?
#ifdef HAVE_XTEST
	m_haveXTest = false;
	int dummy_event, dummy_error, dummy_major, dummy_minor;
	if (XTestQueryExtension(x11Display(), &dummy_event, &dummy_error, &dummy_major, &dummy_minor)) {
		m_fakeKeycode = XKeysymToKeycode(x11Display(), XK_Shift_L);
		if (m_fakeKeycode != 0)
			m_haveXTest = true;
	}
#endif
	//END
}

FullScreen::~FullScreen()
{
}

//BEGIN This code is inspired form the Kaffeine project, in file src/kaffeine.cpp :
void FullScreen::slotFakeKeyEvent()
{
	if (m_haveKWin) {
		// use a better method if we're in a kde environment
		XWarpPointer(x11Display(), None, None, 0, 0, 0, 0, 0, 0);
		XFlush(x11Display());
	} else {
#ifdef HAVE_XTEST
		XTestFakeKeyEvent(x11Display(), m_fakeKeycode, true, 0);
		XTestFakeKeyEvent(x11Display(), m_fakeKeycode, false, 0);
		XFlush(x11Display());
#endif
	}
}
//END

void FullScreen::keyPressEvent(QKeyEvent *event)
{
	QWidget::keyPressEvent(event);

	PlayerInformation *infos = PlayerInformation::instance();
	switch (event->key()) {
		case Qt::Key_Space:
		case Qt::Key_Return:
			if (infos->canPause())
				AmarokApi::playPause();
			else
				AmarokApi::stop();
			break;
		case Qt::Key_P:
		case Qt::Key_PageUp:   AmarokApi::previousTrack();                     break;
		case Qt::Key_N:
		case Qt::Key_PageDown: AmarokApi::nextTrack();                         break;
		case Qt::Key_Left:     if (infos->canSeek()) infos->seekRelative(-10); break;
		case Qt::Key_Right:    if (infos->canSeek()) infos->seekRelative(10);  break;
		case Qt::Key_Up:       AmarokApi::volumeUp();                          break;
		case Qt::Key_Down:     AmarokApi::volumeDown();                        break;
		case Qt::Key_F1:       clickStar(1);                                   break;
		case Qt::Key_F2:       clickStar(2);                                   break;
		case Qt::Key_F3:       clickStar(3);                                   break;
		case Qt::Key_F4:       clickStar(4);                                   break;
		case Qt::Key_F5:       clickStar(5);                                   break;
		case Qt::Key_Q:
		case Qt::Key_Escape:   close();                                        break;
		case Qt::Key_L:        m_coverDisplay->toggleLyrics();                 break;
		case Qt::Key_U:        m_coverDisplay->scrollLyricsUp();               break;
		case Qt::Key_D:        m_coverDisplay->scrollLyricsDown();             break;
		case Qt::Key_T:
			m_coverDisplay->dialogAppeared();
			ThemeManager::chooseTheme(width(), height(), this/*topLevelWidget()*/);
			m_coverDisplay->dialogDisappeared();
			break;
/*
		case Qt::Key_V:
			m_coverDisplay->grabMouse();
			m_coverDisplay->grabKeyboard();
			break;
		case Qt::Key_B:
			m_coverDisplay->releaseMouse();
			m_coverDisplay->releaseKeyboard();
			break;
*/
		default:
			if (!event->text().isEmpty()) // Do not catch Alt, Ctrl... and other modifiers (no display on Alt+Tab...)
				showKeysInformation();
			break;
	}
}

void FullScreen::clickStar(int star)
{
	PlayerInformation *infos = PlayerInformation::instance();

	if (infos->status() == PlayerInformation::Stopped)
		return;

	int rating = star * 2;
	if (infos->rating() == 1)
		rating = 0;
	else if (infos->rating() == rating)
		rating -= 1;
	infos->changeRating(rating);
}

void FullScreen::showKeysInformation()
{
	m_coverDisplay->dialogAppeared();
	KMessageBox::information(
		this,
		"<p>" + i18n("Keyboard shortcuts available in full screen display:") + "</p><table>"
		"<tr><th>" + i18n("Space or Enter:")         + "</th><td>" + i18n("Play/pause")              + "</td></tr>"
		"<tr><th>" + i18n("PageUp/PageDown or P/N:") + "</th><td>" + i18n("Change track")            + "</td></tr>"
		"<tr><th>" + i18n("Left/Right:")             + "</th><td>" + i18n("Seek in track")           + "</td></tr>"
		"<tr><th>" + i18n("Up/Down:")                + "</th><td>" + i18n("Change volume")           + "</td></tr>"
		"<tr><th>" + i18n("F1/F2/F3/F4/F5:")         + "</th><td>" + i18n("Change star rating")      + "</td></tr>"
		"<tr><th>" + i18n("L:")                      + "</th><td>" + i18n("Show/hide lyrics")        + "</td></tr>"
		"<tr><th>" + i18n("U/D:")                    + "</th><td>" + i18n("Scroll lyrics up & down") + "</td></tr>"
		"<tr><th>" + i18n("T:")                      + "</th><td>" + i18n("Choose theme")            + "</td></tr>"
		"<tr><th>" + i18n("Escape or Q:")            + "</th><td>" + i18n("Close full screen")       + "</td></tr>"
		"</table>",
		i18n("%1 - %2").arg("Keyboard Shortcuts", i18n("Kirocker Music Display")),
		/*dontShowAgainName=*/QString(),
		KMessageBox::Notify | KMessageBox::PlainCaption
	);
	m_coverDisplay->dialogDisappeared();
}

void FullScreen::contextMenuEvent(QContextMenuEvent *event)
{
	QWidget::contextMenuEvent(event);

	showKeysInformation();
}

void FullScreen::windowActivationChange(bool oldActive)
{
	static bool wasActivatedLastTime  = false;
	static bool screenSaverWasEnabled = true;
	static bool osdWasEnabled         = true;

	if (isActiveWindow() != wasActivatedLastTime) {
		wasActivatedLastTime = isActiveWindow();
		if (isActiveWindow()) {
			screenSaverWasEnabled = AmarokApi::screenSaverEnabled();
			osdWasEnabled         = AmarokApi::osdEnabled();
			AmarokApi::enableScreenSaver(false);
			AmarokApi::enableOSD(false);
			m_screensaverTimer.start(55000);
		} else {
			AmarokApi::enableScreenSaver(screenSaverWasEnabled);
			AmarokApi::enableOSD(osdWasEnabled);
			m_screensaverTimer.stop();
		}
	}

	QWidget::windowActivationChange(oldActive);
}

#include "fullscreen.moc"
