/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "informationpoller.h"
#include "playerinformation.h"
#include "dcopinterface.h"

#include <kapplication.h>
#include <qevent.h>
#include <qregexp.h>
#include <qdir.h>
#include <kstandarddirs.h>
#include <ksimpleconfig.h>
#include <qdom.h>
#include <kglobal.h>
#include <qfile.h>

//
#include <qtimer.h>

#include <iostream>

const int InformationPoller::NEW_INFO_EVENT     = QEvent::User + 1;
const int InformationPoller::NEW_RATING_EVENT   = QEvent::User + 2;
const int InformationPoller::NEW_PROGRESS_EVENT = QEvent::User + 3;
const int InformationPoller::NEW_LYRICS         = QEvent::User + 4;

InformationPoller::InformationPoller()
// : QThread()
 : QObject(),
   m_failureCounter(0)
{
	m_refreshTimer = new QTimer();
	connect( m_refreshTimer, SIGNAL(timeout()), this, SLOT(poll()) );
	m_refreshTimer->start( 333, false );
}

InformationPoller::~InformationPoller()
{
}

void InformationPoller::run()
{
	while (true) {
		poll();
//		msleep(333);
	}
}

QString InformationPoller::retreiveNextPlaying(bool isLastFm, bool isStream, bool * /*success*/)
{
	// isLastFm is passed as parameter, because it already has been asked to Amarok. We save time by not asking again:
	if (isLastFm)
		return "Last.fm radio.";

	// isStream is passed as parameter, because it already has been asked to Amarok. We save time by not asking again:
	if (isStream)
		return "Internet stream.";

	// TODO ask for random regularily
	bool random = DcopInterface::getBool("amarok", "player", "randomModeStatus()", 0/*success*/);
/*
	if (!success) {
		failed();
		return ""; // Display no error message, the feature is simply not available!
	}
*/
	if (random)
		return "Random playing."; // FIXME "Use 'random dynamic playlist' to get next songs"

	// Ok, "Next playing:" will get useful results, do the work:
	// This code is largely inspired from AmfsXML::

	// http://www.kde-apps.org/content/show.php/Amarok+Full+Screen?content=52641

	// Get Playlist file name and position:
	QString playlistFileName = DcopInterface::getString( "amarok", "playlist", "saveCurrentPlaylist()", 0/*success*/ );
	int     playlistPosition = DcopInterface::getInt(    "amarok", "playlist", "getActiveIndex()",      0/*success*/ );

	// Get the XML:
	QDomDocument document;
	QFile playlistFile(playlistFileName);
	if (!playlistFile.open(IO_ReadOnly))
		return "";
	if (!document.setContent(&playlistFile)) {
		playlistFile.close();
		return "";
	}
	playlistFile.close();

	// Parse playlist for next songs
	const int nextSongCount = 2;
	QDomElement root = document.documentElement();
	QDomNode node = root.firstChild();
	QString nextPlaying;

/*
  <Artist>Slipknot</Artist>
  <Track>1</Track>
  <Title>Prelude 3.0</Title>
  <Album>Vol. 3: The Subliminal Verses</Album>
  <Year>2004</Year>
*/

	for (int i = 0; i <= playlistPosition + nextSongCount && !node.isNull(); i++) {
		if (i > playlistPosition && node.isElement()) {
			QDomElement element = node.toElement();

			// Get description about that song:
			QString artist = element.elementsByTagName("Artist").item(0).toElement().text();
			QString track  = "";//element.elementsByTagName("Track").item(0).toElement().text();    // Do not use the track number, it's too clutered
			QString title  = element.elementsByTagName("Title").item(0).toElement().text();

			if (title.isEmpty()) {
				title = KURL(element.attribute("url")).fileName();
				int index = title.findRev(".");
				if (index >= 0)
					title = title.left(index);
			}

			QString song = "";

			if (!track.isEmpty())
				song = QString("%1. %2").arg(track, title);
			else
				song = title;

			if (!artist.isEmpty() && !song.isEmpty())
				song = QString("%1 - %2").arg(song, artist);
			else if (!artist.isEmpty())
				song = artist;
			else
				song = title;

			nextPlaying += (nextPlaying.isEmpty() ? "" : "\n") + song;
		}
		node = node.nextSibling();
	}

	// TODO: only call this method if there is something playing or paused
	if (nextPlaying.isEmpty())
		return "Last track.";
	else
		return "Next:\n" + nextPlaying;
		//return "Next Playing:\n" + nextPlaying;
	// TODO: Display as text in KickerTip and tooltip, but bold second line and followings in fullscreen
}

void InformationPoller::poll()
{
	PlayerInformation *infos = PlayerInformation::instance();
	bool success = true;

	// Get primary information:
	int  status = DcopInterface::getInt(    "amarok", "player", "status()",     &success );
	KURL url    = DcopInterface::getString( "amarok", "player", "encodedURL()", &success );
	int  rating = DcopInterface::getInt(    "amarok", "player", "rating()",     &success );

	if (!success) {
		failed();
		return;
	}

	bool isLastFm = (url.protocol() == "lastfm");
	bool isStream = (url.protocol() == "http");
	bool radioTrackChanged = false;
/*
	if (isLastFm || isStream) {
		QString nowPlaying = DcopInterface::getString( "amarok", "player", "nowPlaying()", &success );
		if (!success) {
			failed();
			return;
		}
		if (isLastFm && nowPlaying != infos->artist() + " - " + infos->title())
			radioTrackChanged = true;
		if (isStream && nowPlaying != infos->title())
			radioTrackChanged = true;
	}
*/
	if (isLastFm) {
		QString nowPlaying = DcopInterface::getString( "amarok", "player", "nowPlaying()", &success );
		if (!success) {
			failed();
			return;
		}
		if (nowPlaying != infos->artist() + " - " + infos->title())
			radioTrackChanged = true;
	} else if (isStream) {
		QString artist = DcopInterface::getString( "amarok", "player", "artist()", &success );
		QString title  = DcopInterface::getString( "amarok", "player", "title()",  &success );
		if (!success) {
			failed();
			return;
		}
		if (artist != infos->artist() || title != infos->title())
			radioTrackChanged = true;
	}

	// We started or changed track: ask every information again:
	if (status != infos->status() || radioTrackChanged || url.url() != infos->url().url()) { // Two null URLs are always different, so we compare with the string equivalent!
		QString artist    = DcopInterface::getString( "amarok", "player", "artist()",           &success ); // TODO: test success at every lines
		QString album     = DcopInterface::getString( "amarok", "player", "album()",            &success );
		QString year      = DcopInterface::getString( "amarok", "player", "year()",             &success );
		QString track     = DcopInterface::getString( "amarok", "player", "track()",            &success );
		QString title     = DcopInterface::getString( "amarok", "player", "title()",            &success );
		int     duration  = DcopInterface::getInt(    "amarok", "player", "trackTotalTime()",   &success );
		int     position  = DcopInterface::getInt(    "amarok", "player", "trackCurrentTime()", &success );

		QString nextPlaying = (status != PlayerInformation::Stopped ? retreiveNextPlaying(isLastFm, isStream, &success) : "");

		if (!success) {
			failed();
			return;
		}

		// If the track is of a different album than the previous one, then we need to get the new cover:
	//std::cout << "Get album cover art" << std::endl;
		// If artist/album changed, or if last playing artist/album was empty and now stopped (so artist/album is STILL empty but we should now show NO album art):
	//	if (artist != infos->artist() || album != infos->album() || status == PlayerInformation::Stopped) {
			if (status == PlayerInformation::Stopped) {
				infos->setCoverPixmap(QPixmap());
			} else if (isLastFm) {
				retreiveNewLastFmCover();
			} else {
				retreiveNewCover(artist, album, &success);
			}
//		else
//			PlayerInformation::instance()->emitAboutToChangeTrack(isPlaying); // An hack because it's not multi-threaded yet
	//	}

		if (!success) {
			failed();
			return;
		}

		infos->emitAboutToChangeTrack(status, infos->isPrevious(url)); // An hack because it's not multi-threaded yet

		// Change every information at once (the cover has been set at the very end of retreiveNewCover()):
		infos->setStatus((PlayerInformation::Status) status);
		infos->setUrl(url);
		infos->setArtist(artist);
		infos->setAlbum(album);
		infos->setYear(year);
		infos->setTrack(track);
		infos->setTitle(title);
		infos->setRating(rating);
		infos->setDuration(duration);
		infos->setPosition(position);
		infos->setNextPlaying(nextPlaying);

		// Inform listeners that information have changed:
		//kapp->postEvent(infos, new QCustomEvent(NEW_INFO_EVENT));
		infos->customEvent(new QCustomEvent(NEW_INFO_EVENT));

		tryGetLyrics();
	}/* else

*/	if (infos->status() == PlayerInformation::Playing) {
		if (rating != infos->rating()) {
			infos->setRating(rating);
			//kapp->postEvent(infos, new QCustomEvent(NEW_RATING_EVENT));
			infos->customEvent(new QCustomEvent(NEW_RATING_EVENT));
		}
		bool success = true;
		int position = DcopInterface::getInt( "amarok", "player", "trackCurrentTime()", &success );
		if (success && position != infos->position()) {
			infos->setPosition(position);
			//kapp->postEvent(infos, new QCustomEvent(NEW_PROGRESS_EVENT));
			infos->customEvent(new QCustomEvent(NEW_PROGRESS_EVENT));
		}
	}

	if (success)
		m_failureCounter = 0;
	else
		failed();
}

void InformationPoller::failed()
{
	m_failureCounter++;

//	std::cout << "DCOP Call Failed -- Canceling this information retreiving." << std::endl;

	PlayerInformation *infos = PlayerInformation::instance();
	// Ask if Amarok is dead after 1 seconds (3 pollings per second):
	if (infos->isPlaying() && m_failureCounter > 3 && !DcopInterface::isApplicationRegistered("amarok")) {
		infos->setStatus(PlayerInformation::Stopped);
		infos->setUrl(KURL());
		infos->setArtist("");
		infos->setAlbum("");
		infos->setYear("");
		infos->setTrack("");
		infos->setTitle("");
		infos->setRating(0);
		infos->setDuration(0);
		infos->setPosition(0);
		kapp->postEvent(infos, new QCustomEvent(NEW_INFO_EVENT));
	}
}

/**
 * Most of this code comes from the method AmfsXML::getCoverPath() of the application AmarokFS.
 */
void InformationPoller::retreiveNewCover(const QString &artist, const QString &album, bool *success)
{
	QString escapedArtist = QString(artist).replace("'", "''"); // SQLite use double quote ('') to escape simple quote!
	QString escapedAlbum  = QString(album).replace( "'", "''");

	// First try to get the image from the Amarok Images table (USER DEFINED cover):
	QString query = QString("SELECT path FROM images WHERE artist='%1' AND album='%2'").arg(escapedArtist, escapedAlbum);
	QString coverPath = DcopInterface::getStringList("amarok", "collection", "query(QString)", query, success).first();

	// In recent versions of Amarok, coverPath CAN be a relative path. Compute absolute path:
	query = QString("SELECT lastmountpoint FROM devices WHERE id=(SELECT deviceid FROM images WHERE artist='%1' AND album='%2' LIMIT 1)").arg(escapedArtist, escapedAlbum);
//	QString deviceMountPointIdQuery = QString("SELECT deviceid FROM images WHERE artist='%1' and album='%2'").arg(escapedArtist, escapedAlbum);
//	QStringList deviceMountPointId  = DcopInterface::getStringList("amarok", "collection", "query(QString)", deviceMountPointIdQuery);
//	QString deviceMountPointFirstId = deviceMountPointId.first();
//	QString query = QString("SELECT lastmountpoint FROM devices WHERE id='%1'").arg(deviceMountPointFirstId);
	QString deviceMountPoint = DcopInterface::getStringList("amarok", "collection", "query(QString)", query, success).first();
	QString coverAbsolutePath = "/" + deviceMountPoint + coverPath;//.right(coverPath.length() - 1);

	// Get the cached thumbnail image path (usually a 100x100px image, but user configurable)
	QString cachedCoverPath = DcopInterface::getString("amarok", "player", "coverImage()", success);

	//$HOME/.kde/share/apps/amarok/albumcovers/large/.... - some covers are stored here
	QString cachedCoverPathLarge = cachedCoverPath;
	cachedCoverPathLarge.replace(QRegExp("cache/[0-9]*@"), "large/");

	//$HOME/.kde/share/apps/amarok/albumcovers/tagcover/.... - and some are stored here
	QString cachedCoverPathTagcover = cachedCoverPathLarge;
	cachedCoverPathTagcover.replace("large/", "tagcover/");

	if (cachedCoverPathLarge.endsWith("/nocover.png"))
		cachedCoverPathLarge = bigNoCoverPath();

	if (!*success)
		return;

// 	// Debug
// 	std::cout << "coverPath: " << coverPath + "" << std::endl;
// 	std::cout << "deviceMountPoint: " << deviceMountPoint + "" << std::endl;
// 	std::cout << "coverAbsolutePath: " << coverAbsolutePath + "" << std::endl;
// 	std::cout << "cachedCoverPathLarge: " << cachedCoverPathLarge + "" << std::endl;
// 	std::cout << "cachedCoverPathTagcover: " << cachedCoverPathTagcover + "" << std::endl;
// 	std::cout << "cachedCoverPath: " << cachedCoverPath + "" << std::endl;

	QPixmap coverPixmap(coverPath);                // User-defined cover, big, sometimes absolute paths
	if (coverPixmap.isNull()) {
		coverPixmap.load(coverAbsolutePath);       // User-defined cover, big, absolute paths with latest Amarok versions
	}
	if (coverPixmap.isNull()) {
		coverPixmap.load(cachedCoverPathLarge);    // Cached cover image, sometimes in ~/.kde/share/apps/amarok/albumcovers/large OR...
	}
	if (coverPixmap.isNull()) {
		coverPixmap.load(cachedCoverPathTagcover); // ... Cached cover image, sometimes in ~/.kde/share/apps/amarok/albumcovers/tagcover
	}
	if (coverPixmap.isNull()) {
		coverPixmap.load(cachedCoverPath);         // Cached cover image, thumbnail (100x100px by default, but the size can be changed in the Amarok settings)
	}

//	PlayerInformation::instance()->emitAboutToChangeTrack(); // An hack because it's not multi-threaded yet

	// Set:
	PlayerInformation *infos = PlayerInformation::instance();
	infos->setCoverPixmap(coverPixmap);
}

void InformationPoller::retreiveNewLastFmCover()
{
	// Find the image:
	QPixmap lastFmCover;
	QDir dir;
	QStringList folders = kapp->dirs()->resourceDirs("data");
	for (QStringList::Iterator it = folders.begin(); it != folders.end(); ++it) {
		QString imagePath = *it + "/amarok/lastfm_image.png";
		if (dir.exists(imagePath)) {
			lastFmCover = QPixmap(imagePath);
			break;
		}
	}

	// Remove the shadow:
	if (!lastFmCover.isNull()) {
		//[General Options]
		//Cover Preview Size=100
		KSimpleConfig config("amarokrc", /*readOnly=*/true);
		config.setGroup("General Options");
		int size = config.readNumEntry("Cover Preview Size", -1);
		if (size > 0) // The shadow is normally on the bottom and right.
			lastFmCover.resize(size, size);
		else
			lastFmCover.resize(lastFmCover.width() - 6, lastFmCover.height() - 6);
	} else
		lastFmCover = QPixmap(bigNoCoverPath());

	// Set:
	PlayerInformation *infos = PlayerInformation::instance();
	infos->setCoverPixmap(lastFmCover);
}

QString InformationPoller::bigNoCoverPath()
{
	// Try the gorgeous Kirocker Music Display specific image:
	QString nocover = KGlobal::dirs()->findResource("data", "kirocker/images/nocover.png");
	if (QFile::exists(nocover))
		return nocover;

	// If not found, use the Amarok one:
	QDir dir;
	QStringList folders = kapp->dirs()->resourceDirs("data");
	for (QStringList::Iterator it = folders.begin(); it != folders.end(); ++it) {
		QString imagePath = *it + "/amarok/images/nocover.png";
		if (dir.exists(imagePath))
			return imagePath;
	}
	return "";
}

void InformationPoller::tryGetLyrics()
{
	PlayerInformation *infos = PlayerInformation::instance();

	QString lyrics = DcopInterface::getString("amarok", "player", "lyrics()", 0/*success*/);

	if (!lyrics.isEmpty()) {
		// Gor lyrics, decode XML:
		//<?xml version='1.0' encoding='UTF-8'?><lyrics artist='...' title='...' page_url='...'>\nTHE_ACTUAL_LYRICS\n\n </lyrics>
		QDomDocument document;
		if (document.setContent(lyrics)) {
			QDomElement root = document.documentElement();
			lyrics = root.text();
		} else
			lyrics = "";
	} else if (infos->status() != PlayerInformation::Stopped) {
		// No lyrics, try again a few seconds later when Amarok loaded it, or if user it:
		QTimer::singleShot( 3000, this, SLOT(tryGetLyrics()) );
	}

	lyrics = lyrics.stripWhiteSpace();
	if (lyrics != infos->lyrics()) {
		infos->setLyrics(lyrics);
		infos->customEvent(new QCustomEvent(NEW_LYRICS));
	}
}

void InformationPoller::suspendPolling()
{
	m_refreshTimer->stop();
}

void InformationPoller::resumePolling()
{
	if (!m_refreshTimer->isActive())
		m_refreshTimer->start( 333, false );
}

#include "informationpoller.moc"
