/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "taskbarwatcher.h"

#include <qtimer.h>
#include <klocale.h>
#include <kwin.h>

#include <iostream>

/** class TaskBarEntry: */

TaskBarEntry::TaskBarEntry()
 : wid(0)
{
}

TaskBarEntry::TaskBarEntry(WId wid)
{
	this->wid = wid;
	fetch();
}

void TaskBarEntry::fetch()
{
	KWin::WindowInfo info = KWin::windowInfo(wid, NET::WMVisibleName | NET::WMVisibleIconName | NET::WMState | NET::XAWMState, 0);
	if (info.valid()) {
		//title = info.visibleNameWithState();     // Requires NET::WMVisibleName,     NET::WMState and NET::XAWMState passed to KWin::windowInfo().
		title = info.visibleIconNameWithState(); // Requires NET::WMVisibleIconName, NET::WMState and NET::XAWMState passed to KWin::windowInfo().
		icon  = KWin::icon(wid, 16, 16, /*scale=*/true);
	} else {
		title = "";
		icon  = QPixmap();
	}
}

bool TaskBarEntry::operator==(const TaskBarEntry &entry)
{
	return (wid == entry.wid);
}

/** Class TestKWin: */

TaskBarWatcher::TaskBarWatcher()
 : QObject()
{
	QValueList<WId>::ConstIterator it;
	for (it = m_module.windows().begin(); it != m_module.windows().end(); ++it) {
		if (m_module.hasWId(*it)) {
			WId wid = *it;
			if (isBlinking(wid))
				m_taskBarEntries.append(TaskBarEntry(wid));
		}
	}

	if (m_taskBarEntries.count() > 0)
		QTimer::singleShot( 0, this, SLOT(emitChanged()) ); // Keep time for the using class to connect to the changed() signal!

	connect( &m_module, SIGNAL(windowAdded(WId)),                          this, SLOT(windowAdded(WId))                          );
	connect( &m_module, SIGNAL(windowRemoved(WId)),                        this, SLOT(windowRemoved(WId))                        );
	connect( &m_module, SIGNAL(windowChanged(WId, const unsigned long *)), this, SLOT(windowChanged(WId, const unsigned long *)) );
}

TaskBarWatcher::~TaskBarWatcher()
{
}

void TaskBarWatcher::emitChanged()
{
	emit changed();
}

TaskBarEntry *TaskBarWatcher::getTaskBarEntry(WId wid)
{
	// Returns the taskbar entry if it is in the list, or 0 in the counter case:
	QValueList<TaskBarEntry>::iterator it;
	for (it = m_taskBarEntries.begin(); it != m_taskBarEntries.end(); ++it)
		if ((*it).wid == wid)
			return &(*it);
	return 0;
}

bool TaskBarWatcher::isBlinking(WId wid)
{
	// Get information about the window, to know if it is blinking:
	KWin::WindowInfo info = KWin::windowInfo(wid, NET::WMState, 0);
	if (info.valid())
		return (info.state() & NET::DemandsAttention);
	else
		return false;
}

void TaskBarWatcher::windowAdded(WId wid)
{
	// If the newly added window is blinking, add it in the list (while getting other information):
	if (isBlinking(wid)) {
		m_taskBarEntries.append(TaskBarEntry(wid));
		emit changed();
	}
}

void TaskBarWatcher::windowRemoved(WId wid)
{
	// If the removed window was blinking, remove it from the list
	// (obviously, the deleted window do not blink anymore):
	TaskBarEntry *taskBarEntry = getTaskBarEntry(wid);
	if (taskBarEntry) {
		TaskBarEntry &entry = *taskBarEntry;
		m_taskBarEntries.remove(entry);
		emit changed();
	}
}

/*void TaskBarWatcher::changed()
{
	std::cout << "[" << std::endl;
	QValueList<TaskBarEntry>::iterator it;
	for (it = m_taskBarEntries.begin(); it != m_taskBarEntries.end(); ++it)
		std::cout << "  " << (*it).wid << " : " << (*it).icon.width() << " --- " << (*it).title << std::endl;
	std::cout << "]" << std::endl;
}*/

void TaskBarWatcher::windowChanged(WId wid, const unsigned long *properties)
{
	TaskBarEntry *taskBarEntry = getTaskBarEntry(wid);
	if (taskBarEntry) {
		// If the window is in the list and it is not blinking anymore, remove it:
		if ((properties[0] & NET::WMState) && !isBlinking(wid)) {
			m_taskBarEntries.remove(*taskBarEntry);
			emit changed();
		}
		// If the window is in the list and a visible property changed, refresh it:
		else if (properties[0] & (NET::WMName | NET::WMVisibleName | NET::WMIcon | NET::WMIconName | NET::WMVisibleIconName | NET::WMState | NET::XAWMState)) {
			taskBarEntry->fetch();
			emit changed();
		}
	} else {
		// If the window is now blinking, add it in the list (while getting other information):
		if ((properties[0] & NET::WMState) && isBlinking(wid)) {
			m_taskBarEntries.append(TaskBarEntry(wid));
			emit changed();
		}
	}

	return;

//	std::cout << id << " changed" << std::endl;
//	KWin::WindowInfo info = KWin::windowInfo(wid, properties[0], properties[1]);
//	KWin::WindowInfo info(wid, properties[0], properties[1]);
//	std::cout << QString(info.name()) + "" << " : " << (info.state() & NET::DemandsAttention ? " DEMANDS ATTENTION " : " not ") << std::endl;
/*
	QStringList propertyNames;
	// Root:
	if (properties[0] & NET::Supported)				propertyNames += "Supported";
	if (properties[0] & NET::ClientList)			propertyNames += "ClientList";
	if (properties[0] & NET::ClientListStacking)	propertyNames += "ClientListStacking";
	if (properties[0] & NET::NumberOfDesktops)		propertyNames += "NumberOfDesktops";
	if (properties[0] & NET::DesktopGeometry)		propertyNames += "DesktopGeometry";
	if (properties[0] & NET::DesktopViewport)		propertyNames += "DesktopViewport";
	if (properties[0] & NET::CurrentDesktop)		propertyNames += "CurrentDesktop";
	if (properties[0] & NET::DesktopNames)			propertyNames += "DesktopNames";
	if (properties[0] & NET::ActiveWindow)			propertyNames += "ActiveWindow";
	if (properties[0] & NET::WorkArea)				propertyNames += "WorkArea";
	if (properties[0] & NET::SupportingWMCheck)		propertyNames += "SupportingWMCheck";
	if (properties[0] & NET::VirtualRoots)			propertyNames += "VirtualRoots";
	if (properties[0] & NET::KDESystemTrayWindows)	propertyNames += "KDESystemTrayWindows";
	if (properties[0] & NET::CloseWindow)			propertyNames += "CloseWindow";
	if (properties[0] & NET::WMMoveResize)			propertyNames += "WMMoveResize";
	// Window:
	if (properties[0] & NET::WMName)				propertyNames += "WMName";
	if (properties[0] & NET::WMVisibleName)			propertyNames += "WMVisibleName";
	if (properties[0] & NET::WMDesktop)				propertyNames += "WMDesktop";
	if (properties[0] & NET::WMWindowType)			propertyNames += "WMWindowType";
	if (properties[0] & NET::WMState)				propertyNames += "WMState";
	if (properties[0] & NET::WMStrut)				propertyNames += "WMStrut";
	if (properties[0] & NET::WMIconGeometry)		propertyNames += "WMIconGeometry";
	if (properties[0] & NET::WMIcon)				propertyNames += "WMIcon";
	if (properties[0] & NET::WMPid)					propertyNames += "WMPid";
	if (properties[0] & NET::WMHandledIcons)		propertyNames += "WMHandledIcons";
	if (properties[0] & NET::WMPing)				propertyNames += "WMPing";
	if (properties[0] & NET::WMKDESystemTrayWinFor)	propertyNames += "WMKDESystemTrayWinFor";
	if (properties[0] & NET::XAWMState)				propertyNames += "XAWMState";
	if (properties[0] & NET::WMFrameExtents)		propertyNames += "WMFrameExtents";
	if (properties[0] & NET::WMIconName)			propertyNames += "WMIconName";
	if (properties[0] & NET::WMVisibleIconName)		propertyNames += "WMVisibleIconName";
	if (properties[0] & NET::WMGeometry)			propertyNames += "WMGeometry";

	QStringList property2Names;
	if (properties[1] & NET::WM2UserTime)			property2Names += "WM2UserTime";
	if (properties[1] & NET::WM2StartupId)			property2Names += "WM2StartupId";
	if (properties[1] & NET::WM2TransientFor)		property2Names += "WM2TransientFor";
	if (properties[1] & NET::WM2GroupLeader)		property2Names += "WM2GroupLeader";
	if (properties[1] & NET::WM2AllowedActions)		property2Names += "WM2AllowedActions";
	if (properties[1] & NET::WM2RestackWindow)		property2Names += "WM2RestackWindow";
	if (properties[1] & NET::WM2MoveResizeWindow)	property2Names += "WM2MoveResizeWindow";
	if (properties[1] & NET::WM2ExtendedStrut)		property2Names += "WM2ExtendedStrut";
	if (properties[1] & NET::WM2TakeActivity)		property2Names += "WM2TakeActivity";
	if (properties[1] & NET::WM2KDETemporaryRules)	property2Names += "WM2KDETemporaryRules";
	if (properties[1] & NET::WM2WindowClass)		property2Names += "WM2WindowClass";
	if (properties[1] & NET::WM2WindowRole)			property2Names += "WM2WindowRole";
	if (properties[1] & NET::WM2ClientMachine)		property2Names += "WM2ClientMachine";
	if (properties[1] & NET::WM2ShowingDesktop)		property2Names += "WM2ShowingDesktop";

	std::cout << wid << " changed   [Properties: " << propertyNames.join(", ") + "" << "]   [Properties2: " << property2Names.join(", ") + "" << "]" << std::endl;
*/
/*	QValueList<WId>::ConstIterator it;
	for (it = m_module.windows().begin(); it != m_module.windows().end(); ++it) {
		if (m_module.hasWId(*it)) {
			KWin::WindowInfo info(*it);
			std::cout << *it << " : " << info.name() << " : " << (info.state() & NET::DemandsAttention ? " DEMANDS ATTENTION " : " not ") << std::endl;
		}
	}
*/
}

#include "taskbarwatcher.moc"
