/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "themedialogs.h"

#include "themechooser.h"
#include "themeeditor.h"
#include "themeexporter.h"
#include "exportadvises.h"

#include "coverdisplay.h"

#include "thememanager.h"
#include "tools.h"

#include "kcolorcombo2.h"
#include "helplink.h"

#include <klocale.h>
#include <qlayout.h>
#include <kpushbutton.h>
#include <kiconloader.h>
#include <qlistbox.h>
#include <qdesktopwidget.h>
#include <kmessagebox.h>
#include <qstylesheet.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <kurllabel.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include <qbitmap.h>
#include <kseparator.h>
#include <qpainter.h>
#include <qdir.h>
#include <qfile.h>
#include <kfiledialog.h>
#include <qobjectlist.h>
#include <qtextedit.h>
#include <qtextbrowser.h>
#include <qgroupbox.h>
#include <kprogress.h>
#include <ktar.h>
#include <qeventloop.h>
#include <qtooltip.h>

#include <kio/job.h>

#include <iostream>

/** class QListBoxPixmapMargin: */

QListBoxPixmapMargin::QListBoxPixmapMargin(QListBox *listbox, const QPixmap &pix, const QString &text)
 : QListBoxPixmap(listbox, pix, text)
{
}

QListBoxPixmapMargin::QListBoxPixmapMargin(QListBox *listbox, const QPixmap &pix, const QString &text, QListBoxItem *after)
 : QListBoxPixmap(listbox, pix, text, after)
{
}

int QListBoxPixmapMargin::height(const QListBox *lb) const
{
	return QListBoxPixmap::height(lb) + 2 * 2;
}

/** class ThemeChooserDialog: */

ThemeChooserDialog::ThemeChooserDialog(int screenWidth, int screenHeight, QWidget *parent)
 : KDialogBase(KDialogBase::Swallow, i18n("Choose Theme"), KDialogBase::Ok | KDialogBase::Apply | KDialogBase::Cancel,
               KDialogBase::Ok, parent, /*name=*/"ThemeChooserDialog", /*modal=*/true, /*separator=*/true),
   m_screenWidth(screenWidth), m_screenHeight(screenHeight)
{
	setPlainCaption(i18n("%1 - Kirocker Music Display").arg(i18n("Choose Theme")));

	QWidget *page = new QWidget(this);
	QVBoxLayout *topLayout = new QVBoxLayout(page, /*margin=*/0, spacingHint());
	m_chooser = new ThemeChooserWidget(page);
	topLayout->addWidget(m_chooser);

	m_chooser->buttonNew->setIconSet(SmallIconSet("filenew"));
	m_chooser->buttonEdit->setIconSet(SmallIconSet("edit"));
	m_chooser->buttonDelete->setIconSet(SmallIconSet("editdelete"));
	m_chooser->buttonExport->setIconSet(SmallIconSet("fileexport"));
	m_chooser->buttonImport->setIconSet(SmallIconSet("fileimport"));

	loadThemes(Theme::current());

	setMainWidget(page);

	QDesktopWidget desktop;
	int height = desktop.height() * 5 / 7;
	int width  = QMAX(height * 4 / 5, m_chooser->sizeHint().width() * 12 / 10);
	resize(width, height);

	connect( m_chooser->themeList, SIGNAL(doubleClicked(QListBoxItem*)), actionButton(Ok), SLOT(animateClick()) );
	connect( m_chooser->themeList, SIGNAL(returnPressed(QListBoxItem*)), actionButton(Ok), SLOT(animateClick()) );

	connect( m_chooser->themeList, SIGNAL(selectionChanged()), this, SLOT(selectedThemeChanged()) );

	connect( m_chooser->buttonNew,    SIGNAL(clicked()), this, SLOT(slotNew())    );
	connect( m_chooser->buttonEdit,   SIGNAL(clicked()), this, SLOT(slotEdit())   );
	connect( m_chooser->buttonDelete, SIGNAL(clicked()), this, SLOT(slotDelete()) );
	connect( m_chooser->buttonExport, SIGNAL(clicked()), this, SLOT(slotExport()) );
	connect( m_chooser->buttonImport, SIGNAL(clicked()), this, SLOT(slotImport()) );

	connect( m_chooser->getMoreThemes, SIGNAL(leftClickedURL()), this, SLOT(goGetMoreThemes()) );

	selectedThemeChanged();
}

// TODO Gray Apply if index is same as before
// TODO Do not hide mouse when showing dialog

ThemeChooserDialog::~ThemeChooserDialog()
{
}

void ThemeChooserDialog::loadThemes(Theme *themeToSelect)
{
	// If we reload the view, keep the scrolling where it was:
	int x = m_chooser->themeList->contentsX();
	int y = m_chooser->themeList->contentsY();
	int topIndex = m_chooser->themeList->topItem();
	m_chooser->themeList->setUpdatesEnabled(false);
	m_chooser->themeList->verticalScrollBar()->setUpdatesEnabled(false);
	m_chooser->themeList->viewport()->setUpdatesEnabled(false);

	m_chooser->themeList->clear();

	ThemeList themes = ThemeManager::instance()->themes();
	int i = 0;
	QPtrListIterator<Theme> it(themes);
	Theme *theme;
	while ((theme = it.current()) != 0) {
		++it;
		new QListBoxPixmapMargin(m_chooser->themeList, theme->previewPixmap(m_screenWidth, m_screenHeight), theme->themeName());
		if (theme == themeToSelect)
			m_chooser->themeList->setCurrentItem(i);
		i++;
	}

	if (topIndex > 0 || x > 0 || y > 0) {
		// If we reload the view, keep the scrolling where it was:
		m_chooser->themeList->setTopItem(topIndex); // Else, setContentsPos() below does not work!!
		m_chooser->themeList->setContentsPos(x, y);
		// But if the current item moved (renamed...), still make it visible:
		m_chooser->themeList->ensureCurrentVisible();
	}
	// To avoid flickering (list is cleared, then current theme is selected, then we call setTopItem, thenset ContentPos: a LOT of changes):
	m_chooser->themeList->setUpdatesEnabled(true);
	m_chooser->themeList->update();
	m_chooser->themeList->verticalScrollBar()->setUpdatesEnabled(true);
	m_chooser->themeList->verticalScrollBar()->update();
	m_chooser->themeList->viewport()->setUpdatesEnabled(true);
	m_chooser->themeList->viewport()->update();
}

void ThemeChooserDialog::applyChanges()
{
	ThemeManager::instance()->setTheme(selectedTheme());
}

void ThemeChooserDialog::slotApply()
{
	applyChanges();
	KDialogBase::slotApply();
}

void ThemeChooserDialog::slotOk()
{
	applyChanges();
	KDialogBase::slotOk();
}

void ThemeChooserDialog::selectedThemeChanged()
{
	Theme *theme = selectedTheme();
	if (theme)
		m_chooser->buttonDelete->setEnabled(theme->isUserTheme());
}

void ThemeChooserDialog::slotNew()
{
	// Remember the current selected theme in the list:
	int oldSelectedIndex = m_chooser->themeList->currentItem();

	// Create a new folder for the theme, and save a config file with the name:
	QString saveFolder = KGlobal::dirs()->saveLocation("data", "kirocker/themes/");
	QString folderName = Tools::fileNameForNewFile("New Theme", saveFolder, ' ');
	QString location = saveFolder + "/" + folderName;
	QDir dir;
	dir.mkdir(location);
	QFile config(location + "/kirocker-theme.config");
	if (!config.open(IO_WriteOnly))
		return; // TODO Manage that case, eg. if the saveFolder does not exist!
	QTextStream stream(&config);
	stream.setEncoding(QTextStream::UnicodeUTF8);
	stream << "[Theme]\n"
	       << "Name=" << folderName << "\n";
	config.close();

	// Add the theme in the software:
	Theme *newTheme = new Theme(location, folderName);
	ThemeManager::instance()->addTheme(newTheme);
	loadThemes(/*andSelect=*/newTheme);

	// Start editing this new theme:
	bool applyOrOkPressed = slotEdit(/*isNew=*/true);

	// If it was canceled (without any Apply first), remove the theme and select the previous one:
	if (!applyOrOkPressed) {
		doDeletion();
		m_chooser->themeList->setCurrentItem(oldSelectedIndex);
		if (m_chooser->themeList->currentItem())
			m_chooser->themeList->setSelected(m_chooser->themeList->currentItem(), true);
	}
}

Theme *ThemeChooserDialog::selectedTheme()
{
	ThemeList themes = ThemeManager::instance()->themes();
	return themes.at(m_chooser->themeList->currentItem());
}

bool ThemeChooserDialog::slotEdit(bool isNew)
{
	ThemeEditorDialog dialog(m_screenWidth, m_screenHeight, selectedTheme(), (isNew ? i18n("New Theme") : i18n("Edit Theme")), this);
	connect( &dialog, SIGNAL(themeChanged()), this, SLOT(editedThemeChanged()) );
	connect( &dialog, SIGNAL(needCancel()),   this, SLOT(needCancel())         );
	dialog.exec();

	return dialog.applyOrOkPressed();
}

void ThemeChooserDialog::slotDelete()
{
	Theme *theme = selectedTheme();
	int result = KMessageBox::warningYesNo(
		this,
		"<qt>" + i18n("Are you sure you want to definitively remove the theme <b>%1</b>?").arg(theme->themeName()),
		"Remove Theme",
		KStdGuiItem::del(),
		KStdGuiItem::cancel()
	);
	if (result == KMessageBox::Yes)
		doDeletion();
}

void ThemeChooserDialog::doDeletion()
{
	Theme *theme = selectedTheme();
	Tools::deleteRecursively(theme->location());
	ThemeManager::instance()->removeTheme(theme);
	m_chooser->themeList->removeItem(m_chooser->themeList->currentItem()); // Will focus the next/previous item, but NOT select it
	if (m_chooser->themeList->currentItem() > -1)
		m_chooser->themeList->setSelected(m_chooser->themeList->currentItem(), true);
}

/**
 * This code comes from KFileDialog::getSaveFileName() but allow to specify an initial fileName.
 */
#include <krecentdocument.h>
QString ThemeChooserDialog::getSaveFileName(const QString &dir, const QString &filter, QWidget *parent, const QString &caption, const QString &fileName)
{
	bool specialDir = dir.at(0) == ':';
	KFileDialog dlg( specialDir ? dir : QString::null, filter, parent, "filedialog", true);
	if ( !specialDir )
		dlg.setSelection( dir ); // may also be a filename

	dlg.setOperationMode( KFileDialog::Saving );
	//dlg.setCaption(caption.isNull() ? i18n("Save As") : caption);
	dlg.setPlainCaption(i18n("%1 - Kirocker Music Display").arg(caption.isNull() ? i18n("Save As") : caption));

	//BEGIN Added code
	dlg.setSelection(fileName);
	//END

	dlg.exec();

	QString filename = dlg.selectedFile();
	if (!filename.isEmpty())
		KRecentDocument::add(filename);

	return filename;
}

void ThemeChooserDialog::slotExport()
{
	// Ask for theme copyright information:
	Theme *theme = selectedTheme();
	QPixmap preview = *(m_chooser->themeList->pixmap(m_chooser->themeList->currentItem()));
	ThemeExportDialog dialog(theme, preview, this);
	if (dialog.exec() == QDialog::Rejected)
		return;

	// Ask for an archive path:
	QString filter = "*.kirocker-theme.tar.gz|Kirocker Music Display Theme Archive\n*|All files";
	//QString path = KFileDialog::getSaveFileName(/*startDir=*/":export-theme", filter, this, i18n("Export Theme %1").arg(theme->themeName()));
	//QString path = getSaveFileName(/*startDir=*/":export-theme", filter, this, i18n("Export Theme %1").arg(theme->themeName()), theme->folderName().replace("/", "%2f"));
	//if (path.isEmpty())
	//	return;
	QString path;
	for (bool askAgain = true; askAgain; ) {
		path = getSaveFileName(/*startDir=*/":export-theme", filter, this, i18n("Export Theme %1").arg(theme->themeName()), theme->folderName().replace("/", "%2f"));
		if (path.isEmpty()) // User canceled
			return;
		if (QFile::exists(path)) {
			int result = KMessageBox::questionYesNoCancel(
				this,
				"<qt>" + i18n("The file <b>%1</b> already exists. Do you really want to overwrite it?")
					.arg(KURL(path).fileName()),
				i18n("Overwrite File?"),
				KGuiItem(i18n("&Overwrite"), "filesave")
			);
			if (result == KMessageBox::Cancel)
				return;
			else if (result == KMessageBox::Yes)
				askAgain = false;
		} else
			askAgain = false;
	}

	// Figure out the file name, without extension:
	QString fileName = path.mid(path.findRev("/") + 1);
	fileName = fileName.left(fileName.length() - QString(".kirocker-theme.tar.gz").length());

	// Theme files to export in the archive:
	QString normalBackgroundImage = theme->normalBackgroundImagePath();
	QString wideBackgroundImage   = theme->wideBackgroundImagePath();
	QString normalPreviewImage    = theme->normalPreviewPath();
	QString widePreviewImage      = theme->widePreviewPath();
	int fileCount =  /*kirocker-theme.config  &  README:*/2  +
		(normalBackgroundImage.isEmpty() ? 0 : 1) +
		(wideBackgroundImage.isEmpty()   ? 0 : 1) +
		(normalPreviewImage.isEmpty()    ? 0 : 1) +
		(widePreviewImage.isEmpty()      ? 0 : 1);

	// List used frames without duplicates:
	QValueList<Frame*> frames;
	for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
		Frame::Shape shape = (Frame::Shape) i;
		Frame *frame = theme->frame(shape);
		if (frame && !frames.contains(frame))
			frames.append(frame);
	}
	fileCount += frames.count() * 2; // frame.png  &  kirocker-frame.config

	// Initialize the progress dialog:
	KProgressDialog progressDialog(0, 0, i18n("Export Theme"), i18n("Exporting theme <b>%1</b>. Please wait...").arg(theme->themeName()), /*modal=*/true);
	progressDialog.showCancelButton(false);
	progressDialog.setAutoClose(true);
	progressDialog.show();
	KProgress *progress = progressDialog.progressBar();
	progress->setTotalSteps(fileCount);
	progress->setValue(0);

	// Create the archive file, and add a single root folder containing the archive name without the extension:
	KTar tar(path, "application/x-gzip");
	tar.open(IO_WriteOnly);

	// Create the folders ArchiveName/  &  ArchiveName/themes:
	tar.writeDir(fileName, "", "");
	tar.writeDir(fileName + "/themes", "", "");

	// Create file ArchiveName/README:
	QString readmeText =
		i18n("This is a theme for the full-screen display of Kirocker Music Display, version 4 and later.") + "\n" +
		"\n" +
		i18n("Kirocker Music Display is both a panel applet and a full-screen display to see what's you're listening to in Amarok, and control the playback.") + "\n" +
		i18n("You can download Kirocker Music Display here:") + "\n" +
		"http://www.kde-apps.org/content/show.php/Kirocker+Music+Display?content=52869" + "\n" +
		"\n" +
		i18n("To install the theme, just open the archive file (the file ending with \".kirocker-theme.tar.gz\").") + "\n" +
		i18n("Alternatively, you can launch the full-screen display of Kirocker Music Display, open the \"Choose Theme\" window, click \"Import...\" and browse to the archive to install.") + "\n" +
		"\n" +
		i18n("See the About window of the theme in Kirocker Music Display for author & copyright information about the theme and its background image.") + "\n" +
		i18n("Alternatively, open the file kirocker-theme.config in a text editor: information is in the [Author] group.") + "\n";
	QCString readmeData = readmeText.utf8();
	int size = readmeData.size() - 1;
	tar.prepareWriting(fileName + "/README", "", "", size);
	tar.writeData(readmeData, size);
	tar.doneWriting(size);
	progress->advance(1);
	kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);

	// Append the theme folder in ArchiveName/themes/ThemeFolderName/*
	QString themeEmbeddedFolder = fileName + "/themes/" + theme->folderName().replace("/", "%2f");
	tar.writeDir(themeEmbeddedFolder, "", "");

	// And all files in that folder:
	tar.addLocalFile(theme->location() + "kirocker-theme.config", themeEmbeddedFolder + "/kirocker-theme.config");
	progress->advance(1);
	kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
	if (!normalBackgroundImage.isEmpty()) {
		tar.addLocalFile(normalBackgroundImage, themeEmbeddedFolder + "/" + KURL(normalBackgroundImage).fileName());
		progress->advance(1);
		kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
	}
	if (!wideBackgroundImage.isEmpty()) {
		tar.addLocalFile(wideBackgroundImage, themeEmbeddedFolder + "/" + KURL(wideBackgroundImage).fileName());
		progress->advance(1);
		kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
	}
	if (!normalPreviewImage.isEmpty()) {
		tar.addLocalFile(normalPreviewImage, themeEmbeddedFolder + "/" + KURL(normalPreviewImage).fileName());
		progress->advance(1);
		kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
	}
	if (!widePreviewImage.isEmpty()) {
		tar.addLocalFile(widePreviewImage, themeEmbeddedFolder + "/" + KURL(widePreviewImage).fileName());
		progress->advance(1);
		kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
	}

	// If there are frames, create folder ArchiveName/frames:
	if (frames.count() > 0) {
		tar.writeDir(fileName + "/frames/", "", "");
		// And append every frames folders in ArchiveName/frames/FrameName/*:
		for (QValueList<Frame*>::iterator it = frames.begin(); it != frames.end(); ++it) {
			Frame *frame = *it;
			QString frameEmbeddedFolder = fileName + "/frames/" + frame->folderName();
			tar.addLocalFile(frame->location() + "frame.png", frameEmbeddedFolder + "/frame.png");
			progress->advance(1);
			kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
			tar.addLocalFile(frame->location() + "kirocker-frame.config", frameEmbeddedFolder + "/kirocker-frame.config");
			progress->advance(1);
			kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
		}
	}

	// Inform the user the export is already done:
// 	KMessageBox::information(
// 		this,
// 		"<qt>" + i18n("The theme <b>%1</b> has been exported.").arg(theme->themeName()/*, fileName*/),
// 		i18n("%1 - %2").arg("Export Theme", i18n("Kirocker Music Display")),
// 		/*dontShowAgainName=*/QString(),
// 		KMessageBox::Notify | KMessageBox::PlainCaption
// 	);

	ThemeAdviseDialog *advise = new ThemeAdviseDialog(theme, 0);
	advise->show();
}

void ThemeChooserDialog::slotImport()
{
	// It seems the export progress dialog is useless, so I do not bother adding one here when importing!

	// Ask for an archive path:
	QString filter = "*.kirocker-theme.tar.gz|Kirocker Music Display Theme Archive\n*|All files";
	QString path = KFileDialog::getOpenFileName(/*startDir=*/":import-theme", filter, this, i18n("Import Theme"));
	if (path.isEmpty())
		return;

	Theme *firstImportedTheme = ThemeManager::instance()->import(path);

	// Load new themes in the list box:
	loadThemes(firstImportedTheme ? firstImportedTheme : selectedTheme()); // If no theme imported (perhapse only frames were imported), reselect the previous one
}

// void ThemeChooserDialog::badArchiveFormat()
// {
// 	KMessageBox::error(
// 		this,
// 		i18n("The theme cannot be imported because the archive is malformed.") + "\n" +
// 		i18n("Try to download it again or contact its author."),
// 		i18n("%1 - %2").arg("Keyboard Shortcuts", i18n("Kirocker Music Display")),
// 		/*KMessageBox::Notify | */KMessageBox::PlainCaption
// 	);
// }

void ThemeChooserDialog::editedThemeChanged()
{
	// Find current theme:
//	QListBoxPixmapMargin *currentItem = (QListBoxPixmapMargin *) m_chooser->themeList->selectedItem();
	Theme *theme = selectedTheme();

	// Apply theme changes:
	theme->setUsed(false);
	if (theme == Theme::current())
		ThemeManager::instance()->setTheme(theme);

	// Re-sort the theme list if theme name has changed:
	ThemeManager::instance()->reSort();

	// Change the name, and possibly preview:
	loadThemes(theme);

/*
	// Update list view:
	//QListBoxPixmapMargin *item = (QListBoxPixmapMargin *) m_chooser->themeList->selectedItem();
	QListBoxPixmapMargin *newItem = new QListBoxPixmapMargin(m_chooser->themeList, theme->previewPixmap(m_screenWidth, m_screenHeight), theme->themeName(), currentItem);
//	m_chooser->themeList->setSelected(item, false);
//	m_chooser->themeList->setCurrentItem(0);
	delete currentItem;
	m_chooser->themeList->setSelected(newItem, true);
	m_chooser->themeList->setCurrentItem(newItem);
*/
}

void ThemeChooserDialog::goGetMoreThemes()
{
	kapp->invokeBrowser(m_chooser->getMoreThemes->url());
	needCancel();
}

void ThemeChooserDialog::needCancel()
{
	CoverDisplay::focusCoverOnDialogDisappearing = false;
	slotCancel(); // The "Choose Theme" dialog is modal, it is modal to the Kicker panel too (if the full-screen display was launched from the applet)!
	              // Triggering the URL move to a browser, so the panel must still be clickable!
}

/** class ThemeEditorDialog: */

ThemeEditorDialog::ThemeEditorDialog(int screenWidth, int screenHeight, Theme *theme, const QString &title, QWidget *parent)
 : KDialogBase(KDialogBase::Swallow, title, KDialogBase::Ok | KDialogBase::Apply | KDialogBase::Cancel,
               KDialogBase::Ok, parent, /*name=*/"ThemeEditorDialog", /*modal=*/true, /*separator=*/false),
   m_theme(theme), m_applyOrOkPressed(false)
{
	setPlainCaption(i18n("%1 - Kirocker Music Display").arg(title));

	m_temporarTheme = new Theme("", "");
	m_theme->copyTo(m_temporarTheme);

	QWidget *page = new QWidget(this);
	QVBoxLayout *topLayout = new QVBoxLayout(page, /*margin=*/0, spacingHint());
	m_editor = new ThemeEditorWidget(page);
	m_editor->chooseNormalBackgroundImage->setIconSet(SmallIconSet("fileopen"));
	m_editor->removeNormalBackgroundImage->setIconSet(SmallIconSet("locationbar_erase"));
	m_editor->chooseWideBackgroundImage->setIconSet(SmallIconSet("fileopen"));
	m_editor->removeWideBackgroundImage->setIconSet(SmallIconSet("locationbar_erase"));
	topLayout->addWidget(m_editor);
	m_editor->layout()->setMargin(0);

	// Equalise Label Sizes:
	QValueList<QLabel*> labels;
	labels <<
		/*m_editor->themeNameLabel << m_editor->normalBackgroundImageLabel <<*/
		m_editor->backgroundColorLabel << m_editor->textColorLabel << m_editor->nextPlayingColorLabel << m_editor->dateHourColorLabel <<
		m_editor->progressBackgroundLabel << m_editor->progressBarLabel << m_editor->progressBackgroundTextLabel << m_editor->progressBarTextLabel;

	int maxWidth = 0;
	for (QValueList<QLabel*>::const_iterator it = labels.begin(); it != labels.end(); ++it)
		if ((*it)->sizeHint().width() > maxWidth)
			maxWidth = (*it)->sizeHint().width();

	for (QValueList<QLabel*>::const_iterator it = labels.begin(); it != labels.end(); ++it)
		(*it)->setFixedWidth(maxWidth);

	// LOAD VALUES:
	// Theme properties:
	m_editor->themeName->setText(m_temporarTheme->themeName());
	// Screen:
	m_editor->backgroundColor->setColor(m_temporarTheme->backgroundColorSetting());
	m_editor->backgroundColorOpacity->setValue(m_temporarTheme->backgroundColorOpacity());
	m_editor->textColor->setColor(m_temporarTheme->textColorSetting());
	m_editor->nextPlayingColor->setColor(m_temporarTheme->nextPlayingColorSetting());
	m_editor->dateHourColor->setColor(m_temporarTheme->dateHourColorSetting());
	// Progress bar:
	m_editor->progressBackgroundColor->setColor(m_temporarTheme->progressBackgroundColorSetting());
	m_editor->progressBarColor->setColor(m_temporarTheme->progressBarColorSetting());
	m_editor->progressBackgroundTextColor->setColor(m_temporarTheme->progressBackgroundTextColorSetting());
	m_editor->progressBarTextColor->setColor(m_temporarTheme->progressBarTextColorSetting());
	// Shadows:
	m_editor->textShadowEnabled->setChecked(m_temporarTheme->textShadowEnabled());
	m_editor->textShadowColor->setColor(m_temporarTheme->textShadowColorSetting());
	m_editor->nextPlayingShadowEnabled->setChecked(m_temporarTheme->nextPlayingShadowEnabled());
	m_editor->nextPlayingShadowColor->setColor(m_temporarTheme->nextPlayingShadowColorSetting());
	m_editor->dateHourShadowEnabled->setChecked(m_temporarTheme->dateHourShadowEnabled());
	m_editor->dateHourShadowColor->setColor(m_temporarTheme->dateHourShadowColorSetting());
	// Lyrics:
	m_editor->lyricsBackgroundColor->setColor(m_temporarTheme->lyricsBackgroundColorSetting());
	m_editor->lyricsTextColor->setColor(m_temporarTheme->lyricsTextColorSetting());
	m_editor->lyricsScrollBackgroundColor->setColor(m_temporarTheme->lyricsScrollBackgroundColorSetting());
	m_editor->lyricsScrollButtonsColor->setColor(m_temporarTheme->lyricsScrollButtonsColorSetting());


	if (Tools::isWideScreen(screenWidth, screenHeight)) {
		m_editor->normalBackgroundImageLabel->setText("For Normal Screens:<b> </b><br> ");
		m_editor->wideBackgroundImageLabel->setText("<b>For Wide Screens:</b><br><i>(your screen)</i>");
	} else {
		m_editor->normalBackgroundImageLabel->setText("<b>For Normal Screens:</b><br><i>(your screen)</i>");
		m_editor->wideBackgroundImageLabel->setText("For Wide Screens:<b> </b><br> ");
	}

	QString normalMonitor = KGlobal::dirs()->findResource("data", "kirocker/images/monitor-normal.png");
	QString wideMonitor   = KGlobal::dirs()->findResource("data", "kirocker/images/monitor-wide.png");
	m_editor->normalBackgroundImagePreview->setPixmap(QPixmap(normalMonitor));
	m_editor->wideBackgroundImagePreview->setPixmap(QPixmap(wideMonitor));

	m_editor->resolutionsHelpLink->setText("What are the resolutions?");
	m_editor->resolutionsHelpLink->setMessage(
		"<p>Computer screens can have two size ratios: normal like TVs or wide like 16/9 cinema displays.</p>"
		"<p>The background image is stretched to fill the whole screen. "
		"When exporting the theme to make it available to other people, they can see the background distorsed if they have a different screen size ratio. "
		"This is why you can provide a background image for both screen types.</p>"
		"<p>If size ratio is not important for the beautifulness of your background image, then you are not obliged to provide two images. "
		"The image will be used for all screens.</p>"
		"<p>Here are the sizes to use for background images, in pixels:<br>" // From http://en.wikipedia.org/wiki/Image:Vector_Video_Standards2.png
		"<nobr><b>For normal screens</b>: 1600x1200 (recommanded), 1400x1050, 1280x960, 1024x768;</nobr><br>" // = 4/3   = 1.333
		"<nobr><b>For wide screens</b>: 1920x1200, 1680x1050 (recommanded), 1600x1000, 1280x800.</nobr></p>"  // = 16/10 = 1.6
	);

// 	m_editor->framesHelpLink->setText("Learn some design tips...");
// 	m_editor->framesHelpLink->setMessage(
// 		"<p>Having a frame at the bottom of the screen visually separate primary information (the track information) and secondary information (the date & time, the task bar). "
// 		"But it is also visually more appealing when Lyrics are shown. It avoids to have an empty rectangle at the bottom of the screen.</p>"
// 		"<p>MORE TIPS TO COME.</p>"
// 	);

//	m_editor->framesHelpLink->hide();

	m_editor->manageFrames->hide();
	m_editor->framesHelpLink->setText("How to create new frames?");
	connect( m_editor->framesHelpLink, SIGNAL(leftClickedURL()), this, SLOT(howToCreateCustomFrames()) );
// 	m_editor->framesHelpLink->setMessage(
// 		"<p>. "
// 		".</p>"
// 		"<p>.</p>"
// 	);

	QValueList<QLabel*> frameLabels;
	m_editor->framesScrollView->viewport()->setPaletteBackgroundColor(paletteBackgroundColor());
	QWidget *widget = new QWidget(m_editor->framesScrollView->viewport());
	QVBoxLayout* layout = new QVBoxLayout(widget);
	m_editor->framesScrollView->addChild(widget);
	for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
		FrameEditor *frameEditor = new FrameEditor(m_temporarTheme, (Frame::Shape) i, widget);
		layout->addWidget(frameEditor);
		layout->addStretch();
		if (i == Frame::Screen || i == Frame::PluginIconBar || i == Frame::Stars)
			frameEditor->hide();
		if (Frame::lineAfterShape((Frame::Shape) i)) {
			QFrame *line = new KSeparator(KSeparator::HLine, widget);
			line->setPalette(palette());
			layout->addWidget(line);
		}
		frameLabels << frameEditor->frameNameLabel;
		m_frameEditors[i] = frameEditor;
		connect( frameEditor, SIGNAL(somethingChanged()), this, SLOT(somethingChanged()) );
	}

	layout->setMargin(KDialog::marginHint());
	layout->setSpacing(KDialog::spacingHint());

	/*int */maxWidth = 0;
	for (QValueList<QLabel*>::const_iterator it = frameLabels.begin(); it != frameLabels.end(); ++it)
		if ((*it)->sizeHint().width() > maxWidth)
			maxWidth = (*it)->sizeHint().width();

	for (QValueList<QLabel*>::const_iterator it = frameLabels.begin(); it != frameLabels.end(); ++it)
		(*it)->setFixedWidth(maxWidth);

	//int minWidth = ((QWidget *) labels[0]->parent()->parent())->sizeHint().width();
	int minWidth = m_frameEditors[0]->minimumSizeHint().width()/* - 2 * m_editor->framesScrollView->frameWidth() - m_editor->framesScrollView->verticalScrollBar()->width()*/;
	m_editor->framesScrollView->setMinimumWidth(/*m_editor->framesScrollView->frameWidth() * 2 +*/ /*widget->minimumSizeHint().width()*/ minWidth);


	// CONNECT TO CHANGES:
	// Theme properties:
	connect( m_editor->themeName,                   SIGNAL(textChanged(const QString&)), this, SLOT(somethingChanged()) );
	// Screen:
	connect( m_editor->backgroundColor,             SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->backgroundColorOpacity,      SIGNAL(valueChanged(int)),           this, SLOT(somethingChanged()) );
	connect( m_editor->textColor,                   SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->nextPlayingColor,            SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->dateHourColor,               SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	// Progress bar:
	connect( m_editor->progressBackgroundColor,     SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->progressBarColor,            SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->progressBackgroundTextColor, SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->progressBarTextColor,        SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	// Shadows:
	connect( m_editor->textShadowEnabled,           SIGNAL(stateChanged(int)),           this, SLOT(somethingChanged()) );
	connect( m_editor->textShadowColor,             SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->nextPlayingShadowEnabled,    SIGNAL(stateChanged(int)),           this, SLOT(somethingChanged()) );
	connect( m_editor->nextPlayingShadowColor,      SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->dateHourShadowEnabled,       SIGNAL(stateChanged(int)),           this, SLOT(somethingChanged()) );
	connect( m_editor->dateHourShadowColor,         SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	// Lyrics:
	connect( m_editor->lyricsBackgroundColor,       SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->lyricsTextColor,             SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->lyricsScrollBackgroundColor, SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );
	connect( m_editor->lyricsScrollButtonsColor,    SIGNAL(changed(const QColor&)),      this, SLOT(somethingChanged()) );

	connect( m_editor->pbAbout,                     SIGNAL(clicked()),                   this, SLOT(aboutTheme())       );


	connect( m_editor->chooseNormalBackgroundImage, SIGNAL(clicked()),                   this, SLOT(chooseNormalBackgroundImage()) );
	connect( m_editor->removeNormalBackgroundImage, SIGNAL(clicked()),                   this, SLOT(removeNormalBackgroundImage()) );
	connect( m_editor->chooseWideBackgroundImage,   SIGNAL(clicked()),                   this, SLOT(chooseWideBackgroundImage())   );
	connect( m_editor->removeWideBackgroundImage,   SIGNAL(clicked()),                   this, SLOT(removeWideBackgroundImage())   );


	// Finalize:
	setDefaultColors();
	somethingChanged();
	backgroundImageChanged();
	setMainWidget(page);

	m_editor->removeNormalBackgroundImage->setEnabled(m_temporarTheme->hasNormalBackgroundImage());
	m_editor->removeWideBackgroundImage->setEnabled(m_temporarTheme->hasWideBackgroundImage());

	m_editor->themeName->selectAll();
}

void ThemeEditorDialog::howToCreateCustomFrames()
{
	kapp->invokeBrowser("http://slaout.linux62.org/kirocker/custom-frames.html");
	slotCancel(); // The "Choose Theme" dialog is modal, it is modal to the Kicker panel too (if the full-screen display was launched from the applet)!
	              // Triggering the URL move to a browser, so the panel must still be clickable!
	emit needCancel();
}


ThemeEditorDialog::~ThemeEditorDialog()
{
}

void ThemeEditorDialog::somethingChanged()
{
////std::cout << "somethingChanged() BEGIN, opacity:" << m_temporarTheme->backgroundColorOpacity() << std::endl;
	// Remember some settings (for below):
	QColor backgroundColor        = m_temporarTheme->backgroundColor();
	int    backgroundColorOpacity = m_temporarTheme->backgroundColorOpacity();
	QColor textColor              = m_temporarTheme->textColor();

//	std::cout << "Something changed in the theme editor." << std::endl;
	m_temporarTheme->setThemeName(m_editor->themeName->text());

	// Screen:
	m_temporarTheme->setBackgroundColor(m_editor->backgroundColor->color());
	m_temporarTheme->setBackgroundColorOpacity(m_editor->backgroundColorOpacity->value());
	m_temporarTheme->setTextColor(m_editor->textColor->color());
	m_temporarTheme->setNextPlayingColor(m_editor->nextPlayingColor->color());
	m_temporarTheme->setDateHourColor(m_editor->dateHourColor->color());
	// Progress bar:
	m_temporarTheme->setProgressBackgroundColor(m_editor->progressBackgroundColor->color());
	m_temporarTheme->setProgressBarColor(m_editor->progressBarColor->color());
	m_temporarTheme->setProgressBackgroundTextColor(m_editor->progressBackgroundTextColor->color());
	m_temporarTheme->setProgressBarTextColor(m_editor->progressBarTextColor->color());
	// Shadows:
	m_temporarTheme->setTextShadowEnabled(m_editor->textShadowEnabled->isOn());
	m_temporarTheme->setTextShadowColor(m_editor->textShadowColor->color());
	m_temporarTheme->setNextPlayingShadowEnabled(m_editor->nextPlayingShadowEnabled->isOn());
	m_temporarTheme->setNextPlayingShadowColor(m_editor->nextPlayingShadowColor->color());
	m_temporarTheme->setDateHourShadowEnabled(m_editor->dateHourShadowEnabled->isOn());
	m_temporarTheme->setDateHourShadowColor(m_editor->dateHourShadowColor->color());
	// Lyrics:
	m_temporarTheme->setLyricsBackgroundColor(m_editor->lyricsBackgroundColor->color());
	m_temporarTheme->setLyricsTextColor(m_editor->lyricsTextColor->color());
	m_temporarTheme->setLyricsScrollBackgroundColor(m_editor->lyricsScrollBackgroundColor->color());
	m_temporarTheme->setLyricsScrollButtonsColor(m_editor->lyricsScrollButtonsColor->color());

	m_editor->textShadowColor->setEnabled(m_editor->textShadowEnabled->isOn());
	m_editor->nextPlayingShadowColor->setEnabled(m_editor->nextPlayingShadowEnabled->isOn());
	m_editor->dateHourShadowColor->setEnabled(m_editor->dateHourShadowEnabled->isOn());

	// Frames:
	for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
		int paddingTop    = m_frameEditors[i]->getPaddingTop();
		int paddingLeft   = m_frameEditors[i]->getPaddingLeft();
		int paddingRight  = m_frameEditors[i]->getPaddingRight();
		int paddingBottom = m_frameEditors[i]->getPaddingBottom();
		Frame *frame = m_frameEditors[i]->getFrame();
		m_temporarTheme->setFrame((Frame::Shape) i, frame);
		m_temporarTheme->setFramePaddings((Frame::Shape) i, paddingTop, paddingLeft, paddingRight, paddingBottom);
	}

	setDefaultColors();

	// Recreate preview images if some part of them have changed:
	if (    backgroundColor        != m_temporarTheme->backgroundColor()
	     || backgroundColorOpacity != m_temporarTheme->backgroundColorOpacity()
	     || textColor              != m_temporarTheme->textColor() ) {
		recreatePreview();
		backgroundImageChanged();
	}
////std::cout << "somethingChanged() END, opacity:" << m_temporarTheme->backgroundColorOpacity() << std::endl;
}

void ThemeEditorDialog::setDefaultColors()
{
	// Screen:
	m_editor->backgroundColor->setDefaultColor(m_temporarTheme->defaultBackgroundColor());
	m_editor->textColor->setDefaultColor(m_temporarTheme->defaultTextColor());
	m_editor->nextPlayingColor->setDefaultColor(m_temporarTheme->defaultNextPlayingColor());
	m_editor->dateHourColor->setDefaultColor(m_temporarTheme->defaultDateHourColor());
	// Progress bar:
	m_editor->progressBackgroundColor->setDefaultColor(m_temporarTheme->defaultProgressBackgroundColor());
	m_editor->progressBarColor->setDefaultColor(m_temporarTheme->defaultProgressBarColor());
	m_editor->progressBackgroundTextColor->setDefaultColor(m_temporarTheme->defaultProgressBackgroundTextColor());
	m_editor->progressBarTextColor->setDefaultColor(m_temporarTheme->defaultProgressBarTextColor());
	// Shadows:
	m_editor->textShadowColor->setDefaultColor(m_temporarTheme->defaultTextShadowColor());
	m_editor->nextPlayingShadowColor->setDefaultColor(m_temporarTheme->defaultNextPlayingShadowColor());
	m_editor->dateHourShadowColor->setDefaultColor(m_temporarTheme->defaultDateHourShadowColor());
	// Lyrics:
	m_editor->lyricsBackgroundColor->setDefaultColor(m_temporarTheme->defaultLyricsBackgroundColor());
	m_editor->lyricsTextColor->setDefaultColor(m_temporarTheme->defaultLyricsTextColor());
	m_editor->lyricsScrollBackgroundColor->setDefaultColor(m_temporarTheme->defaultLyricsScrollBackgroundColor());
	m_editor->lyricsScrollButtonsColor->setDefaultColor(m_temporarTheme->defaultLyricsScrollButtonsColor());
}

bool ThemeEditorDialog::applyOrOkPressed()
{
	return m_applyOrOkPressed;
}

void ThemeEditorDialog::applyChanges()
{
	m_applyOrOkPressed = true;

	// If the spinbox has changed but is still focused, no "changed" event has been triggered.
	// Force the change, to get the new value:
	if (m_editor->backgroundColorOpacity->hasFocus()) {
		m_editor->backgroundColorOpacity->clearFocus();
		//somethingChanged();
		m_editor->backgroundColorOpacity->setFocus();
	}

	for (int i = 0; i < Frame::SHAPE_COUNT; i++)
		m_frameEditors[i]->aboutToApplyChanges();

	renameThemeFolderIfPossible();

	m_temporarTheme->copyTo(m_theme);
	m_theme->save();

	emit themeChanged();
}

void ThemeEditorDialog::renameThemeFolderIfPossible()
{
	// If the folder cannot be renamed, there is nothing to do:
	if (!m_temporarTheme->isUserTheme())
		return;

	// Escape the theme name to a real folder name:
	QString newFolderName = m_temporarTheme->themeName();
	newFolderName = newFolderName.replace("/", "%2f");
	if (newFolderName.isEmpty())
		newFolderName = "_";

	// If the folder is already named like the theme name, no rename to do:
	if (m_temporarTheme->folderName() == newFolderName)
		return;

	// If the folder already exists (but is not assigned to that theme: see above), find another folder name:
	QString oldLocation = m_temporarTheme->location();
	if (oldLocation.endsWith("/"))
		oldLocation = oldLocation.left(oldLocation.length() - 1);
	QString location = oldLocation.left(oldLocation.findRev("/"));
	newFolderName = Tools::fileNameForNewFile(newFolderName, location, ' ');

	// Do the folder rename:
	syncMoveAs(KURL::encode_string(oldLocation), KURL::encode_string(location + "/" + newFolderName));

	// Save the new folder name and location to the theme
	m_temporarTheme->setLocationAndFolder(location + "/" + newFolderName + "/", newFolderName);
}

void ThemeEditorDialog::slotApply()
{
	applyChanges();
	KDialogBase::slotApply();
}

void ThemeEditorDialog::slotOk()
{
	applyChanges();
	KDialogBase::slotOk();
}

void ThemeEditorDialog::aboutTheme()
{
	QString message = "<h1>" + QStyleSheet::escape(m_temporarTheme->themeName())   + "</h1>";

	if (!m_temporarTheme->authorName().isEmpty() || !m_temporarTheme->authorEMail().isEmpty() || !m_temporarTheme->authorURL().isEmpty()) {
		message += "<p>";
		if (!m_temporarTheme->authorName().isEmpty())
			message += "<b>Author:</b> " + QStyleSheet::escape(m_temporarTheme->authorName()) +
			           (!m_temporarTheme->authorEMail().isEmpty() || !m_temporarTheme->authorURL().isEmpty() ? "<br>" : "");
		if (!m_temporarTheme->authorEMail().isEmpty())
			message += "<b>E-Mail:</b> <a href='mailto:" + QStyleSheet::escape(m_temporarTheme->authorEMail()) + "'>" + QStyleSheet::escape(m_temporarTheme->authorEMail()) + "</a>" +
			           (!m_temporarTheme->authorURL().isEmpty() ? "<br>" : "");
		if (!m_temporarTheme->authorURL().isEmpty())
			message += "<b>Website:</b> <a href='" + QStyleSheet::escape(m_temporarTheme->authorURL()) + "'>" + QStyleSheet::escape(m_temporarTheme->authorURL()) + "</a>";
		message += "</p>";
	} else if (m_temporarTheme->copyrightInformation().isEmpty())
		message += "<p>No information about its author and copyright.</p>";

	if (!m_temporarTheme->copyrightInformation().isEmpty())
		message += "<p><b>Copyright information:</b><br>" +
		           Tools::tagURLs(QStyleSheet::escape(m_temporarTheme->copyrightInformation())).replace("\n", "<br>") + "</p>";

	KMessageBox::information(this, message, "About Theme", /*dontDhowAgain=*/QString::null, KMessageBox::Notify | KMessageBox::AllowLink);
}



// SHOULD be in Tools, but we have to create a new QObject to handle the signal:

void ThemeEditorDialog::syncCopyAs(const QString &folder, const QString &newFolder)
{
	m_copyFinished = false;
	KIO::CopyJob *copyJob = KIO::copyAs(KURL(folder), KURL(newFolder), /*showProgressInfo=*/false);
	connect( copyJob,  SIGNAL(result(KIO::Job*)), this, SLOT(slotCopyingDone(KIO::Job*)) );
	while (!m_copyFinished)
		//kapp->processEvents();
		kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
}

void ThemeEditorDialog::syncMoveAs(const QString &folder, const QString &newFolder)
{
	m_copyFinished = false;
	KIO::CopyJob *copyJob = KIO::moveAs(KURL(folder), KURL(newFolder), /*showProgressInfo=*/false);
	connect( copyJob,  SIGNAL(result(KIO::Job*)), this, SLOT(slotCopyingDone(KIO::Job*)) );
	while (!m_copyFinished)
		//kapp->processEvents();
		kapp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
}

void ThemeEditorDialog::slotCopyingDone(KIO::Job *)
{
	m_copyFinished = true;
}




void ThemeEditorDialog::chooseNormalBackgroundImage()
{
	QString filter = "*.png *.jpg *.jpeg *.gif|Image files\n*|All files";
	QString fileName = KFileDialog::getOpenFileName(/*startDir=*/":choose-image", filter, this, "Choose Normal Background Image");
	if (!fileName.isEmpty()) {
		if (!m_temporarTheme->normalBackgroundImagePath().isEmpty())
			QFile::remove(m_temporarTheme->normalBackgroundImagePath());
		QString extension = fileName.mid(fileName.findRev("."));
		if (!extension.isEmpty()) {
			QString destination = m_temporarTheme->location() + "background" + extension;
			//std::cout << "Copy: " << fileName << std::endl
			//          << "To: "   << destination << std::endl;
			syncCopyAs(fileName, destination);
			m_editor->removeNormalBackgroundImage->setEnabled(true);
			backgroundImageChanged();
			recreatePreview();
		}
	}
}

void ThemeEditorDialog::chooseWideBackgroundImage()
{
	QString filter = "*.png *.jpg *.jpeg *.gif|Image files\n*|All files";
	QString fileName = KFileDialog::getOpenFileName(/*startDir=*/":choose-image", filter, this, "Choose Normal Background Image");
	if (!fileName.isEmpty()) {
		if (!m_temporarTheme->wideBackgroundImagePath().isEmpty())
			QFile::remove(m_temporarTheme->wideBackgroundImagePath());
		QString extension = fileName.mid(fileName.findRev("."));
		if (!extension.isEmpty()) {
			QString destination = m_temporarTheme->location() + "background-wide" + extension;
			//std::cout << "Copy: " << fileName << std::endl
			//          << "To: "   << destination << std::endl;
			syncCopyAs(fileName, destination);
			m_editor->removeWideBackgroundImage->setEnabled(true);
			backgroundImageChanged();
			recreatePreview();
		}
	}
}

void ThemeEditorDialog::removeNormalBackgroundImage()
{
	int result = KMessageBox::warningYesNo(this, "Are you sure you want to definitively remove the normal background image?", "Remove Normal Background Image", KStdGuiItem::del(), KStdGuiItem::cancel());
	if (result == KMessageBox::Yes) {
		QFile::remove(m_temporarTheme->normalBackgroundImagePath());
		m_editor->removeNormalBackgroundImage->setEnabled(false);
		backgroundImageChanged();
		recreatePreview();
	}
}

void ThemeEditorDialog::removeWideBackgroundImage()
{
	int result = KMessageBox::warningYesNo(this, "Are you sure you want to definitively remove the wide background image?", "Remove Wide Background Image", KStdGuiItem::del(), KStdGuiItem::cancel());
	if (result == KMessageBox::Yes) {
		QFile::remove(m_temporarTheme->wideBackgroundImagePath());
		m_editor->removeWideBackgroundImage->setEnabled(false);
		backgroundImageChanged();
		recreatePreview();
	}
}

void ThemeEditorDialog::backgroundImageChanged()
{
	// Recreate both previews:
	QPixmap normal = m_temporarTheme->backgroundPixmap(151, 113/*, useCache=false*/);
	QPixmap wide   = m_temporarTheme->backgroundPixmap(151, 94/*, useCache=false*/);

	QPainter painter;

	QPixmap normalMonitor(*m_editor->normalBackgroundImagePreview->pixmap());
	painter.begin(&normalMonitor);
	painter.drawPixmap(23, 16, normal);
	if (m_temporarTheme->hasNormalBackgroundImage()) {
		QPixmap icon = kapp->iconLoader()->loadIcon("image", KIcon::Desktop, 16, KIcon::DefaultState);
		painter.drawPixmap(23 + 10, 16 + 5, icon);
		QToolTip::add(m_editor->normalBackgroundImagePreview, i18n("A normal background image is defined."));
	} else if (m_temporarTheme->hasWideBackgroundImage())
		QToolTip::add(m_editor->normalBackgroundImagePreview, i18n("The wide background image is used for normal-ratio screens."));
	else
		QToolTip::add(m_editor->normalBackgroundImagePreview, "");
	painter.end();
	m_editor->normalBackgroundImagePreview->setPixmap(normalMonitor);

	QPixmap wideMonitor(*m_editor->wideBackgroundImagePreview->pixmap());
	painter.begin(&wideMonitor);
	painter.drawPixmap(23, 35, wide);
	if (m_temporarTheme->hasWideBackgroundImage()) {
		QPixmap icon = kapp->iconLoader()->loadIcon("image", KIcon::Desktop, 16, KIcon::DefaultState);
		painter.drawPixmap(23 + 10, 35 + 5, icon);
		QToolTip::add(m_editor->wideBackgroundImagePreview, i18n("A wide background image is defined."));
	} else if (m_temporarTheme->hasNormalBackgroundImage())
		QToolTip::add(m_editor->wideBackgroundImagePreview, i18n("The normal background image is used for wide-ratio screens."));
	else
		QToolTip::add(m_editor->wideBackgroundImagePreview, "");
	painter.end();
	m_editor->wideBackgroundImagePreview->setPixmap(wideMonitor);
}

void ThemeEditorDialog::recreatePreview()
{
	m_theme->invalidPreview();
	m_temporarTheme->invalidPreview();
	m_theme->setUsed(false);
	m_temporarTheme->setUsed(false);

////std::cout << "recreatePreview(), opacity:" << m_temporarTheme->backgroundColorOpacity() << std::endl;

	// Remove the old normal preview, if any:
	if (!m_temporarTheme->normalPreviewPath().isEmpty())
		QFile::remove(m_temporarTheme->normalPreviewPath());
	// Compute a new one, if there is an normal background image to preview:
	if (!m_temporarTheme->normalBackgroundImagePath().isEmpty()) {
		QPixmap preview = m_temporarTheme->backgroundPixmap(100, 75/*, useCache=false*/);
		QString fileName = m_temporarTheme->normalBackgroundImagePath();
		QString extension = fileName.mid(fileName.findRev("."));
		if (!extension.isEmpty()) {
			QString destination = m_temporarTheme->location() + "background.preview" + extension;
			QString format = "PNG";
			if (extension == ".jpg" || extension == ".jpeg")
				format = "JPEG";
			if (extension == ".gif")
				format = "GIF";
			preview.save(destination, format);
////std::cout << "Save " << destination << std::endl;
		}
	}

	// Remove the old wide preview, if any:
	if (!m_temporarTheme->widePreviewPath().isEmpty())
		QFile::remove(m_temporarTheme->widePreviewPath());
	// Compute a new one, if there is an wide background image to preview:
	if (!m_temporarTheme->wideBackgroundImagePath().isEmpty()) {
		QPixmap preview = m_temporarTheme->backgroundPixmap(100, 63/*, useCache=false*/);
		QString fileName = m_temporarTheme->wideBackgroundImagePath();
		QString extension = fileName.mid(fileName.findRev("."));
		if (!extension.isEmpty()) {
			QString destination = m_temporarTheme->location() + "background-wide.preview" + extension;
			QString format = "PNG";
			if (extension == ".jpg" || extension == ".jpeg")
				format = "JPEG";
			if (extension == ".gif")
				format = "GIF";
			preview.save(destination, format);
////std::cout << "Save " << destination << std::endl;
		}
	}
}

/** class FrameEditor: */

FrameEditor::FrameEditor(Theme *theme, Frame::Shape shape, QWidget *parent)
 : FrameChooser(parent), m_theme(theme), m_shape(shape)
{
	// Set the label text:
	frameNameLabel->setText(Frame::shapeLabel((Frame::Shape) shape));

	// Fill frame list & select the current one:
	if (frameList->listBox()) // A lot of frames are semi-transparent white, sowe can't see them otherwise:
		frameList->listBox()->setPaletteBackgroundColor(paletteBackgroundColor());
	int frameWidth = 55;
	int frameHeight = 40;
	frameList->setFixedSize(frameWidth + 40, frameHeight + 10);
	QPixmap transparent = QPixmap(frameWidth, frameHeight);
	transparent.fill(Qt::white);
	transparent.setMask(transparent.createHeuristicMask());
	frameList->insertItem(transparent);
	QPtrListIterator<Frame> it(Frame::list());
	Frame *frame;
	while ((frame = it.current()) != 0) {
		++it;
		frameList->insertItem(frame->pixmap(frameWidth, frameHeight, 0, 0, 0, 0));
		if (theme->frame(shape) == frame)
			frameList->setCurrentItem(frameList->count() - 1);
	}
	connect( frameList, SIGNAL(activated(int)), this, SLOT(frameChanged(int)) );

	// Set current paddings:
	int top;
	int bottom;
	int left;
	int right;
	theme->framePaddings((Frame::Shape) shape, &top, &left, &right, &bottom);
	paddingTop->setValue(top);
	paddingLeft->setValue(left);
	paddingRight->setValue(right);
	paddingBottom->setValue(bottom);

	bool hasFrame = (theme->frame(shape) != 0);
	enablePaddings(hasFrame);

	connect( paddingTop,    SIGNAL(valueChanged(int)), this, SIGNAL(somethingChanged()) );
	connect( paddingLeft,   SIGNAL(valueChanged(int)), this, SIGNAL(somethingChanged()) );
	connect( paddingRight,  SIGNAL(valueChanged(int)), this, SIGNAL(somethingChanged()) );
	connect( paddingBottom, SIGNAL(valueChanged(int)), this, SIGNAL(somethingChanged()) );
}

FrameEditor::~FrameEditor()
{
}

void FrameEditor::frameChanged(int index)
{
	enablePaddings(index > 0);

	Frame *frame = (index > 0 ? frameForIndex(index) : 0);

	if (index > 0) {
		int defaultPadding        = 5;  // Padding for normal frames
		int defaultOutsidePadding = 10; // Padding for sides of the frame that are outside the screen
		int defaultCoverPadding   = 3;  // Padding of the cover rect so for frames that need a padding on this shape
// 		if ( Frame::haveDefaultPaddings(m_shape) &&
// 		     (!frame->noDefaultPaddingForBlocks() || (m_shape != Frame::ProgressBar && m_shape != Frame::Cover && m_shape != Frame::Lyrics)) ) {
// 			paddingTop->setValue(defaultPadding - (m_shape == Frame::MiddleBar || m_shape == Frame::BelowCover || m_shape == Frame::BottomBar ? frame->borderTop() : 0));
// 			paddingLeft->setValue(defaultPadding);
// 			paddingRight->setValue(defaultPadding);
// 			paddingBottom->setValue(defaultPadding - (m_shape == Frame::PluginIconBar || m_shape == Frame::MiddleBar || m_shape == Frame::BelowCover ? frame->borderBottom() : 0));

			// The default padding:
			int paddingTop    = defaultPadding;
			int paddingLeft   = defaultPadding;
			int paddingRight  = defaultPadding;
			int paddingBottom = defaultPadding;
			// The cover has a special small padding for the bottom of it to be exactly at the middle of the bottom of the cover and the top of the cover reflexion:
			if (m_shape == Frame::Cover || m_shape == Frame::CoverOverlay) {
				paddingTop    = defaultCoverPadding;
				paddingLeft   = defaultCoverPadding;
				paddingRight  = defaultCoverPadding;
				paddingBottom = defaultCoverPadding;
			}
			// Those screen-wide bars have big left and right paddings for any round-corner to be invisible, and no top and bottom paddings, for each zone to stick to the other zone edge:
			if (m_shape == Frame::PluginIconBar || m_shape == Frame::MiddleBar || m_shape == Frame::BelowCover || m_shape == Frame::BottomBar) {
				paddingTop    = 0;
				paddingLeft   = defaultOutsidePadding;
				paddingRight  = defaultOutsidePadding;
				paddingBottom = 0;
			}
			// Some frames do only define a shadow or a lighting glow, they do not have padding:
			// The cover & text rect already contains the defaultCoverPadding padding, no we can't have another one:
			if (  (frame->noDefaultPaddingForBlocks() && (m_shape == Frame::ProgressBar || m_shape == Frame::Cover || m_shape == Frame::CoverOverlay || m_shape == Frame::Lyrics))
			    || m_shape == Frame::CoverAndTextInfos) {
				paddingTop    = 0;
				paddingLeft   = 0;
				paddingRight  = 0;
				paddingBottom = 0;
			}
			this->paddingTop->setValue(paddingTop);
			this->paddingLeft->setValue(paddingLeft);
			this->paddingRight->setValue(paddingRight);
			this->paddingBottom->setValue(paddingBottom);
// 		} else {
// 			paddingTop->setValue(0 - (m_shape == Frame::MiddleBar || m_shape == Frame::BelowCover || m_shape == Frame::BottomBar ? frame->borderTop() : 0));
// 			paddingLeft->setValue(0);
// 			paddingRight->setValue(0);
// 			paddingBottom->setValue(0 - (m_shape == Frame::PluginIconBar || m_shape == Frame::MiddleBar || m_shape == Frame::BelowCover ? frame->borderBottom() : 0));
// 		}
	} else {
		paddingTop->setValue(0);
		paddingLeft->setValue(0);
		paddingRight->setValue(0);
		paddingBottom->setValue(0);
	}

	emit somethingChanged();
}

void FrameEditor::enablePaddings(bool enable)
{
	paddingLabel->setEnabled(enable);
	paddingTopLabel->setEnabled(enable);
	paddingTop->setEnabled(enable);
	paddingLeftLabel->setEnabled(enable);
	paddingLeft->setEnabled(enable);
	paddingRightLabel->setEnabled(enable);
	paddingRight->setEnabled(enable);
	paddingBottomLabel->setEnabled(enable);
	paddingBottom->setEnabled(enable);
}

Frame *FrameEditor::frameForIndex(int index)
{
	QPtrListIterator<Frame> it(Frame::list());
	Frame *frame;
	int i = 0;
	while ((frame = it.current()) != 0) {
		++it;
		++i; // (i == 0) is "no frame"
		if (i == index)
			return frame;
	}
	return 0;
}

void FrameEditor::aboutToApplyChanges()
{
	if (paddingTop->hasFocus()) {
		paddingTop->clearFocus();
		paddingTop->setFocus();
	}
	if (paddingLeft->hasFocus()) {
		paddingLeft->clearFocus();
		paddingLeft->setFocus();
	}
	if (paddingRight->hasFocus()) {
		paddingRight->clearFocus();
		paddingRight->setFocus();
	}
	if (paddingBottom->hasFocus()) {
		paddingBottom->clearFocus();
		paddingBottom->setFocus();
	}
}

int FrameEditor::getPaddingTop()
{
	return paddingTop->value();
}

int FrameEditor::getPaddingLeft()
{
	return paddingLeft->value();
}

int FrameEditor::getPaddingRight()
{
	return paddingRight->value();
}

int FrameEditor::getPaddingBottom()
{
	return paddingBottom->value();
}

Frame *FrameEditor::getFrame()
{
	return frameForIndex(frameList->currentItem());
}

/** class ThemeExportDialog: */

ThemeExportDialog::ThemeExportDialog(Theme *theme, const QPixmap &preview, QWidget *parent)
 : KDialogBase(KDialogBase::Plain, i18n("Export Theme"), KDialogBase::Ok | KDialogBase::Cancel,
               KDialogBase::Ok, parent, /*name=*/"ThemeExportDialog", /*modal=*/true, /*separator=*/false),
   m_theme(theme)
{
	setPlainCaption(i18n("%1 - Kirocker Music Display").arg(i18n("Export Theme")));

//	QWidget *page = new QWidget(/*this*/plainPage());
	QVBoxLayout *topLayout = new QVBoxLayout(/*page*/plainPage(), /*margin=*/0, spacingHint());
	m_exporter = new ThemeExporterWidget(/*page*/plainPage());
	topLayout->addWidget(m_exporter);
	m_exporter->layout()->setMargin(0);

	m_exporter->themePreview->setPixmap(preview);
	m_exporter->exportThemeName->setText(i18n("Export <b>%1</b>").arg(m_theme->themeName()));
	m_exporter->authorName->setText(m_theme->authorName());
	m_exporter->authorEMail->setText(m_theme->authorEMail());
	m_exporter->authorURL->setText(m_theme->authorURL());
	m_exporter->copyrightInformation->setText(m_theme->copyrightInformation());
/*
	QValueList<QWidget*> widgets;
	widgets <<
		m_exporter->authorNameLabel << m_exporter->authorEMailLabel << m_exporter->authorURLLabel << m_exporter->templateButton;

	int maxWidth = 0;
	for (QValueList<QWidget*>::const_iterator it = widgets.begin(); it != widgets.end(); ++it)
		if ((*it)->sizeHint().width() > maxWidth)
			maxWidth = (*it)->sizeHint().width();

	for (QValueList<QWidget*>::const_iterator it = widgets.begin(); it != widgets.end(); ++it)
		(*it)->setFixedWidth(maxWidth);

	int spacerWidth = m_exporter->authorName->x() - m_exporter->gbAuthor->x();
	//m_exporter->titleSpacer->changeSize(spacerWidth, 10, QSizePolicy::Fixed, QSizePolicy::Fixed);
	std::cout << ">>>>>>>>" << spacerWidth << " = " << m_exporter->authorName->x() << " - " << m_exporter->gbAuthor->x() << std::endl;
	m_exporter->titleSpacer->setFixedWidth(spacerWidth);
*/
//	setMainWidget(page);

	plainPage()->setMinimumSize(m_exporter->sizeHint().width() * 2, m_exporter->sizeHint().height());

	connect( m_exporter->templateButton, SIGNAL(clicked()), this, SLOT(slotTemplate()) );
}

ThemeExportDialog::~ThemeExportDialog()
{
}

void ThemeExportDialog::showEvent(QShowEvent *event)
{
	KDialogBase::showEvent(event);

	QValueList<QWidget*> widgets;
	widgets <<
		m_exporter->authorNameLabel << m_exporter->authorEMailLabel << m_exporter->authorURLLabel << m_exporter->templateButton;

	int maxWidth = 0;
	for (QValueList<QWidget*>::const_iterator it = widgets.begin(); it != widgets.end(); ++it)
		if ((*it)->sizeHint().width() > maxWidth)
			maxWidth = (*it)->sizeHint().width();

	for (QValueList<QWidget*>::const_iterator it = widgets.begin(); it != widgets.end(); ++it)
		(*it)->setFixedWidth(maxWidth);

	//int spacerWidth = m_exporter->authorName->x() - m_exporter->gbAuthor->x();
	//int spacerWidth = m_exporter->authorName->mapTo(plainPage(), QPoint(0, 0)).x() - spacingHint();//m_exporter->gbAuthor->mapTo(plainPage(), QPoint(0, 0)).x();
	int spacerWidth = m_exporter->authorNameLabel->mapTo(plainPage(), QPoint(0, 0)).x() + maxWidth;//m_exporter->gbAuthor->mapTo(plainPage(), QPoint(0, 0)).x();
	//m_exporter->titleSpacer->changeSize(spacerWidth, 10, QSizePolicy::Fixed, QSizePolicy::Fixed);
	m_exporter->titleSpacer->setFixedWidth(spacerWidth);


	plainPage()->setMinimumSize(m_exporter->sizeHint().width() * 2, m_exporter->sizeHint().height());

	m_exporter->authorName->setFocus();
}

void ThemeExportDialog::slotTemplate()
{
	QString templateMessage =
		"The background image comes from **IMAGE WEB ADDRESS** under **LICENSE NAME**." "\n"
		"All credits go to the creator of the wallpaper, **ARTIST NAME**: **ARTIST WEB ADDRESS**";
	if (m_exporter->copyrightInformation->text().isEmpty())
		m_exporter->copyrightInformation->setText(templateMessage);
	else
		m_exporter->copyrightInformation->setText(m_exporter->copyrightInformation->text() + "\n" + templateMessage);

	// Select the first thing to replace:
	int para  = 0;
	int index = 0;
	m_exporter->copyrightInformation->find("**IMAGE WEB ADDRESS**", /*caseSensitive=*/true, /*wholeWordMatches=*/false, /*forward=*/true, &para, &index);

	m_exporter->copyrightInformation->setFocus();
}

void ThemeExportDialog::slotOk()
{
	m_theme->setAuthorName(m_exporter->authorName->text());
	m_theme->setAuthorEMail(m_exporter->authorEMail->text());
	m_theme->setAuthorURL(m_exporter->authorURL->text());
	m_theme->setCopyrightInformation(m_exporter->copyrightInformation->text());
	m_theme->save();

	KDialogBase::slotOk();
}








//!----------------------------------------------------------------------------------------------------------




/** class ThemeAdviseDialog: */

ThemeAdviseDialog::ThemeAdviseDialog(Theme *theme, QWidget *parent)
 : KDialogBase(KDialogBase::Plain, i18n("Export Theme Advises"), KDialogBase::Close,
               KDialogBase::Close, parent, /*name=*/"ThemeAdviseDialog", /*modal=*/false, /*separator=*/true),
   m_theme(theme)
{
	setPlainCaption(i18n("%1 - Kirocker Music Display").arg(i18n("Export Theme Advises")));
	setIcon(kapp->iconLoader()->loadIcon("kirocker", KIcon::Desktop, 16, KIcon::DefaultState));

	setWFlags(Qt::WStyle_StaysOnTop/* | Qt::WX11BypassWM*/);

	QVBoxLayout *topLayout = new QVBoxLayout(/*page*/plainPage(), /*margin=*/0, spacingHint());
	m_adviser = new ExportAdvisesWidget(/*page*/plainPage());
	topLayout->addWidget(m_adviser);
	m_adviser->layout()->setMargin(0);

	m_adviser->title->setText(m_adviser->title->text().arg(m_theme->themeName()));
	m_adviser->buttonCopy->setIconSet(SmallIconSet("editcopy"));
	m_adviser->textLabel3->setText(m_adviser->textLabel3->text().arg(m_theme->themeName()));

	QString description =
		i18n("This is a theme for the full-screen display of Kirocker Music Display, version 4 and later.") + "\n" +
		"\n" +
		i18n("Kirocker Music Display is both a panel applet and a full-screen display to see what's you're listening to in Amarok, and control the playback.") + "\n" +
		i18n("You can download Kirocker Music Display here:") + "\n" +
		"[url]http://www.kde-apps.org/content/show.php/Kirocker+Music+Display?content=52869[/url]" + "\n" +
		"\n" +
		i18n("To install the theme, just open the archive file (the file ending with \".kirocker-theme.tar.gz\").") + "\n" +
		i18n("Alternatively, you can launch the full-screen display of Kirocker Music Display, open the \"Choose Theme\" window, click \"Import...\" and browse to the archive to install.");

	if (!m_theme->copyrightInformation().isEmpty())
		description += "\n\n[b]Copyright information:[/b]\n" +
		               Tools::tagURLsBBCode(m_theme->copyrightInformation());

	m_adviser->descriptionExample->setText(description);

	plainPage()->setMinimumSize(m_adviser->sizeHint().width() * 3 / 2, m_adviser->sizeHint().height());

	connect( m_adviser->kdeLookOrgLink, SIGNAL(leftClickedURL()), this, SLOT(slotGoKdeLookOrg()) );
	connect( m_adviser->buttonCopy,     SIGNAL(clicked()),        this, SLOT(slotCopy())         );
}

ThemeAdviseDialog::~ThemeAdviseDialog()
{
}

void ThemeAdviseDialog::slotGoKdeLookOrg()
{
	QString url = "http://kde-look.org/content/add.php?new=Save&cname=" + m_theme->themeName() + "%20Kirocker%20Theme";
	kapp->invokeBrowser(url);
}

void ThemeAdviseDialog::slotCopy()
{
	m_adviser->descriptionExample->selectAll();
	m_adviser->descriptionExample->copy();
}

#include "themedialogs.moc"
