/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "thememanager.h"

#include "settings.h"
#include "themedialogs.h"
#include "tools.h"

#include <qpixmap.h>
#include <qimage.h>
#include <qdir.h>
#include <qstringlist.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kstandarddirs.h>
#include <ksimpleconfig.h>
#include <kimageeffect.h>
#include <ktar.h>
#include <kmessagebox.h>
#include <klocale.h>

#include <iostream>

/** Class Theme: */

Theme::Theme(const QString &location, const QString &folderName)
 : m_location(location), m_folderName(folderName), m_backgroundColorOpacity(80), m_backgroundPixmap(0), m_previewPixmap(0)
{
	if (!m_location.endsWith("/"))
		m_location += "/";

	load();
}

Theme::~Theme()
{
	setUsed(false);
}

// Theme information:
QString Theme::location()                    const { return m_location;   }
QString Theme::folderName()                  const { return m_folderName; }
QString Theme::themeName()                   const { return m_themeName;  }
// Screen:
QColor  Theme::backgroundColor()             const { if (m_backgroundColor.isValid())             return m_backgroundColor;             else return defaultBackgroundColor();             }
int     Theme::backgroundColorOpacity()      const { return m_backgroundColorOpacity; }
QColor  Theme::textColor()                   const { if (m_textColor.isValid())                   return m_textColor;                   else return defaultTextColor();                   }
QColor  Theme::nextPlayingColor()            const { if (m_nextPlayingColor.isValid())            return m_nextPlayingColor;            else return defaultNextPlayingColor();            }
QColor  Theme::dateHourColor()               const { if (m_dateHourColor.isValid())               return m_dateHourColor;               else return defaultDateHourColor();               }
// Progress bar:
QColor  Theme::progressBackgroundColor()     const { if (m_progressBackgroundColor.isValid())     return m_progressBackgroundColor;     else return defaultProgressBackgroundColor();     }
QColor  Theme::progressBarColor()            const { if (m_progressBarColor.isValid())            return m_progressBarColor;            else return defaultProgressBarColor();            }
QColor  Theme::progressBackgroundTextColor() const { if (m_progressBackgroundTextColor.isValid()) return m_progressBackgroundTextColor; else return defaultProgressBackgroundTextColor(); }
QColor  Theme::progressBarTextColor()        const { if (m_progressBarTextColor.isValid())        return m_progressBarTextColor;        else return defaultProgressBarTextColor();        }
// Shadows:
bool    Theme::textShadowEnabled()           const { return m_textShadowEnabled;        }
QColor  Theme::textShadowColor()             const { if (m_textShadowColor.isValid())             return m_textShadowColor;             else return defaultTextShadowColor();             }
bool    Theme::nextPlayingShadowEnabled()    const { return m_nextPlayingShadowEnabled; }
QColor  Theme::nextPlayingShadowColor()      const { if (m_nextPlayingShadowColor.isValid())      return m_nextPlayingShadowColor;      else return defaultNextPlayingShadowColor();      }
bool    Theme::dateHourShadowEnabled()       const { return m_dateHourShadowEnabled;    }
QColor  Theme::dateHourShadowColor()         const { if (m_dateHourShadowColor.isValid())         return m_dateHourShadowColor;         else return defaultDateHourShadowColor();         }
// Lyrics:
QColor  Theme::lyricsBackgroundColor()       const { if (m_lyricsBackgroundColor.isValid())       return m_lyricsBackgroundColor;       else return defaultLyricsBackgroundColor();       }
QColor  Theme::lyricsTextColor()             const { if (m_lyricsTextColor.isValid())             return m_lyricsTextColor;             else return defaultLyricsTextColor();             }
QColor  Theme::lyricsScrollBackgroundColor() const { if (m_lyricsScrollBackgroundColor.isValid()) return m_lyricsScrollBackgroundColor; else return defaultLyricsScrollBackgroundColor(); }
QColor  Theme::lyricsScrollButtonsColor()    const { if (m_lyricsScrollButtonsColor.isValid())    return m_lyricsScrollButtonsColor;    else return defaultLyricsScrollButtonsColor();    }
// Author:
QString Theme::authorName()                  const { return m_authorName;           }
QString Theme::authorEMail()                 const { return m_authorEMail;          }
QString Theme::authorURL()                   const { return m_authorURL;            }
QString Theme::copyrightInformation()        const { return m_copyrightInformation; }

// Pixmap:

QString Theme::normalBackgroundImagePath() const
{
	QStringList locations;
	locations << "background.png"      << "background.jpg"      << "background.jpeg"      << "background.gif";
	return imagePath(locations);
}

QString Theme::wideBackgroundImagePath() const
{
	QStringList locations;
	locations << "background-wide.png" << "background-wide.jpg" << "background-wide.jpeg" << "background-wide.gif";
	return imagePath(locations);
}

bool Theme::hasNormalBackgroundImage() const
{
	return !normalBackgroundImagePath().isEmpty();
}

bool Theme::hasWideBackgroundImage() const
{
	return !wideBackgroundImagePath().isEmpty();
}

QString Theme::normalPreviewPath() const
{
	QStringList locations;
	locations << "background.preview.png"      << "background.preview.jpg"      << "background.preview.jpeg"      << "background.preview.gif";
	return imagePath(locations);
}

QString Theme::widePreviewPath() const
{
	QStringList locations;
	locations << "background-wide.preview.png" << "background-wide.preview.jpg" << "background-wide.preview.jpeg" << "background-wide.preview.gif";
	return imagePath(locations);
}

QString Theme::imagePath(const QStringList &locations) const
{
	for (uint i = 0; i < locations.count(); i++)
		if (QFile::exists(m_location + locations[i]))
			return m_location + locations[i];

	return QString::null;
}

const QPixmap &Theme::backgroundPixmap(int width, int height)
{
////std::cout << "backgroundPixmap(" << width << ", " << height << ", " << themeName() << "), opacity: " << backgroundColorOpacity() << std::endl;
	if (m_backgroundPixmap) {
		if (m_backgroundPixmap->width() == width && m_backgroundPixmap->height() == height)
			return *m_backgroundPixmap;

		delete m_backgroundPixmap;
		m_backgroundPixmap = 0;
	}

	QStringList locations;
	if (Tools::isWideScreen(width, height))
		locations << "background-wide.png" << "background-wide.jpg" << "background-wide.jpeg" << "background-wide.gif"
		          << "background.png"      << "background.jpg"      << "background.jpeg"      << "background.gif";
	else
		locations << "background.png"      << "background.jpg"      << "background.jpeg"      << "background.gif"
		          << "background-wide.png" << "background-wide.jpg" << "background-wide.jpeg" << "background-wide.gif";
	m_backgroundPixmap = computePixmap(width, height, locations);

	return *m_backgroundPixmap;
}

QPixmap *Theme::computePixmap(int width, int height, const QStringList &locations) const
{
////std::cout << "computePixmap(" << width << ", " << height << ", " << themeName() << "), opacity: " << backgroundColorOpacity() << std::endl;
	QPixmap *pixmap = 0;

	if (backgroundColorOpacity() >= 100) {
		pixmap = new QPixmap(width, height);
		pixmap->fill(backgroundColor());
	} else {
		// Load the first adequate image:
		QImage image;
		uint i = 0;
		while (image.isNull() && i < locations.count()) {
			image.load(m_location + locations[i]);
			i++;
		}
		// If no image has been found, do as if background color were 100% opaque:
		if (image.isNull()) {
			pixmap = new QPixmap(width, height);
			pixmap->fill(backgroundColor());
		}
		// If an image has been found, blend:
		else {
			image = image.smoothScale(width, height);
			if (backgroundColorOpacity() > 0)
				KImageEffect::blend(backgroundColor(), image, backgroundColorOpacity() / 100.0);
			pixmap = new QPixmap();
			pixmap->convertFromImage(image);
		}
	}

	return pixmap;
}

const QPixmap &Theme::previewPixmap(int width, int height)
{
	if (m_previewPixmap) {
		if (m_previewPixmap->width() == width && m_previewPixmap->height() == height)
			return *m_previewPixmap;

		delete m_previewPixmap;
		m_previewPixmap = 0;
	}

	double size = 75.0;
	int widthPreview  = (int) (width  >= height ? size : size * width  / height);
	int heightPreview = (int) (height >= width  ? size : size * height / width);

	QStringList locations;
	if (Tools::isWideScreen(width, height))
		locations << "background-wide.preview.png" << "background-wide.preview.jpg" << "background-wide.preview.jpeg" << "background-wide.preview.gif"
		          << "background.preview.png"      << "background.preview.jpg"      << "background.preview.jpeg"      << "background.preview.gif";
	else
		locations << "background.preview.png"      << "background.preview.jpg"      << "background.preview.jpeg"      << "background.preview.gif"
		          << "background-wide.preview.png" << "background-wide.preview.jpg" << "background-wide.preview.jpeg" << "background-wide.preview.gif";
	m_previewPixmap = computePixmap(widthPreview, heightPreview, locations);

	return *m_previewPixmap;
}

void Theme::invalidPreview()
{
	delete m_previewPixmap;
	m_previewPixmap = 0;
}

void Theme::setUsed(bool used) // TODO Rename reset()
{
	if (!used && m_backgroundPixmap || m_previewPixmap) {
		delete m_backgroundPixmap;
		m_backgroundPixmap = 0;
		delete m_previewPixmap;
		m_previewPixmap = 0;
	}
}

void Theme::setLocationAndFolder(const QString &location, const QString &folderName)
{
	if (m_location == location && m_folderName == folderName)
		return;

	m_location   = location;
	m_folderName = folderName;

	if (!m_location.endsWith("/"))
		m_location += "/";

	if (this == Theme::current()) {
		Settings::setCurrentTheme(folderName);
		Settings::writeConfig();
	}
}

void Theme::copyTo(Theme *theme) const
{
	// Theme information:
	theme->setLocationAndFolder(m_location, m_folderName); // In order to save the folder change if it is the current theme!
	//theme->m_location   = m_location;
	//theme->m_folderName = m_folderName;
	theme->m_themeName  = m_themeName;
	// Screen:
	theme->m_backgroundColor             = m_backgroundColor;
	theme->m_backgroundColorOpacity      = m_backgroundColorOpacity;
	theme->m_textColor                   = m_textColor;
	theme->m_nextPlayingColor            = m_nextPlayingColor;
	theme->m_dateHourColor               = m_dateHourColor;
	// Progress bar:
	theme->m_progressBackgroundColor     = m_progressBackgroundColor;
	theme->m_progressBarColor            = m_progressBarColor;
	theme->m_progressBackgroundTextColor = m_progressBackgroundTextColor;
	theme->m_progressBarTextColor        = m_progressBarTextColor;
	// Shadows:
	theme->m_textShadowEnabled           = m_textShadowEnabled;
	theme->m_textShadowColor             = m_textShadowColor;
	theme->m_nextPlayingShadowEnabled    = m_nextPlayingShadowEnabled;
	theme->m_nextPlayingShadowColor      = m_nextPlayingShadowColor;
	theme->m_dateHourShadowEnabled       = m_dateHourShadowEnabled;
	theme->m_dateHourShadowColor         = m_dateHourShadowColor;
	// Lyrics:
	theme->m_lyricsBackgroundColor       = m_lyricsBackgroundColor;
	theme->m_lyricsTextColor             = m_lyricsTextColor;
	theme->m_lyricsScrollBackgroundColor = m_lyricsScrollBackgroundColor;
	theme->m_lyricsScrollButtonsColor    = m_lyricsScrollButtonsColor;
	// Author:
	theme->m_authorName           = m_authorName;
	theme->m_authorEMail          = m_authorEMail;
	theme->m_authorURL            = m_authorURL;
	theme->m_copyrightInformation = m_copyrightInformation;
//	// Pixmaps:
//	theme->m_backgroundPixmap = m_backgroundPixmap;
//	theme->m_previewPixmap    = m_previewPixmap;

	for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
		theme->m_frames[i]              = m_frames[i];
		theme->m_framesPaddingTop[i]    = m_framesPaddingTop[i];
		theme->m_framesPaddingLeft[i]   = m_framesPaddingLeft[i];
		theme->m_framesPaddingRight[i]  = m_framesPaddingRight[i];
		theme->m_framesPaddingBottom[i] = m_framesPaddingBottom[i];
	}
}

// GET:
// Screen:
QColor Theme::backgroundColorSetting()             const { return m_backgroundColor;             }
QColor Theme::textColorSetting()                   const { return m_textColor;                   }
QColor Theme::nextPlayingColorSetting()            const { return m_nextPlayingColor;            }
QColor Theme::dateHourColorSetting()               const { return m_dateHourColor;               }
// Progress bar:
QColor Theme::progressBackgroundColorSetting()     const { return m_progressBackgroundColor;     }
QColor Theme::progressBarColorSetting()            const { return m_progressBarColor;            }
QColor Theme::progressBackgroundTextColorSetting() const { return m_progressBackgroundTextColor; }
QColor Theme::progressBarTextColorSetting()        const { return m_progressBarTextColor;        }
// Shadows:
QColor Theme::textShadowColorSetting()             const { return m_textShadowColor;             }
QColor Theme::nextPlayingShadowColorSetting()      const { return m_nextPlayingShadowColor;      }
QColor Theme::dateHourShadowColorSetting()         const { return m_dateHourShadowColor;         }
// Lyrics:
QColor Theme::lyricsBackgroundColorSetting()       const { return m_lyricsBackgroundColor;       }
QColor Theme::lyricsTextColorSetting()             const { return m_lyricsTextColor;             }
QColor Theme::lyricsScrollBackgroundColorSetting() const { return m_lyricsScrollBackgroundColor; }
QColor Theme::lyricsScrollButtonsColorSetting()    const { return m_lyricsScrollButtonsColor;    }

// DEFAULT COLORS:
// Screen:
QColor Theme::defaultBackgroundColor()             const { if (m_textColor.isValid()       &&  Tools::tooDark(m_textColor))       return Qt::white; else return Qt::black;   }
QColor Theme::defaultTextColor()                   const { if (m_backgroundColor.isValid() && !Tools::tooDark(m_backgroundColor)) return Qt::black; else return Qt::white;   }
QColor Theme::defaultNextPlayingColor()            const { return Tools::mixColors(textColor(), backgroundColor(), 0.75); }
QColor Theme::defaultDateHourColor()               const { return textColor(); }
// Progress bar:
QColor Theme::defaultProgressBackgroundColor()     const { return KGlobalSettings::baseColor();            }
QColor Theme::defaultProgressBarColor()            const { return KGlobalSettings::highlightColor();       }
QColor Theme::defaultProgressBackgroundTextColor() const { return KGlobalSettings::textColor();            }
QColor Theme::defaultProgressBarTextColor()        const { return KGlobalSettings::highlightedTextColor(); }
// Shadows:
QColor Theme::defaultTextShadowColor()             const { return QColor(qGray(textColor().rgb())        < 128 ? Qt::white : Qt::black); }
QColor Theme::defaultNextPlayingShadowColor()      const { return QColor(qGray(nextPlayingColor().rgb()) < 128 ? Qt::white : Qt::black); }
QColor Theme::defaultDateHourShadowColor()         const { return QColor(qGray(dateHourColor().rgb())    < 128 ? Qt::white : Qt::black); }
// Lyrics:
QColor Theme::defaultLyricsBackgroundColor()       const { if (m_lyricsTextColor.isValid())       return (Tools::tooDark(m_lyricsTextColor)       ? Qt::white : Qt::black); else return backgroundColor(); }
QColor Theme::defaultLyricsTextColor()             const { if (m_lyricsBackgroundColor.isValid()) return (Tools::tooDark(m_lyricsBackgroundColor) ? Qt::white : Qt::black); else return textColor();       }
QColor Theme::defaultLyricsScrollBackgroundColor() const { return lyricsBackgroundColor();                                                            }
QColor Theme::defaultLyricsScrollButtonsColor()    const { return Tools::mixColors(lyricsScrollBackgroundColor(), lyricsTextColor(), 0.75);    }

// SET:
// Theme information:
void Theme::setThemeName(const QString &name)                   { m_themeName                   = name;        }
// Screen:
void Theme::setBackgroundColor(const QColor &color)             { m_backgroundColor             = color;       }
void Theme::setBackgroundColorOpacity(int opacity)              { m_backgroundColorOpacity      = opacity;     }
void Theme::setTextColor(const QColor &color)                   { m_textColor                   = color;       }
void Theme::setNextPlayingColor(const QColor &color)            { m_nextPlayingColor            = color;       }
void Theme::setDateHourColor(const QColor &color)               { m_dateHourColor               = color;       }
// Progress bar:
void Theme::setProgressBackgroundColor(const QColor &color)     { m_progressBackgroundColor     = color;       }
void Theme::setProgressBarColor(const QColor &color)            { m_progressBarColor            = color;       }
void Theme::setProgressBackgroundTextColor(const QColor &color) { m_progressBackgroundTextColor = color;       }
void Theme::setProgressBarTextColor(const QColor &color)        { m_progressBarTextColor        = color;       }
// Shadows:
void Theme::setTextShadowEnabled(bool enable)                   { m_textShadowEnabled           = enable;      }
void Theme::setTextShadowColor(const QColor &color)             { m_textShadowColor             = color;       }
void Theme::setNextPlayingShadowEnabled(bool enable)            { m_nextPlayingShadowEnabled    = enable;      }
void Theme::setNextPlayingShadowColor(const QColor &color)      { m_nextPlayingShadowColor      = color;       }
void Theme::setDateHourShadowEnabled(bool enable)               { m_dateHourShadowEnabled       = enable;      }
void Theme::setDateHourShadowColor(const QColor &color)         { m_dateHourShadowColor         = color;       }
// Lyrics:
void Theme::setLyricsBackgroundColor(const QColor &color)       { m_lyricsBackgroundColor       = color;       }
void Theme::setLyricsTextColor(const QColor &color)             { m_lyricsTextColor             = color;       }
void Theme::setLyricsScrollBackgroundColor(const QColor &color) { m_lyricsScrollBackgroundColor = color;       }
void Theme::setLyricsScrollButtonsColor(const QColor &color)    { m_lyricsScrollButtonsColor    = color;       }
// Author:
void Theme::setAuthorName(const QString &name)                  { m_authorName                  = name;        }
void Theme::setAuthorEMail(const QString &eMail)                { m_authorEMail                 = eMail;       }
void Theme::setAuthorURL(const QString &url)                    { m_authorURL                   = url;         }
void Theme::setCopyrightInformation(const QString &information) { m_copyrightInformation        = information; }

void Theme::load()
{
	KSimpleConfig config(m_location + "kirocker-theme.config", /*readOnly=*/true);

	QColor *defaultColor = new QColor();

	config.setGroup("Theme");
	m_themeName = config.readEntry("Name", "");

	config.setGroup("Colors");
	// Screen:
	m_backgroundColor             = config.readColorEntry("Background",               defaultColor);
	m_backgroundColorOpacity      = config.readNumEntry(  "BackgroundOpacity",        80);
	m_textColor                   = config.readColorEntry("Text",                     defaultColor);
	m_nextPlayingColor            = config.readColorEntry("NextPlaying",              defaultColor);
	m_dateHourColor               = config.readColorEntry("DateHour",                 defaultColor);
	// Progress bar:
	m_progressBackgroundColor     = config.readColorEntry("ProgressBackground",       defaultColor);
	m_progressBarColor            = config.readColorEntry("ProgressBar",              defaultColor);
	m_progressBackgroundTextColor = config.readColorEntry("ProgressBackgroundText",   defaultColor);
	m_progressBarTextColor        = config.readColorEntry("ProgressBarText",          defaultColor);
	// Shadows:
	m_textShadowEnabled           = config.readBoolEntry( "TextShadowEnabled",        false);
	m_textShadowColor             = config.readColorEntry("TextShadow",               defaultColor);
	m_nextPlayingShadowEnabled    = config.readBoolEntry( "NextPlayingShadowEnabled", false);
	m_nextPlayingShadowColor      = config.readColorEntry("NextPlayingShadow",        defaultColor);
	m_dateHourShadowEnabled       = config.readBoolEntry( "DateHourShadowEnabled",    false);
	m_dateHourShadowColor         = config.readColorEntry("DateHourShadow",           defaultColor);
	// Lyrics:
	m_lyricsBackgroundColor       = config.readColorEntry("LyricsBackground",         defaultColor);
	m_lyricsTextColor             = config.readColorEntry("LyricsText",               defaultColor);
	m_lyricsScrollBackgroundColor = config.readColorEntry("LyricsScrollBackground",   defaultColor);
	m_lyricsScrollButtonsColor    = config.readColorEntry("LyricsScrollButtons",      defaultColor);

	config.setGroup("Author");
	m_authorName           = config.readEntry("Name",                 "");
	m_authorEMail          = config.readEntry("EMail",                "");
	m_authorURL            = config.readEntry("URL",                  "");
	m_copyrightInformation = config.readEntry("CopyrightInformation", "");

	// Frames:
	for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
		QString frameGroup = Frame::shapeName((Frame::Shape) i) + "Frame";
		config.setGroup(frameGroup);
		m_frames[i] = Frame::forName(config.readEntry("Name", ""));
		m_framesPaddingTop[i]    = config.readNumEntry("PaddingTop",    0);
		m_framesPaddingLeft[i]   = config.readNumEntry("PaddingLeft",   0);
		m_framesPaddingRight[i]  = config.readNumEntry("PaddingRight",  0);
		m_framesPaddingBottom[i] = config.readNumEntry("PaddingBottom", 0);
	}

	delete defaultColor;
	defaultColor = 0;
}

void Theme::save()
{
	KSimpleConfig config(m_location + "kirocker-theme.config", /*readOnly=*/false);

	config.setGroup("Theme");
	config.writeEntry("Name",                     m_themeName);

	config.setGroup("Colors");
	// Screen:
	config.writeEntry("Background",               m_backgroundColor);
	config.writeEntry("BackgroundOpacity",        m_backgroundColorOpacity);
	config.writeEntry("Text",                     m_textColor);
	config.writeEntry("NextPlaying",              m_nextPlayingColor);
	config.writeEntry("DateHour",                 m_dateHourColor);
	// Progress bar:
	config.writeEntry("ProgressBackground",       m_progressBackgroundColor);
	config.writeEntry("ProgressBar",              m_progressBarColor);
	config.writeEntry("ProgressBackgroundText",   m_progressBackgroundTextColor);
	config.writeEntry("ProgressBarText",          m_progressBarTextColor);
	// Shadows:
	config.writeEntry("TextShadowEnabled",        m_textShadowEnabled);
	config.writeEntry("TextShadow",               m_textShadowColor);
	config.writeEntry("NextPlayingShadowEnabled", m_nextPlayingShadowEnabled);
	config.writeEntry("NextPlayingShadow",        m_nextPlayingShadowColor);
	config.writeEntry("DateHourShadowEnabled",    m_dateHourShadowEnabled);
	config.writeEntry("DateHourShadow",           m_dateHourShadowColor);
	// Lyrics:
	config.writeEntry("LyricsBackground",         m_lyricsBackgroundColor);
	config.writeEntry("LyricsText",               m_lyricsTextColor);
	config.writeEntry("LyricsScrollBackground",   m_lyricsScrollBackgroundColor);
	config.writeEntry("LyricsScrollButtons",      m_lyricsScrollButtonsColor);

	config.setGroup("Author");
	config.writeEntry("Name",                     m_authorName);
	config.writeEntry("EMail",                    m_authorEMail);
	config.writeEntry("URL",                      m_authorURL);
	config.writeEntry("CopyrightInformation",     m_copyrightInformation);

	// Frames:
	for (int i = 0; i < Frame::SHAPE_COUNT; i++) {
		QString frameGroup = Frame::shapeName((Frame::Shape) i) + "Frame";
		config.setGroup(frameGroup);
		config.writeEntry("Name",         (m_frames[i] ? m_frames[i]->folderName() : ""));
		config.writeEntry("PaddingTop",    m_framesPaddingTop[i]);
		config.writeEntry("PaddingLeft",   m_framesPaddingLeft[i]);
		config.writeEntry("PaddingRight",  m_framesPaddingRight[i]);
		config.writeEntry("PaddingBottom", m_framesPaddingBottom[i]);
	}
	setUsed(false);
}

bool Theme::isUserTheme() const
{
	// FIXME: It's better to see if the location folder can be renamed and/or files under it can be modified!
	return m_location.startsWith(QDir::home().path());
}

bool Theme::hasFrame(Frame::Shape shape)
{
	return (m_frames[shape] != 0);
}

Frame *Theme::frame(Frame::Shape shape)
{
	return m_frames[shape];
}

void Theme::framePaddings(Frame::Shape shape, int *paddingTop, int *paddingLeft, int *paddingRight, int *paddingBottom)
{
	*paddingTop    = m_framesPaddingTop[shape];
	*paddingLeft   = m_framesPaddingLeft[shape];
	*paddingRight  = m_framesPaddingRight[shape];
	*paddingBottom = m_framesPaddingBottom[shape];
}

void Theme::setFramePaddings(Frame::Shape shape, int paddingTop, int paddingLeft, int paddingRight, int paddingBottom)
{
	m_framesPaddingTop[shape]    = paddingTop;
	m_framesPaddingLeft[shape]   = paddingLeft;
	m_framesPaddingRight[shape]  = paddingRight;
	m_framesPaddingBottom[shape] = paddingBottom;
}

void Theme::setFrame(Frame::Shape shape, Frame *frame)
{
	m_frames[shape] = frame;
}

QRect Theme::realFrameRect(Frame::Shape shape, const QRect &rect)
{
	Frame *frame = m_frames[shape];
	if (frame)
		return QRect(rect.x() - m_framesPaddingLeft[shape],
		             rect.y() - m_framesPaddingTop[shape],
		             rect.width() + m_framesPaddingLeft[shape] + m_framesPaddingRight[shape],
		             rect.height() + m_framesPaddingTop[shape] + m_framesPaddingBottom[shape]);
	else
		return QRect();
}

Theme *Theme::current()
{
	return ThemeManager::instance()->currentTheme();
}

/*bool operator<(const Theme &theme1, const Theme &theme2)
{
	return theme1.themeName().compare(theme2.themeName());
}*/

/** Class ThemeList: */

ThemeList::ThemeList()
 : QPtrList<Theme>()
{
}

int ThemeList::compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2)
{
	Theme *theme1 = (Theme *) item1;
	Theme *theme2 = (Theme *) item2;
	return theme1->themeName().lower().compare(theme2->themeName().lower());
}


/** Class ThemeManager: */

ThemeManager *ThemeManager::s_instance = 0;

ThemeManager::ThemeManager()
 : QObject(), m_currentTheme(0)
{
}

ThemeManager::~ThemeManager()
{
}

void ThemeManager::loadThemeList()
{
	QString currentThemeFolderName = Settings::currentTheme();

	// Load Theme List:
	QStringList directories = KGlobal::dirs()->resourceDirs("data"); // eg. { "/home/seb/.kde/share/apps/", "/usr/share/apps/" }
	// For each data folder:
	for (QStringList::Iterator it = directories.begin(); it != directories.end(); ++it) {
		// For each theme folder in this data folder:
		QDir dir(*it + "kirocker/themes/", /*nameFilder=*/"", /*sortSpec=*/QDir::Name | QDir::IgnoreCase, /*filterSpec=*/QDir::Dirs | QDir::NoSymLinks);
		QStringList files = dir.entryList();
		for (QStringList::Iterator it2 = files.begin(); it2 != files.end(); ++it2)
			if (*it2 != "." && *it2 != ".." && forName(*it2) == 0) { // Don't reload already-loaded themes
				Theme *theme = new Theme(*it + "kirocker/themes/" + *it2, *it2);
				m_themes.inSort(theme);
				if (*it2 == currentThemeFolderName)
					m_currentTheme = theme;
			}
	}

	// Load the Current Theme (from config file, default one, or first in the list):
	if (!m_currentTheme && m_themes.count() > 0)
		m_currentTheme = m_themes.at(0); // TODO

	// Load a Default Theme if none in the list:
	if (!m_currentTheme)
		m_currentTheme = new Theme("", "");
}

void ThemeManager::addTheme(Theme *theme)
{
	m_themes.inSort(theme);
}

void ThemeManager::removeTheme(Theme *theme)
{
	// Free memory:
	theme->setUsed(false);

	// If the current theme was the one we remove, select the following or previous one:
	if (m_currentTheme == theme) {
		int index = m_themes.find(theme);
		if (index != -1 && index == ((int) m_themes.count()) - 1 && index > 0) // Next theme not available
			setTheme(m_themes.at(index - 1)); // Set to the previous theme
		else if (index != -1 && index < ((int) m_themes.count()) - 1)
			setTheme(m_themes.at(index + 1)); // Set to the next theme
		else
			setTheme(0);
		Settings::setCurrentTheme(theme->folderName());
		Settings::writeConfig();
	}

	// Remove the theme:
	m_themes.remove(theme);
	delete theme;
}

void ThemeManager::reSort()
{
	m_themes.sort();
}

Theme *ThemeManager::forName(const QString name)
{
	QPtrListIterator<Theme> it(m_themes);
	Theme *theme;
	while ((theme = it.current()) != 0) {
		++it;
		if (theme->folderName() == name)
			return theme;
	}
	return 0;
}

Theme *ThemeManager::currentTheme()
{
	if (!m_currentTheme)
		loadThemeList();

	return m_currentTheme;
}

ThemeManager *ThemeManager::instance()
{
	if (!s_instance)
		s_instance = new ThemeManager();

	return s_instance;
}

const ThemeList &ThemeManager::themes() const
{
	return m_themes;
}

void ThemeManager::setTheme(Theme *theme)
{
	m_currentTheme = theme;
	emit themeChanged();
	Settings::setCurrentTheme(theme->folderName());
	Settings::writeConfig();
}

Theme *ThemeManager::import(const QString &path)
{
	// Open the archive file:
	KTar tar(path, "application/x-gzip");
	if (!tar.open(IO_ReadOnly))
		return badArchiveFormat();

	// Check existence of ArchiveName/themes:
	QStringList entries = tar.directory()->entries();
	if (entries.count() != 1 || !tar.directory()->entry(entries[0])->isDirectory())
		return badArchiveFormat();
	const KArchiveDirectory *rootFolder = ((const KArchiveDirectory *) tar.directory()->entry(entries[0])); // rootFolder = ArchiveName/
	if (!rootFolder->isDirectory())
		return badArchiveFormat();
//	if (!rootFolder->entries().contains("themes")) // Checking ArchiveName/themes/
//		return badArchiveFormat();                 // NO: because an archive could contains only additionnal frames
	if (rootFolder->entries().contains("themes") && !rootFolder->entry("themes")->isDirectory()) // Checking ArchiveName/themes/
		return badArchiveFormat();

	// Unpack each frames (before themes because themes need the frames), load them and add them to the frame list:
	if (rootFolder->entries().contains("frames") && rootFolder->entry("frames")->isDirectory()) {// Checking ArchiveName/frames/
		const KArchiveDirectory *framesFolder = ((const KArchiveDirectory *) rootFolder->entry("frames")); // rootFolder = ArchiveName/
		QStringList frames = framesFolder->entries();
		for (QStringList::Iterator it = frames.begin(); it != frames.end(); ++it) {
			QString frameName = *it;
			if (Frame::forName(frameName) == 0) // If it isn't installed, import it:
				((const KArchiveDirectory *) framesFolder->entry(frameName))->copyTo(KGlobal::dirs()->saveLocation("data", "kirocker/frames/" + frameName + "/"));
		}
	}

	// If fine, list all ThemeFolderNames
	Theme *firstImportedTheme = 0;
	const KArchiveDirectory *themesFolder = ((const KArchiveDirectory *) rootFolder->entry("themes"));
	QStringList themes = themesFolder->entries();
	for (QStringList::Iterator it = themes.begin(); it != themes.end(); ++it) {
		// Unpack the theme folder:
		QString sourceFolderName = *it;
		QString destinationPath = KGlobal::dirs()->saveLocation("data", "kirocker/themes/");
		// If the folder theme already exists, ask to overwrite or not!
		QString destinationFolderName = sourceFolderName;
		bool isReplacing = false;
		if (QFile::exists(destinationPath + sourceFolderName)) {
			int result = KMessageBox::questionYesNo(
				0,
				"<qt>" + i18n("The theme <b>%1</b> already exists.").arg(sourceFolderName) + "\n" +
				         i18n("Do you want to overwrite it (eg. if it is a newer version)?"),
				i18n("Overwrite Theme?"),
				KGuiItem(i18n("&Overwrite"), "filesave"),
				KGuiItem(i18n("I&nstall Aside"))
			);
			if (result == KMessageBox::Yes)
				isReplacing = true;
			else // Do not overwrite
				destinationFolderName = Tools::fileNameForNewFile(destinationFolderName, destinationPath, ' ');
		}
		((const KArchiveDirectory *) themesFolder->entry(sourceFolderName))->copyTo(destinationPath + destinationFolderName + "/");
		// Load the theme and add it to the theme list:
		if (!isReplacing) {
			Theme *theme = new Theme(destinationPath + destinationFolderName, destinationFolderName);
			ThemeManager::instance()->addTheme(theme);
			if (firstImportedTheme == 0)
				firstImportedTheme = theme;
		} else if (firstImportedTheme == 0)
			firstImportedTheme = ThemeManager::instance()->forName(destinationFolderName);
	}

	return firstImportedTheme;
}

Theme *ThemeManager::badArchiveFormat()
{
	KMessageBox::error(
		0,
		i18n("The theme cannot be imported because the archive is malformed.") + "\n" +
		i18n("Try to download it again or contact its author."),
		i18n("%1 - %2").arg("Keyboard Shortcuts", i18n("Kirocker Music Display")),
		/*KMessageBox::Notify | */KMessageBox::PlainCaption
	);

	return 0;
}

void ThemeManager::chooseTheme(int screenWidth, int screenHeight, QWidget *parent)
{
	static ThemeChooserDialog *dialog = 0;
	static int lastScreenWidth  = 0;
	static int lastScreenHeight = 0;

	if (dialog == 0 || lastScreenWidth != screenWidth || lastScreenHeight != screenHeight) {
		if (dialog != 0) {
			delete dialog;
			dialog = 0;
		}

		dialog = new ThemeChooserDialog(screenWidth, screenHeight, parent);
		lastScreenWidth  = screenWidth;
		lastScreenHeight = screenHeight;
	}

	dialog->exec();

	// FIXME This recenters the dialog, and select the current theme:
	delete dialog;
	dialog = 0;
}

#include "thememanager.moc"
