/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef THEMEMANAGER_H
#define THEMEMANAGER_H

#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qcolor.h>
#include <qptrlist.h>

#include "framemanager.h"

class QPixmap;
class QWidget;

/**
 * @author Sébastien Laoût <slaout@linux62.org>
 */
class Theme
{
  public:
	// Constructor:
	Theme(const QString &location, const QString &folderName);
	~Theme();

	// GETTERS:
	// Theme information:
	QString location() const;
	QString folderName() const;
	QString themeName() const;
	// Screen:
	QColor  backgroundColor() const;
	int     backgroundColorOpacity() const;
	QColor  textColor() const;
	QColor  nextPlayingColor() const;
	QColor  dateHourColor() const;
	// Progress bar:
	QColor  progressBackgroundColor() const;
	QColor  progressBarColor() const;
	QColor  progressBackgroundTextColor() const;
	QColor  progressBarTextColor() const;
	// Shadows:
	bool    textShadowEnabled() const;
	QColor  textShadowColor() const;
	bool    nextPlayingShadowEnabled() const;
	QColor  nextPlayingShadowColor() const;
	bool    dateHourShadowEnabled() const;
	QColor  dateHourShadowColor() const;
	// Lyrics:
	QColor  lyricsBackgroundColor() const;
	QColor  lyricsTextColor() const;
	QColor  lyricsScrollBackgroundColor() const;
	QColor  lyricsScrollButtonsColor() const;
	// Author:
	QString authorName() const;
	QString authorEMail() const;
	QString authorURL() const;
	QString copyrightInformation() const;

	// Pixmap:
	const QPixmap &backgroundPixmap(int width, int height);
	const QPixmap &previewPixmap(int width, int height);
	void invalidPreview();
	void setUsed(bool used);
	QString normalBackgroundImagePath() const;
	QString wideBackgroundImagePath() const;
	bool hasNormalBackgroundImage() const;
	bool hasWideBackgroundImage() const;
	QString normalPreviewPath() const;
	QString widePreviewPath() const;
	QString imagePath(const QStringList &locations) const;

	// For the editor to store a temporar version:
	void copyTo(Theme *theme) const;

	bool isUserTheme() const;

	void setLocationAndFolder(const QString &location, const QString &folderName);

	// Help:
	static Theme *current();

	// GET:
	// Screen:
	QColor  backgroundColorSetting() const;
	QColor  textColorSetting() const;
	QColor  nextPlayingColorSetting() const;
	QColor  dateHourColorSetting() const;
	// Progress bar:
	QColor  progressBackgroundColorSetting() const;
	QColor  progressBarColorSetting() const;
	QColor  progressBackgroundTextColorSetting() const;
	QColor  progressBarTextColorSetting() const;
	// Shadows:
	QColor  textShadowColorSetting() const;
	QColor  nextPlayingShadowColorSetting() const;
	QColor  dateHourShadowColorSetting() const;
	// Lyrics:
	QColor  lyricsBackgroundColorSetting() const;
	QColor  lyricsTextColorSetting() const;
	QColor  lyricsScrollBackgroundColorSetting() const;
	QColor  lyricsScrollButtonsColorSetting() const;

	// DEFAULT COLORS:
	// Screen:
	QColor defaultBackgroundColor() const;
	QColor defaultTextColor() const;
	QColor defaultNextPlayingColor() const;
	QColor defaultDateHourColor() const;
	// Progress bar:
	QColor defaultProgressBackgroundColor() const;
	QColor defaultProgressBarColor() const;
	QColor defaultProgressBackgroundTextColor() const;
	QColor defaultProgressBarTextColor() const;
	// Shadows:
	QColor  defaultTextShadowColor() const;
	QColor  defaultNextPlayingShadowColor() const;
	QColor  defaultDateHourShadowColor() const;
	// Lyrics:
	QColor  defaultLyricsBackgroundColor() const;
	QColor  defaultLyricsTextColor() const;
	QColor  defaultLyricsScrollBackgroundColor() const;
	QColor  defaultLyricsScrollButtonsColor() const;

	// SET:
	// Theme information:
	void setThemeName(const QString &name);
	// Screen:
	void setBackgroundColor(const QColor &color);
	void setBackgroundColorOpacity(int opacity);
	void setTextColor(const QColor &color);
	void setNextPlayingColor(const QColor &color);
	void setDateHourColor(const QColor &color);
	// Progress bar:
	void setProgressBackgroundColor(const QColor &color);
	void setProgressBarColor(const QColor &color);
	void setProgressBackgroundTextColor(const QColor &color);
	void setProgressBarTextColor(const QColor &color);
	// Shadows:
	void setTextShadowEnabled(bool enable);
	void setTextShadowColor(const QColor &color);
	void setNextPlayingShadowEnabled(bool enable);
	void setNextPlayingShadowColor(const QColor &color);
	void setDateHourShadowEnabled(bool enable);
	void setDateHourShadowColor(const QColor &color);
	// Lyrics:
	void setLyricsBackgroundColor(const QColor &color);
	void setLyricsTextColor(const QColor &color);
	void setLyricsScrollBackgroundColor(const QColor &color);
	void setLyricsScrollButtonsColor(const QColor &color);
	// Author:
	void setAuthorName(const QString &name);
	void setAuthorEMail(const QString &eMail);
	void setAuthorURL(const QString &url);
	void setCopyrightInformation(const QString &information);

	// Disk Management:
	void load();
	void save();

  protected:
	QPixmap *computePixmap(int width, int height, const QStringList &locations) const;

  private:
	// Theme information:
	QString m_location;
	QString m_folderName;
	QString m_themeName;
	// Screen:
	QColor  m_backgroundColor;
	int     m_backgroundColorOpacity;
	QColor  m_textColor;
	QColor  m_nextPlayingColor;
	QColor  m_dateHourColor;
	// Progress bar:
	QColor  m_progressBackgroundColor;
	QColor  m_progressBarColor;
	QColor  m_progressBackgroundTextColor;
	QColor  m_progressBarTextColor;
	// Shadows:
	bool    m_textShadowEnabled;
	QColor  m_textShadowColor;
	bool    m_nextPlayingShadowEnabled;
	QColor  m_nextPlayingShadowColor;
	bool    m_dateHourShadowEnabled;
	QColor  m_dateHourShadowColor;
	// Lyrics:
	QColor  m_lyricsBackgroundColor;
	QColor  m_lyricsTextColor;
	QColor  m_lyricsScrollBackgroundColor;
	QColor  m_lyricsScrollButtonsColor;
	// Author:
	QString m_authorName;
	QString m_authorEMail;
	QString m_authorURL;
	QString m_copyrightInformation;

	// Pixmap & preview:
	QPixmap *m_backgroundPixmap;
	QPixmap *m_previewPixmap;

	// Frames:
	Frame *m_frames[Frame::SHAPE_COUNT];
	int    m_framesPaddingTop[Frame::SHAPE_COUNT]; // TODO: struct FramePaddings
	int    m_framesPaddingLeft[Frame::SHAPE_COUNT];
	int    m_framesPaddingRight[Frame::SHAPE_COUNT];
	int    m_framesPaddingBottom[Frame::SHAPE_COUNT];

  public:
	bool hasFrame(Frame::Shape shape);
	Frame *frame(Frame::Shape shape);
	void framePaddings(Frame::Shape shape, int *paddingTop, int *paddingLeft, int *paddingRight, int *paddingBottom);
	void setFramePaddings(Frame::Shape shape, int paddingTop, int paddingLeft, int paddingRight, int paddingBottom);
	void setFrame(Frame::Shape shape, Frame *frame);
	QRect realFrameRect(Frame::Shape shape, const QRect &rect);
};

class ThemeList : public QPtrList<Theme>
{
  public:
	ThemeList();
  protected:
	int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2);
};

//bool operator<(const Theme &theme1, const Theme &theme2);

/**
 * @author Sébastien Laoût <slaout@linux62.org>
 */
class ThemeManager : public QObject
{
  Q_OBJECT
  private:
	ThemeManager();
	~ThemeManager();

  signals:
	void themeChanged();

  public:
	Theme *currentTheme();

	void loadThemeList();

	static ThemeManager *instance();
	static void chooseTheme(int screenWidth, int screenHeight, QWidget *parent);

	const ThemeList &themes() const;
	void addTheme(Theme *theme);
	void removeTheme(Theme *theme);
	void reSort();
	Theme *forName(const QString name);

	void setTheme(Theme *theme);

	Theme *import(const QString &path);
	Theme *badArchiveFormat();

  private:
	ThemeList  m_themes;
	Theme     *m_currentTheme;

	static ThemeManager *s_instance;
};

#endif // THEMEMANAGER_H
