/***************************************************************************
 *   Copyright (C) 2007 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "tools.h"

#include <qregexp.h>
#include <qdir.h>

#include <math.h>

QPixmap Tools::addTransparentPixelsOnTop(const QPixmap &source, int nbPixels)
{
	// Verify if it is possible:
	if (nbPixels <= 0 || source.isNull())
		return source;

	// Create the images:
	QImage resultImage(source.width(), nbPixels + source.height(), 32/*bits*/);
	QImage sourceImage = source.convertToImage();
	resultImage.setAlphaBuffer(true);

	// Clear the top part by making it fully transparent:
	uint *p;
	for (int row = 0; row < nbPixels; ++row) {
		for (int column = 0; column < resultImage.width(); ++column) {
			p = (uint *)resultImage.scanLine(row) + column;
			*p = 0; // qRgba(0, 0, 0, 0)
		}
	}

	// Copy the source image byte per byte to the right part:
	uint *q;
	for (int row = 0; row < sourceImage.height(); ++row) {
		for (int column = 0; column < sourceImage.width(); ++column) {
			p = (uint *)resultImage.scanLine(row + nbPixels) + column;
			q = (uint *)sourceImage.scanLine(row) + column;
			*p = *q;
		}
	}

	// And return the result:
	QPixmap result;
	result.convertFromImage(resultImage);
	return result;
}

QPixmap Tools::addTransparentPixelsOnLeftRight(const QPixmap &source, int nbPixelsLeft, int nbPixelsRight)
{
	// Verify if it is possible:
	if (nbPixelsLeft + nbPixelsRight <= 0 || source.isNull())
		return source;

	// Create the images:
	QImage resultImage(nbPixelsLeft + source.width() + nbPixelsRight, source.height(), 32/*bits*/);
	QImage sourceImage = source.convertToImage();
	resultImage.setAlphaBuffer(true);

	// Clear the top part by making it fully transparent:
	uint *p;
	for (int row = 0; row < resultImage.height(); ++row) {
		for (int column = 0; column < nbPixelsLeft; ++column) {
			p = (uint *)resultImage.scanLine(row) + column;
			*p = 0; // qRgba(0, 0, 0, 0)
		}
		for (int column = 0; column < nbPixelsRight; ++column) {
			p = (uint *)resultImage.scanLine(row) + resultImage.width() - 1 - column;
			*p = 0; // qRgba(0, 0, 0, 0)
		}
	}

	// Copy the source image byte per byte to the right part:
	uint *q;
	for (int row = 0; row < sourceImage.height(); ++row) {
		for (int column = 0; column < sourceImage.width(); ++column) {
			p = (uint *)resultImage.scanLine(row) + column + nbPixelsLeft;
			q = (uint *)sourceImage.scanLine(row) + column;
			*p = *q;
		}
	}

	// And return the result:
	QPixmap result;
	result.convertFromImage(resultImage);
	return result;
}

QImage Tools::reflexionImage(QImage &image)
{
	if (image.width() > 0 && image.height() > 0) {
		image = image.convertDepth(32, /*converstionFlags=*/0); // The code below needs this assumption to work
		int reflexionWidth  = image.width();
		int reflexionHeight = image.height() * 2 / 3;
		QImage reflexion(reflexionWidth, reflexionHeight, 32);
		reflexion.setAlphaBuffer(true);

		uchar lineOpacity; // [0; 255]
		int x;
		QRgb *srcLine; // pixel is an unsigned int: 4 bytes. Most significant byte is alpha channel
		QRgb *destLine;
		for (int y = 0; y < reflexionHeight; ++y) {
			// 1/exp(0) = 1
			//   down to:
			// 1/exp(5) = 0.00...
			lineOpacity = (uchar) (128 * (1.0 / exp(5.0 * y / reflexionHeight))); //128 - (128 * y / reflexionHeight); // Decreasing opacity from 50% on top to 0% on bottom
			srcLine  = (QRgb *) image.scanLine(image.height() - 1 - y);
			destLine = (QRgb *) reflexion.scanLine(y);
			for (x = 0; x < reflexionWidth; ++x) {
				QRgb &srcPixel  = (QRgb &) *(srcLine  + x);
				QRgb &destPixel = (QRgb &) *(destLine + x);
				destPixel = qRgba(qRed(srcPixel), qGreen(srcPixel), qBlue(srcPixel), lineOpacity * (int) qAlpha(srcPixel) / 255);
			}
		}
		return reflexion;
	} else
		return QImage();
}

QPixmap Tools::reflexionPixmap(QImage &image)
{
	QImage reflexion = reflexionImage(image);
	QPixmap result;
	result.convertFromImage(reflexion);
	return result;
}

QImage Tools::smoothScale(QImage &image, int width, int height)
{
	if (image.isNull())
		return image;

	QImage scaled = image.smoothScale(width, height);

	int originalWidth  = image.width();
	int originalHeight = image.height();

	if (originalHeight > 0 && height > 0 && originalWidth > 0 && width > 0)
		for (int y = 0; y < height; y++)
			scaled.setPixel(width - 1, y, image.pixel(originalWidth - 1, y * originalHeight / height));

	return scaled;
}

QColor Tools::mixColors(const QColor &color1, const QColor &color2, double color1Proportion)
{
	float color2Proportion = 1.0 - color1Proportion;
	QColor mixedColor;
	mixedColor.setRgb( (int) (color1Proportion * color1.red()   + color2Proportion * color2.red()),
	                   (int) (color1Proportion * color1.green() + color2Proportion * color2.green()),
	                   (int) (color1Proportion * color1.blue()  + color2Proportion * color2.blue()) );
	return mixedColor;
}

bool Tools::tooDark(const QColor &color)
{
	int dontCare, value;
	color.getHsv(/*hue:*/dontCare, /*saturation:*/dontCare, value);
	return value < 175;
}

bool Tools::isWideScreen(int width, int height)
{
	double ratio = height / (double) width;
	return !(ratio > 0.7 && ratio < 0.8); // Not wide screen if ratio is roughly 3/4 (== 0.75)
}

// The following is adapted from KStringHanlder::tagURLs
// The adaptation lies in the change to urlEx
// Thanks to Richard Heck
QString Tools::tagURLs(const QString &text)
{
	QRegExp urlEx("(www\\.(?!\\.)|([a-zA-z]+)://)[\\d\\w\\./,:_~\\?=&;#@\\-\\+\\%\\$]+[\\d\\w/]");

	QString richText(text);
	int urlPos = 0;
	int urlLen;
	while ((urlPos = urlEx.search(richText, urlPos)) >= 0) {
		urlLen = urlEx.matchedLength();
		QString href = richText.mid(urlPos, urlLen);
		// Qt doesn't support (?<=pattern) so we do it here
		if ((urlPos > 0) && richText[urlPos-1].isLetterOrNumber()) {
			urlPos++;
			continue;
		}
		QString anchor = "<a href=\"" + href + "\">" + href + "</a>";
		richText.replace(urlPos, urlLen, anchor);
		urlPos += anchor.length();
	}
	return richText;
}

// Same as above, but modified to tag using [url][/url] instead of HTML:
QString Tools::tagURLsBBCode(const QString &text)
{
	QRegExp urlEx("(www\\.(?!\\.)|([a-zA-z]+)://)[\\d\\w\\./,:_~\\?=&;#@\\-\\+\\%\\$]+[\\d\\w/]");

	QString richText(text);
	int urlPos = 0;
	int urlLen;
	while ((urlPos = urlEx.search(richText, urlPos)) >= 0) {
		urlLen = urlEx.matchedLength();
		QString href = richText.mid(urlPos, urlLen);
		// Qt doesn't support (?<=pattern) so we do it here
		if ((urlPos > 0) && richText[urlPos-1].isLetterOrNumber()) {
			urlPos++;
			continue;
		}
		QString anchor = "[url]" + href + "[/url]";
		richText.replace(urlPos, urlLen, anchor);
		urlPos += anchor.length();
	}
	return richText;
}


/** @Return a new filename that doesn't already exist in @p destFolder.
 * If @p wantedName alread exist in @p destFolder, a dash and a number will be added before the extenssion.
 * Id there were already such a number in @p wantedName, it is incremented until a free filename is found.
 *
 * This code comes from BasKet Note Pads, modified to take a separator parameter (here, we use space instead of dash: more human).
 */
QString Tools::fileNameForNewFile(const QString &wantedName, const QString &destFolder, const char separator)
{
	QString fileName  = wantedName;
	QString fullName  = destFolder + fileName;
	QString extension = "";
	int     number    = 2;
	QDir    dir;

	// First check if the file do not exists yet (simplier and more often case)
	dir = QDir(fullName);
	if ( ! dir.exists(fullName) )
		return fileName;

	// Find the file extension, if it exists : Split fileName in fileName and extension
	// Example : fileName == "note5-3.txt" => fileName = "note5-3" and extension = ".txt"
	int extIndex = fileName.findRev('.');
	if (extIndex != -1 && extIndex != int(fileName.length()-1))  { // Extension found and fileName do not ends with '.' !
		extension = fileName.mid(extIndex);
		fileName.truncate(extIndex);
	} // else fileName = fileName and extension = ""

	// Find the file number, if it exists : Split fileName in fileName and number
	// Example : fileName == "note5-3" => fileName = "note5" and number = 3
	int extNumber = fileName.findRev(separator);
	if (extNumber != -1 && extNumber != int(fileName.length()-1))  { // Number found and fileName do not ends with '-' !
		bool isANumber;
		int  theNumber = fileName.mid(extNumber + 1).toInt(&isANumber);
		if (isANumber) {
			number = theNumber;
			fileName.truncate(extNumber);
		} // else :
	} // else fileName = fileName and number = 2 (because if the file already exists, the genereated name is at last the 2nd)

	QString finalName;
	for (/*int number = 2*/; ; ++number) { // TODO: FIXME: If overflow ???
		finalName = fileName + separator + QString::number(number) + extension;
		fullName = destFolder + finalName;
		dir = QDir(fullName);
		if ( ! dir.exists(fullName) )
			break;
	}

	return finalName;
}


/** Delete the folder @p folderOrFile recursively (to remove sub-folders and child files too).
 *
 * This code comes from BasKet Note Pads.
 */
void Tools::deleteRecursively(const QString &folderOrFile)
{
	if (folderOrFile.isEmpty())
		return;

	QFileInfo fileInfo(folderOrFile);
	if (fileInfo.isDir()) {
		// Delete the child files:
		QDir dir(folderOrFile, QString::null, QDir::Name | QDir::IgnoreCase, QDir::All | QDir::Hidden);
		QStringList list = dir.entryList();
		for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
			if ( *it != "." && *it != ".." )
				deleteRecursively(folderOrFile + "/" + *it);
		// And then delete the folder:
		dir.rmdir(folderOrFile);
	} else
		// Delete the file:
		QFile::remove(folderOrFile);
}

