/*
 * klick - an advanced metronome for jack
 *
 * Copyright (C) 2007  Dominic Sacré  <dominic.sacre@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "options.h"
#include "main.h"

#include <string>
#include <iostream>
#include <boost/tokenizer.hpp>
#include <cstdlib>
#include <unistd.h>

using namespace std;
typedef boost::char_separator<char> char_sep;
typedef boost::tokenizer<char_sep> tokenizer;


Options::Options()
  : auto_connect(false),
    follow_transport(false),
    click_sample(1),
    emphasis(EMPHASIS_NORMAL),
    volume(1.0),
    frequency(1.0),
    transport_enabled(false),
    transport_master(false),
    delay(0.0f),
    preroll(PREROLL_NONE),
    tempo_multiplier(1.0),
    verbose(false)
{
}


void Options::print_version(ostream & out)
{
    out << "klick " VERSION << endl << endl;
}


void Options::print_usage(ostream & out)
{
    out << "Usage:" << endl
        << "  klick [ options ] [meter] tempo[-tempo2/accel] [pattern]" << endl
        << "  klick [ options ] -f filename" << endl
        << "  klick [ options ] -j" << endl
        << endl
        << "Options:" << endl
        << "  -f filename       load tempomap from file" << endl
        << "  -j                no tempomap, just follow jack transport" << endl
        << "  -n name           set jack client name" << endl
        << "  -p port,..        jack port(s) to connect to" << endl
        << "  -P                automatically connect to hardware ports" << endl
        << "  -s number         use built-in sounds 1 (sine), 2 (noise) or 3 (bell/click)" << endl
        << "  -S file[,file]    load sounds from file(s)" << endl
        << "  -e                no emphasized beats" << endl
        << "  -E                emphasized beats only" << endl
        << "  -v multiplier     adjust playback volume" << endl
        << "  -w multiplier     adjust playback pitch" << endl
        << "  -t                enable jack transport" << endl
        << "  -T                become transport master (implies -t)" << endl
        << "  -d seconds        delay before starting playback" << endl
        << "  -c bars           pre-roll. use -c 0 for 2 beats" << endl
        << "  -l label          start playback at the given label" << endl
        << "  -x multiplier     multiply tempo by the given factor" << endl
        << "  -h                show this help" << endl
        << endl
        << "Tempomap file format:" << endl
        << "  [label:] bars [meter] tempo [pattern] [volume]" << endl
        << "  ..." << endl;
}


void Options::parse(int argc, char *argv[])
{
    int c;
    char optstring[] = "+f:jn:p:Ps:S:eEv:w:tTd:c:l:x:Vh";
    char *end;

    if (argc < 2) {
        // run with no arguments, print usage
        print_version();
        print_usage();
        throw Exit(EXIT_SUCCESS);
    }

    while ((c = getopt(argc, argv, optstring)) != -1) {
        switch (c) {
            case 'f':
                filename = string(::optarg);
                break;

            case 'j':
                follow_transport = true;
                break;

            case 'n':
                client_name = string(::optarg);
                break;

            case 'p':
              { char_sep sep(",");
                tokenizer tok(string(::optarg), sep);
                for (tokenizer::iterator i = tok.begin(); i != tok.end(); ++i) {
                    connect_ports.push_back(*i);
                }
              } break;

            case 'P':
                auto_connect = true;
                break;

            case 's':
                click_sample = strtoul(::optarg, &end, 10);
                if (*end != '\0' || click_sample < 1 || click_sample > 3) {
                    throw InvalidArgument("click sample");
                }
                break;

            case 'S':
              { char_sep sep(",");
                string str(::optarg);
                tokenizer tok(str, sep);
                tokenizer::iterator i = tok.begin();
                click_filename_emphasis = *i++;
                switch (distance(tok.begin(), tok.end())) {
                  case 1:
                    click_filename_normal = click_filename_emphasis;
                    break;
                  case 2:
                    click_filename_normal = *i;
                    break;
                  default:
                    throw InvalidArgument("sample file names");
                }
                click_sample = CLICK_SAMPLE_FROM_FILE;
              } break;

            case 'e':
                emphasis = EMPHASIS_NONE;
                break;

            case 'E':
                emphasis = EMPHASIS_ALL;
                break;

            case 'v':
                volume = strtof(::optarg, &end);
                if (*end != '\0') throw InvalidArgument("volume");
                break;

            case 'w':
                frequency = strtof(::optarg, &end);
                if (*end != '\0') throw InvalidArgument("frequency");
                break;

            case 't':
                transport_enabled = true;
                break;

            case 'T':
                transport_master = true;
                transport_enabled = true;
                break;

            case 'd':
                delay = strtod(::optarg, &end);
                if (*end != '\0' || delay < 0.0f) throw InvalidArgument("delay");
                break;

            case 'c':
                preroll = strtoul(::optarg, &end, 10);
                if (*end != '\0') throw InvalidArgument("pre-roll");
                break;

            case 'l':
                start_label = string(::optarg);
                break;

            case 'x':
                tempo_multiplier = strtof(::optarg, &end);
                if (*end != '\0' || tempo_multiplier <= 0.0f) {
                    throw InvalidArgument("tempo multiplier");
                }
                break;

            case 'V':
                verbose = true;
                break;

            case 'h':
                print_version();
                print_usage();
                throw Exit(EXIT_SUCCESS);
                break;

            default:
                print_usage(cerr);
                throw Exit(EXIT_FAILURE);
                break;
        }
    }

    // all remaining arguments make up the "tempomap"
    for (int n = ::optind; n < argc; n++) {
        cmdline += string(argv[n]);
        if (n < argc - 1) cmdline += " ";
    }

    if (follow_transport && (filename.length() || cmdline.length())) {
        throw CmdlineError("can't use tempo/tempomap together with -j option");
    }

    if (filename.length() && cmdline.length()) {
        throw CmdlineError("can't use tempomap from file and command line at the same time");
    }

    if ((!follow_transport) && filename.empty() && cmdline.empty()) {
        throw CmdlineError("no tempo specified");
    }
}
