/***************************************************************************
 * optionsmidi.cpp: implementation of Timidity class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "settings.h"
#include "timidity.h"

Timidity::Timidity(QWidget* parent, const char* name)
: QWidget(parent, name),
  midiTimidity(i18n("&Launch Timidity"), this),
  timidityPathLabel(i18n("Timidity path"), this),
  timidityPath(this), timidityButton("...", this),
  bufferLabel(i18n("Buffer"), this),
  buffer(0, 10, 1, this),
  fragmentLabel(i18n("Fragment"), this),
  fragment(0, 10, 1,this),
  outputLabel(i18n("Output"), this),
  output(this),
  chorusLabel(i18n("Chorus"), this),
  chorus(this),
  delayLabel(i18n("Delay"), this),
  delay(this), reverbLabel(i18n("Reverb"), this),
  reverb(this),
  vlpfLabel(i18n("Voice LPF effect"), this),
  vlpf(this),
  noiseLabel(i18n("Noise shaping"), this),
  noise(this),
  resampLabel(i18n("Resample"), this),
  resamp(this),
  frequencyLabel(i18n("Resampling frequency"), this),
  frequency(0, 44100, 1000, this),
  realtimeEnableLabel(i18n("Realtime priority enable"), this),
  realtimeEnable(this),
  realtimeLabel(i18n("Realtime priority"), this),
  realtime(0, 100, 10, this), timidityLayout(this)
{
	timidityPathLayout.addWidget(&timidityPathLabel);
	timidityPathLayout.addWidget(&timidityPath);
	timidityPathLayout.addWidget(&timidityButton);
	
	bufferLayout.addWidget(&bufferLabel);
	bufferLayout.addWidget(&buffer);
	
	fragmentLayout.addWidget(&fragmentLabel);
	fragmentLayout.addWidget(&fragment);
	
	outputLayout.addWidget(&outputLabel);
	outputLayout.addWidget(&output);
	
	output.insertItem("Default");
	output.insertItem("ALSA");
	output.insertItem("EsounD");
	output.insertItem("JACK");
	output.insertItem("aRts");
	
	chorusLayout.addWidget(&chorusLabel);
	chorusLayout.addWidget(&chorus);
	
	chorus.insertItem("Disable this effect");
	chorus.insertItem("Enable MIDI chorus effect control.");
	chorus.insertItem("Surround  sound, chorus detuned to a lesser degree (default).");
	
	delayLayout.addWidget(&delayLabel);
	delayLayout.addWidget(&delay);
	
	delay.insertItem("Disable delay effect.");
	delay.insertItem("Left delay.");
	delay.insertItem("Right delay.");
	delay.insertItem("Swap left & right.");
	
	reverbLayout.addWidget(&reverbLabel);
	reverbLayout.addWidget(&reverb);
	
	reverb.insertItem("Disable MIDI reverb effect control.");
	reverb.insertItem("Enable  MIDI  reverb  effect control. This effect is only available in stereo.");
	reverb.insertItem("Global reverb effect.");
	reverb.insertItem("Freeverb MIDI reverb effect control (default).");
	reverb.insertItem("Global freeverb effect.");
	
	vlpfLayout.addWidget(&vlpfLabel);
	vlpfLayout.addWidget(&vlpf);
	
	vlpf.insertItem("Disable LPF effect.");
	vlpf.insertItem("Chamberlin resonant LPF (12dB/oct) (default).");
	vlpf.insertItem("Moog resonant low-pass VCF (24dB/oct).");
	
	noiseLayout.addWidget(&noiseLabel);
	noiseLayout.addWidget(&noise);
	
	noise.insertItem("No noise shaping.");
	noise.insertItem("Traditional noise shaping.");
	noise.insertItem("Overdrive-like soft-clipping + new noise shaping.");
	noise.insertItem("Tube-amplifier-like soft-clipping + new noise shaping.");
	noise.insertItem("New noise shaping.");
	
	resampLayout.addWidget(&resampLabel);
	resampLayout.addWidget(&resamp);
	
	resamp.insertItem("No interpolation.");
	resamp.insertItem("Linear interpolation.");
	resamp.insertItem("Cubic spline interpolation.");
	resamp.insertItem("Lagrange method.");
	resamp.insertItem("Newton polynomial interpolation.");
	resamp.insertItem("Modified Gauss effect (default).");
	
	frequencyLayout.addWidget(&frequencyLabel);
	frequencyLayout.addWidget(&frequency);
	
	realtimeEnableLayout.addWidget(&realtimeEnableLabel);
	realtimeEnableLayout.addWidget(&realtimeEnable);
	
	realtimeLayout.addWidget(&realtimeLabel);
	realtimeLayout.addWidget(&realtime);
	
	timidityLayout.addWidget(&midiTimidity);
	timidityLayout.addLayout(&timidityPathLayout);
	timidityLayout.addLayout(&bufferLayout);
	timidityLayout.addLayout(&fragmentLayout);
	timidityLayout.addLayout(&outputLayout);
	timidityLayout.addLayout(&chorusLayout);
	timidityLayout.addLayout(&delayLayout);
	timidityLayout.addLayout(&reverbLayout);
	timidityLayout.addLayout(&vlpfLayout);
	timidityLayout.addLayout(&noiseLayout);
	timidityLayout.addLayout(&resampLayout);
	timidityLayout.addLayout(&frequencyLayout);
	timidityLayout.addLayout(&realtimeEnableLayout);
	timidityLayout.addLayout(&realtimeLayout);
	
	fillOptions();
	
	connect(&timidityButton, SIGNAL(clicked()), SLOT(search()));
	connect(&midiTimidity, SIGNAL(clicked()), SLOT(launchTimidity()));
}

void Timidity::writeConfig()
{
	Settings::config->setGroup("Timidity");
	
	Settings::config->writeEntry("Path", timidityPath.text());
	
	Settings::config->writeEntry("Buffer", buffer.value());
	
	Settings::config->writeEntry("Fragment", fragment.value());
	
	switch(output.currentItem()) {
		case 0:
			Settings::config->writeEntry("Output", "d");
			break;
		case 1:
			Settings::config->writeEntry("Output", "s");
			break;
		case 2:
			Settings::config->writeEntry("Output", "e");
			break;
		case 3:
			Settings::config->writeEntry("Output", "j");
			break;
		case 4:
			Settings::config->writeEntry("Output", "r");
			break;
	}
	
	switch(chorus.currentItem()) {
		case 0:
			Settings::config->writeEntry("Chorus", "d");
			break;
		case 1:
			Settings::config->writeEntry("Chorus", "n");
			break;
		case 2:
			Settings::config->writeEntry("Chorus", "s");
			break;
	}
	
	switch(delay.currentItem()) {
		case 0:
			Settings::config->writeEntry("Delay", "d");
			break;
		case 1:
			Settings::config->writeEntry("Delay", "l");
			break;
		case 2:
			Settings::config->writeEntry("Delay", "r");
			break;
		case 3:
			Settings::config->writeEntry("Delay", "b");
			break;
	}
	
	switch(reverb.currentItem()) {
		case 0:
			Settings::config->writeEntry("Reverb", "d");
			break;
		case 1:
			Settings::config->writeEntry("Reverb", "n");
			break;
		case 2:
			Settings::config->writeEntry("Reverb", "g");
			break;
		case 3:
			Settings::config->writeEntry("Reverb", "f");
			break;
		case 4:
			Settings::config->writeEntry("Reverb", "G");
			break;
	}
	
	switch(vlpf.currentItem()) {
		case 0:
			Settings::config->writeEntry("Vlpf", "d");
			break;
		case 1:
			Settings::config->writeEntry("Vlpf", "c");
			break;
		case 2:
			Settings::config->writeEntry("Vlpf", "m");
			break;
	}
	
	
	Settings::config->writeEntry("Noise", noise.currentItem());
	
	switch(resamp.currentItem()) {
		case 0:
			Settings::config->writeEntry("Resample", "d");
			break;
		case 1:
			Settings::config->writeEntry("Resample", "l");
			break;
		case 2:
			Settings::config->writeEntry("Resample", "c");
			break;
		case 3:
			Settings::config->writeEntry("Resample", "L");
			break;
		case 4:
			Settings::config->writeEntry("Resample", "n");
			break;
		case 5:
			Settings::config->writeEntry("Resample", "G");
			break;
	}
	
	Settings::config->writeEntry("Frequency", frequency.value());
	
	Settings::config->writeEntry("RealTime", realtime.value());
	
	Settings::config->writeEntry("RealTimeEnable", realtimeEnable.isChecked());
	
	Settings::config->setGroup("MIDI");
	
	Settings::config->writeEntry("Timidity", midiTimidity.isChecked());
}

bool Timidity::isTimidityWanted() const
{
	return midiTimidity.isChecked();
}

void Timidity::buildCmdLine(QProcess& process) const
{
	process.clearArguments();
	
	if (timidityPath.text() != "")
		process.addArgument(timidityPath.text());
	else
		process.addArgument("/usr/bin/timidity");
	
	process.addArgument("-iA");
	
	QString buffer;
	buffer.setNum(this->buffer.value());
	QString fragment;
	fragment.setNum(this->fragment.value());
	process.addArgument("-B" + buffer + "," + fragment);
	
	switch(output.currentItem()) {
		case 0:
			process.addArgument("-Od");
			break;
		case 1:
			process.addArgument("-Os");
			
			break;
		case 2:
			process.addArgument("-Oe");
			break;
		case 3:
			process.addArgument("-Oj");
			break;
		case 4:
			process.addArgument("-Or");
			break;
	}
	
	switch(chorus.currentItem()) {
		case 0:
			process.addArgument("-EFchorus=d");
			break;
		case 1:
			process.addArgument("-EFchorus=n");
			break;
		case 2:
			process.addArgument("-EFchorus=s");
			break;
	}
	
	switch(delay.currentItem()) {
		case 0:
			process.addArgument("-EFdelay=d");
			break;
		case 1:
			process.addArgument("-EFdelay=l");
			break;
		case 2:
			process.addArgument("-EFdelay=r");
			break;
		case 3:
			process.addArgument("-EFdelay=b");
			break;
	}
	
	QString noise;
	noise.setNum(this->noise.currentItem());
	process.addArgument("-EFns=" + noise);
	
	switch(resamp.currentItem()) {
		case 0:
			process.addArgument("-EFresamp=d");
			break;
		case 1:
			process.addArgument("-EFresamp=l");
			break;
		case 2:
			process.addArgument("-EFresamp=c");
			break;
		case 3:
			process.addArgument("-EFresamp=L");
			break;
		case 4:
			process.addArgument("-EFresamp=n");
			break;
		case 5:
			process.addArgument("-EFresamp=G");
			break;
	}
	
	switch(reverb.currentItem()) {
		case 0:
			process.addArgument("-EFreverb=d");
			break;
		case 1:
			process.addArgument("-EFreverb=n");
			break;
		case 2:
			process.addArgument("-EFreverb=g");
			break;
		case 3:
			process.addArgument("-EFreverb=f");
			break;
		case 4:
			process.addArgument("-EFreverb=G");
			break;
	}
	
	switch(vlpf.currentItem()) {
		case 0:
			process.addArgument("-EFvlpf=d");
			break;
		case 1:
			process.addArgument("-EFvlpf=c");
			break;
		case 2:
			process.addArgument("-EFvlpf=m");
			break;
	}
	
	QString realtime;
	
	if (realtimeEnable.isChecked()) {
		realtime.setNum(this->realtime.value());
		process.addArgument("--realtime-priority=" + realtime);
	} else realtime = "no realtime enabled";
	
	QString frequency;
	frequency.setNum(this->frequency.value());
	process.addArgument("-s " + frequency);
}

void Timidity::search()
{
	QString temp = QFileDialog::getOpenFileName(timidityPath.text(), "timidity", this, "open file dialog", "Choose timidity" );
	
	if (!temp.isEmpty()) {
		timidityPath.setText(temp);
	}
}

void Timidity::launchTimidity()
{
	if (midiTimidity.isChecked() == false)
		return ;
	
	if (KMessageBox::questionYesNo(0, "KTabEdit tries to launch the timidity sequencer with alsa emulation(it must be compiled with alsa support)","KTabEdit") == KMessageBox::Yes)
		return ;
	
	midiTimidity.setChecked(false);
}

void Timidity::fillOptions()
{
	midiTimidity.setChecked(false);
	
	timidityPath.setText("/usr/bin/timidity");
	
	// Maximum number of buffers fragments
	//
	buffer.setValue(2);
	
	// Number of fragments in interactive mode
	//
	fragment.setValue(8);
	
	// Output to ALSA
	//
	output.setCurrentItem(1);
	
	// Disable this effect
	//
	chorus.setCurrentItem(0);
	
	// Disabled delay effect
	//
	delay.setCurrentItem(0);
	
	// Disable MIDI reverb effect control
	//
	reverb.setCurrentItem(0);
	
	// Disable LPF effect
	//
	vlpf.setCurrentItem(0);
	
	// No noise shaping
	//
	noise.setCurrentItem(0);
	
	// Resample linear interporlation
	//
	resamp.setCurrentItem(1);
	
	// Frequency high quality
	//
	frequency.setValue(44100);
	
	// Realtime is disabled by default
	//
	realtimeEnable.setChecked(false);
	
	// Realtime priority
	//
	realtime.setValue(50);
}
