/***************************************************************************
 * convertascii.h: implementation of ConvertAscii class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "convertascii.h"
#include "settings.h"

#include "tabbar.h"
#include "tabcolumn.h"
#include "tabsong.h"
#include "tabtrack.h"

ConvertAscii::ConvertAscii(TabSong *song): ConvertBase(song)
{
	Settings::config->setGroup("ASCII");
	durMode = Settings::config->readNumEntry("DurationDisplay", 3);
	pageWidth = Settings::config->readNumEntry("PageWidth", 72);

	// Clever expression to determine minimal duration to put one
	// blank for (i.e. we put only one blank for this duration and any
	// durations less than this).
	oneBlankDuration = (durMode > 0) ? (120 >> (durMode - 1)) : 0;
}

ConvertAscii::~ConvertAscii()
{
}

bool ConvertAscii::save(QString fileName)
{
	// Initialize output stream
	QFile f(fileName);
	if (!f.open(IO_WriteOnly))
		return FALSE;
	QTextStream s(&f);
	stream = &s;

	// Print out header
	writeHeader();

	// Print out track data
	QListIterator<TabTrack> it(*song);
	for (int n = 1; it.current(); ++it) {
		TabTrack *trk = it.current();
		writeTrack(trk, n);
		n++; // Numerical track counter
	}

	f.close();

	return TRUE;
}

bool ConvertAscii::load(QString)
{
	// GREYFIX: todo loading from ASCII tabs
    return FALSE;
}

void ConvertAscii::writeHeader()
{
	writeCentered(song->title());
	(*stream) << endl;
	writeCentered("Author: " + song->author());
	writeCentered("Transcribed by: " + song->transcriber());
	// GREYFIX - comments?
	(*stream) << "Tempo: " << song->tempo() << endl << endl;
}

void ConvertAscii::writeTrack(TabTrack *trk, int n)
{
	QString tmp;

	startTrack(trk, n);
	startRow(trk);

	uint bar = 0;

// 	for (uint x = 0; x < trk->getTabColumnSize(); x++) {
// 
// 		// If this bar's not last
// 		if (bar + 1 < trk->getTabBarSize()) {
// /*			if ((uint) trk->getVectTabBar(bar+1).start == x) {*/
// 			if ((uint) 0 == x) {
// 
// 				// Time for next bar
// 				bar++;
// 				flushBar(trk);
// 			}
// 		}
// 
// 		TabColumn tab = trk->getVectTabColumn(x);
// 		addColumn(trk, &tab);
// 		trk->setVectTabColumn(x, tab);
// // 		addColumn(trk, &(trk->getTabColumn(x)));
// 	}

	flushBar(trk);
	flushRow(trk);
}

void ConvertAscii::startTrack(TabTrack *trk, int n)
{
	(*stream) << "Track " << n << ": " << trk->name() << endl << endl;
	// GREYFIX - channel, bank, patch, string, frets data

	minstart = 1;
	for (int i = 0; i < trk->nbStrings(); i++)
		if (Settings::noteName(trk->tune(i) % 12).length() > 1)
			minstart = 2;
}

void ConvertAscii::startRow(TabTrack *trk)
{
	for (int i = 0; i < trk->nbStrings(); i++) {
		if (trk->mode() == FretTab) {
			row[i] = Settings::noteName(trk->tune(i) % 12);
			while (row[i].length() < minstart)
				row[i] += ' ';
		} else {
			row[i] = drum_abbr[trk->tune(i)];
		}
		row[i] += "|-";
	}
	rowBars = 0;
}

void ConvertAscii::writeCentered(QString l)
{
	for (int i = 0; i < (pageWidth-(int) l.length()) / 2; i++)
		(*stream) << ' ';
	(*stream) << l << endl;
}

void ConvertAscii::addColumn(TabTrack *trk, TabColumn *col)
{
	bool lng = FALSE;

	// Check if column contains any 'long' (2-digit) values
	if (trk->mode() == DrumTab) {
		for (int i = 0; i < trk->nbStrings(); i++)
			if (col->getNbFret(i) >= 10)
				lng = TRUE;
	}

	// Determine spaces for duration
	int spaces = col->getDuration() / oneBlankDuration;
	if (spaces < 1)  spaces = 1;

	// Render column
	for (int i = 0; i < trk->nbStrings(); i++) {

		// Digits
		switch (col->getNbFret(i)) {
		case NULL_NOTE:
			bar[i] += lng ? "--" : "-";
			break;
		case DEAD_NOTE:
			bar[i] += lng ? "-X" : "X";
			break;
		default:
			if (trk->mode() == DrumTab) {
				bar[i] += "o";
			} else {
				if ((lng) && (col->getNbFret(i) < 10))
					bar[i] += '-';
				bar[i] += QString::number(col->getNbFret(i));
			}
			break;
		}

		// Space for duration
		for (int j = 0; j < spaces; j++)
			bar[i] += '-';
	}
}

void ConvertAscii::flushBar(TabTrack *trk)
{
/*	// Close bar with vertical pipe symbol
	for (int i = 0; i < trk->nbStrings(); i++)
		bar[i] += '|';

	// If we won't overfill page width or if we have no bars yet, add
	// bar[] to row[]
	if (rowBars == 0 || (row[0].length() + bar[0].length() <= pageWidth)) {
		for (int i = 0; i < trk->nbStrings(); i++) {
			row[i] += bar[i];
			bar[i] = "";
		}
		rowBars++;
	}

	// If we have to flush row, do it
	if (row[0].length() + bar[0].length() >= pageWidth) {
		flushRow(trk);
		startRow(trk);
	}

	// If we still have bar to flush, do it now
	if (bar[0].length() > 0) {
		for (int i = 0; i < trk->nbStrings(); i++) {
			row[i] += bar[i];
			bar[i] = "";
		}
		rowBars++;
	}*/
}

void ConvertAscii::flushRow(TabTrack *trk)
{
	if (rowBars > 0) {
		for (int i = trk->nbStrings() - 1; i >= 0; i--)
			(*stream) << row[i] << endl;

		(*stream) << endl;
	}
}
