/***************************************************************************
 * guitarpro.h: implementation of GuitarPro class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "guitarpro4.h"
#include "tabsong.h"
#include "logger.h"

GuitarPro4::GuitarPro4()
: instruments(64), trackProperties(0)
{
}

GuitarPro4::~GuitarPro4()
{
}

inline void GuitarPro4::dummies(int i)
{
	buf+=i;
	size-=i;
}
unsigned char GuitarPro4::getByte()
{
	unsigned char i = buf[0];
	dummies(1);
	
	return i;
}

inline unsigned long GuitarPro4::getLong()
{
	unsigned long i = buf[0] | buf[1]<<8 | buf[2]<<16 | buf[3]<<24;
	dummies(4);
	
	return i;
}

inline QString GuitarPro4::getString()
{
	unsigned long max = getLong();
	QString temp;
	
	if (max > 0) {
		unsigned char i = getByte();
		
		if (i != max - 1)
			throw "Wrong string length";
		
		temp = QString::fromLocal8Bit(reinterpret_cast<const char *>(buf), i);
		dummies(i);
	} else {
		max = getByte();
	}
	
	return temp;
}

inline QString GuitarPro4::getDelphiString()
{
	unsigned long max = getLong();
	QString temp;
	
	temp = QString::fromLocal8Bit(reinterpret_cast<const char *>(buf), max);
	dummies(max);
	
	return temp;
}

inline QString GuitarPro4::getStringNoInc(int i)
{
	int nb = getByte();
	QString temp;
	
	ktabLog << "getStringNoInc " << i << "\n";
	
	temp = QString::fromLocal8Bit(reinterpret_cast<const char *>(buf), nb);
	
	if (i)
		dummies(i);
	else
		dummies(nb);
	
	temp.setLength(nb);
	
	return temp;
}

inline QString GuitarPro4::getChordName()
{
	QString temp;
	unsigned int max = 20;
	
	temp = QString::fromLocal8Bit(reinterpret_cast<const char *>(buf), 21);
	dummies(21);
	
	if (temp[0] < max) {
		max = temp[0].latin1();
	}
	
	for(uint i = 1; i <= max ; i++) {
		temp[i-1] = temp[i] ;
	}
	
	temp[max] = 0;
	
	return temp;
}

inline bool GuitarPro4::isChar(unsigned char c)
{
	return (((c)>='a' && (c)<='z' ) || ( (c)>='A' && (c)<='Z' ));
}

unsigned char GuitarPro4::getBeatDuration()
{
	unsigned char beatDuration = getByte();
	
	if (beatDuration > 128)
		beatDuration-=256;
	
	return beatDuration;
}

// This code is traducted from the python code of songwrite
//
TabSong GuitarPro4::loadFile(const QString &fileName)
{
	unsigned char command, beatduration;
	unsigned int i, j, k, noteVolume, nbbeats, noteLength;
	bool rest = false;
	TabSong tabSong;
	TabChord tabChord;
	
	// Open file
	//
	QFile gp3file(fileName);
	
	if (!gp3file.open(IO_ReadOnly)) {
		ktabLog << "Cannot open file" << "\n";
		throw "Cannot open file";
	}
	
	// Read whole file in a buffer
	//
	totalsize = size = gp3file.size();
	buffer = new unsigned char[size];
	QDataStream fstream(&gp3file);
	fstream.readRawBytes(reinterpret_cast<char *>(buffer),size );
	gp3file.close();
	
	readHeader(tabSong);
	
	readMidi();
	
	readBars(tabSong);
	
	readTracks(tabSong);
	
	QListIterator<TabTrack> tracks(tabSong);
	
	i = 0;
	while (i < nbbars) {
		j = 0;
	
		ktabLog << "Lecture de la mesure " << i << "/" << nbbars << "\n" << "\n";
		
		tracks.toFirst();
		
		while (j < nbtrack) {
			noteVolume = 0xCC;
			
			nbbeats = getLong();
			
			ktabLog << "Analyse de la piste " << j << " (" << nbbeats << " nombre de notes)" << "\n" << "\n";
			
			TabTrack *track = tracks.current();
			++tracks;
			
			TabBar tempBar(4, 4);
			
			TabBarProperties* tempProperties = dynamic_cast<TabBarProperties*>(&tempBar);
			*tempProperties = trackProperties[i];
			
			k = 0;
			
			while (k < nbbeats) {
				TabTimes notes;
				noteLength = 0;
				rest = false;
				
// 				notes.setNbFrets(track->getNbFrets());
				
				command = getByte();
				
				ktabLog << k << " Commande = " << static_cast<int>(command) << "\n";
				
				if (command & 0x40) {
					// Beat status
					getByte();
					// If byte == 0x00 empty
					// If byte == 0x02 rest
					ktabLog << "REST found " <<  noteLength << "\n";
					rest = true;
				}
				
				if (command & 0x01) {
					ktabLog << "Dotted time" << "\n";
					notes.setDotted(true);
				}
				
				beatduration = getBeatDuration();
				
				if (command & 0x20) {
					notes.setNTuplet(getLong() - 1);
					ktabLog << "NTuplet " << notes.nTuplet() << "\n";
				}
				
				if (command & 0x02) {
					ktabLog << "Accord\n";
					tabChord = readChords();
					
					notes.setChord(true);
					
					TabChord* tempChord = dynamic_cast<TabChord*>(&notes);
					*tempChord = tabChord;
				}
				
				if (command & 0x04) {
					notes.setText(getString());
					ktabLog << "Text found " << notes.text() << "\n";
				}
				
				if (command & 0x08) {
					readEffectsOnBeat(notes);
				}
				
				if (command & 0x10) {
					readChanges(tempBar);
				}
				
				readNotes(rest, beatduration, notes, noteLength);
				
				tempBar.insertTimes(notes);
				
				k++;
			}
			j++;
			
			track->addBar(tempBar);
		}
		i++;
	}
	
	delete buffer;
	
	return tabSong;
}

void GuitarPro4::readHeader(TabSong& tabSong)
{
	// Check file signature
	//
	const char *GTP_ID3 = "FICHIER GUITAR PRO";
	int i;
	long noticeNbLines;
	QString notice;
	
	buf = buffer;
	
	if (strncmp(reinterpret_cast<const char*>(buf + 1), GTP_ID3, strlen(GTP_ID3))) {
		ktabLog << "Not a Guitar Pro file!" << "\n" << GTP_ID3 <<"!=" << reinterpret_cast<char *>(buf + 1) << "\n";
		delete buffer;
		throw "Not a Guitar Pro file!";
	}

	// Now it's time to parse the whole thing
	// First the header: general infos
	dummies(31);
	
	tabSong.setTitle(getString());
	ktabLog << "Title = " << tabSong.title() << "\n";
	
	tabSong.setSubTitle(getString());
	ktabLog << "Subtitle = " << tabSong.subTitle() << "\n";

	tabSong.setArtist(getString());
	ktabLog << "Artist = " << tabSong.artist() << "\n";

	tabSong.setAlbum(getString());
	ktabLog << "Album = " << tabSong.album() << "\n";

	tabSong.setAuthor(getString());
	ktabLog << "Author = " << tabSong.author() << "\n";

	tabSong.setCopyright(getString());
	ktabLog << "Copyright = " << tabSong.copyright() << "\n";

	tabSong.setTranscriber(getString());
	ktabLog << "Transcriber = " << tabSong.transcriber() << "\n";

	tabSong.setInstructions(getString());
	ktabLog << "Instruction = " << tabSong.instructions() << "\n";

	noticeNbLines = getLong();
	ktabLog << "Nombre de lignes des commentaires :" << noticeNbLines << "\n";
	
	i = 0;
	while (i < noticeNbLines) {
		notice+=getString();
		notice+="\n";
		i++;
	}
	tabSong.setComments(notice);
	ktabLog << "Commentaires =" << notice << "\n";
	
	tabSong.setShuffle(static_cast<bool>(getByte()));
	ktabLog << "Shuffle =" << tabSong.isShuffle() << "\n";
	
	tabSong.setLyricsTrack(getLong());
	ktabLog << "lyricsTrack = " << tabSong.lyricsTrack() << "\n";
	for (i = 0; i < 5; i++) {
		tabSong.setLyricsMeasure(i, getLong());
		tabSong.setLyrics(i, getDelphiString());
		
		ktabLog << "paroles = " << tabSong.lyricsMeasure(i) << "\n";
		ktabLog << "paroles = " << tabSong.lyrics(i) << "\n";
	}
	
	tabSong.setTempo(getLong());
	ktabLog << "Tempo =" << tabSong.tempo() << "\n";
	
	tabSong.setKey(getByte());
	ktabLog << "Key =" << tabSong.nameOfKey() << "\n";
	
	tabSong.setOctave(getLong());
	ktabLog << "octave =" << tabSong.octave() << "\n";
}

void GuitarPro4::readMidi()
{
	// Lecture des instruments
	//
	uchar i = 0;
	
	ktabLog << "Lecture des instruments =" << "\n";
	while (i < 64) {
		instruments[i].setPatch(getLong());
		instruments[i].setVolume(getByte());
		instruments[i].setBalance(getByte());
		instruments[i].setChorus(getByte());
		instruments[i].setReverb(getByte());
		instruments[i].setPhaser(getByte());
		instruments[i].setTremolo(getByte());
		
		ktabLog << "Instrument n " << (int)i << " " << instruments[i].patch() << " Volume " << (int)instruments[i].volume() 
		<< " balance " << (int)instruments[i].balance() << " chorus " << (int)instruments[i].chorus() << " reverb " << (int) instruments[i].reverb() 
		<< " phaser " << (int)instruments[i].phaser() << " tremolo " << (int)instruments[i].tremolo() << "\n";
		 
		dummies(2);
		i++;
	}
}

void GuitarPro4::readBars(TabSong& tabSong)
{
	unsigned int i;
	
	nbbars = getLong();
	ktabLog << "\n\n" << "Nombre de mesures =" << nbbars << "\n";
	
	nbtrack = getLong();
	ktabLog << "Nombre de pistes =" << nbtrack << "\n";
	
	// Read track infos: name
	//
	tabSong.clear();
	
	trackProperties.resize(nbbars);
	
	unsigned char barFlag;
	unsigned char barRythm1 = 4, barRythm2 = 4;
	
	i = 0;
	while (i < nbbars) {
		barFlag = getByte();
		
		ktabLog << "Mesure =" << i << "\n";
		
		if (barFlag & 1) {
			barRythm1 = getByte();
			ktabLog << "Bar rythm 1 =" << static_cast<int>(barRythm1) << "\n";
		}
		
		if (barFlag & 2) {
			barRythm2 = getByte();
			ktabLog << "Bar rythm 2 =" << static_cast<int>(barRythm2) << "\n";
		}
		
		trackProperties[i].setTimeSignature(0, barRythm1);
		trackProperties[i].setTimeSignature(1, barRythm2);
		
		if (barFlag & 4) {
			ktabLog << "Open repeat barFlag & 4" << "\n";
			trackProperties[i].setRepeatStart(true);
		}
		
		if (barFlag & 8) {
			ktabLog << "Repeat close barFlag & 8" << "\n";
			trackProperties[i].setRepetitions(getByte());
		}
		
		if (barFlag & 0x10) {
			ktabLog << "Open alt barFlag & 16" << "\n";
			trackProperties[i].setAlternateEndings(getByte());
		}
		
		if (barFlag & 0x20) {
			ktabLog << "Marker barFlag & 32" << "\n";
			QString marker(getString());
			QColor color(getByte(), getByte(), getByte());
			
			ktabLog << "Marker =" << marker << "Couleur r :" << color << "\n";
			
			// Empty byte
			//
			getByte();
			
			trackProperties[i].setMarker(marker);
			trackProperties[i].setColor(color);
		}
		
		if (barFlag & 0x40) {
			ktabLog << "Armor change barFlag & 64" << "\n";
			// Armor jump
			//
			trackProperties[i].setKey(getByte());
			// Minor
			//
			trackProperties[i].setMinor(getByte());
			
			ktabLog << "Key =" << trackProperties[i].nameOfKey() << "\n";
			
			ktabLog << "Key =" << trackProperties[i].isMinor() << "\n";
		}
		
		if (barFlag & 0x80) {
			ktabLog << "Double bar barFlag & 0x80" << "\n";
			trackProperties[i].setDoubleBar(true);
		}
		
		i++;
	}
}

void GuitarPro4::readTracks(TabSong& tabSong)
{
	unsigned int i = 0;
	unsigned long tunning[7];
	
	ktabLog << "Track" << "\n";
	
	while (i < nbtrack) {
		TrackMode mode = FretTab;
		
		unsigned char header = getByte();
		
		if (header & 1) {
			ktabLog << "drum track" << "\n";
			mode = DrumTab;
		}
		
		if (header & 2) {
			ktabLog << "12 strings track" << "\n";
			mode = TwelveStringsTab;
		}
		
		if (header & 4) {
			ktabLog << "banjo track" << "\n";
			mode = BanjoTab;
		}
		
		QString track = getStringNoInc(40);
		ktabLog << "Track Name =" << track << "\n";
		
		long nbStrings = getLong();
		ktabLog << "Nb Strings =" << nbStrings << "\n";
		
		if (mode == DrumTab)
			nbStrings = 4;
		
		ktabLog << "Tuning =\n";
		
		int j = 0;
		while (j < 7) {
			tunning[j] = getLong();
			j++;
		}
		
		int midiPort = getLong();
		ktabLog << "Midi port =" << midiPort << "\n";
		
		int channel = getLong();
		ktabLog << "Channel 1 =" << channel << "\n";
		
		int channelEffects = getLong();
		ktabLog << "Channel effects =" << channelEffects << "\n";
		
		int nbFrets = getLong();
		ktabLog << "Nb frets =" << nbFrets << "\n";
		
		int capo = getLong();
		ktabLog << "Capo =" << capo << "\n";
		
		QColor color(getByte(), getByte(), getByte());
		ktabLog << "Color =" << "Couleur :" << color << "\n";
		
		// Empty byte
		//
		getByte();
		
		TabTrack* temp = new TabTrack(mode, track, 0, 0, instruments[channel].patch(), nbStrings, nbFrets);
		TrackMidiProperties* trackMidiProperties = temp;
		
		*trackMidiProperties = instruments[channel];
		
		for (j = 0; j < nbStrings; j++)
			temp->setTune(nbStrings - 1 - j, tunning[j]);
		
		temp->setCapo(capo);
		temp->setColor(color);
		temp->setChannel(channel);
		temp->setChannelEffects(channelEffects);
		temp->setMidiPort(midiPort);
		
		tabSong.append(temp);
		
		i++;
	}
}

void GuitarPro4::readEffectsOnBeat(TabTimes& times)
{
	uchar command = getByte();
	uchar command2 = getByte();
	ktabLog << "Beat effect " << (int)command << "\n";
	
	if (command & 0x20) {
		uchar effect = getByte();
		
		if (!effect) {
			ktabLog << "Tremolo bar" << "\n";
			unsigned int value = getLong();
			
			times.setTremolo(value);
		} else {
			switch(effect) {
				case 1:
					ktabLog << "Tapping\n";
					
					times.setTapping(true);
					break;
				case 2:
					ktabLog << "slapping\n";
					
					times.setSlapping(true);
					break;
				case 3:
					ktabLog << "Popping\n";
					
					times.setPopping(true);
					break;
// 				default:
// 					throw "Unexpected value in effect";
			}
		}
	}
	
	if (command2 & 0x04) {
		ktabLog << "Note effect : Bend" << "\n";
			
		// Type of the bend
		//
		times.setBendType(static_cast<TabBend::BendType>(getByte()));
		
		//Vertical height
		//
		times.setBendValue(getLong());
		
		// num points
		//
		int numPoints = getLong();
		
		QValueList<TabBendPoint> list;
		
		for (int i = 0; i <numPoints; i++) {
			TabBendPoint point;
			
			point.setPosition(getLong());
			point.setHeight(getLong());
			point.setVibrato(static_cast<TabVibrato>(getByte()));
			
			list.append(point);
		}
		
		times.setBendPoint(list);
	}
	
	if (command & 0x40) {
		ktabLog << "Strocke" << "\n";
		
		times.setStroke(true);
		
		int duration = getByte();
		ktabLog << "duration " << duration << "\n";
		if (duration)
			times.setUpStroke(6 - duration);
		
		duration = getByte();
		ktabLog << "duration " << duration << "\n";
		if (duration)
			times.setDownStroke(6 - duration);
	}
	
	if (command2 & 0x01) {
		// Rasgueado
		//
		ktabLog << "rasgueado\n";
		times.setRasgueado(true);
	}
	
	if (command2 & 0x02) {
		// Pickstroke
		ktabLog << "pickstroke\n";
		times.setPickStroke(static_cast<PickStroke>(getByte()));
	}
	
	if (command & 0x01) {
		times.setVibrato(true);
		ktabLog << "Vibrato" << "\n";
	}
	
	if (command & 0x02) {
		times.setWideVibrato(true);
		ktabLog << "Wide vibrato" << "\n";
	}
	
	if (command & 0x04) {
		times.setNaturalHarmonic(true);
		ktabLog << "Natural harmonic" << "\n";
	}
	
	if (command & 0x08) {
		times.setArtificialHarmonic(true);
		ktabLog << "Other harmonic" << "\n";
	}
	
	if (command & 0x10) {
		times.setFadeIn(true);
		ktabLog << "Fade in" << "\n";
	}
}

void GuitarPro4::readEffectsOnNote(TabNote& note)
{
	ktabLog << " Special note\n";
	uchar effect_1 = getByte();
	uchar effect_2 = getByte();
	
	ktabLog << " Effect 1 =" << (int)effect_1 << "\n";
	ktabLog << " Effect 2 =" << (int)effect_2 << "\n";
	
	if (effect_1 & 0x01) {
		ktabLog << "Note effect : Bend" << "\n";
		
		// Absolute time position
		//
		note.setBendType(static_cast<TabBend::BendType>(getByte()));
		
		//Vertical height
		//
		note.setBendValue(getLong());
		
		ktabLog << "bendValue = " << note.bendValue() << "\n";
		
		// num points
		//
		int numPoints = getLong();
		
		QValueList<TabBendPoint> list;
		
		for (int i = 0; i <numPoints; i++) {
			TabBendPoint point;
			
			point.setPosition(getLong());
			ktabLog << "bendPosition = " << point.position() << "\n";
			point.setHeight(getLong());
			ktabLog << "bendHeight = " << point.height() << "\n";
			point.setVibrato(static_cast<TabVibrato>(getByte()));
			ktabLog << "bendVibrato = " << point.tabVibrato() << "\n";
			
			list.append(point);
		}
		
		note.setBendPoint(list);
	}
	
	
	if (effect_1 & 0x04) {
		ktabLog << "Note effect : Slide" << "\n";
		note.setSlide(true);
	}
	
	if (effect_1 & 0x10) {
		ktabLog << "Note effect : Grace note" << "\n";
		// Fret
		note.setGraceNoteFret(getByte());
		// Dynamic
		note.setGraceNoteDynamic(static_cast<Dynamic>(getByte()));
		// Transition
		note.setGraceNoteTransition(static_cast<GraceNoteTransition>(getByte()));
		// Duration
		note.setGraceNoteDuration(getByte());
	}
	
	if (effect_2 & 0x04) {
		ktabLog << "tremolo picking\n";
		note.setTremoloPicking(getByte());
	}
	
	if (effect_2 & 0x08) {
		ktabLog << "Note effect : Slide" << "\n";
		note.setSlide(static_cast<Slide>(getByte()));
	}
	
	if (effect_2 & 0x10) {
		// Harmonic
		//
		note.setHarmonic(static_cast<Harmonic>(getByte()));
	}
	
	if (effect_2 & 0x20) {
		// Trill
		//
		ktabLog << "Note effect : Trill" << "\n";
		
		// Fret
		//
		note.setFretTrill(getByte());
		
		// Duration
		//
		note.setPeriod(getByte());
	}
	
	if (effect_1 & 0x08) {
		ktabLog << "Note effect : Let Ring" << "\n";
		note.setLetRing(true);
	}
	
	if (effect_1 & 0x02) {
		ktabLog << "Note effect : Pull" << "\n";
		note.setHammerOnPullOff(true);
	}
	
	if (effect_2 & 0x40) {
		// Left hand vibrato
		note.setLeftHandVibrato(true);
	}
	
	if (effect_2 & 0x02) {
		// Palm mute
		note.setPalmMute(true);
	}
	
	if (effect_2 & 0x01) {
		// Staccato
		note.setStaccato(true);
	}
}

void GuitarPro4::readChanges(TabBar& bar)
{
	ktabLog << "Change" << "\n";
	
	uchar new_inst = getByte();
// 	ktabLog << "Instrument " << static_cast<int>(new_inst) << "\n";
	
	uchar new_vol = getByte();
// 	ktabLog << "Volume " << static_cast<int>(new_vol) << "\n";
	
	uchar new_pan = getByte();
// 	ktabLog << "Pan " << static_cast<int>(new_pan) << "\n";
	
	uchar new_chorus = getByte();
// 	ktabLog << "Chorus " << static_cast<int>(new_chorus) << "\n";
	
	uchar new_reverb = getByte();
// 	ktabLog << "Reverb " << static_cast<int>(new_reverb) << "\n";
	
	uchar new_phaser = getByte();
// 	ktabLog << "Phaser " << static_cast<int>(new_phaser) << "\n";
	
	uchar new_tremolo = getByte();
// 	ktabLog << "Tremolo " << static_cast<int>(new_tremolo) << "\n";
	
	int new_tempo = getLong();
// 	ktabLog << "Tempo " << static_cast<int>(new_tempo) << "\n";
	
	if (new_vol != 255) {
		bar.setInstrument(new_inst);
		dummies(1);
	}
	
	if (new_pan != 255) {
		bar.setBalance(new_pan);
		dummies(1);
	}
	
	if (new_chorus != 255) {
		bar.setChorus(new_chorus);
		dummies(1);
	}
	
	if (new_reverb != 255) {
		bar.setReverb(new_reverb);
		dummies(1);
	}
	
	if (new_phaser != 255) {
		bar.setPhaser(new_phaser);
		dummies(1);
	}
	
	if (new_tremolo != 255) {
		bar.setTremolo(new_tremolo);
		dummies(1);
	}
	
	if (new_tempo != -1) {
		bar.setTempo(new_tempo);
		dummies(1);
	}

// 	if (new_vol != 255) dummies(1);
// 	if (new_pan != 255) dummies(1);
// 	if (new_chorus != 255) dummies(1);
// 	if (new_reverb != 255) dummies(1);
// 	if (new_phaser != 255) dummies(1);
// 	if (new_tremolo != 255) dummies(1);
// 	if (new_tempo != -1) dummies(1);
	
	// Apply to all
	//
	getByte();
}

TabChord GuitarPro4::readChords()
{
	uchar header = getByte();
	TabChord chord;
	
	if ((header & 0x01) == 0) {
		chord.setName(getString());
		
		int base = getLong();
		
		chord.setBaseFret(base);
		
		if (base != 0) {
			for (int i = 0; i < 6; i++) {
				chord.setFret(i, getLong());
			}
		}
	} else {
		
		
		// Sharp
		//
		chord.setSharp(getByte());
		ktabLog << "sharp = " << chord.isSharp() << "\n";
		
		// Skip 3 bytes
		//
		dummies(3);
		
		// GP4Root
		//
		chord.setRoot(getByte());
		ktabLog << "root = " << chord.root() << "\n";
		
		// Chord type
		//
		chord.setType(getByte());
		ktabLog << "type = " << chord.type() << "\n";
		
		// 9, 11, 13
		//
		chord.setNineElevenThirteen(static_cast<TabChord::ExtentionChord>(getByte()));
		
		ktabLog << "9, 11, 13 = " << chord.nineElevenThirteen() << "\n";
		
		// Bass
		//
		chord.setBass(getLong());
		ktabLog << "bass = " << chord.bass() << "\n";
		
		// Augmented or diminished
		//
		chord.setAugmentedDiminished(static_cast<TabChord::AugmentedDiminished>(getLong()));
		ktabLog << "+ - = " << chord.augmentedDiminished() << "\n";
		
		// Added Note
		//
		chord.setAddedNote(getByte());
		ktabLog << "added note = " << chord.addedNote() << "\n";
		
		// Name
		//
		chord.setName(getChordName());
		ktabLog << "chord name = " << chord.name() << "\n";
		
		// Skip 2 bytes
		//
		dummies(2);
		
		// Tonality five
		//
		chord.setTonalityFive(getByte());
		ktabLog << "tonality five = " << chord.tonalityFive() << "\n";
		
		// Tonality nine
		//
		chord.setTonalityNine(getByte());
		ktabLog << "tonality nine = " << chord.tonalityNine() << "\n";
		
		// Tonality eleven
		//
		chord.setTonalityEleven(getByte());
		ktabLog << "tonality eleven = " << chord.tonalityEleven() << "\n";
		
		// Base fret
		//
		chord.setBaseFret(getLong());
		ktabLog << "base fret = " << chord.baseFret() << "\n";
		
		// Reads the 6 frets
		//
		for (int i = 0; i < 7; i++) {
			chord.setFret(i, getLong());
			ktabLog << "fret = " << chord.fret(i) << "\n";
		}
		
		// Num barres
		//
		chord.setNumBarres(getByte());
		ktabLog << "num barres = " << chord.numBarres() << "\n";
		
		// Fret of barre
		//
		for (int i = 0; i < 5; i++) {
			chord.setBarreFret(i, getByte());
			ktabLog << "fret of barre = " << chord.barreFret(i) << "\n";
		}
		
		// Barre start
		//
		for (int i = 0; i < 5; i++) {
			chord.setBarreStart(i, getByte());
			ktabLog << "bare start = " << chord.barreStart(i) << "\n";
		}
		
		// Barre end
		//
		for (int i = 0; i < 5; i++) {
			chord.setBarreEnd(i, getByte());
			ktabLog << "bare end = " << chord.barreEnd(i) << "\n";
		}
		
		dummies(8);
		
		// Omission : gives the notes present in the chord
		for (int i = 0; i < 7; i++) {
			chord.setFingering(i, getByte());
			ktabLog << " " << chord.fingering(i) << "\n";
		}
		
		
		// Show diagram
		//
		chord.setShowDiagram(getByte());
	}
	
	return chord;
}

void GuitarPro4::readNotes(bool rest, uchar beatduration, TabTimes& times, uint noteLength)
{
	char strings_fill = getByte(), numberOfStrings = 0, i;
	
	if (rest) {
		switch(beatduration) {
		case 0:
			times.setDuration(Quarter);
			break;
		case 1:
			times.setDuration(Eighth);
			break;
		case 2:
			times.setDuration(Sixteenth);
			break;
		case 3:
			times.setDuration(ThirthSecond);
			break;
		case 4:
			// Quadruple croche
			times.setDuration(SixtyFourth);
			break;
		case 0xff:
			times.setDuration(Half);
			break;
		case 0xfe:
			times.setDuration(Whole);
			break;
		}
		
		times.setRest(true);
		
		rest = false;
	}
	
	for (i = 0; i < 7; i++) {
		if (strings_fill & (1 << i)) {
			numberOfStrings++;
		}
	}
	
	ktabLog << "String played = " << (int)numberOfStrings << "\n";
	
	QValueList<char> notes_strings;
	
	if (strings_fill & 64) notes_strings.append(0);
	if (strings_fill & 32) notes_strings.append(1);
	if (strings_fill & 16) notes_strings.append(2);
	if (strings_fill & 8)  notes_strings.append(3);
	if (strings_fill & 4)  notes_strings.append(4);
	if (strings_fill & 2)  notes_strings.append(5);
	if (strings_fill & 1)  notes_strings.append(6);
	QValueList<char>::iterator it = notes_strings.begin();
	
	i = 0;
	while (i < numberOfStrings) {
		TabNote note;
		
		uchar note_special = getByte();
		uchar note_alteration = 0;
		uchar note_value = 0;
		char chord = 0;
		
		ktabLog << "Note header = " << (int)note_special << "\n";
		
		if (note_special & 0x20) {
			note_alteration = getByte();
			ktabLog << "note altration " << static_cast<int>(note_alteration)  << "\n";
			
			if (note_alteration == 3) {
				note.setDeadNote(true);
				
				ktabLog << "Dead note" << "\n";
			} else if (note_alteration == 2) { 
				note.setTieNote(true);
				
				ktabLog << "Note is linked from another note" << "\n";
			}
			
		}
		
		if (note_special & 0x01) {
			ktabLog << "note duration != beat duration" << "\n";
			
			uchar note_duration = getByte();
			if(note_duration > 128) note_duration-=256;
			
			ktabLog << static_cast<int>(note_duration) << "\n";
			
			// NTuplet
			//
			note.setNTuplet(getByte() - 1);
		}
		
		if (note_special & 0x10) {
// 			ktabLog << "Note nuance" << "\n";
			note.setDynamic(static_cast<Dynamic>(getByte()));
		}
		
		if (note_special & 0x20) {
			note_value = getByte();
			chord = (*it);
			ktabLog << "Valeur de la note =" << static_cast<int>(note_value) << " corde = " << 5 -  static_cast<int>(chord) << "\n";
			if (note_value == 255) note_value = 0;
		}
		
		if (note_special & 0x20) {
			
// 			ktabLog << "Ajout de la note :" << static_cast<int>(note_value) << " corde : " << 5 -  static_cast<int>(chord) << "\n";
// 			ktabLog << "beat = " << (int)beatduration << "\n";
			switch(beatduration) {
			case 0:
				note.setDuration(Quarter);
				times.setDuration(Quarter);
				break;
			case 1:
				note.setDuration(Eighth);
				times.setDuration(Eighth);
				break;
			case 2:
				note.setDuration(Sixteenth);
				times.setDuration(Sixteenth);
				break;
			case 3:
				note.setDuration(ThirthSecond);
				times.setDuration(ThirthSecond);
				break;
			case 4:
				// Quadruple croche
				note.setDuration(SixtyFourth);
				times.setDuration(SixtyFourth);
				break;
			case 0xff:
				note.setDuration(Half);
				times.setDuration(Half);
				break;
			case 0xfe:
				note.setDuration(Whole);
				times.setDuration(Whole);
				break;
// 			default:
// 				ktabLog << "beat " << beatduration << "\n";
			}
			
			note.setFret(note_value);
			
			noteLength = 1;
		}
		
		
		if (note_special & 0x80) {
			ktabLog << "fingerprint" << "\n";
			
			// Left hand
			//
			note.setFingeringLeft(static_cast<TabNote::Fingering>(getByte()));
			
			// Right hand
			//
			note.setFingeringRight(static_cast<TabNote::Fingering>(getByte()));
		}
		
		if (note_special & 0x40) {
			ktabLog << "Accentuated note" << "\n";
			note.setAccentuated(true);
		}
		
		if (note_special & 0x04) {
			ktabLog << "Ghost note" << "\n";
			note.setGhostNote(true);
		}
		
		if (note_special & 0x08) {
			readEffectsOnNote(note);
		}
		
		if (note_special & 0x02) {
			ktabLog << "Dotted note" << "\n";
			note.setDotted(true);
		}
		
		times.setNotes(chord, note);
		++it;
		i++;
	}
}

bool GuitarPro4::saveFile(const QString &, const TabSong &)
{
    return false;
}
