/***************************************************************************
 * kguitar_part.cpp: implementation of KGuitarPart class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "assistant/assistant.h"
#include "assistant/timiditylauncher.h"
#include "convertascii.h"
#include "guitarpro.h"
#include "guitarpro4.h"
#include "kguitartab.h"
#include "melodyeditor.h"
#include "options.h"
#include "optionsexportascii.h"
#include "optionsexportmusixtex.h"
#include "powertab.h"
#include "setsong.h"
#include "settings.h"
#include "songprint.h"
#include "songview.h"
#include "tabsong.h"
#include "texnotes.h"
#include "textab.h"
#include "trackdrag.h"
#include "trackeditview.h"
#include "tracklist.h"
#include "xmlmusic.h"

#include "kguitar_part.h"

typedef KParts::GenericFactory<KGuitarPart> KGuitarPartFactory;
K_EXPORT_COMPONENT_FACTORY(libktabeditpart, KGuitarPartFactory)

QString drum_abbr[128];

KGuitarPart::KGuitarPart(QWidget *parentWidget, const char * /*widgetName*/, QObject *parent, const char *name, const QStringList & /*args*/)
: KParts::ReadWritePart(parent, name),
  factory(),
  scheduler(0),
  transport(0),
  metronome(0),
  timidityLauncher(0)
{
	Settings::config = KGuitarPartFactory::instance()->config();
	
	cmdHist = new KCommandHistory(actionCollection());
	
	setInstance(KGuitarPartFactory::instance());
	
	// Launch the assistant for the midi configuration
	//
	if (Settings::assistant() == false) {
		Assistant assistant(factory);
		assistant.exec();
		
		timidityLauncher = assistant.timidityLauncher();
		scheduler        = assistant.midiScheduler();
	} else {
		if (Settings::midiTimidity()) {
			timidityLauncher = new TimidityLauncher(parentWidget, "timidityLauncher");
			
			if (!timidityLauncher->launchTimidity())
				KMessageBox::error(0, "Couldn't launch timidity " + timidityLauncher->errorMsg());
		}
	}
	
	// Configure the midi devices
	//
	midiConfigure();
	
	// Custom main widget
	//
	songView = new SongView(scheduler, actionCollection(), this, cmdHist, parentWidget);
	
	// notify the part that this is our internal widget
	//
	setWidget(songView);
	
	setupActions();
	setupAccels();
	
	// SET UP RESPONSES FOR VARIOUS TRACK CHANGES
	// TODO : connect(songView->getTrackView(), SIGNAL(trackChanged(TabTrack *)), SLOT(updateToolbars(TabTrack *)));
	connect(QApplication::clipboard(), SIGNAL(dataChanged()), SLOT(clipboardDataChanged()));
	connect(songView, SIGNAL(statusBar(const QString&)), this, SLOT(updateStatusBar(const QString&)));
	
	setXMLFile("ktabedit/ktabedit_part.rc");
	
	setReadWrite(true);
	setModified(false);
	
	// Read config
	//
	readOptions();
	
	readMidiNames();
}

KGuitarPart::~KGuitarPart()
{
	saveOptions();
	
	if (timidityLauncher) {
		timidityLauncher->kill();
		delete timidityLauncher;
	}
	
	delete cmdHist;
	delete viewMelodyEditorAct;
	delete viewScoreAct;
	delete saveOptionAct;
}

void KGuitarPart::setReadWrite(bool rw)
{
	songView->setReadOnly(!rw);
	
	if (rw)
		connect(songView->getTrackEditView(), SIGNAL(songChanged(bool)), this, SLOT(setModified(bool)));
	else
		disconnect(songView->getTrackEditView(), SIGNAL(songChanged(bool)), this, SLOT(setModified(bool)));
	
	ReadWritePart::setReadWrite(rw);
}

void KGuitarPart::setModified(bool modified)
{
	// enable or disable Save action based on modified
	//
	KAction *save = actionCollection()->action(KStdAction::stdName(KStdAction::Save));
	save->setEnabled(modified);
	
	// in any event, we want our parent to do it's thing
	//
	ReadWritePart::setModified(modified);
}

KAboutData *KGuitarPart::createAboutData()
{
	KAboutData *aboutData = new KAboutData("ktabedit", I18N_NOOP("KTabEdit"), VERSION);
	aboutData->addAuthor("KTabEdit development team", 0, 0);
	
	return aboutData;
}

void KGuitarPart::fileOpenAs()
{
	// this slot is called whenever the File->Open menu is selected,
	// the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
	// button is clicked
	QString file_name = KFileDialog::getOpenFileName();
	
	if (file_name.isEmpty() == false)
		openURL(file_name);
}

// Reimplemented method from KParts to open file m_file
bool KGuitarPart::openFile()
{
	QFileInfo fi(m_file);
	
	FileImport *file = NULL;
	bool success = false;
	QString ext = fi.extension();
	
	if (!fi.isFile())
	{
		KMessageBox::sorry(0, i18n("No file specified, please select a file."));
		return FALSE;
	}
	
	if (!fi.isReadable())
	{
		KMessageBox::sorry(0, i18n("You have no permission to read this file."));
		return FALSE;
	}
	
	ext = ext.lower();
	
	if (ext == "ptb")
		file = new PowerTab;
	
	if (ext == "kg")
		file = new KGuitarTab;
	else
		if (ext == "tab")
		{
			ConvertAscii converter(songView);
			success = converter.load(m_file);
		}
		else
			if (ext == "gtp")
				file = new GuitarPro;
			else
				if (((ext[0] == 'g') && (ext[1] == 'p') && (ext[2] == '3'))  && (ext.length() == 3))
				{
					file = new GuitarPro;
				}
				else
					if (ext == "xml")
						file = new XMLMusic;
					else
						if (((ext[0] == 'g') && (ext[1] == 'p') && (ext[2] == '4'))  && (ext.length() == 3))
						{
							file = new GuitarPro4;
						}
	try
	{
		if (file) {
			*dynamic_cast<TabSong*>(songView) = file->loadFile(m_file);
			delete file;
		}
		
		success = true;
		
		songView->refreshView();
		cmdHist->clear();
	}
	catch(const char *msg)
	{
		setWinCaption(i18n("Unnamed"));
		KMessageBox::sorry(0, i18n("Can't load or import song!"
									"It may be a damaged/wrong file format or, "
									"if you're trying experimental importers "
									"it may be a flaw with the import code.") + msg);
	}
	
	return success;
}

bool KGuitarPart::exportOptionsDialog(QString ext)
{
	// Skip dialog if user set appopriate option
	//
	if (!Settings::config->readBoolEntry("AlwaysShow", TRUE))
		return TRUE;
	
	KDialogBase opDialog(0, 0, TRUE, i18n("Additional Export Options"), KDialogBase::Help|KDialogBase::Default|KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok);
	
	QVBox *box = opDialog.makeVBoxMainWidget();
	OptionsPage *op;
	
	if (ext == "tab")
		op = new OptionsExportAscii(Settings::config, (QFrame *) box);
	else if (ext == "tex")
		op = new OptionsExportMusixtex(Settings::config, (QFrame *) box);
	else
	{
		kdWarning() << "Weird exportOptionsDialog() call! Wrong extension " << ext << endl;
		return FALSE;
	}
	
	connect(&opDialog, SIGNAL(defaultClicked()), op, SLOT(defaultBtnClicked()));
	connect(&opDialog, SIGNAL(okClicked()), op, SLOT(applyBtnClicked()));
	
	bool res = (opDialog.exec() == QDialog::Accepted);
	
	delete op;
	
	return res;
}

// Reimplemented method from KParts to current song to file m_file
//
bool KGuitarPart::save()
{
	switch (KMessageBox::warningYesNoCancel(0, i18n("Save changes to document ?"))) {
	case KMessageBox::Yes :
		// save document here. If saving fails, return false;
		//
		return saveFile();
	case KMessageBox::No :
		return true;
	default: // cancel
		return false;
	}
	
	return false;
}

// Reimplemented method from KParts to current song to file m_file
//
bool KGuitarPart::saveFile()
{
	// if we aren't read-write, return immediately
	//
	if (isReadWrite() == false)
		return false;
	
	FileImport *file = NULL;
	QFileInfo *fi = new QFileInfo(m_file);
	QString ext = fi->extension().lower();
	bool success = false;
	
	// GREYFIX: some sort of dirty hack - workaround the KDE default
	// save, not saveAs without file name
	if (m_file.isEmpty()) {
		fileSaveAs();
		return false;
	}
	
	if (ext == "kg") {
		file = new KGuitarTab;
	} else if (ext == "tab") {
		Settings::config->setGroup("ASCII");
	
		if (exportOptionsDialog(ext))
		{
			ConvertAscii converter(songView);
			success = converter.save(m_file);
		}
		else
			return FALSE;
	}
	else if (ext == "gtp")
		file = new GuitarPro;
	else if ((ext[0] == 'g') && (ext[1] == 'p') && ((ext[2] == '3') || (ext[2] == '4'))  && (ext.length() == 3))
		file = new GuitarPro;
	else if (ext == "tex")
	{
		Settings::config->setGroup("MusiXTeX");
		if (exportOptionsDialog(ext))
		{
			switch (Settings::texExportMode())
			{
			case 0:
				file = new TexTab;
				break;
			case 1:
				file = new TexNotes;
				break;
			}
		}
		else
			return FALSE;
	}
	else if (ext == "xml")
		file = new XMLMusic;
	
	try
	{
		if (file)
		{
			file->saveFile(m_file, *dynamic_cast<TabSong*>(songView));
	
			setWinCaption(m_file);
			cmdHist->clear();
		}
	}
	catch (const char *msg)
	{
		KMessageBox::sorry(0, i18n("Can't save song in %1 format").arg(ext));
	}
	
	delete file;
	
	return success;
}

void KGuitarPart::fileSaveAs()
{
	QString filter =
		"*.kg|" + i18n("KGuitar files") + " (*.kg)\n"
		"*.tab|" + i18n("ASCII files") + " (*.tab)\n"
		"*.mid|" + i18n("MIDI files") + " (*.mid)\n"
		"*.tse3|" + i18n("TSE3MDL files") + " (*.tse3)\n"
		"*.gtp|" + i18n("Guitar Pro files") + " (*.gtp)\n"
		"*.gp3|" + i18n("Guitar Pro 3 files") + " (*.gp3)\n"
		"*.gp4|" + i18n("Guitar Pro 4 files") + " (*.gp4)\n"
		"*.xml|" + i18n("MusicXML files") + " (*.xml)\n"
		"*.tex|" + i18n("MusiXTeX") + " (*.tex)\n"
		"*|" + i18n("All files");
	
	QString file_name = KFileDialog::getSaveFileName(QString::null, filter);
	
	if (file_name.isEmpty() == false)
		saveAs(file_name);
}

// Updates possibility of actions, depending on freshly selected
// track. For drum track, lots of actions are unavailable.
void KGuitarPart::updateToolbars(TabTrack *)
{
/*	switch (songView->getTrackView()->getTrack()->trackMode())
	{
	case DrumTab:
		insChordAct->setEnabled(FALSE);
		natHarmAct->setEnabled(FALSE);
		artHarmAct->setEnabled(FALSE);
		break;
	default:
		insChordAct->setEnabled(TRUE);
		natHarmAct->setEnabled(TRUE);
		artHarmAct->setEnabled(TRUE);
	}*/
}

void KGuitarPart::filePrint()
{
	KPrinter printer(true, QPrinter::ScreenResolution);
	
	songView->print(&printer);
}

void KGuitarPart::options()
{
	Options op(factory, scheduler, KGuitarPartFactory::instance()->config());
	op.exec();
	
	songView->setMidiScheduler(scheduler);
	songView->drawBackground();
}

void KGuitarPart::readOptions()
{
	KConfig *config = KGuitarPartFactory::instance()->config();
	
	viewMelodyEditorAct->setChecked(config->readBoolEntry("Visible", TRUE));
	viewMelodyEditor();
	
	viewScoreAct->setChecked(FALSE);	// LVIFIX: enable before commit
	viewScore();
}

void KGuitarPart::saveOptions()
{
	Settings::config->setGroup("MelodyEditor");
	Settings::config->writeEntry("Visible", viewMelodyEditorAct->isChecked());
	Settings::config->sync();
}

void KGuitarPart::readMidiNames()
{
	drum_abbr[35] = QString("BD1");
	drum_abbr[36] = QString("BD2");
	drum_abbr[38] = QString("SD1");
	drum_abbr[40] = QString("SD2");
	
	drum_abbr[39] = QString("HCL"); // Hand clap
	
	drum_abbr[42] = QString("CHH");
	drum_abbr[44] = QString("PHH");
	drum_abbr[46] = QString("OHH");
	
	drum_abbr[49] = QString("CR1"); // Crash cymbal
	drum_abbr[57] = QString("CR2");
	
	drum_abbr[51] = QString("RI1"); // Ride cymbal
	drum_abbr[59] = QString("RI2");
	
	drum_abbr[54] = QString("TBR"); // Tamborine
	drum_abbr[55] = QString("SPL"); // Splash cymbal
	
	drum_abbr[41] = QString("TL2");
	drum_abbr[43] = QString("TL1");
	drum_abbr[45] = QString("TM2");
	drum_abbr[47] = QString("TM1");
	drum_abbr[48] = QString("TH2");
	drum_abbr[50] = QString("TH1");
}

void KGuitarPart::setWinCaption(const QString& caption)
{
	emit setWindowCaption(caption);
}

void KGuitarPart::viewMelodyEditor()
{
	songView->showMelodyEditor(viewMelodyEditorAct->isChecked());
}

void KGuitarPart::viewScore()
{
	songView->viewScore(viewScoreAct->isChecked());
}

void KGuitarPart::setupActions()
{
	// Setup standard actions
	//
	KStdAction::saveAs(this, SLOT(fileSaveAs()), actionCollection());
	KStdAction::save  (this, SLOT(save())      , actionCollection());
	KStdAction::print (this, SLOT(filePrint()) , actionCollection());
	
	preferencesAct = KStdAction::preferences(this, SLOT(options()), actionCollection(), "pref_options");
	
	// View
	//
	viewMelodyEditorAct = new KToggleAction(i18n("Show Melody Editor"), "melodyeditor", KKeySequence("Shift+M").keyCodeQt(),
											this, SLOT(viewMelodyEditor()), actionCollection(), "view_melodyEditor");
	viewScoreAct = new KToggleAction(i18n("Show Score"), "score", KKeySequence("Shift+S").keyCodeQt(), this, SLOT(viewScore()),
										actionCollection(), "view_score");
	
	// Track
	//
	saveOptionAct = new KAction(i18n("&Save Options"), 0, this, SLOT(saveOptions()), actionCollection(), "save_options");
	
	pasteAct = KStdAction::paste(songView, SLOT(slotPaste()), actionCollection());
}

void KGuitarPart::setupAccels()
{
}

void KGuitarPart::clipboardDataChanged()
{
	pasteAct->setEnabled(TrackDrag::canDecode(*QApplication::clipboard()->data()));
}

void KGuitarPart::updateStatusBar(const QString& msg)
{
	emit setStatusBarText(msg);
}

/*
	Configure the midi card with TSE3
*/
void KGuitarPart::midiConfigure()
{
	// The scheduler is already configure
	//
	if (scheduler) {
		return ;
	}
	
	try {
		scheduler = factory.createScheduler();
	} catch (TSE3::MidiSchedulerError e) {
		QString temp = e.what();
		
		kdDebug() << "error while creating the midi scheduler !!\n" << temp << "\n";
		KMessageBox::error(0, "Couldn't create a MidiScheduler for this platform " + temp);
		
		scheduler = 0;
		
		return ;
	}
}
