/***************************************************************************
 * optionsmidi.cpp: implementation of OptionsMidi class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "optionsmidi.h"
#include "settings.h"

OptionsMidi::OptionsMidi(TSE3::MidiSchedulerFactory& midiFactory, TSE3::MidiScheduler* &scheduler, KConfig *conf, QWidget *parent, const char *name)
: OptionsPage(conf, parent, name),
  midiport(this),
  midiportLabel(&midiport, i18n("MIDI &output port"), this),
  midiRefresh(i18n("&Refresh"), this),
  midiTimidity(i18n("&Launch Timidity"), this),
  midiLayout(this, 10, 5),
  midiButton(this),
  factory(midiFactory),
  scheduler(scheduler),
  timidity(false)
{
	midiport.setSorting(-1);
	midiport.setFrameStyle(QFrame::Panel | QFrame::Sunken);
	midiport.addColumn(i18n("Name"));
	midiport.addColumn(i18n("Port"));
	midiport.addColumn(i18n("Port Number"));
	midiport.addColumn(i18n("is Readable"));
	midiport.addColumn(i18n("is Writeable"));
	
	fillMidiBox();
	
	connect(&midiRefresh, SIGNAL(clicked()), SLOT(fillMidiBox()));
	connect(&midiTimidity, SIGNAL(clicked()), SLOT(launchTimidity()));
	
	midiButton.addWidget(&midiRefresh);
	
	// Set widget layout
	//
	midiLayout.addWidget(&midiportLabel);
	midiLayout.addWidget(&midiport, 1);
	midiLayout.addLayout(&midiButton);
	midiLayout.addWidget(&midiTimidity);
	midiLayout.activate();
	
	midiTimidity.setChecked(Settings::midiTimidity());
}

void OptionsMidi::fillMidiBox()
{
	QString temp, read, write;
	QListViewItem* lastItem = 0;
	
	midiport.clear();
	
	// The scheduler is already configured so deleted it
	//
	if (scheduler)
		delete scheduler ;
	
	// Try to create a new scheduler
	//
	try {
		scheduler = factory.createScheduler();
	} catch (TSE3::MidiSchedulerError e) {
		QString temp = e.what();
		
		kdDebug() << "error while creating the midi scheduler !!\n" << temp << "\n";
		KMessageBox::error(0, "Couldn't create a MidiScheduler for this platform " + temp);
		
		scheduler = 0;
		
		return ;
	}
	
	scheduler->portNumbers(portNums);
	
	for (uint i = 0; i < scheduler->numPorts(); i++) {
		if (scheduler->portReadable(portNums[i]))
			read = "Yes";
		else
			read = "No";
		
		if (scheduler->portWriteable(portNums[i]))
			write = "Yes";
		else
			write = "No";
		
		lastItem = new QListViewItem(&midiport, lastItem, scheduler->portName(portNums[i]), scheduler->portType(portNums[i]), temp.setNum(portNums[i]), read, write);
		
		// Select the current midi card
		//
		if (portNums[i] == Settings::midiPortNumber()) {
			midiport.setCurrentItem(lastItem);
		}
	}
}

void OptionsMidi::launchTimidity()
{
	if (midiTimidity.isChecked() == false) {
		timidity = false;
		return ;
	}
	
	if (KMessageBox::questionYesNo(0, "Restart KTabEdit and it tries to launch the timidity sequencer with alsa emulation(it must be compiled with alsa support)","KTabEdit") == KMessageBox::Yes) {
		timidity = true;
		return ;
	}
	
	timidity = false;
	midiTimidity.setChecked(false);
}

void OptionsMidi::defaultBtnClicked()
{
}

void OptionsMidi::applyBtnClicked()
{
	config->setGroup("MIDI");
	config->writeEntry("Timidity", timidity);
	
	if (midiport.isSelected(midiport.currentItem())) {
		config->writeEntry("Name", midiport.currentItem()->text(0));
		config->writeEntry("Port", midiport.currentItem()->text(1));
		config->writeEntry("PortNumber", midiport.currentItem()->text(2));
		config->writeEntry("Readable", midiport.currentItem()->text(3));
		config->writeEntry("Writeable", midiport.currentItem()->text(4));
	}
}
