/***************************************************************************
 * playbacktracker.cpp: implementation of PlayBacktracker class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "logger.h"
#include "playbacktracker.h"
#include "settings.h"
#include "songview.h"
#include "tabtrack.h"
#include "track/trackpos.h"

const uint PlaybackTracker::playBackTrackCmd = 14;

PlaybackTracker::PlaybackTracker(SongView *sv)
: TransportCallback(),
  songView(sv),
  drumTrack(sv->isDrumTrack()),
  drumTrackNb(sv->drumTrack()),
  midiPort(Settings::midiPortNumber()),
  trackPos(sv->count())
{
}

void PlaybackTracker::resetCursor()
{
	trackPos.resize(songView->count());
	drumTrack = songView->isDrumTrack();
	drumTrackNb = songView->drumTrack();
	
	for (uint i = 0; i < trackPos.count(); i++) {
		trackPos[i].setBar(0);
		trackPos[i].setChord(0);
		trackPos[i].setTimes(0);
	}
}

void PlaybackTracker::Transport_MidiOut(TSE3::MidiCommand c)
{
	uint realChannel;
	
	if (c.status == playBackTrackCmd) {
		ktabLog << "[PlaybackTracker::MidiOut] : " << c.status << " " << c.port << "\n";
		realChannel = midiChanToIntChan(c.channel);
		
		TabTrack& tempTrack = *songView->at(realChannel);
		TabBar tempBar = tempTrack.bar(trackPos[realChannel].bar());
		TabTimes tempTimes = tempBar.times(trackPos[realChannel].times());
		
		// Go to the next time
		//
		nextTime(tempBar, realChannel);
		
		songView->playbackCursor(realChannel, trackPos[realChannel]);
	}
	
}

void PlaybackTracker::Transport_MidiIn(TSE3::MidiCommand) 
{
}

uint PlaybackTracker::midiChanToIntChan(uint channel)
{
	// Check the real channel
	//
	if (drumTrack) {
		if (drumTrackNb != 9) {
			if (drumTrackNb < 9) {
				if (channel == 9)
					channel = drumTrackNb;
				else if (channel > drumTrackNb && channel < 9)
					channel--;
				else if (channel >= 9)
					channel++;
			} else {
				if (channel == 9)
					channel = drumTrackNb;
				else if (channel >= 9 && channel < drumTrackNb)
					channel++;
			}
		}
	}
	
	return channel;
}

void PlaybackTracker::nextTime(const TabBar& bar, uint channel)
{
	TrackPos tempPos(trackPos[channel]);
	TabTrack& tempTrack = *songView->at(channel);
	
	// Go to the next time
	//
	tempPos.setChord(0);
	tempPos.incTimes();
		
	if (tempPos.times() == bar.count() && tempPos.bar() + 1 < tempTrack.count()) {
		tempPos.setTimes(0);
		tempPos.incBar();
	} else {
		if (tempPos.bar() + 1 == tempTrack.count()) {
			trackPos[channel] = tempPos;
			
			return ;
		}
	}
	
	trackPos[channel] = tempPos;
}
