/***************************************************************************
 * songview.cpp: implementation of SongView class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "chord.h"
#include "chordlist.h"
#include "chordlistitem.h"
#include "clipboard.h"
#include "melodyeditor.h"
#include "playbacktracker.h"
#include "settings.h"
#include "settrack.h"
#include "settabfret.h"
#include "settabdrum.h"
#include "setsong.h"
#include "settings.h"
#include "songprint.h"
#include "songview.h"
#include "songviewcommands.h"
#include "trackeditview.h"
#include "tracklist.h"
#include "trackpane.h"
#include "trackdrag.h"
#include "tablatureplayer/tabsongplayer.h"


SongView::SongView(TSE3::MidiScheduler* midiScheduler, KActionCollection *parentAction, KXMLGUIClient *_XMLGUIClient, KCommandHistory *_cmdHist,
				   QWidget *parent, const char *name)
: QWidget(parent, name),
  TabSong(i18n("Unnamed"), 120),
  canvas(0),
  ro(false),
  midiInUse(false),
  midiStopPlay(false),
  actionCollection(parentAction),
  scheduler(midiScheduler),
  tabSong(0),
  playbackTracker(0)
{
	canvas = new QCanvas();
	
	// Create a new track
	//
	append(new TabTrack(FretTab, i18n("Guitar"), 1, 0, 25, 6, 24));
	
	// Add a new bar to the track
	//
	TabBar emptyBar(4,4);
	TabTimes emptyTimes;
	
	emptyBar.insertTimes(emptyTimes);
	at(0)->addBar(emptyBar);
	
	split = new QSplitter(this);
	split->setOrientation(QSplitter::Vertical);
	
	trackEditView = new TrackEditView(canvas, scheduler, this, _XMLGUIClient, _cmdHist, split);
	connect(trackEditView, SIGNAL(stopMidiPlaying()), this, SLOT(stopPlay()));
	connect(trackEditView, SIGNAL(songChanged()), this, SIGNAL(songChanged()));
	connect(trackEditView, SIGNAL(statusBar(const QString&)), this, SIGNAL(statusBar(const QString&)));
	
	splitv = new QSplitter(split);
	splitv->setOrientation(QSplitter::Horizontal);
	
	trackList = new TrackList(this, _XMLGUIClient, splitv);
	trackList->setSelected(trackList->firstChild(), TRUE);
	trackPane = new TrackPane(this, trackList->header()->height(), trackList->firstChild()->height(), splitv);
	
	trackEditView->setTrackPane(trackPane);
	
	melodyEditor = new MelodyEditor(trackEditView, split);
	
	connect(trackList, SIGNAL(trackChanged(TabTrack*)), trackEditView, SLOT(selectTrack(TabTrack*)));
	connect(trackPane, SIGNAL(trackChanged(TabTrack*)), trackEditView, SLOT(selectTrack(TabTrack*)));
	connect(trackPane, SIGNAL(newBarSelected(uint)), trackEditView, SLOT(selectBar(uint)));
	connect(trackEditView, SIGNAL(paneChanged()), trackPane, SLOT(update()));
	
	layout = new QVBoxLayout(this);
	layout->addWidget(split);
	
	cmdHist = _cmdHist;
	
	songPrint = new SongPrint();
	
	sngPropAct = new KAction(i18n("P&roperties..."), 0, this, SLOT(songProperties()), actionCollection, "song_properties");
	
	// Cut and paste
	//
	KStdAction::cut(this, SLOT(slotCut()), actionCollection);
	KStdAction::copy(this, SLOT(slotCopy()), actionCollection);
	KStdAction::selectAll(this, SLOT(slotSelectAll()), actionCollection);
	
	// Track
	//
	trkNewAct = new KAction(i18n("&New..."), 0, this, SLOT(trackNew()), actionCollection, "track_new");
	trkDeleteAct = new KAction(i18n("&Delete"), 0, this, SLOT(trackDelete()), actionCollection, "track_delete");
	trkBassLineAct = new KAction(i18n("&Generate Bass Line"), 0, this, SLOT(trackBassLine()), actionCollection, "track_bassline");
	trkPropAct = new KAction(i18n("P&roperties..."), 0, this, SLOT(trackProperties()), actionCollection, "track_properties");
	rhythmerAct = new KAction(i18n("&Rhythm..."), "rhythmer", KKeySequence("Shift+R").keyCodeQt(), trackEditView, SLOT(rhythmer()), actionCollection, "rhythmer");
	insChordAct = new KAction(i18n("&Chord..."), "chord", KKeySequence("Shift+C").keyCodeQt(), trackEditView, SLOT(insertChord()), actionCollection, "insert_chord");
	arrTrkAct = new KAction(i18n("&Arrange Track"), KKeySequence("Shift+A").keyCodeQt(), trackEditView, SLOT(arrangeTracks()), actionCollection,  "arrange_trk");
	insertBarAct = new KAction(i18n("&insert bar"), "insert bar", KKeySequence("").keyCodeQt(), trackEditView, SLOT(insertBar()), actionCollection, "insert_bar");
	deleteBarAct = new KAction(i18n("&delete bar"), "delete bar", KKeySequence("").keyCodeQt(), trackEditView, SLOT(deleteColumn()), actionCollection, "delete_bar");
	
	// Duration
	//
	len1Act = new KAction(i18n("Whole"), "note1", KKeySequence("Ctrl+1").keyCodeQt(), trackEditView, SLOT(setLength1()), actionCollection, "set_len1");
	len2Act = new KAction("1/2", "note2", KKeySequence("Ctrl+2").keyCodeQt(), trackEditView, SLOT(setLength2()), actionCollection, "set_len2");
	len4Act = new KAction("1/4", "note4", KKeySequence("Ctrl+3").keyCodeQt(), trackEditView, SLOT(setLength4()), actionCollection, "set_len4");
	len8Act = new KAction("1/8", "note8", KKeySequence("Ctrl+4").keyCodeQt(), trackEditView, SLOT(setLength8()), actionCollection, "set_len8");
	len16Act = new KAction("1/16", "note16", KKeySequence("Ctrl+5").keyCodeQt(), trackEditView, SLOT(setLength16()), actionCollection, "set_len16");
	len32Act = new KAction("1/32", "note32", KKeySequence("Ctrl+6").keyCodeQt(), trackEditView, SLOT(setLength32()), actionCollection, "set_len32");
	lenDottedAct = new KToggleAction ("Dotted time", KKeySequence("D").keyCodeQt(), trackEditView, SLOT(setDotted()), actionCollection, "set_dotted");
	lenRestAct = new KToggleAction ("Rest", KKeySequence("R").keyCodeQt(), trackEditView, SLOT(setRest()), actionCollection, "set_rest");
	
	// effects
	//
	keySigAct = new KAction(i18n("Key signature"), "keysig", KKeySequence("Shift+K").keyCodeQt(), trackEditView, SLOT(keySig()),
				actionCollection, "key_sig");
	timeSigAct = new KAction(i18n("Time signature"), "timesig", KKeySequence("Shift+T").keyCodeQt(), trackEditView, SLOT(timeSig()),
				actionCollection, "time_sig");
	arcAct = new KAction(i18n("Link with previous column"), "arc", KKeySequence("L").keyCodeQt(), trackEditView, SLOT(linkPrev()),
				actionCollection, "link_prev");
	legatoAct = new KAction(i18n("Legato (hammer on/pull off)"), "fx_legato", KKeySequence("P").keyCodeQt(), trackEditView, SLOT(addLegato()),
				actionCollection, "fx_legato");
	slideAct = new KAction(i18n("Slide"), "fx_slide", KKeySequence("S").keyCodeQt(), trackEditView, SLOT(addSlide()),
				actionCollection, "fx_slide");
	letRingAct = new KAction(i18n("Let Ring"), "fx_let_ring", KKeySequence("I").keyCodeQt(), trackEditView, SLOT(addLetRing()),
				actionCollection, "fx_let_ring");
	natHarmAct = new KAction(i18n("Natural harmonic"), "fx_harmonic", KKeySequence("H").keyCodeQt(), trackEditView, SLOT(addHarmonic()),
				actionCollection, "fx_nat_harm");
	artHarmAct = new KAction(i18n("Artificial harmonic"), "fx_harmonic", KKeySequence("R").keyCodeQt(), trackEditView, SLOT(addArtHarm()),
				actionCollection, "fx_art_harm");
	palmMuteAct = new KAction(i18n("Palm muting"), "fx_palmmute", KKeySequence("M").keyCodeQt(), trackEditView, SLOT(palmMute()),
				actionCollection, "fx_palmmute");
	tripletAct = new KAction(i18n("Triplet note"), "fx_triplet", KKeySequence("T").keyCodeQt(), trackEditView, SLOT(ntuplet()),
				actionCollection, "fx_triplet");
	halfBendAct = new KAction(i18n("&half bend"), "half bend", KKeySequence("").keyCodeQt(), trackEditView, SLOT(halfBend()), actionCollection, "fx_half_bend");
	fullBendAct = new KAction(i18n("&full bend"), "full bend", KKeySequence("").keyCodeQt(), trackEditView, SLOT(fullBend()), actionCollection, "fx_full_bend");
	halfBendReleaseAct = new KAction(i18n("&half bend release"), "half bend release", KKeySequence("").keyCodeQt(), trackEditView, SLOT(halfBendRelease()), 
				actionCollection, "fx_half_bend_release");
	fullBendReleaseAct = new KAction(i18n("&full bend release"), "full bend release", KKeySequence("").keyCodeQt(), trackEditView, SLOT(fullBendRelease()), 
				actionCollection, "fx_full_bend_release");
	halfPreBendAct = new KAction(i18n("&half pre-bend"), "half pre-bend", KKeySequence("").keyCodeQt(), trackEditView, SLOT(halfPreBend()), actionCollection, 
				"fx_half_pre_bend");
	fullPreBendAct = new KAction(i18n("&full pre-bend"), "full pre-bend", KKeySequence("").keyCodeQt(), trackEditView, SLOT(fullPreBend()), actionCollection, 
				"fx_full_pre_bend");
	halfPreBendRelease = new KAction(i18n("&half pre-bend and release"), "half pre-bend and release", KKeySequence("").keyCodeQt(), trackEditView, 
				SLOT(halfPreBendRelease()), actionCollection, "fx_half_pre_bend_release");
	fullPreBendRelease = new KAction(i18n("&full pre-bend and release"), "full pre-bend and release", KKeySequence("").keyCodeQt(), trackEditView, 
				SLOT(fullPreBendRelease()), actionCollection, "fx_full_pre_bend_release");
	unisonBend = new KAction(i18n("&unison bend"), "unison bend", KKeySequence("").keyCodeQt(), trackEditView, SLOT(unisonBend()), actionCollection,
				"fx_unison_bend");
	vibratoAct = new KAction(i18n("&vibrato"), "vibrato", KKeySequence("").keyCodeQt(), trackEditView, SLOT(vibrato()), actionCollection, "fx_vibrato");
	wideVibratoAct = new KAction(i18n("&wide vibrato"), "wide vibrato", KKeySequence("").keyCodeQt(), trackEditView, SLOT(wideVibrato()), actionCollection,
				"fx_wide_vibrato");
	trillAct = new KAction(i18n("&trill"), "trill", KKeySequence("").keyCodeQt(), trackEditView, SLOT(trill()), actionCollection, "fx_trill");
	tappingAct = new KAction(i18n("&tapping"), "tapping", KKeySequence("").keyCodeQt(), trackEditView, SLOT(tapping()), actionCollection, "fx_tapping");
	tremoloBarAct = new KAction(i18n("&tremolo bar"), "tremolo bar", KKeySequence("").keyCodeQt(), trackEditView, SLOT(tremoloBar()), actionCollection, 
				"fx_tremolo_bar");
	muffledStringsAct = new KAction(i18n("&muffled strings"), "muffled strings", KKeySequence("").keyCodeQt(), trackEditView, SLOT(muffledStrings()),
				actionCollection, "fx_muffled_strings");
	pickSlideAct = new KAction(i18n("&pick slide"), "pick slide", KKeySequence("").keyCodeQt(), trackEditView, SLOT(pickSlide()), actionCollection, 
				"fx_pick_slide");
	tremoloPickingAct = new KAction(i18n("&tremolo picking"), "tremolo picking", KKeySequence("").keyCodeQt(), trackEditView, SLOT(tremoloPicking()), 
				actionCollection, "fx_tremolo_picking");
	
	// Midi plays
	//
	midiPlaySongAct = new KAction(i18n("&Play / stop"), "1rightarrow", KKeySequence("Space").keyCodeQt(), this, SLOT(playSong()),
				actionCollection, "midi_playsong");
	midiStopPlayAct = new KAction(i18n("&Stop"), "player_stop", KKeySequence("Ctrl+Shift+P").keyCodeQt(), this, SLOT(stopPlay()),
				actionCollection, "midi_stopplay");
	
	// N-tuplets (seconds, triplets, ...)
	//
	ntuplets = new QComboBox(this, "read-only");
	
	ntuplets->insertItem("2");
	ntuplets->insertItem("3");
	ntuplets->insertItem("4");
	ntuplets->insertItem("5");
	ntuplets->insertItem("6");
	ntuplets->insertItem("7");
	ntuplets->insertItem("8");
	ntuplets->insertItem("9");
	ntuplets->insertItem("10");
	ntuplets->insertItem("11");
	ntuplets->insertItem("4:3");
	ntuplets->insertItem("7:6");
	
	ntuplets->setCurrentItem(1);
	
	nTupletsAct = new KWidgetAction(ntuplets, i18n("&n-tuplets"), KShortcut(), trackEditView, SLOT(ntuplet()), actionCollection, "fx_ntuplets");
	
	connect(ntuplets, SIGNAL(activated(int)), trackEditView, SLOT(ntupletSelected(int)));
	
	// Setup accel
	//
	mainAccel = new KAccel(trackEditView);
	
	mainAccel->insert("key_left",i18n("Move cursor left"), i18n("Left"), Qt::Key_Left, trackEditView, SLOT(keyLeft()));
	mainAccel->insert("key_right",i18n("Move cursor right"), i18n("Right"), Qt::Key_Right, trackEditView, SLOT(keyRight()));
	mainAccel->insert("key_home",i18n("Move cursor to the beginning of bar"),i18n("Home"), Qt::Key_Home, trackEditView, SLOT(keyHome()));
	mainAccel->insert("key_end",i18n("Move cursor to the end of bar"), i18n("End"), Qt::Key_End, trackEditView, SLOT(keyEnd()));
	mainAccel->insert("key_CtrlHome",i18n("Move cursor to the beginning of track"),  i18n("Ctrl+Home"), KKeySequence("Ctrl+Home").keyCodeQt(), trackEditView, 
			  SLOT(keyCtrlHome()));
	mainAccel->insert("key_CtrlEnd",i18n("Move cursor to the end of track"),  i18n("Ctrl+End"), KKeySequence("Ctrl+End").keyCodeQt(), trackEditView, 
			  SLOT(keyCtrlEnd()));
	mainAccel->insert("key_ShiftLeft",i18n("Move and select left"), i18n("Shift+Left"), Qt::Key_Shift + Qt::Key_Left, trackEditView, SLOT(selectLeft()));
	mainAccel->insert("key_ShiftRight", i18n("Move and select right"), i18n("Shift+Right"), Qt::Key_Shift + Qt::Key_Right, trackEditView, SLOT(selectRight()));
	mainAccel->insert("key_up", i18n("Move cursor up"), i18n("Up"), Qt::Key_Up, trackEditView, SLOT(moveUp()));
	mainAccel->insert("key_down",i18n("Move cursor down"), i18n("Down"), Qt::Key_Down, trackEditView, SLOT(moveDown()));
	mainAccel->insert("key_CtrlUp",i18n("Transpose up"), i18n("Ctrl+Up"), Qt::Key_Control + Qt::Key_Up, trackEditView, SLOT(transposeUp()));
	mainAccel->insert("key_CtrlDown",i18n("Transpose down"),i18n( "Ctrl+Down"), Qt::Key_Control + Qt::Key_Down, trackEditView, SLOT(transposeDown()));
	
	// Other keys
	//
	mainAccel->insert("key_x", i18n("Dead note"), i18n("X"), Qt::Key_X, trackEditView, SLOT(deadNote()));
	mainAccel->insert("key_del",i18n("Delete note"), i18n("Delete"), Qt::Key_Delete, trackEditView, SLOT(deleteNote()));
	mainAccel->insert("key_CtrlDel",i18n("Delete column"), i18n("Ctrl+Delete"), Qt::Key_Control + Qt::Key_Delete, trackEditView, SLOT(deleteColumn()));
	mainAccel->insert("key_ins", i18n("Insert column"), i18n("Insert"), Qt::Key_Insert, trackEditView, SLOT(insertColumn()));
	mainAccel->insert("key_period",i18n("Dotted note"), i18n("Period"), Qt::Key_Period, trackEditView, SLOT(dotNote()));
	mainAccel->insert("key_t",i18n("Triplet note"), i18n("T"), Qt::Key_T, trackEditView, SLOT(ntuplet()));
	mainAccel->insert("key_equal",i18n("More duration"), i18n("Equal"), Qt::Key_Equal, trackEditView, SLOT(keyPlus()));
	mainAccel->insert("key_minus",i18n("Less duration"), i18n("Minus"), Qt::Key_Minus, trackEditView, SLOT(keyMinus()));
	
	// Key '0' - '9'
	mainAccel->insert("key_1",i18n("Key 1"), i18n("1"), Qt::Key_1, trackEditView, SLOT(key1()));
	mainAccel->insert("key_2",i18n("Key 2"), i18n("2"), Qt::Key_2, trackEditView, SLOT(key2()));
	mainAccel->insert("key_3",i18n("Key 3"), i18n("3"), Qt::Key_3, trackEditView, SLOT(key3()));
	mainAccel->insert("key_4",i18n("Key 4"), i18n("4"), Qt::Key_4, trackEditView, SLOT(key4()));
	mainAccel->insert("key_5",i18n("Key 5"), i18n("5"), Qt::Key_5, trackEditView, SLOT(key5()));
	mainAccel->insert("key_6",i18n("Key 6"), i18n("6"), Qt::Key_6, trackEditView, SLOT(key6()));
	mainAccel->insert("key_7",i18n("Key 7"), i18n("7"), Qt::Key_7, trackEditView, SLOT(key7()));
	mainAccel->insert("key_8",i18n("Key 8"), i18n("8"), Qt::Key_8, trackEditView, SLOT(key8()));
	mainAccel->insert("key_9",i18n("Key 9"), i18n("9"), Qt::Key_9, trackEditView, SLOT(key9()));
	mainAccel->insert("key_0",i18n("Key 0"), i18n("0"), Qt::Key_0, trackEditView, SLOT(key0()));
	
	playbackTracker = new PlaybackTracker(this);
}

SongView::~SongView()
{
	delete mainAccel;
	delete melodyEditor;
	delete trackEditView;
	delete trackList;
	delete trackPane;
	delete layout;
	delete splitv;
	delete split;
	delete songPrint;
	
	delete nTupletsAct;
	delete ntuplets;
	
	delete midiPlaySongAct;	
	delete midiStopPlayAct;
	delete tremoloPickingAct;
	delete muffledStringsAct;
	delete pickSlideAct;
	delete tremoloBarAct;
	delete tappingAct;
	delete trillAct;
	delete wideVibratoAct;
	delete vibratoAct;
	
	delete unisonBend;
	delete fullPreBendRelease;
	delete halfPreBendRelease;
	delete fullPreBendAct;
	delete halfPreBendAct;
	delete fullBendReleaseAct;
	delete halfBendReleaseAct;
	delete fullBendAct;
	delete halfBendAct;
	delete tripletAct;
	delete palmMuteAct;
	delete artHarmAct;
	delete natHarmAct;
	delete letRingAct;
	delete slideAct;
	delete legatoAct;
	delete arcAct;
	delete timeSigAct;
	delete keySigAct;
	
	delete lenRestAct;
	delete lenDottedAct;
	delete len32Act;
	delete len16Act;
	delete len8Act;
	delete len4Act;
	delete len2Act;
	delete len1Act;
	
	delete deleteBarAct;
	delete insertBarAct;
	delete arrTrkAct;
	delete insChordAct;
	delete rhythmerAct;
	delete trkPropAct;
	delete trkBassLineAct;
	delete trkDeleteAct;
	delete trkNewAct;
	
	delete playbackTracker;
}

// Refreshes all the views and resets all minor parameters in the
// song. Should be called every time when the song's just got loaded
// or imported.
void SongView::refreshView()
{
	trackEditView->setCurrentTrack(first());
	trackList->updateList();
}

void SongView::drawBackground()
{
	melodyEditor->drawBackground();
}

void SongView::viewScore(bool i)
{
	trackEditView->viewScore(i);
}

void SongView::showMelodyEditor(bool i)
{
	if (i)
		melodyEditor->show();
	else
		melodyEditor->hide();
}

// Creates a new track in the song
//
bool SongView::trackNew()
{
	uint freeChannel = 0;
	
	TabTrack* newTrack = new TabTrack(FretTab, "", freeChannel, 0, 25, 6, 24);
	
	// Special case - if user declined track properties dialog during
	// track creation, then he doesn't seem to want the new track, so
	// we'll destroy it.
	//
	if (!setTrackProperties(*newTrack)) {
		delete newTrack;
		return false;
	}
	
	cmdHist->addCommand(new AddTrackCommand(*this, *newTrack, *trackEditView, *trackList, *trackPane));
	emit statusBar("Track added");
	
	return true;
}

// Deletes the currently selected track in the song
//
void SongView::trackDelete()
{
	// Check that we won't delete the only last track in the list
	if (getFirst() != getLast()) {
		cmdHist->addCommand(new DeleteTrackCommand(*this, *trackEditView, *trackList, *trackPane));
		emit statusBar("Track deleted");
	}
}

// Sets current track's properties
//
bool SongView::trackProperties()
{
	bool res = false;
	
	TabTrack *newtrk = new TabTrack(*(trackEditView->getTrack()));
	SetTrack *st = new SetTrack(*newtrk, *this);
	
	if (st->exec()) {
		newtrk->setName(st->title->text());
		newtrk->setChannel(st->channel->value());
		newtrk->setBank(st->bank->value());
// 		newtrk->setPatch(st->patch->value());
		newtrk->setPatch(st->patch->currentItem());
		newtrk->setMode((TrackMode) st->mode->currentItem());
		
		newtrk->setVolume(st->tabmidi->SliderVolume->value());
		newtrk->setBalance(st->tabmidi->SliderPan->value() + 8);
		newtrk->setChorus(st->tabmidi->SliderChorus->value());
		newtrk->setReverb(st->tabmidi->SliderReverb->value());
		
		// Fret tab
		//
		if (st->mode->currentItem() == FretTab) {
			SetTabFret *fret = (SetTabFret *) st->modespec;
			
			newtrk->setNbStrings(fret->string());
			newtrk->setNbFrets( fret->frets());
			
			for (int i = 0; i < newtrk->nbStrings(); i++)
				newtrk->setTune(i, fret->tune(i));
		}
		
		// Drum tab
		//
		if (st->mode->currentItem() == DrumTab) {
			SetTabDrum *drum = (SetTabDrum *) st->modespec;
			newtrk->setNbStrings(drum->drums());
			newtrk->setNbFrets( 0);
			
			for (int i = 0; i < newtrk->nbStrings(); i++)
				newtrk->setTune(i, drum->tune(i));
		}
		
		cmdHist->addCommand(new SetTrackPropCommand(*trackEditView, *trackList, *trackPane, *(trackEditView->getTrack()), *newtrk));
		
		emit statusBar("Track properties changed");
		
		res = true;
	}
	
	delete st;
	delete newtrk;
	
	return res;
}

// Generates a new track with a basic bass line, derived from current
// track's rhythm
//
void SongView::trackBassLine()
{
/*	const TabTrack& currentTrack = *trackEditView->getTrack();
	
	if (currentTrack.mode() == DrumTab) {
		KMessageBox::sorry(this, i18n("Can't generate a bass line from drum track"));
		return;
	}
	
	if (trackNew() == true) {
		TabTrack& newTrack = *trackEditView->getTrack();
// 		newTrack.tabColumnResize(currentTrack.getTabColumnSize());
		ChordSelector cs(currentTrack);
		
		int note;
		
		for (uint i = 0; i < currentTrack.getTabColumnSize(); i++) {
			for (uint k = 0; k < currentTrack.nbStrings(); k++)
				cs.setApp(k, currentTrack.getVectTabColumn(i).getNbFret(k));
				
				cs.detectChord();
				
				if ((ChordListItem *)cs.getChordList().item(0)) {
					note = ((ChordListItem *)cs.getChordList().item(0))->tonic();
					kdDebug() << "Column " << i << ", detected tonic " << Settings::noteName(note) << endl;
				} else {
					note = -1;
					kdDebug() << "Column " << i << ", EMPTY " << endl;
				}
				
				TabColumn tab = newTrack.getVectTabColumn(i);
				
				for (uint k = 0; k < MAX_STRINGS; k++) {
					tab.setNbFret(k, -1);
					tab.setEffect(k, 0);
				}
	
				tab.setDuration(currentTrack.getVectTabColumn(i).getDuration());
				tab.setFlags(currentTrack.getVectTabColumn(i).getFlags());
				
				// GREYFIX: make a better way of choosing a fret. This way
				// it can, for example, be over max frets number.
				if (note >= 0) {
					tab.setNbFret(0, note - newTrack.tune(0) % 12);
	
					if (newTrack.getVectTabColumn(i).getNbFret(0) < 0)
						tab.setNbFret(0, tab.getNbFret(0) + 12);
				}
				
				newTrack.setVectTabColumn(i, tab);
		}
	}
	
	trackEditView->arrangeTracks();*/
}

// Sets track's properties called from trackNew
//
bool SongView::setTrackProperties(TabTrack& track)
{
	bool res = false;
	SetTrack *st = new SetTrack(track, *this);
	
	if (st->exec()) {
		track.setName(st->title->text());
		track.setChannel(st->channel->value());
		track.setBank(st->bank->value());
// 		track.setPatch(st->patch->value());
		track.setPatch(st->patch->currentItem());
		track.setMode((TrackMode) st->mode->currentItem());
		
		track.setVolume(st->tabmidi->SliderVolume->value());
		track.setBalance(st->tabmidi->SliderPan->value());
		track.setChorus(st->tabmidi->SliderChorus->value());
		track.setReverb(st->tabmidi->SliderReverb->value());
		
		// Fret tab
		if (st->mode->currentItem() == FretTab) {
			SetTabFret *fret = (SetTabFret *) st->modespec;
			track.setNbStrings(fret->string());
			track.setNbFrets(fret->frets());
			
			for (int i = 0; i < trackEditView->getTrack()->nbStrings(); i++)
				track.setTune(i, fret->tune(i));
		}

		// Drum tab
		if (st->mode->currentItem() == DrumTab) {
			SetTabDrum *drum = (SetTabDrum *) st->modespec;
			track.setNbStrings(drum->drums());
			track.setNbFrets(0);
			
			for (int i = 0; i < trackEditView->getTrack()->nbStrings(); i++)
				track.setTune(i, drum->tune(i));
		}
		
// 		trackEditView->selectTrack(trackEditView->getTrack()); // artificially needed to emit newTrackSelected()
/*		trackList->updateList();
		trackPane->updateList();*/
		res = true;
	}
	
	delete st;
	
	return res;
}

// Dialog to set song's properties
void SongView::songProperties()
{
	SetSong ss(*dynamic_cast<TabProperties *>(this));

	if (ss.exec()) {
		cmdHist->addCommand(new SetSongPropCommand(*this, ss.getProperties()));
		emit statusBar("Song properties changed");
	}
}

// Start playing the song or stop it if it already plays
//
void SongView::playSong()
{
	kdDebug() << "playTheSong\n";
	
	if (!scheduler) {
		kdDebug() << "unable to find a midi scheduler\n";
		return ;
	}
	
	trackEditView->beginPlaying();
	trackEditView->stopPlaying();
	
	if (tabSong) {
		tabSong->stopPlaying();
		
		while (tabSong->running())
			;
		
		delete tabSong;
	}
	
	tabSong = new TabSongPlayer(*this, playbackTracker);
	
	tabSong->setScheduler(scheduler);
	tabSong->start();
}

void SongView::stopPlay()
{
	kdDebug() << "stopTheSong\n";
	
	trackEditView->stopPlaying();
	
	if (tabSong) {
		tabSong->stopPlaying();
		
		while (tabSong->running())
			;
		
		delete tabSong;
		
		tabSong = 0;
	}
}

void SongView::slotCut()
{
// 	if (!trackEditView->getTrack()->getSel()) {
// 		KMessageBox::error(this, i18n("There is no selection!"));
// 		return;
// 	}

// 	QApplication::clipboard()->setData(new TrackDrag(highlightedTabs()));
// 	trackEditView->deleteColumn(i18n("Cut to clipboard"));
}

void SongView::slotCopy()
{
	Clipboard clipDlg(trackEditView->getBarPosition() + 1, trackEditView->getBarNumber(), this, "clipboard copy");
	
	if (clipDlg.exec() == QDialog::Accepted) {
		TabTrack tempTrack;
		TabTrack* currentTrack;
		uint from;
		uint to;
		
		currentTrack = trackEditView->getTrack();
		from = clipDlg.getFrom() - 1;
		to = clipDlg.getTo();
		
		for (uint i = from; i < to; i++) {
			TabBar tempBar = currentTrack->bar(i);
			
			tempTrack.addBar(tempBar);
		}
		
		*dynamic_cast<TrackProperties*>(&tempTrack) = *dynamic_cast<TrackProperties*>(currentTrack);
		
		QApplication::clipboard()->setData(new TrackDrag(tempTrack));
		emit statusBar("Bar(s) copy");
	}
}

void SongView::slotPaste()
{
	TabTrack tempTrack;
	
	if (TrackDrag::decode(*QApplication::clipboard()->data(), tempTrack))
		insertTabs(tempTrack);
	
	trackEditView->update();
}

void SongView::slotSelectAll()
{
/*	trackEditView->getTrack()->setXSel(0);
	trackEditView->getTrack()->setTabColumn(trackEditView->getTrack()->getTabColumnSize() - 1);
	trackEditView->getTrack()->setSel(TRUE);

	trackEditView->update();*/
}

TabTrack *SongView::highlightedTabs()
{
/*	if (!trackEditView->getTrack()->getSel())  return NULL;

	TabTrack* trk = trackEditView->getTrack();
	TabTrack* newtrk = new TabTrack(trk->mode(), "ClipboardTrack", trk->getChannel(),
									trk->getBank(), trk->getPatch(), trk->nbStrings(), trk->nbFrets());
									
	for (int i = 0; i < trk->nbStrings(); i++)
		newtrk->setTune(i,trk->tune(i));

	uint pdelta, pstart, pend;

	if (trk->getTabColumn() <= trk->getXSel()) {
		pend = trk->getXSel();
		pstart = trk->getTabColumn();
	} else {
		pend = trk->getTabColumn();
		pstart = trk->getXSel();
	}

	pdelta = pend - pstart + 1;

	newtrk->tabColumnResize(pdelta);
	int _s = pstart;

	for (uint i = 0; i < pdelta; i++) {
		TabColumn tab = newtrk->getVectTabColumn(i);
		
		for (uint k = 0; k < MAX_STRINGS; k++) {
				tab.setNbFret(k, -1);
				tab.setEffect(k, 0);
		}

		tab.setDuration(trk->getVectTabColumn(_s).getDuration());
		tab.setFlags(trk->getVectTabColumn(_s).getFlags());

		for (uint k = 0; k < newtrk->nbStrings(); k++) {
			tab.setNbFret(k, trk->getVectTabColumn(_s).getNbFret(k));
			tab.setEffect(k, trk->getVectTabColumn(_s).getEffect(k));
		}

		newtrk->setVectTabColumn(i, tab);

		_s++;
	}
*/
	return NULL;
}

void SongView::insertTabs(TabTrack& tempTrack)
{
	QString msg;
	bool err = false, errtune = false;
	TabTrack* currentTrack = trackEditView->getTrack();
	
	if (currentTrack->mode() != tempTrack.mode()) {
		msg += i18n("The clipboard data hasn't the same track mode.\n");
		err = true;
	}
	
	if (currentTrack->nbStrings() != tempTrack.nbStrings()) {
		msg += i18n("The clipboard data hasn't the same number of strings.\n");
		err = true;
	} else {
		for (int i = 0; i < currentTrack->nbStrings(); i++) {
			if (currentTrack->tune(i) != tempTrack.tune(i))  errtune = true;
			if (errtune) break;
		}
		
		if (errtune) {
			msg += i18n("The clipboard data hasn't the same tuneing.\n");
			err = true;
		}
	}
	
	if (currentTrack->nbFrets() != tempTrack.nbFrets()) {
		msg += i18n("The clipboard data hasn't the same number of frets.\n");
		err = true;
	}
	
	if (err) {
		msg += i18n("\n\nI'll improve this code. So some of these problems\n");
		msg += i18n("will be solved in the future.");
		KMessageBox::error(this, msg);
		return;
	}
	
	cmdHist->addCommand(new InsertTabsCommand(*trackEditView, *currentTrack, tempTrack, trackEditView->getBarPosition()));
	emit statusBar("Bar(s) paste");
}

void SongView::print(KPrinter* printer)
{
	printer->setFullPage(true);
	
	QMessageBox::information( this, "KTabEdit",
	"The printing support is not yet finished.\n"
	"Don't print a document in your printer but in a pdf or ps file instead !" );
	   
	if (printer->setup(this))
		songPrint->printSong(printer, this);
}

/*
	Move the cursor to the position and refresh the view.
*/
void SongView::playbackCursor(int track, const TrackPos& position)
{
	TabTrack *trk = at(track);
	
	if (trackEditView->getTrack() == trk) {
		trackEditView->setCursor(position);
		kapp->postEvent(trackEditView, new QKeyEvent(QEvent::KeyRelease, 0, 0, 0));
	}
}

const TrackEditView* SongView::getTrackEditView() const
{
	return trackEditView;
}
