/***************************************************************************
 * songviewcommands.cpp: implementation of SetSongPropCommand class
 *                                         SetTrackPropCommand
 *                                         InsertTabsCommand
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

/*
  Undo/Redo commands for SongView
*/
#include "songviewcommands.h"
#include "tabsong.h"
#include "tabproperties.h"
#include "tabtrack.h"
#include "trackeditview.h"
#include "tracklist.h"
#include "trackpane.h"
#include "trackproperties.h"


SongView::AddTrackCommand::AddTrackCommand(TabSong& tabSong, TabTrack& tabTrack, TrackEditView& tv, TrackList& list, TrackPane& pane)
: KNamedCommand(i18n("Add track")), newTabSong(tabSong), oldTrack(*tv.getTrack()), newTrack(tabTrack),
  trackView(tv), trackList(list), trackPane(pane)
{
}

void SongView::AddTrackCommand::execute()
{
	newTabSong.append(&newTrack);
	
	trackView.setCurrentTrack(&newTrack);
	
	trackList.updateList();
	trackPane.updateList();
}

void SongView::AddTrackCommand::unexecute()
{
	trackView.setCurrentTrack(&oldTrack);
	newTabSong.removeLast();
	
	trackList.updateList();
	trackPane.updateList();
}

SongView::DeleteTrackCommand::DeleteTrackCommand(TabSong& tabSong, TrackEditView& tv, TrackList& list, TrackPane& pane)
: KNamedCommand(i18n("Delete track")), newTabSong(tabSong), oldTrack(*tv.getTrack()), trackView(tv),
  trackList(list), trackPane(pane)
{
}

void SongView::DeleteTrackCommand::execute()
{
	TabTrack* newTrack;
	
	// If we delete the last track, make sure we'll get the item
	//
	if (newTabSong.last() == trackView.getTrack()) {
		newTrack = newTabSong.prev();
	} else {
		newTabSong.findRef(trackView.getTrack());
		newTrack = newTabSong.next();
	}
	
	newTabSong.remove(trackView.getTrack());
	trackView.setCurrentTrack(newTrack);
	
	trackList.updateList();
	trackPane.updateList();
}

void SongView::DeleteTrackCommand::unexecute()
{
	TabTrack* newTrack = new TabTrack(oldTrack);
	
	newTabSong.append(newTrack);
	trackView.setCurrentTrack(newTrack);
	
	trackList.updateList();
	trackPane.updateList();
}

SongView::SetSongPropCommand::SetSongPropCommand(SongView &_song, const TabProperties &_prop)
: KNamedCommand(i18n("Set song properties")), song(_song), oldProperties(_song),
  newProperties(_prop)
{
}

void SongView::SetSongPropCommand::execute()
{
	TabProperties *prop1 = &song;
	TabProperties *prop2 = &newProperties;
	
	*prop1 = *prop2;
	
	emit song.songChanged();
}

void SongView::SetSongPropCommand::unexecute()
{
	TabProperties *prop1 = &song;
	TabProperties *prop2 = &oldProperties;
	
	*prop1 = *prop2;
	
	emit song.songChanged();
}

SongView::SetTrackPropCommand::SetTrackPropCommand(TrackEditView &_tv, TrackList &_tl, TrackPane &_tp, TabTrack &_trk, TabTrack &_newtrk)
: KNamedCommand(i18n("Set track properties")), oldTrack(_trk), currentTrack(_trk), newTrack(_newtrk),
  trackView(_tv), trackList(_tl), trackPane(_tp), string(trackView.string())
{
}

void SongView::SetTrackPropCommand::execute()
{
	TrackProperties *properties1 = dynamic_cast<TrackProperties *>(&currentTrack);
	TrackProperties *properties2 = dynamic_cast<TrackProperties *>(&newTrack);
	
	*properties1 = *properties2;
	properties1->setNbStrings(properties2->nbStrings());
	properties1->setNbFrets(properties2->nbFrets());
	
	trackView.setString(properties2->nbStrings() - 1);
	
// 	trackView.drawAllTrack();
	trackList.updateList();
	trackPane.updateList();
}

void SongView::SetTrackPropCommand::unexecute()
{
	TrackProperties *oldProperties = dynamic_cast<TrackProperties *>(&currentTrack);
	TrackProperties *newProperties = dynamic_cast<TrackProperties *>(&oldTrack);
	
	*oldProperties = *newProperties;
	oldProperties->setNbStrings(newProperties->nbStrings());
	oldProperties->setNbFrets(newProperties->nbFrets());
	
	currentTrack = oldTrack;
	
	trackView.setString(string);
// 	trackView.drawAllTrack();
	trackList.updateList();
	trackPane.updateList();
}

SongView::InsertTabsCommand::InsertTabsCommand(TrackEditView& tv, TabTrack& trk, TabTrack& tabs, uint bar)
: KNamedCommand(i18n("Insert from clipboard")), barPosition(bar), currentTrack(trk), tempTrack(tabs),
  oldTrack(trk), trackView(tv)
{
}

void SongView::InsertTabsCommand::execute()
{
	for (uint i = 0; i < tempTrack.count(); i++) {
		TabBar tempBar = tempTrack.bar(i);
		currentTrack.insertBar(barPosition + i, tempBar);
	}
	
// 	trackView.drawAllTrack();
}

void SongView::InsertTabsCommand::unexecute()
{
	currentTrack = oldTrack;
	
// 	trackView.drawAllTrack();
}
