#include "playbacktracker.h"
#include "settings.h"
#include "tabsongplayer.h"

TabSongPlayer::TabSongPlayer(const TabSong& tempSong, PlaybackTracker* tracker)
: scheduler(0),
  playbackTracker(tracker),
  stop(false)
{
	TabSong* thisSong = this;
	*thisSong = tempSong;
}

void TabSongPlayer::setScheduler(TSE3::MidiScheduler* midiScheduler)
{
	scheduler = midiScheduler;
}

void TabSongPlayer::stopPlaying()
{
	stop = true;
}

inline long TabSongPlayer::calcDuration(const TabTimes& times) const
{
	long temp = 0;
	const Duration duration = times.duration();
	
	switch (duration) {
	case NoneDuration:
		break;
	case SixtyFourth:
		temp = (TSE3::Clock::PPQN / 32);
		break;
	case HundredTwentyEighth:
		temp = (TSE3::Clock::PPQN / 16);
		break;
	case ThirthSecond:  // 1/32, triple croche
		temp = (TSE3::Clock::PPQN / 8);
		break;
	case Sixteenth:  // 1/16, double croche
		temp = (TSE3::Clock::PPQN / 4);
		break;
	case Eighth:  // 1/8, croche
		temp = (TSE3::Clock::PPQN / 2);
		break;
	case Quarter:  // 1/4 - a long vertical line, so we need to find the highest note, noire
		temp = TSE3::Clock::PPQN;
		break;
	case Half: // 1/2, blanche
		temp = (2 * TSE3::Clock::PPQN);
		break;
	case Whole: // whole, ronde
		temp = (4 * TSE3::Clock::PPQN);
		break;
	}
	
	if (times.isDotted() == true)
		temp = temp + temp / 2;
	
	// For the triplet double time duration
	//
	if (times.nTuplet() == 2)
		temp*=2;
	else if (times.nTuplet() > 2 && times.nTuplet() < 8)
		temp*=4;
	else if (times.nTuplet() > 8 && times.nTuplet() < 12)
		temp*=8;
	
	return temp / ((int)times.nTuplet() + 1);
}

inline long TabSongPlayer::calcLink(uint barNb, uint timeNb, uint chord, const TabTrack& tempTrack) const
{
	long time = 0;
	long restTime = 0;
	
	// Check in the next time if the next note is linked with this note
	//
	for (uint i = barNb; i < tempTrack.count(); i++) {
		const TabBar tempBar = tempTrack.bar(i);
		
		for (uint j = (i == barNb ? timeNb + 1 : 0); j < tempBar.count(); j++) {
			const TabTimes tempTimes = tempBar.times(j);
			
			if (tempTimes.notes(chord).isTieNote()) {
				time+=calcDuration(tempTimes) + restTime;
				restTime = 0;
			} else
				if (tempTimes.isRest())
					restTime+=calcDuration(tempTimes);
				else
					return time;
		}
	}
	
	return time;
}

inline bool TabSongPlayer::checkLinkAndRest(uint barNb, uint timeNb, uint chord, const TabTrack& tempTrack) const
{
	// Check in the next time if the next note is linked with this note
	//
	for (uint i = barNb; i < tempTrack.count(); i++) {
		const TabBar tempBar = tempTrack.bar(i);
		
		for (uint j = (i == barNb ? timeNb + 1 : 0); j < tempBar.count(); j++) {
			const TabTimes tempTimes = tempBar.times(j);
			
			if (tempTimes.notes(chord).isTieNote())
				return false;
			else if (!tempTimes.isRest())
					return true;
		}
	}
	
	return false;
}

void TabSongPlayer::run()
{
	if (!scheduler)
		return ;
	
	TSE3::PhraseEdit phraseEdit;
	uint i, j, k, l, pos;
	long countTime;
	QValueVector<long> time(0);
	QValueVector<long> nextTime(0);
	long lastTime = 0;
	int note = 0;
	bool drumTrack = false;
	uint drumTrackNb = 0;
	
	// Check if there is a drum track
	//
	for (i = 0; i < count(); i++) {
		TabTrack* tempTrack = at(i);
		
		if (tempTrack->mode() == DrumTab) {
			drumTrack = true;
			drumTrackNb = i;
			
			break;
		}
	}
	
	for (i = 0; i < count(); i++) {
		TabTrack* tempTrack = at(i);
		
		pos = i;
		
		if (drumTrack) {
			if (drumTrackNb != 9) {
				if (drumTrackNb < 9) {
					if (i == drumTrackNb)
						pos = 9;
					else if (i > drumTrackNb && i < 9)
						pos = i - 1;
					else if (i >= 9)
						pos = i + 1;
				} else {
					if (i == drumTrackNb)
						pos = 9;
					else if (i >= 9 && i < drumTrackNb)
						pos = i + 1;
				}
			}
		}
		
		// Select the instrument
		//
		phraseEdit.insert(TSE3::MidiEvent(
					TSE3::MidiCommand(
						TSE3::MidiCommand_ProgramChange, 
						pos, Settings::midiPortNumber(), tempTrack->patch())
					, 0)
				);
		
		phraseEdit.insert(TSE3::MidiEvent(
					TSE3::MidiCommand(
						TSE3::MidiCommand_ControlChange,
						pos, Settings::midiPortNumber(), TSE3::MidiControl_ReverbDepth, tempTrack->reverb())
					, 0)
		);
		
		phraseEdit.insert(TSE3::MidiEvent(
					TSE3::MidiCommand(
						TSE3::MidiCommand_ControlChange,
						pos, Settings::midiPortNumber(), TSE3::MidiControl_PanMSB, tempTrack->balance())
					, 0)
		);
		
		phraseEdit.insert(TSE3::MidiEvent(
					TSE3::MidiCommand(
						TSE3::MidiCommand_ControlChange,
						pos, Settings::midiPortNumber(), TSE3::MidiControl_ChorusDepth, tempTrack->chorus())
					, 0)
		);
		
		phraseEdit.insert(TSE3::MidiEvent(
					TSE3::MidiCommand(
						TSE3::MidiCommand_ControlChange,
						pos, Settings::midiPortNumber(), TSE3::MidiControl_ChannelVolumeMSB, tempTrack->volume() + 20)
					, 0)
		);
	}
	
	for (j = 0; j < count(); j++) {
		TabTrack* tempTrack = at(j);
		
		countTime = 0;
		pos = j;
		
		if (drumTrack) {
			if (j == drumTrackNb)
				pos = 9;
			else if (j > drumTrackNb && j < 9)
				pos = j - 1;
			else if (j >= 9)
				pos = j + 1;
		}
		
		time.resize(tempTrack->nbStrings());
		nextTime.resize(tempTrack->nbStrings());
		
		for (i = 0; i < tempTrack->nbStrings(); i++) {
			time[i] = nextTime[i] = 0;
		}
		
		for (k = 0; k < tempTrack->count(); k++) {
			const TabBar tempBar = tempTrack->bar(k);
			
			for (l = 0; l < tempBar.count(); l++) {
				const TabTimes tempTimes = tempBar.times(l);
				
				// Add duration for the others chords that aren't played
				//
				for (i = 0; i < tempTrack->nbStrings(); i++) {
					// Don't add linked note and rest between
					// this note and the linked note
					//
					if ((!tempTimes.notes(i).isTieNote() && !tempTimes.isRest()) || (tempTimes.isRest() && checkLinkAndRest(k, l, i, *tempTrack)))
						nextTime[i]+=calcDuration(tempTimes);
				}
				
				if (tempTimes.isRest() == false) {
					// Add the notes to be played
					//
					for (i = 0; i < tempTrack->nbStrings(); i++) {
						
						if (!tempTimes.notes(i).isEmpty() && !tempTimes.notes(i).isTieNote()) {
							if (drumTrack && pos == 9)
								note = tempTimes.notes(i).fret();
							else
								note = tempTrack->tune(tempTrack->nbStrings() - i - 1) + tempTimes.notes(i).fret();
							
							nextTime[i]+=calcLink(k, l, i, *tempTrack);
							
							phraseEdit.insert(TSE3::MidiEvent(
								TSE3::MidiCommand(
									TSE3::MidiCommand_NoteOn,
									pos, Settings::midiPortNumber(), note, 0x60)
								, time[i])
							);
							
							phraseEdit.insert(TSE3::MidiEvent(
								TSE3::MidiCommand(
									TSE3::MidiCommand_NoteOff,
									pos, Settings::midiPortNumber(), note, 0x60)
								, nextTime[i] - 1)
							);
						
						}
					}
				}
				
				time = nextTime;
				
				for (i = 0; i < tempTrack->nbStrings(); i++) {
					// Check the greatter last time
					//
					if (lastTime < time[i])
						lastTime = time[i];
				}
				
				// Event for the time tracking
				//
				countTime+=calcDuration(tempTimes);
				
				phraseEdit.insert(TSE3::MidiEvent(
							TSE3::MidiCommand(
								PlaybackTracker::playBackTrackCmd,
								pos, Settings::midiPortNumber(), 0x0, 0x0)
							, countTime)
						);
			}
		}
	}
	
	for (i = 0; i < count(); i++) {
		pos = i;
		
		if (drumTrack) {
			if (i == drumTrackNb)
				pos = 9;
			else if (i > drumTrackNb && i < 9)
				pos = i - 1;
			else if (i >= 9)
				pos = i + 1;
		}
		
		phraseEdit.insert(TSE3::MidiEvent(
			TSE3::MidiCommand(
				TSE3::MidiCommand_NoteOn, 
				pos, Settings::midiPortNumber(), 0, 0)
			, lastTime, 0, lastTime + TSE3::Clock::PPQN)
		);
	}
	
	// Assemble the song
	//
	TSE3::Song    song(1);
	
	song.tempoTrack()->insert(TSE3::Event<TSE3::Tempo>(TSE3::Tempo(tempo()), 0));
	
	TSE3::Phrase *phrase = phraseEdit.createPhrase(song.phraseList());
	TSE3::Part   *part   = new TSE3::Part(0, phraseEdit.lastClock());
	part->setPhrase(phrase);
	song[0]->insert(part);
	
	TSE3::Metronome metronome;
	TSE3::Transport transport(&metronome, scheduler);
	
	transport.attachCallback(playbackTracker);
	
	// Set the position of the cursor to (0, 0, 0)
	//
	playbackTracker->resetCursor();
	
	// PLay the song
	//
	transport.play(&song, 0);
	
	do {
		transport.poll();
		
		if (stop)
			transport.stop();
	} while(transport.status() != TSE3::Transport::Resting && !stop);
	
	if (stop)
		transport.stop();
}
