#include "global.h"
#include "settings.h"
#include "trackproperties.h"
#include "trackbarview.h"
#include "trackbarlineview.h"
#include "trackclassicalclefview.h"
#include "trackdurationview.h"
#include "trackpos.h"
#include "tracktabclefview.h"
#include "tracktimesview.h"
#include "tracktimesclassicalview.h"
#include "tracktimesignatureview.h"

bool TrackBarView::DefaultFontInit = false;
QFont TrackBarView::DefaultTextFont;
QFontMetrics TrackBarView::DefaultTextFontInfo(DefaultTextFont);
int TrackBarView::DefaultTextWidth = 0;
int TrackBarView::DefaultTextHeight = 0;

TrackBarView::TrackBarView(uint nb, const TabBar& bar, const TrackProperties& properties, uint strings, int resolutionDPI, QCanvas* canvas)
: Unit(resolutionDPI),
  WholeSize(0.0),
  HalfSize(0.0),
  QuarterSize(0.0),
  EighthSize(0.0),
  SixteenthSize(0.0),
  ThirthSecondSize(0.0),
  SixtyFourthSize(0.0),
  HundredTwentyEighthSize(0.0),
  canvasView(canvas),
  tabBar(bar),
  trackProperties(properties),
  nbStrings(strings),
  clef(false),
  armor(false),
  time(false),
  tempo(false),
  minimalBarSize(0),
  barSize(0),
  barReduce(0),
  partitionView(0),
  tablatureView(0),
  classicalClefView(0),
  tablatureClefView(0),
  timeSignatureView(0),
  partition(false),
  partitionClef(false),
  tablature(false),
  tablatureClef(false),
  displayBarNumberView(false),
  barNumber(nb),
  barNumberView(0),
  textFont(DefaultTextFont),
  textFontInfo(DefaultTextFontInfo),
  textWidth(DefaultTextWidth),
  textHeight(DefaultTextHeight),
  effects(true)
{
	if (DefaultFontInit == false) {
		textFont = DefaultTextFont = QFont(Settings::textAndTabFont(), Settings::textAndTabSize(), Settings::textAndTabWeight(), 
										   Settings::textAndTabItalic());
		
		textFontInfo = DefaultTextFontInfo = QFontMetrics(DefaultTextFont);
		textWidth = DefaultTextWidth = DefaultTextFontInfo.boundingRect("8").width();
		textHeight = DefaultTextHeight = DefaultTextFontInfo.boundingRect("8").height();
		DefaultFontInit = true;
	}
	
	textWidth = convertDPIPixel(DefaultTextWidth, 75);
	textHeight = convertDPIPixel(DefaultTextHeight, 75);
	
	linkView[0] = linkView[1] = 0;
	
	fretView.setAutoDelete(true);
	durationView.setAutoDelete(true);
	noteClassicalView.setAutoDelete(true);
	durationClassicalView.setAutoDelete(true);
	
	calcDurationSize();
}

TrackBarView::TrackBarView(uint nb, const TabBar& bar, const TrackProperties& properties, uint strings, const QFont& font, int resolutionDPI, QCanvas* canvas)
: Unit(resolutionDPI),
  WholeSize(0.0),
  HalfSize(0.0),
  QuarterSize(0.0),
  EighthSize(0.0),
  SixteenthSize(0.0),
  ThirthSecondSize(0.0),
  SixtyFourthSize(0.0),
  HundredTwentyEighthSize(0.0),
  canvasView(canvas),
  tabBar(bar),
  trackProperties(properties),
  nbStrings(strings),
  clef(false),
  armor(false),
  time(false),
  tempo(false),
  minimalBarSize(0),
  barSize(0),
  partitionView(0),
  tablatureView(0),
  tablatureClefView(0),
  timeSignatureView(0),
  partition(false),
  partitionClef(false),
  tablature(false),
  tablatureClef(false),
  displayBarNumberView(false),
  barNumber(nb),
  barNumberView(0),
  textFont(font),
  textFontInfo(font),
  textWidth(convertDPIPixel(textFontInfo.boundingRect("8").width(), 75)),
  textHeight(convertDPIPixel(textFontInfo.boundingRect("8").height(), 75)),
  effects(true)
{
	if (DefaultFontInit == false) {
		DefaultTextFont = QFont(Settings::textAndTabFont(), Settings::textAndTabSize(), Settings::textAndTabWeight()
				        , Settings::textAndTabItalic());
		DefaultTextFontInfo = QFontMetrics(DefaultTextFont);
		DefaultTextWidth = DefaultTextFontInfo.boundingRect("8").width();
		DefaultTextHeight = DefaultTextFontInfo.boundingRect("8").height();
		DefaultFontInit = true;
	}
	
	linkView[0] = linkView[1] = 0;
	fretView.setAutoDelete(true);
	durationView.setAutoDelete(true);
	
	calcDurationSize();
}

TrackBarView::~TrackBarView()
{
	if (barNumberView)
		delete barNumberView;
	
	if (partition)
		delete partitionView;

	if (partitionClef)
		delete classicalClefView;
	
	if (partition && tablature) {
		if (linkView[0]) {
			delete linkView[0];
			delete linkView[1];
		}
	}
	
	if (tablatureView)
		delete tablatureView;
	
	if (tablatureClefView)
		delete tablatureClefView;
	
	if (timeSignatureView)
		delete timeSignatureView;
}

// Calc the size of the duration in this bar. Check the minimal
// duration time and adapt the size
// 
//
void TrackBarView::calcDurationSize()
{
	// Check if there is hundredtwentyeighth or sixtyfourth or thirthsecond or sixteenth
	//
	Duration smallDuration = Whole;
	
	for (uint i = 0; i < tabBar.count(); i++) {
		Duration d = tabBar.times(i).duration();
		
		if (d != NoneDuration && d < smallDuration)
			smallDuration = d;
	}
	
	float fontSize = pixelToCm<float>(static_cast<int>(textWidth * 1.5));
	float step     = fontSize * 1.20;

	switch (smallDuration) {
	case NoneDuration:
	case Whole:
	case Half:
	case Quarter:
		QuarterSize = fontSize * 2.5;
		HalfSize = QuarterSize + step;
		WholeSize = HalfSize + step;
		break;
	case Eighth:
		EighthSize = fontSize * 2.5;
		QuarterSize = EighthSize + step;
		HalfSize = QuarterSize + step;
		WholeSize = HalfSize + step;
		break;
	case Sixteenth:
		SixteenthSize = fontSize * 2.5;
		EighthSize = SixteenthSize + step;
		QuarterSize = EighthSize + step;
		HalfSize = QuarterSize + step;
		WholeSize = WholeSize + step;
		break;
	case ThirthSecond:
		ThirthSecondSize = fontSize * 2.5;
		SixteenthSize = ThirthSecondSize + step;
		EighthSize = SixteenthSize + step;
		QuarterSize = EighthSize + step;
		HalfSize = QuarterSize + step;
		WholeSize = HalfSize + step;
		break;
	case SixtyFourth:
		SixtyFourthSize = fontSize * 2.5;
		ThirthSecondSize = SixtyFourthSize + step;
		SixteenthSize = ThirthSecondSize + step;
		EighthSize = SixteenthSize + step;
		QuarterSize = EighthSize + step;
		HalfSize = QuarterSize + step;
		WholeSize = HalfSize + step;
		break;
	case HundredTwentyEighth:
		HundredTwentyEighthSize = fontSize * 2.5;
		SixtyFourthSize = HundredTwentyEighthSize + step;
		ThirthSecondSize = SixtyFourthSize + step;
		SixteenthSize = ThirthSecondSize + step;
		EighthSize = SixteenthSize + step;
		QuarterSize = EighthSize + step;
		HalfSize = QuarterSize + step;
		break;
	}
}

uint TrackBarView::clefSize() const
{
	uint size = 0;
	
	// Calc the clef size
	//
	if (clef) {
		size = cmToPixel<uint>(0.4);
		
		// Space between the clef and either the time signature
		// either the first note
		//
		size+=cmToPixel<uint>(0.4);
	}
	
	return size;
}

uint TrackBarView::armorSize() const
{
	uint size = 0;
	
	// Calc the size of the armor
	//
	if (armor) {
		char armorNb = tabBar.key();
		
		if (armorNb < 0)
			armorNb = -armorNb;
		
		size = (cmToPixel<uint>(0.1) * armorNb) + cmToPixel<uint>(0.1);
	}
	
	return size;
}

uint TrackBarView::timesSize() const
{
	uint size = 0;
	
	for (uchar i = 0; i < tabBar.count(); i++)
		size+=timeSpace(tabBar.times(i));
	
	return size;
}

uint TrackBarView::timeSpace(const TabTimes& times) const
{
	float timeSize = 0.0;
	
	switch (times.duration()) {
	case NoneDuration:
		break;
	case Whole:
		timeSize = WholeSize;
		break;
	case Half:
		timeSize = HalfSize;
		break;
	case Quarter:
		timeSize = QuarterSize;
		break;
	case Eighth:
		timeSize = EighthSize;
		break;
	case Sixteenth:
		timeSize = SixteenthSize;
		break;
	case ThirthSecond:
		timeSize = ThirthSecondSize;
		break;
	case SixtyFourth:
		timeSize = SixtyFourthSize;
		break;
	case HundredTwentyEighth:
		timeSize = HundredTwentyEighthSize;
		break;
	default:
		break;
	}
	
	if (times.isDotted())
		timeSize+=(timeSize / 2.0);
	
	return cmToPixel<uint>(timeSize) + ((barSize - minimalBarSize) / tabBar.count()) - (barReduce / tabBar.count());
}

double TrackBarView::dtimesSize() const
{
	double size = 0.0;
	
	for (uchar i = 0; i < tabBar.count(); i++)
		size+=dtimeSpace(tabBar.times(i));
	
	return size;
}

double TrackBarView::dtimeSpace(const TabTimes& times) const
{
	double timeSize = 0.0;
	
	switch (times.duration()) {
	case NoneDuration:
		break;
	case Whole:
		timeSize = WholeSize;
		break;
	case Half:
		timeSize = HalfSize;
		break;
	case Quarter:
		timeSize = QuarterSize;
		break;
	case Eighth:
		timeSize = EighthSize;
		break;
	case Sixteenth:
		timeSize = SixteenthSize;
		break;
	case ThirthSecond:
		timeSize = ThirthSecondSize;
		break;
	case SixtyFourth:
		timeSize = SixtyFourthSize;
		break;
	case HundredTwentyEighth:
		timeSize = HundredTwentyEighthSize;
		break;
	default:
		break;
	}
	
	if (times.isDotted())
		timeSize+=(timeSize / 2.0);
	
	return cmToPixel<double>(timeSize) + ((barSize - minimalBarSize) / static_cast<double>(tabBar.count())) - (barReduce / 
		   static_cast<double>(tabBar.count()));
}

// Calc the minimal size of a bar
//
uint TrackBarView::minimalSize()
{
	if (minimalBarSize)
		return minimalBarSize;
	
	double dTablatureSize = 0.0;
	uint tablatureSize = 0;
	uint partitionSize = 0;
	
	if (tablature) {
		// Space between the separator and the word "TAB"
		//
		tablatureSize = cmToPixel<uint>(0.1);
		dTablatureSize = cmToPixel<double>(0.1);
		
		if (tablatureClef && tablatureClefView) {
			tablatureSize+=static_cast<int>(tablatureClefView->width()) + cmToPixel<uint>(0.1);
			dTablatureSize+=tablatureClefView->width() + cmToPixel<double>(0.1);
		}
		
		if (time && timeSignatureView) {
			tablatureSize+=static_cast<int>(timeSignatureView->width()) + cmToPixel<uint>(0.1);
			dTablatureSize+=timeSignatureView->width() + cmToPixel<double>(0.1);
		}
		
		if (tablatureClef && !time) {
			tablatureSize+=cmToPixel<uint>(0.1);
			dTablatureSize+=cmToPixel<double>(0.1);
		}
		
		tablatureSize+=timesSize();
		dTablatureSize+=dtimesSize();
	}
	
	if (partition) {
/*		partitionSize = cmToPixel<uint>(0.1);
		
		if (tablatureClef && tablatureClef)
			partitionSize+=tablatureClefView->width() + cmToPixel<uint>(0.1);
		
		if (time)
			partitionSize+=timeSignatureView->width() + cmToPixel<uint>(0.1);
		
		partitionSize+=clefSize();
		
		partitionSize+=armorSize();
		
		partitionSize+=timeSignatureSize();
		
		partitionSize+=timesSize();*/
	}
	
	return minimalBarSize = (tablatureSize < partitionSize ? partitionSize : static_cast<uint>(dTablatureSize + 0.5));
}

void TrackBarView::reduceToWidth(uint width)
{
	if (width >= minimalSize())
		return ;
	
	barReduce = minimalSize() - width;
}

uint TrackBarView::height() const
{
	uint height = 0;
	
	if (partition)
		height = cmToPixel<uint>(0.14) * nbStrings;
	
	if (tablature)
		height+=cmToPixel<uint>(0.26) * nbStrings;
	
	if (tablature && partition)
		height+=cmToPixel<uint>(1.7);
	
	return height;
}

// draw bar lines at xpos,ypos width w for all strings of track trk
//
void TrackBarView::draw(QPoint position)
{
	double durationCount = 0;
	barPosition = position;
	
	// Display the bar number upside the partition and the tablature
	//
	if (displayBarNumberView) {
		QString temp;
		
		barNumberView->move(position.x(), position.y() - 2 * textHeight);
		barNumberView->show();
	}
	
	// Draw the partition
	//
	if (partition) {
		uint lineHeight = partitionView->lineSpace();

		partitionView->draw(position, barSize - barReduce);
		
		// FIXME used to accel the printing
		//
		if (effects) {
			if (partitionClef)
				classicalClefView->draw(position);
		}
		
		position.setY(position.y() + partitionView->height());
		
		QPoint tempPoint(position);
		
		// FIXME used to accel the printing
		//
		if (effects) {
		// Space between the clef and the first note
		//
		position.setX(static_cast<int>(position.x() + classicalClefView->width() + cmToPixel(0.2)));
		}
		
		// Display each time in the bar
		//
		for (uint i = 0; i < tabBar.count(); i++) {
			TabTimes tempTimes = tabBar.times(i);
			uint space = timeSpace(tempTimes);
			QPoint temp(position);
			temp.setX(temp.x() + space / 2);
			
			TrackTimesClassicalView* timesClassicalView = new TrackTimesClassicalView(tempTimes,
				       	trackProperties, textFont, nbStrings, lineHeight, space, DPI(), canvasView);
			
			noteClassicalView.append(timesClassicalView);
			
			durationCount+=tempTimes.realDuration();
			
			// Draw the time duration bars
			//
			if (i + 1 < tabBar.count() && durationCount < (tabBar.timeSignature(0) / 4) * (Whole / tabBar.timeSignature(1)) && !tempTimes.isRest()) {
				timesClassicalView->draw(position, tabBar.times(i + 1).duration());
			} else {
				// Special case it is the last note and not link with a previous group of notes
				//
				if (i + 1 == tabBar.count() && durationCount == tempTimes.realDuration() && !tempTimes.isRest()) {
					timesClassicalView->draw(position, NoneDuration);
				} else {
					if (!tempTimes.isRest())
						timesClassicalView->draw(position, NoneDuration);
					
					durationCount = 0.0;
				}
			}
			
			position.setX(position.x() + space);
		}
		
		position = tempPoint;
	}
	
	if (partition && tablature) {
		if (linkView[0]) {
			delete linkView[0];
			delete linkView[1];
		}
		
		linkView[0] = new QCanvasLine(canvasView);
		linkView[1] = new QCanvasLine(canvasView);
		
		
		linkView[0]->setPoints(position.x(), position.y(), position.x(), position.y() + cmToPixel<uint>(1.0));
		linkView[1]->setPoints(position.x() + barSize - 1 - barReduce, position.y(), position.x() + barSize - 1 - barReduce, position.y() + cmToPixel<uint>(1.0));
		
		linkView[0]->show();
		linkView[1]->show();
		
		position.setY(position.y() + cmToPixel<uint>(1.0));
	}
	
	if (tablature) {
		durationCount = 0.0;
		uint height = textHeight + cmToPixel<uint>(0.05);
		
		tablatureView->draw(position, barSize - barReduce);
		
		position.setX(position.x() + cmToPixel<uint>(0.1));
		
		// Display the word TAB
		//
		if (tablatureClef) {
			tablatureClefView->draw(position);
			position.setX(position.x() + static_cast<int>(tablatureClefView->width()) + cmToPixel<uint>(0.1));
		}
		
		// Display the time signature
		//
		if (time) {
			timeSignatureView->draw(position);
			position.setX(position.x() + static_cast<int>(timeSignatureView->width()) + cmToPixel<uint>(0.1));
		}
		
		if (!tablatureClef && !time)
			position.setX(position.x() + cmToPixel<uint>(0.1));
		
		QPoint temp(position);
		
		// Display each time in the bar
		//
		for (uint i = 0; i < tabBar.count(); i++) {
			TabTimes tempTimes = tabBar.times(i);
			uint space = timeSpace(tempTimes);
			QPoint temp(position);
			temp.setX(temp.x() + space / 2);
			
			TrackDurationView* timesDurationView = new TrackDurationView(tempTimes, nbStrings, height, space, DPI(), canvasView);
			TrackTimesView* timesView = new TrackTimesView(tempTimes, textFont, nbStrings, height, DPI(), canvasView);
			
			timesView->draw(position);
			fretView.append(timesView);
			durationView.append(timesDurationView);
			
			durationCount+=tempTimes.realDuration();
			
			if (!tempTimes.isRest())
				drawEffect(tempTimes, position);
			
			if (i + 1 < tabBar.count() && durationCount < (tabBar.timeSignature(0) / 4) * (Whole / tabBar.timeSignature(1)) && !tempTimes.isRest()) {
				timesDurationView->draw(temp, tabBar.times(i + 1).duration());
			} else {
				// Special case it is the last note and not link with a previous group of notes
				//
				if (i + 1 == tabBar.count() && durationCount == tempTimes.realDuration() && !tempTimes.isRest()) {
					timesDurationView->draw(temp, NoneDuration);
				} else {
					if (!tempTimes.isRest())
						timesDurationView->draw(temp, NoneDuration);
					
					durationCount = 0;
				}
			}
			
			position.setX(position.x() + space);
		}
		
		drawAllTuplets(tabBar, temp);
	}
}


// FIXME correct the class design
//
void TrackBarView::drawVibrato(unsigned int x, unsigned int y)
{
	QPointArray points(6);

	for (unsigned int i = 0; i < 6; i++) {
		x+=3;
		
		if (i % 2) y-=3;
		else y+=3;

		points.setPoint(i, x, y);
	}

	QCanvasPolygon *temp = new QCanvasPolygon(canvasView);
	temp->setPoints(points);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawWideVibrato(unsigned int x, unsigned int y)
{
	QPointArray points(6);
	QPen pen(Qt::black, 2);
	
	for (unsigned int i = 0; i < 6; i++) {
		x+=3;
		
		if (i % 2) y-=3;
		else y+=3;

		points.setPoint(i, x, y);
	}
	
	QCanvasPolygon *temp = new QCanvasPolygon(canvasView);
	temp->setPoints(points);
	temp->setPen(pen);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawTremoloBar(unsigned int , unsigned int )
{
}

void TrackBarView::drawNaturalHarmonic(unsigned int x, unsigned int y)
{
	QPointArray a(4);
	
	// initialize diamond shape
	//
	a.setPoint(0, x, y);
	a.setPoint(1, x + 5, y + 5);
	a.setPoint(2, x + 2 * 5, y);
	a.setPoint(3, x + 5, y - 5);
	
	QCanvasPolygon *temp = new QCanvasPolygon(canvasView);
	temp->setPoints(a);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawArtificialHarmonic(unsigned int x, unsigned int y)
{
	QPointArray a(4);
	
	// initialize diamond shape
	//
	a.setPoint(0, x, y);
	a.setPoint(1, x + 5, y + 5);
	a.setPoint(2, x + 2 * 5, y);
	a.setPoint(3, x + 5, y - 5);
	
	// draw filled diamond
	//
	QBrush blbr(Qt::black);
	QCanvasPolygon *temp = new QCanvasPolygon(canvasView);
	temp->setBrush(blbr);
	temp->setPoints(a);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawTapping(unsigned int x, unsigned int y)
{
	QCanvasText *temp = new QCanvasText("T", canvasView);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawSlapping(unsigned int x, unsigned int y)
{
	QCanvasText *temp = new QCanvasText("S", canvasView);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawPopping(unsigned int x, unsigned int y)
{
	QCanvasText *temp = new QCanvasText("P", canvasView);
	temp->move(x + 3, y - 12);
	temp->show();
}

void TrackBarView::drawFadeIn(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 10, y - 5);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 10, y + 5);
	temp->show();
}

void TrackBarView::drawUpStroke(unsigned int x, unsigned int y, bool ras)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x, y + 10);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x - 5, y + 5);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 5, y + 5);
	temp->show();
	
	if (ras) {
		QCanvasText *temp = new QCanvasText("r", canvasView);
		temp->move(x + 15, y);
		temp->show();
	}
	
}

void TrackBarView::drawDownStroke(unsigned int x, unsigned int y, bool ras)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x, y + 10);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y + 10, x - 5, y + 5);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y + 10, x + 5, y + 5);
	temp->show();
	
	if (ras) {
		QCanvasText *temp = new QCanvasText("r", canvasView);
		temp->move(x + 15, y);
		temp->show();
	}
	
}

void TrackBarView::drawDip(unsigned int x, unsigned int y, const QString &s)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 5, y + 5);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 5, x + 10, y);
	temp->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawDive(unsigned int x, unsigned int y, const QString &s)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 10, y + 5);
	temp->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawReleaseUp(unsigned int x, unsigned int y, const QString &s)
{
	QPen pen(Qt::DotLine);
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPen(pen);
	temp->setPoints(x, y, x + 10, y);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPen(pen);
	temp->setPoints(x + 10, y, x + 10, y - 5);
	temp->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawInvertedDip(unsigned int x, unsigned int y, const QString &s)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 5, y - 5);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y - 5, x + 10, y);
	temp->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawReturn(unsigned int x, unsigned int y, const QString &s)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y, x + 10, y - 5);
	temp->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawReleaseDown(unsigned int x, unsigned int y, const QString &s)
{
	QPen pen(Qt::DotLine);
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPen(pen);
	temp->setPoints(x, y - 5, x + 10, y - 5);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPen(pen);
	temp->setPoints(x + 10, y - 5, x + 10, y);
	temp->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawBend(unsigned int x, unsigned  int y, const QString &s)
{
	QCanvasPixmapArray *temp = new QCanvasPixmapArray(locate("data", "ktabedit/pics/bend.png"));
	QCanvasSprite *sprite = new QCanvasSprite(temp, canvasView);
	sprite->move(x + 3, y - 4);
	sprite->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawBendAndRelease(unsigned int x, unsigned  int y, const QString &s)
{
	QCanvasPixmapArray *temp = new QCanvasPixmapArray(locate("data", "ktabedit/pics/bendrelease.png"));
	QCanvasSprite *sprite = new QCanvasSprite(temp, canvasView);
	sprite->move(x + 3, y - 4);
	sprite->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawPreBend(unsigned int x, unsigned  int y, const QString &s)
{
	QCanvasPixmapArray *temp = new QCanvasPixmapArray(locate("data", "ktabedit/pics/prebendrelease.png"));
	QCanvasSprite *sprite = new QCanvasSprite(temp, canvasView);
	sprite->move(x + 3, y - 4);
	sprite->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawPreBendAndRelease(unsigned int x, unsigned  int y, const QString &s)
{
	QCanvasPixmapArray *temp = new QCanvasPixmapArray(locate("data", "ktabedit/pics/prebendrelease.png"));
	QCanvasSprite *sprite = new QCanvasSprite(temp, canvasView);
	sprite->move(x + 3, y - 4);
	sprite->show();
	
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawText(unsigned int x, unsigned int y, const QString &s)
{
	QCanvasText *text = new QCanvasText(s, canvasView);
	text->move(x + 3, y - 12);
	text->show();
}

void TrackBarView::drawSlide(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 10, x + 15, y + 5);
	temp->show();
}

void TrackBarView::drawLegatoSlide(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 10, x + 15, y + 5);
	temp->show();

	QCanvasEllipse *ellipse = new QCanvasEllipse(15, 10, 0, 180 * 16, canvasView);
	ellipse->move(x + 4, y - 5);
	ellipse->show();
}

void TrackBarView::drawOutDownwardSlide(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 10, x + 15, y + 5);
	temp->show();
}

void TrackBarView::drawOutUpwardSlide(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 10, x + 15, y + 5);
	temp->show();
}

void TrackBarView::drawSlideInBelow(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 10, x + 15, y + 5);
	temp->show();
}

void TrackBarView::drawSlideInAbove(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x + 5, y + 10, x + 15, y + 5);
	temp->show();
}

void TrackBarView::drawTremoloPicking(unsigned int x, unsigned int y, unsigned int nbTremolo)
{
	y+=nbStrings * 8;
	
	for (uchar i = 0; i < nbTremolo; i++) {
		QCanvasLine *temp = new QCanvasLine(canvasView);
		temp->setPoints(x - 5, y + 10, x - 15, y + 5);
		temp->show();
		y+=4;
	}
}

void TrackBarView::drawLegato(unsigned int x, unsigned int y, const TabTimes &currentTimes, const TabTimes &nextTimes, unsigned int i)
{
	QCanvasEllipse *ellipse = new QCanvasEllipse(15, 10, 0, 180 * 16, canvasView);
	ellipse->move(x + 4, y - 5);
	ellipse->show();
	
	QString string = "";

	if (currentTimes.notes(i).fret() < nextTimes.notes(i).fret()) {
		string = "HO";
	} else {
	 	string = "PO";
	}

	drawText(x, y, string);
}

void TrackBarView::drawPalmMute(unsigned int x, unsigned int y)
{
	drawText(x + 3, y - 12, "P.M.");
}

void TrackBarView::drawTieNote(unsigned int x, unsigned int y)
{
	QCanvasEllipse *ellipse = new QCanvasEllipse(15, 10, 0, 180 * 16, canvasView);
	ellipse->move(x + 4, y);
	ellipse->show();
}

void TrackBarView::drawStaccato(unsigned int x, unsigned int y)
{
	drawText(x + 3, y - 12, ".");
}

void TrackBarView::drawTrillNote(unsigned int x, unsigned int y, unsigned int note)
{
	QString string;
	
	string = string.setNum(note);
	
	drawText(x + 3, y - 12, "TR");
	drawText(x + 15, y, "(" + string + ")");
}

void TrackBarView::drawGraceNote(unsigned int x, unsigned int y, unsigned int note)
{
	QString string;
	
	if (note != 255)
		string = string.setNum(note);
	else
		string = "x";
	
	drawText(x, y, string);
}

void TrackBarView::drawAccentuated(unsigned int x, unsigned int y)
{
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y - 5, x + 10, y);
	temp->show();
	
	temp = new QCanvasLine(canvasView);
	temp->setPoints(x, y + 5, x + 10, y);
	temp->show();
}

void TrackBarView::drawEffect(const TabTimes &tempTimes, const QPoint &position)
{
	if (tempTimes.isVibrato()) {
		int x = position.x() + 3;
		
		drawVibrato(x, position.y() - 7);
	}
	
	if (tempTimes.isWideVibrato()) {
		int x = position.x() + 3;
		
		drawWideVibrato(x, position.y() - 7);
	}
	
	if (tempTimes.tremolo()) {
		int x = position.x() + 3;
		
		drawTremoloBar(x, position.y() - 7);
	}
	
	if (tempTimes.isNaturalHarmonic()) {
		// leftmost point of diamond
		//
		int x = position.x() - 3;
		
		drawNaturalHarmonic(x, position.y() - 15);
	}
	
	if (tempTimes.isArtificialHarmonic()) {
		// leftmost point of diamond
		//
		int x = position.x() - 3;
		
		drawArtificialHarmonic(x, position.y() - 15);
	}
	
	if (tempTimes.isTapping()) {
		int x = position.x() - 3;
		
		drawTapping(x, position.y() - 15);
	}
	
	if (tempTimes.isSlapping()) {
		int x = position.x() - 3;
		
		drawSlapping(x, position.y() - 15);
	}
	
	if (tempTimes.isPopping()) {
		int x = position.x() - 3;
		
		drawPopping(x, position.y() - 15);
	}
	
	if (tempTimes.isFadeIn()) {
		int x = position.x() - 3;
		
		drawFadeIn(x, position.y() - 15);
	}
	
	if (tempTimes.isStroke()) {
		int x = position.x() + 10;
		
		if (tempTimes.upStroke())
			drawUpStroke(x, position.y() - 15, tempTimes.isRasgueado());
		else
			drawDownStroke(x, position.y() - 15, tempTimes.isRasgueado());
	}
	
	switch (tempTimes.bendType()) {
		case TabBend::NoneBend:
			break;
		case TabBend::Bend:
		case TabBend::BendRelease:
		case TabBend::BendReleaseBend:
		case TabBend::PreBend:
		case TabBend::PreBendRelease:
			kdDebug() << "Warning you can't have a bend in a TabTime class !\n";
			break;
		case TabBend::Dip: {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawDip(x, y, tempTimes.bendValueToString());
			break;
		}
		case TabBend::Dive: {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawDive(x, y, tempTimes.bendValueToString());
			break;
		}
		case TabBend::ReleaseUp: {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawReleaseUp(x, y, tempTimes.bendValueToString());
			break;
		}
		case TabBend::InvertedDip: {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawInvertedDip(x, y, tempTimes.bendValueToString());
			break;
		}
		case TabBend::Return: {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawReturn(x, y, tempTimes.bendValueToString());
			break;
		}
		case TabBend::ReleaseDown: {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawReleaseDown(x, y, tempTimes.bendValueToString());
			break;
		}
		default:
			kdDebug() << "bend : " << (int)tempTimes.bendType() << " there is a problem :p" << endl;
			break;
	}
	
	for (int j = 0; j < nbStrings; j++) {
		TabNote note = tempTimes.notes(j);
		
		// Check if there is a bend effect
		//
		switch (note.bendType()) {
			case TabBend::NoneBend:
				break;
			case TabBend::Bend: {
				int x = position.x() + 3;
				int y = position.y() + ((j - 1) * 7);
				
				drawBend(x, y, note.bendValueToString());
			} break;
			case TabBend::BendRelease: {
				int x = position.x() + 3;
				int y = position.y() + ((j - 1) * 7);
				
				drawBendAndRelease(x, y, note.bendValueToString());
			} break;
			case TabBend::BendReleaseBend: {
				int x = position.x() + 3;
				int y = position.y() + ((j - 1) * 7);
				
				drawBendAndRelease(x, y, "BendReleaseBend");
			}
			case TabBend::PreBend: {
				int x = position.x() + 3;
				int y = position.y() + ((j - 1) * 7);
				
				drawPreBend( x, y, note.bendValueToString());
			} break;
			case TabBend::PreBendRelease: {
				int x = position.x() + 3;
				int y = position.y() + ((j - 1) * 7);
				
				drawPreBendAndRelease( x, y, note.bendValueToString());
			} break;
			default:
				kdDebug() << "bend : " << (int)note.bendType() << " there is a problem :p" << endl;
				break;
		}
		
		switch (note.harmonic()) {
			case NoneHarmonic:
				break;
			case Natural: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawNaturalHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "N.H.");
			} break;
			case Tapped: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawArtificialHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "T.H.");
			} break;
			case Pitch: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawArtificialHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "P.H.");
			} break;
			case Semi: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawArtificialHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "S.H.");
			} break;
			case Artificial5: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawArtificialHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "A.H. 5");
			} break;
			case Artificial7: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawArtificialHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "A.H. 7");
			} break;
			case Artificial12: {
				// leftmost point of diamond
				//
				int x = position.x() - 3;
				
				drawArtificialHarmonic(x, position.y() - 15);
				drawText(x + 3, position.y() - 20, "A.H. 12");
			} break;
			default:
				kdDebug() << "harmonic : " << (int)note.harmonic() << " there is a problem :p" << endl;
				break;
		}
		
		switch (note.slide()) {
		case NoneSlide:
			break;
		case ShiftSlide: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawSlide(x, y);
		} break;
		case LegatoSlide: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawLegatoSlide(x, y);
		} break;
		case OutDownwardSlide: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawOutDownwardSlide(x, y);
		} break;
		case OutUpwardSlide: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawOutUpwardSlide(x, y);
		} break;
		case SlideInBelow: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawSlideInBelow(x, y);
		} break;
		case SlideInAbove: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawSlideInAbove(x, y);
		} break;
		default:
			kdDebug() << "slide : " << (int)note.slide() << " there is a problem :p" << endl;
			break;
		}
		
		switch (note.tremoloPicking()) {
		case 0:
			break;
		case 1: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawTremoloPicking(x, y, 1);
		} break;
		case 2: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawTremoloPicking(x, y, 2);
		} break;
		case 3: {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawTremoloPicking(x, y, 3);
		} break;
		default:
			kdDebug() << "tremolo picking : " << (int)note.tremoloPicking() << " there is a problem :p" << endl;
			break;
		}
		
		if (note.isSlide() && note.slide() == NoneSlide) {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawSlide(x, y);
		}
		
		if (note.isHammerOnPullOff()) {
			// Hammer on, Pull off
			//
			//int x = position.x() + 3;
			//int y = position.y() + ((j - 1) * 7);
			
// 			drawLegato(x, y, tempTimes, nextTimes, j);
		}
			
		if (note.isPalmMute()) {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawPalmMute(x, y);
		}
		
		if (note.isTieNote()) {
			// Hammer on, Pull off
			//
			int x = position.x() - 26 + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawTieNote(x, y);
		}
		
		if (note.isStaccato()) {
			int x = position.x() + 3;
			int y = position.y() + ((j - 1) * 7);
			
			drawStaccato(x, y);
		}
		
		if (note.isAccentuated()) {
			int x = position.x() + 3;
			int y = position.y() - 10;
			
			drawAccentuated(x, y);
		}
		
		if (note.graceNoteDuration()) {
			int x = position.x() - 6;
			int y = position.y() + ((j) * 7);
			
			drawGraceNote(x, y, note.graceNoteFret());
		}
		
		if (note.fretTrill()) {
			int x = position.x() - 6;
			int y = position.y() + ((j) * 7);
			
			drawTrillNote(x, y, note.fretTrill());
		}
	}
}

void TrackBarView::drawTuplet(unsigned int ntupletLength, unsigned int ntuplet, const QPoint &position)
{
	unsigned int queueLength = (nbStrings * textHeight) + cmToPixel(0.5);
	QString s;
	
	// Draw the tuplet line
	//
	QCanvasLine *temp = new QCanvasLine(canvasView);
	temp->setPoints(position.x() , position.y() + queueLength, ntupletLength, position.y() + queueLength);
	temp->show();
	
	drawText(position.x() + (ntupletLength - position.x()) / 2, position.y() + queueLength + textWidth,
		 s.setNum(ntuplet + 1));
}

void TrackBarView::drawAllTuplets(const TabBar &tabBar, const QPoint &position)
{
	uint i = 0;
	uint j = 0;
	uint tuplet = 0, tupletCount = 0;
	int x = position.x();
	float timeCount = 0.0;
	QString temp;
	QPoint drawPosition(position);

	// For each times in the bar
	//
	for (i = 0; i < tabBar.count(); i++) {
		TabTimes tempTimes = tabBar.times(i);
	
		x = x + timeSpace(tempTimes);

		// Draw the old tuplet
		//
		if (tempTimes.nTuplet() != tuplet) {
			// Is it a tuplet ?
			//
			if (tuplet != 0) {
				if (tupletCount > tuplet) {
					drawTuplet(x, tuplet, drawPosition);
				} else {
					// Draw only the tuplet number
					//
					for (j = 0; j < tupletCount; j++) {
						unsigned int queueLength = nbStrings * 8 + 2;
						
						drawText(x, position.y() + queueLength + textWidth, temp.setNum(tuplet + 1));
					}
				}
			}
			
			if ((tuplet = tempTimes.nTuplet()) != 0) {
				// Set the postion of the new tuplet
				//
				drawPosition.setX(x);

				tupletCount = 1;
				timeCount+=tempTimes.duration();
				
				if (tempTimes.isDotted() == true)
					timeCount+=(tempTimes.duration() / 2);
			} else {
				tupletCount = 0;
				timeCount = 0.0;
			}
		} else if (tuplet != 0) {
			tupletCount++;
			
			// Check the number of tuplet times
			//
			timeCount+=tempTimes.duration();
			
			if (tempTimes.isDotted() == true)
				timeCount+=(tempTimes.duration() / 2);
			
			// Need at least tuplet + 1 elements for a complete tuplet
			//
			if (tuplet != 0 && tupletCount >= tuplet + 1) {
				if (timeCount == 15 * (tuplet + 1)   || timeCount == 30 * (tuplet + 1)   ||
					timeCount == 60 * (tuplet + 1)   || timeCount == 120 * (tuplet + 1)  ||
					timeCount == 240 * (tuplet + 1)  || timeCount == 480 * (tuplet + 1)  ||
					timeCount == 22.5 * (tuplet + 1) || timeCount == 45 * (tuplet + 1)   ||
					timeCount == 90 * (tuplet + 1)   || timeCount == 180 * (tuplet + 1)  ||
					timeCount == 360 * (tuplet + 1)  || timeCount == 720 * (tuplet + 1)) {
					drawTuplet(x, tuplet, drawPosition);
					
					timeCount = 0.0;
					tupletCount = 0;
				}
			}
		}
	}
	
	// Draw the last tuplets
	//
	if (tuplet != 0) {
		if (tupletCount >= tuplet + 1) {
			drawTuplet(tupletCount, tuplet, drawPosition);
		} else {
			// Draw only the tuplet number
			//
			for (uint j = 0; j < tupletCount; j++) {
				unsigned int queueLength = nbStrings * 8 + 2;
				
				drawText(x, position.y() + queueLength + textWidth, temp.setNum(tuplet + 1));
			}
		}
	}
}

void TrackBarView::displayPartition(bool t)
{
	if (partitionView) {
		delete partitionView;
		delete classicalClefView;
	}
	
	if (t) {
		// Font used for the text
		//
		uint height = static_cast<uint>(textHeight * 0.85 + cmToPixel<uint>(0.05));
		
		partitionView = new TrackBarLineView(5, height, DPI(), canvasView);
		classicalClefView = new TrackClassicalClefView(DPI(), canvasView);
	} else {
		partitionView = 0;
		classicalClefView = 0;
	}
	
	partition = t;
}

bool TrackBarView::isDisplayPartition() const
{
	return partition;
}

void TrackBarView::displayPartitionClef(bool t)
{
	if (partitionClef)
		delete classicalClefView;

	if (t) {
		classicalClefView = new TrackClassicalClefView(DPI(), canvasView);
	} else {
		classicalClefView = 0;
	}

	partitionClef = t;
}

bool TrackBarView::isDisplayPartitionClef() const
{
	return partitionClef;
}

void TrackBarView::displayTablature(bool t)
{
	if (tablatureView)
		delete tablatureView;
	
	if (t) {
		// Font used for the text
		//
		uint height = textHeight + cmToPixel<uint>(0.05);
		
		tablatureView = new TrackBarLineView(nbStrings, height, DPI(), canvasView);
	} else tablatureView = 0;
	
	tablature = t;
}

bool TrackBarView::isDisplayTablature() const
{
	return tablature;
}

void TrackBarView::displayTablatureClef(bool t, const QFont& clefFont)
{
	if (tablatureClefView)
		delete tablatureClefView;
	
	if (t && tablature) {
		tablatureClefView = new TrackTabClefView(clefFont, tablatureView->height(), DPI(), canvasView);
	}
	
	tablatureClef = t;
}

bool TrackBarView::isDisplayTablatureClef() const
{
	return tablatureClef;
}

void TrackBarView::displayTimeSignature(bool t, const QFont& timeFont)
{
	if (timeSignatureView)
		delete timeSignatureView;
	
	if (t && tablature) {
		timeSignatureView = new TrackTimeSignatureView(tabBar, timeFont, tablatureView->height(), DPI(), canvasView);
	}
	
	time = t;
}

bool TrackBarView::isDisplayTimeSignature() const
{
	return time;
}

void TrackBarView::displayBarNumber(bool t)
{
	QString temp;
	
	if (displayBarNumberView)
		delete barNumberView;

	if (t)
		barNumberView = new QCanvasText(temp.setNum(barNumber), textFont, canvasView);
	else
		barNumberView = 0;

	displayBarNumberView = t;

}

bool TrackBarView::isDisplayBarNumber() const
{
	return displayBarNumberView;
}

/*
	Return the position of a time
*/
QPoint TrackBarView::positionTablature(const TrackPos& cursorPos) const
{
	if (!tablature || cursorPos.times() > tabBar.count())
		return QPoint(0, 0);
	
	QPoint temp(barPosition);
	
	if (partition)
		temp.setY(temp.y() + partitionView->height() + cmToPixel<uint>(1.0));
	
	// Calc the chord position
	//
	temp.setY(static_cast<int>(temp.y() + (cursorPos.chord() * textHeight) - (0.5 * textHeight)));
	
	// Space between the bar separator and the first note
	//
	temp.setX(temp.x() + cmToPixel<uint>(0.1));
	
	// There is a clef
	//
	if (tablatureClef && tablatureClefView)
		temp.setX(temp.x() + static_cast<int>(tablatureClefView->width()) + cmToPixel<uint>(0.1));
	
	// There is a time signature
	//
	if (time && timeSignatureView)
		temp.setX(temp.x() + static_cast<int>(timeSignatureView->width()) + cmToPixel<uint>(0.1));
	
	// Display the word TAB
	//
	if (tablatureClef && !time)
		temp.setX(temp.x() + cmToPixel<uint>(0.1));
	
	// Display each time in the bar
	//
	for (uint i = 0; i < cursorPos.times(); i++) {
		TabTimes tempTimes = tabBar.times(i);
		uint space = timeSpace(tempTimes);
		
		temp.setX(temp.x() + space);
	}

	return temp;
}

/*
	Return the position of a time
*/
QPoint TrackBarView::positionPartiton(const TrackPos& ) const
{
	if (!partition)
		return QPoint(0, 0);
	
	QPoint temp(barPosition);
	
	return temp;
}

/*
	Return the position of the bar viewer
*/
QPoint TrackBarView::positionViewer() const
{
	return barPosition;
}
