/***************************************************************************
 * trackbarview.h: implementation of TrackBarView class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#ifndef TRACKBARVIEW_H
#define TRACKBARVIEW_H

#include <header.h>

#include "tabbar.h"
#include "unit.h"

class TrackBarLineView;
class TrackClassicalClefView;
class TrackDurationView;
class TrackPos;
class TrackProperties;
class TrackTabClefView;
class TrackTimesView;
class TrackTimesClassicalView;
class TrackTimeSignatureView;

// TODO : - Remove uggly effects
//        - Use the lilypod font
//
class TrackBarView : public Unit
{
public:
	TrackBarView(uint nb, const TabBar& bar, const TrackProperties& properties, uint strings, int resolutionDPI, QCanvas* canvas);
	TrackBarView(uint nb, const TabBar& bar, const TrackProperties& properties, uint strings, const QFont& font, int resolutionDPI, QCanvas* canvas);
	virtual ~TrackBarView();
	
	inline void displayClef(bool t)
	{ clef = t; }
	
	inline bool isDisplayClef() const
	{ return clef; }
	
	inline void displayArmor(bool t)
	{ armor = t; }
	
	inline bool isDisplayArmor() const
	{ return armor; }
	
	void displayTimeSignature(bool t, const QFont& timeFont);
	bool isDisplayTimeSignature() const;
	
	inline void displayTempo(bool t)
	{ tempo = t; }
	
	inline bool isDisplayTempo() const
	{ return tempo; }
	
	inline uint size() const
	{ return barSize; }
	
	inline void setSize(uint t)
	{ if (t < minimalSize()) return ;
	  else barSize = t; }
	
	void displayPartition(bool t);
	bool isDisplayPartition() const;

	void displayPartitionClef(bool t);
	bool isDisplayPartitionClef() const;

	void displayTablature(bool t);
	bool isDisplayTablature() const;
	
	void displayTablatureClef(bool t, const QFont& clefFont);
	bool isDisplayTablatureClef() const;
	
	void displayBarNumber(bool t);
	bool isDisplayBarNumber() const;
	
	/*
		Return the minimal size of the bar
	*/
	uint minimalSize();
	
	/*
		Reduce the spaces to adapt the bar to the width
	*/
	void reduceToWidth(uint width);
	
	/*
		Return the height of the bar
	*/
	uint height() const;
	
	/*
		Draw the bar on the display
	*/
	void draw(QPoint position);
	
	/*
		Return the position of a time
	*/
	QPoint positionTablature(const TrackPos& cursorPos) const;
	
	/*
		Return the position of a time
	*/
	QPoint positionPartiton(const TrackPos& cursorPos) const;
	
	/*
		Return the position of the bar viewer
	*/
	QPoint positionViewer() const;
	
	/*
		Draw the guitar Fx
	*/
	void setDrawEffects(bool b)
	{ effects = b; }
	
	/*
		Are we drawing guitar Fx
	*/
	bool isDrawEffects() const
	{ return effects; }
	
private:
	// Calc the space of the duration
	//
	void calcDurationSize();
	
	// Return the size of the clef
	//
	uint clefSize() const;
	
	// Return the size of the armor
	//
	uint armorSize() const;
	
	// Return the size of all the times
	// in the current bar
	//
	uint timesSize() const;
	
	// Calc the space between a time and an other time
	//
	uint timeSpace(const TabTimes& times) const;
	
	double dtimesSize() const;
	double dtimeSpace(const TabTimes& times) const;
	
	// Draw the fret number in a time
	//
	void drawTimeChords(const TabTimes& times, const QPoint& position);
	

	// FIXME :
	//  - remove some of these functions from this class
	//  - Support lilypond font
	//
	void drawVibrato(unsigned int x, unsigned int y);
	void drawWideVibrato(unsigned int x, unsigned int y);
	void drawTremoloBar(unsigned int x, unsigned int y);
	void drawNaturalHarmonic(unsigned int x, unsigned int y);
	void drawArtificialHarmonic(unsigned int x, unsigned int y);
	void drawTapping(unsigned int x, unsigned int y);
	void drawSlapping(unsigned int x, unsigned int y);
	void drawPopping(unsigned int x, unsigned int y);
	void drawFadeIn(unsigned int x, unsigned int y);
	void drawUpStroke(unsigned int x, unsigned int y, bool ras);
	void drawDownStroke(unsigned int x, unsigned int y, bool ras);
	void drawDip(unsigned int x, unsigned int y, const QString &s);
	void drawDive(unsigned int x, unsigned int y, const QString &s);
	void drawReleaseUp(unsigned int x, unsigned int y, const QString &s);
	void drawInvertedDip(unsigned int x, unsigned int y, const QString &s);
	void drawReturn(unsigned int x, unsigned int y, const QString &s);
	void drawReleaseDown(unsigned int x, unsigned int y, const QString &s);
	void drawBend(unsigned int x, unsigned  int y, const QString &s);
	void drawBendAndRelease(unsigned int x, unsigned  int y, const QString &s);
	void drawPreBend(unsigned int x, unsigned  int y, const QString &s);
	void drawPreBendAndRelease(unsigned int x, unsigned  int y, const QString &s);
	void drawText(unsigned int x, unsigned int y, const QString &s);
	void drawSlide(unsigned int x, unsigned int y);
	void drawLegatoSlide(unsigned int x, unsigned int y);
	void drawOutDownwardSlide(unsigned int x, unsigned int y);
	void drawOutUpwardSlide(unsigned int x, unsigned int y);
	void drawSlideInBelow(unsigned int x, unsigned int y);
	void drawSlideInAbove(unsigned int x, unsigned int y);
	void drawTremoloPicking(unsigned int x, unsigned int y, unsigned int nbTremolo);
	void drawLegato(unsigned int x, unsigned int y, const TabTimes &currentTimes, const TabTimes &nextTimes, unsigned int i);
	void drawPalmMute(unsigned int x, unsigned int y);
	void drawTieNote(unsigned int x, unsigned int y);
	void drawStaccato(unsigned int x, unsigned int y);
	void drawAccentuated(unsigned int x, unsigned int y);
	void drawTrillNote(unsigned int x, unsigned int y, unsigned int note);
	void drawGraceNote(unsigned int x, unsigned int y, unsigned int note);
	void drawEffect(const TabTimes &tabTimes, const QPoint &position);

	// Draw the tuplet
	//
	void drawTuplet(unsigned int ntupletLength ,unsigned int ntuplet, const QPoint &position);

	// Draw all the tuplets in the current bar
	//
	void drawAllTuplets(const TabBar &tabBar, const QPoint &position);
	
	static bool DefaultFontInit;
	static QFont DefaultTextFont;
	static QFontMetrics DefaultTextFontInfo;
	static int DefaultTextWidth;
	static int DefaultTextHeight;
	
	float WholeSize;
	float HalfSize;
	float QuarterSize;
	float EighthSize;
	float SixteenthSize;
	float ThirthSecondSize;
	float SixtyFourthSize;
	float HundredTwentyEighthSize;
	
	QCanvas* canvasView;
	TabBar tabBar;
	const TrackProperties trackProperties;
	const int nbStrings;
	bool clef;
	bool armor;
	bool time;
	bool tempo;
	uint minimalBarSize;
	uint barSize;
	uint barReduce;
	TrackBarLineView* partitionView;
	TrackBarLineView* tablatureView;
	TrackClassicalClefView* classicalClefView;
	TrackTabClefView* tablatureClefView;
	TrackTimeSignatureView* timeSignatureView;
	QPtrList<TrackTimesView> fretView;
	QPtrList<TrackDurationView> durationView;
	QPtrList<TrackTimesClassicalView> noteClassicalView;
	QPtrList<TrackDurationView> durationClassicalView;
	QCanvasLine* linkView[2];
	bool partition;
	bool partitionClef;
	bool tablature;
	bool tablatureClef;
	bool displayBarNumberView;
	uint barNumber;
	QCanvasText* barNumberView;
	
	QFont textFont;
	QFontMetrics textFontInfo;
	int textWidth;
	int textHeight;
	
	QPoint barPosition;
	
	bool effects;
};

#endif
