#include "tabtimes.h"
#include "trackdurationview.h"

const double TrackDurationView::queueSpace = 0.1;

TrackDurationView::TrackDurationView(const TabTimes& times, uint strings, uint space, int width, int resolutionDPI, QCanvas* canvas)
: Unit(resolutionDPI),
  tabTimes(times),
  nbStrings(strings),
  lineSpaces(space),
  durationWidth(width),
  durationCanvas(canvas)
{
	lineList.setAutoDelete(true);
}

TrackDurationView::~TrackDurationView()
{
}

void TrackDurationView::draw(QPoint position, Duration nextDuration)
{
	drawVerticalLine(position);
	
	if (nextDuration != NoneDuration)
		drawFirstDuration(position, nextDuration);
	else
		drawLastDuration(position);
}

void TrackDurationView::drawVerticalLine(QPoint position)
{
	unsigned int queueLength = nbStrings * lineSpaces;
	int j;
	
	QCanvasLine* line = new QCanvasLine(durationCanvas);
	
	switch (tabTimes.duration()) {
	case HundredTwentyEighth:
	case SixtyFourth:
	case ThirthSecond:  // 1/32 Triple croche
	case Sixteenth:  // 1/16 Double croche
	case Eighth:  // 1/8 Croche
	case Quarter: // 1/4 - a long vertical line, so we need to find the highest note
		for (j = nbStrings - 1;((j >= 0) && (tabTimes.notes(j).isEmpty()) && (tabTimes.notes(j).isDeadNote() == false)); j--);
		
		// If it's an empty measure at all - draw the vertical line from bottom
		//
		if (j < 0)  j = 1;
		
		line->setPoints(position.x(), position.y() + j * lineSpaces, position.x(), position.y() + queueLength);
		line->show();
		
		break;		// required to prevent print preview artefact
	case Half: // 1/2 Blanche
		line->setPoints(position.x(), position.y() + queueLength - 5, position.x(), position.y() + queueLength);
		line->show();
		
	case Whole: // whole Ronde
	case NoneDuration:
		break;
	}
	
	lineList.append(line);
}

void TrackDurationView::drawFirstDuration(QPoint position, Duration nextDuration)
{
	uint queueLength = nbStrings * lineSpaces;
	uint nbLines = 0, nextNbLines = 0;
	
	switch (tabTimes.duration()) {
	case HundredTwentyEighth:
		nbLines = 5;
		break;
	case SixtyFourth:
		nbLines = 4;
		break;
	case ThirthSecond:  // 1/32 Triple croche
		nbLines = 3;
		break;
	case Sixteenth:  // 1/16 Double croche
		nbLines = 2;
		break;
	case Eighth:  // 1/8 Croche
		nbLines = 1;
		break;
	case Quarter: // 1/4 - a long vertical line, so we need to find the highest note
	case Half: // 1/2 Blanche
	case Whole: // whole Ronde
	case NoneDuration:
		break; 
	}
	
	for (uint i = 0; i < nbLines; i++) {
		QCanvasLine* line = new QCanvasLine(durationCanvas);
		
		line->setPoints(position.x(), position.y() + queueLength - (i * cmToPixel(queueSpace)),
						position.x() + durationWidth / 2, position.y() + queueLength - (i * cmToPixel(queueSpace)));
		line->show();
		
		lineList.append(line);
	}
	
	switch (nextDuration) {
	case HundredTwentyEighth:
		nextNbLines = 5;
		break;
	case SixtyFourth:
		nextNbLines = 4;
		break;
	case ThirthSecond:  // 1/32 Triple croche
		nextNbLines = 3;
		break;
	case Sixteenth:  // 1/16 Double croche
		nextNbLines = 2;
		break;
	case Eighth:  // 1/8 Croche
		nextNbLines = 1;
		break;
	case Quarter: // 1/4 - a long vertical line, so we need to find the highest note
	case Half: // 1/2 Blanche
	case Whole: // whole Ronde
	case NoneDuration:
		break; 
	}
	
	if (nextNbLines < nbLines)
		nbLines = nextNbLines;
	
	// Draw the tablature duration
	//
	for (uint i = 0; i < nextNbLines; i++) {
		QCanvasLine* line = new QCanvasLine(durationCanvas);
		
		line->setPoints(position.x() + durationWidth / 2, position.y() + queueLength - (i * cmToPixel(queueSpace)),
						position.x() + durationWidth, position.y() + queueLength - (i * cmToPixel(queueSpace)));
		line->show();
		
		lineList.append(line);
	}
}

void TrackDurationView::drawLastDuration(QPoint position)
{
	uint queueLength = nbStrings * lineSpaces;
	uint nbLines = 0;
	
	switch (tabTimes.duration()) {
	case HundredTwentyEighth:
		nbLines = 5;
		break;
	case SixtyFourth:
		nbLines = 4;
		break;
	case ThirthSecond:  // 1/32 Triple croche
		nbLines = 3;
		break;
	case Sixteenth:  // 1/16 Double croche
		nbLines = 2;
		break;
	case Eighth:  // 1/8 Croche
		nbLines = 1;
		break;
	case Quarter: // 1/4 - a long vertical line, so we need to find the highest note
	case Half: // 1/2 Blanche
	case Whole: // whole Ronde
	case NoneDuration:
		break; 
	}
	
	for (uint i = 0; i < nbLines; i++) {
		QCanvasLine* line = new QCanvasLine(durationCanvas);
		
		line->setPoints(position.x() - durationWidth / 2, position.y() + queueLength - (i * cmToPixel(queueSpace)),
						position.x(), position.y() + queueLength - (i * cmToPixel(queueSpace)));
		line->show();
		
		lineList.append(line);
	}
}
