/***************************************************************************
 * trackeditview.cpp: implementation of TrackEditView class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "chord.h"
#include "fingers.h"
#include "fretboard.h"
#include "keysig.h"
#include "rhythmer.h"
#include "settings.h"
#include "songview.h"
#include "tabbar.h"
#include "tabsong.h"
#include "tabtimesplayer.h"
#include "timesig.h"
#include "trackbarview.h"
#include "trackcursorview.h"
#include "trackeditview.h"
#include "trackpageview.h"
#include "trackpane.h"
#include "trackpos.h"
#include "trackprint.h"
#include "tracktitleview.h"
#include "tracktuningview.h"
#include "trackview.h"
#include "trackviewcommands.h"
#include "trackwordsmusicview.h"

#include "kprofilemethod.h"
int pr_theMethod = 0;

TrackEditView::TrackEditView(QCanvas* canvas, TSE3::MidiScheduler* midiScheduler, TabSong* song, KXMLGUIClient* xml, KCommandHistory* cmd, QWidget *parent, const char *name)
: QCanvasView(parent, name),
  tabSong(song),
  xmlGUIClient(xml),
  cmdHist(cmd),
  playbackCursor(false),
  change(false),
  size(632, 0),
  widthBar(582),
  tuplet(2),
  scheduler(midiScheduler),
  notePlayer(0),
  trackPane(0),
  space(0)
{
	KPrinter printer(true, QPrinter::ScreenResolution);
	
	// Sets the frame style to style
	//
	setFrameStyle(Panel | Sunken);
	
	// Sets the color role used for painting the background of the widget
	//
	setBackgroundMode(PaletteBase);
	
	dpi = printer.resolution();
	
	kdDebug() << "dpi = " << dpi << "\n";
	
	trackView = new TrackView(canvas, song, dpi, 1.016, 1.016, QSize(static_cast<int>(8.26 * dpi), static_cast<int>(11.7 * dpi)), false);
	
	// set the canvas
	//
	setCanvas(canvas);
	
	// Standard palette
	//
	viewport()->setBackgroundMode(PaletteBase);
	
	setFocusPolicy(QWidget::StrongFocus);
	
	setCurrentTrack(tabSong->first());

	space = trackView->cursorPosition().y();
	
	// Display the current time
	//
	repaintCell();
	
	currentTime();
}

TrackEditView::~TrackEditView()
{
}

void TrackEditView::selectTrack(TabTrack *trk)
{
	if(cursorPos.chord() >= currentTrack->nbStrings()) {
		cursorPos.setChord(0);
	}
	
	cursorPos.setBar(0);
	cursorPos.setTimes(0);
	
	setCurrentTrack(trk);
	repaintCell();
	
	emit statusBar("Track displayed");
}

void TrackEditView::setCurrentTrack(TabTrack* tabTrack)
{
	cursorPos.setChord(0);
	cursorPos.setBar(0);
	cursorPos.setTimes(0);
	
	currentTrack = tabTrack;
	
	PROFILE_METHOD_BEGIN( pr_theMethod );
	
	trackView->setCurrentTrack(tabTrack);
	trackView->refreshCursor(cursorPos);
	
	PROFILE_METHOD_END( pr_theMethod );
	PROFILE_METHOD_PRINT( pr_theMethod, "REFRESH VIEW" );
	
	repaintCell();
	currentTime();
	
	emit trackChanged(tabTrack);
}

/*
	Repaint the current bar and choose if the other bars must
	drawed
*/
void TrackEditView::repaintCell()
{
	PROFILE_METHOD_BEGIN( pr_theMethod );
	
	trackView->setCurrentTrack(currentTrack);
	
	PROFILE_METHOD_END( pr_theMethod );
	PROFILE_METHOD_PRINT( pr_theMethod, "REFRESH VIEW" );
	
	setContentsPos(0, trackView->cursorPosition().y() - space);
}

// Process a mouse press of fret "fret" in current column on string
// "num". Depending on given "button" mouse state flags, additional
// things may happen.
//
void TrackEditView::melodyEditorPress(int num, int fret, ButtonState button = NoButton)
{
	if (button & LeftButton)
		melodyEditorAction(num, fret, 0);
	if (button & MidButton)
		melodyEditorAction(num, fret, 1);
	if (button & RightButton)
		melodyEditorAction(num, fret, 2);
}

// Execute one of melody editors actions, as defined in
// globalMelodyEditorAction array
//
void TrackEditView::melodyEditorAction(int /*num*/, int /*fret*/, int /*action*/)
{
/*	// GREYFIX: make it *one* undo transaction
	switch (Settings::melodyEditorAction(action)) {
	case 0: // no action
		break;
	case 1: // set note
		setFinger(num, fret);
		break;
	case 3: // set 022 power chord
		setFinger(num + 2, fret + 2);
	case 2: // set 02 power chord
		setFinger(num + 1, fret + 2);
		setFinger(num, fret);
		break;
	case 5: // set 0022 power chord
		setFinger(num + 3, fret + 2);
		setFinger(num + 2, fret + 2);
	case 4: // set 00 power chord
		setFinger(num + 1, fret);
		setFinger(num, fret);
		break;
	case 6: // delete note
		setFinger(num, NULL_NOTE);
		break;
	}*/
}

// Process a mouse release in melody editor. Depending on given
// "button" mouse state flags, additional things, such as proceeding
// to next column, may happen.
void TrackEditView::melodyEditorRelease(ButtonState button)
{
	if (((button & LeftButton)  && (Settings::melodyEditorAdvance(0))) ||
		((button & MidButton)   && (Settings::melodyEditorAdvance(1))) ||
		((button & RightButton) && (Settings::melodyEditorAdvance(2))))  {
/*		if (currentTrack->getSelectionMode()) {
			currentTrack->setSelectionMode(false);
			repaintCell();
		}*/
		moveRight();
	}
}

void TrackEditView::halfBend()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::Bend);
		bend.setBendValue(150);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(0);
		list.append(point);
		
		point.setPosition(30);
		point.setHeight(150);
		list.append(point);
		
		point.setPosition(60);
		point.setHeight(150);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::fullBend()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::Bend);
		bend.setBendValue(150);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(0);
		list.append(point);
		
		point.setPosition(30);
		point.setHeight(100);
		list.append(point);
		
		point.setPosition(60);
		point.setHeight(100);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::halfBendRelease()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::BendRelease);
		bend.setBendValue(150);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(0);
		list.append(point);
		
		point.setPosition(15);
		point.setHeight(150);
		list.append(point);
		
		point.setPosition(30);
		point.setHeight(150);
		list.append(point);
		
		point.setPosition(45);
		point.setHeight(0);
		list.append(point);
		
		point.setPosition(60);
		point.setHeight(0);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::fullBendRelease()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::BendRelease);
		bend.setBendValue(100);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(0);
		list.append(point);
		
		point.setPosition(15);
		point.setHeight(100);
		list.append(point);
		
		point.setPosition(30);
		point.setHeight(100);
		list.append(point);
		
		point.setPosition(0);
		point.setHeight(45);
		list.append(point);
		
		point.setPosition(60);
		point.setHeight(0);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::halfPreBend()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::PreBend);
		bend.setBendValue(150);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(150);
		list.append(point);
		
		point.setPosition(60);
		point.setHeight(150);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::fullPreBend()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::PreBend);
		bend.setBendValue(100);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(100);
		list.append(point);
		
		point.setPosition(60);
		point.setHeight(100);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::halfPreBendRelease()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::PreBendRelease);
		bend.setBendValue(150);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(150);
		list.append(point);
		
		point.setPosition(20);
		point.setHeight(150);
		list.append(point);
		
		point.setPosition(40);
		point.setHeight(0);
		list.append(point);
		
		
		point.setPosition(60);
		point.setHeight(0);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::fullPreBendRelease()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
		TabBend bend;
		
		bend.setBendType(TabBend::PreBendRelease);
		bend.setBendValue(100);
		
		QValueList<TabBendPoint> list;
		TabBendPoint point;
		
		point.setPosition(0);
		point.setHeight(100);
		list.append(point);
		
		point.setPosition(20);
		point.setHeight(100);
		list.append(point);
		
		point.setPosition(40);
		point.setHeight(0);
		list.append(point);
		
		
		point.setPosition(60);
		point.setHeight(0);
		list.append(point);
		
		bend.setBendPoint(list);
		
		cmdHist->addCommand(new BendNoteCommand(*this, *currentTrack, cursorPos, bend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::unisonBend()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, UnisonBend));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::vibrato()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, Vibrato));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::wideVibrato()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, WideVibrato));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::trill()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, Trill));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::tapping()
{
}

void TrackEditView::tremoloBar()
{
}

void TrackEditView::muffledStrings()
{
}

void TrackEditView::pickSlide()
{
}

void TrackEditView::tremoloPicking()
{
}

void TrackEditView::setLength(Duration length)
{
	//only if needed
	//
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).duration() != length) {
		cmdHist->addCommand(new SetLengthCommand(*this, *currentTrack, cursorPos, length));
		change = true;
	}
}

void TrackEditView::linkPrev()
{
// 	cmdHist->addCommand(new SetFlagCommand(*this, *currentTrack, cursorPos, TrackEditView::SetFlagCommand::Arc));
	change = true;
}

void TrackEditView::addHarmonic()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, NaturalHarmonic));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::addArtHarm()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, ArtificialHarmonic));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::addLegato()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, Legato));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::addSlide()
{
	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty()) {
// 		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, Slide));
		change = true;
	} else
		emit statusBar("There is no note");
}

void TrackEditView::addLetRing()
{
/*	if (!currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty())
		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, LetRing));
	else
		cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, StopRing));*/
	
	change = true;
}

// Call the chord constructor dialog and may be parse something from it
void TrackEditView::insertChord()
{
	QMemArray<int> a(currentTrack->nbStrings());
	ChordSelector cs(scheduler, *currentTrack);
	int i;
	
	emit stopMidiPlaying();
	
	if (notePlayer) {
		notePlayer->stopPlaying();
		
		// Check if the old note is played
		//
		while (!notePlayer->finished())
			;
		
		delete notePlayer;
	}
	
	// required to detect chord from tabulature
	//
	cs.detectChord();
	
	// set fingering right if frets > 5
	//
	for (i = 0; i < currentTrack->nbStrings(); i++)
		a[i] = cs.app(i);
	
	cs.getFingering().setFingering(a);
	
	if (cs.exec()) {
		for (i = 0; i < currentTrack->nbStrings(); i++)
			a[i] = cs.app(i);
		
		cmdHist->addCommand(new InsertStrumCommand(*this, *currentTrack, cursorPos, a));
		change = true;
	}
	
	// Display the current time
	//
	currentTime();
}

// Call rhythm construction dialog and may be parse something from it
void TrackEditView::rhythmer()
{
	Rhythmer r;
	
	if (r.exec()) {
		cmdHist->addCommand(new InsertRhythm(*this, *currentTrack, *r.quantized, cursorPos));
		change = true;
	}
}

void TrackEditView::resizeEvent(QResizeEvent *e)
{
// 	QGridView::resizeEvent(e);
	QScrollView::resizeEvent(e);
	size = e->size();
	
	if (size.width() < 632)
		size.setWidth(632);
	
// 	setCellWidth(size.width());
// 	setCellHeight(42 + currentTrack->nbStrings() * 8);
}

void TrackEditView::setDotted()
{
	cmdHist->addCommand(new DottedNoteCommand(*this, *currentTrack, cursorPos));
	emit columnChanged();
}

void TrackEditView::setRest()
{
	cmdHist->addCommand(new RestNoteCommand(*this, *currentTrack, cursorPos));
	emit columnChanged();
}

void TrackEditView::keySig()
{
/*// 	short oldsig = currentTrack->getVectTabBar(0).getKeySignature();

	if ((oldsig <= -8) || (8 <= oldsig)) {
		oldsig = 0;
	}

	SetKeySig *sks = new SetKeySig();
	sks->sig->setCurrentItem(7 - oldsig);

	if (sks->exec()) {
		int newsig = sks->sig->currentItem();
// 		TabBar tab = currentTrack->getVectTabBar(0);
		tab.setKeySignature((short) (7 - newsig));
// 		currentTrack->setVectTabBar(0, tab);
	}

	//lastnumber = -1;*/
}

void TrackEditView::timeSig()
{
	TabBar currentBar = currentTrack->bar(cursorPos.bar());
	SetTimeSig *sts = new SetTimeSig();

	sts->getTime1().setValue(currentBar.timeSignature(0));

	switch (currentBar.timeSignature(1))
	{
		case 1:	 sts->getTime2().setCurrentItem(0); break;
		case 2:	 sts->getTime2().setCurrentItem(1); break;
		case 4:	 sts->getTime2().setCurrentItem(2); break;
		case 8:	 sts->getTime2().setCurrentItem(3); break;
		case 16: sts->getTime2().setCurrentItem(4); break;
		case 32: sts->getTime2().setCurrentItem(5); break;
	}

	if (sts->exec()) {
		int time1 = sts->getTime1().value();
		int time2 = ((QString) sts->getTime2().currentText()).toUInt();

		cmdHist->addCommand(new SetTimeSigCommand(*this, *currentTrack, cursorPos.bar(), sts->getToEnd().isChecked(), time1, time2));
	}
}

// Move cursor left one column
//
void TrackEditView::keyLeft()
{
	moveLeft();
}

// Move cursor right one column
//
void TrackEditView::keyRight()
{
	moveRight();
}

// Move cursor to the beginning of bar, breaking selection
//
void TrackEditView::keyHome()
{
/*	if (currentTrack->getSelectionMode()) {
		currentTrack->setSelectionMode(false);
		repaintCell();
	} else {*/
		moveHome();
// 	}
}

// Move cursor to the ending of bar, breaking selection
//
void TrackEditView::keyEnd()
{
/*	if (currentTrack->getSelectionMode()) {
		currentTrack->setSelectionMode(false);
		repaintCell();
	} else {*/
		moveEnd();
// 	}
}

// Move cursor to the very beginning of the song, breaking selection
//
void TrackEditView::keyCtrlHome()
{
/*	if (currentTrack->getSelectionMode()) {
		currentTrack->setSelectionMode(false);
		repaintCell();
	} else {*/
		moveCtrlHome();
// 	}
}

// Move cursor to the very end of the song
//
void TrackEditView::keyCtrlEnd()
{
	moveCtrlEnd();
}

void TrackEditView::moveLeft()
{
	TabBar currentBar;
	
	if (cursorPos.times() > 0) {
		cursorPos.decTimes();
		currentBar = currentTrack->bar(cursorPos.bar());
	} else if (cursorPos.bar() > 0) {
		cursorPos.decBar();
		currentBar = currentTrack->bar(cursorPos.bar());
		
		cursorPos.setTimes(currentBar.count() - 1);
	}
	
	if (currentBar.isOverTaken()) {
		emit statusBar("time overtake !");
	}
	
	trackView->refreshCursor(cursorPos);
	canvas()->update();
	
	// Center the view
	//
	setContentsPos(0, trackView->cursorPosition().y() - space);
	
	// Display the current time
	//
	currentTime();
}

// Move to the right
// I test if it is the last bar
//	YES : 
//	- add a new bar
//	- add a new note
//	- go to the next note
//
//	NO :
//	- go to the next note
//	- go to the next bar
//
void TrackEditView::moveRight()
{
	TabBar currentBar = currentTrack->bar(cursorPos.bar());
	
	// Is it the last bar ?
	//
	if (currentTrack->count() == (static_cast<uint>(cursorPos.bar()) + 1)) {
		// Is it the last note ?
		//
		if (currentBar.count() == (static_cast<uint>(cursorPos.times()) + 1)) {
			// Is it full ?
			//
			if (currentBar.isFull()) {
				// Add a new bar
				//
				cmdHist->addCommand(new InsertColumnCommand(*this, *currentTrack, cursorPos));
				
				repaintCell();
				
// 				emit columnChanged();
			} else {
				// Add a new note
				//
				cmdHist->addCommand(new InsertNoteCommand(*this, *currentTrack, cursorPos));
				
				repaintCell();
				
// 				emit columnChanged();
			}
			
		} else {
			// Go to the next note
			//
			cursorPos.incTimes();
		}
	} else {
		// Is it the last note ?
		//
		if (currentBar.count() == (static_cast<uint>(cursorPos.times()) + 1)) {
			// Is it full ?
			//
			if (currentBar.isFull()) {
				// Go to the next bar
				//
				cursorPos.setTimes(0);
				cursorPos.incBar();
			} else {
				// You can add a new note and we can rearrange
				// the whole track
				//
				
				// Add a new note
				//
				cmdHist->addCommand(new InsertNoteCommand(*this, *currentTrack, cursorPos));
				
				repaintCell();
				
// 				emit columnChanged();
			}
		} else {
			cursorPos.incTimes();
		}
	}
	
	if (currentBar.isOverTaken()) {
		emit statusBar("time overtake !");
	}
	
	
	trackView->refreshCursor(cursorPos);
	canvas()->update();
	
	// Center the view
	//
	setContentsPos(0, trackView->cursorPosition().y() - space);
	
	// Display the current time
	//
	currentTime();
	emit columnChanged();
}

// Move to the first note and first bar
//
void TrackEditView::moveHome()
{
	cursorPos.setBar(0);
	cursorPos.setTimes(0);
	
	repaintCell();
	
	emit columnChanged();
}

// Move to the first note and last bar
//
void TrackEditView::moveEnd()
{
	if (currentTrack->count() > 0)
		cursorPos.setBar(currentTrack->count() - 1);
	else
		cursorPos.setBar(0);
	
	cursorPos.setTimes(0);
	
	trackView->refreshCursor(cursorPos);
/*	if (const TrackBarView* barView = barViewList.at(cursorPos.bar()))
		cursorView->refresh(barView->positionTablature(cursorPos));*/
	
	repaintCell();
	
	emit columnChanged();
}

// Move to the beginning of the bar
//
void TrackEditView::moveCtrlHome()
{
	cursorPos.setTimes(0);
	
	repaintCell();
	
	emit columnChanged();
}

// Move to the end of the bar
//
void TrackEditView::moveCtrlEnd()
{
	TabBar tempBar = currentTrack->bar(cursorPos.bar());
	
	cursorPos.setTimes(tempBar.count() - 1);
	
	repaintCell();
	
	emit columnChanged();
}

void TrackEditView::selectLeft()
{
	moveLeft();
}

void TrackEditView::selectRight()
{
	moveRight();
}

void TrackEditView::moveUp()
{
	if(cursorPos.chord() > 0) {
		cursorPos.decChord();
	}
	
	trackView->refreshCursor(cursorPos);
	canvas()->update();
}

void TrackEditView::transposeUp()
{
// 	if (currentTrack->getTabString()+1 < currentTrack->nbStrings())
// 		moveFinger(currentTrack->getTabString(), 1);
	repaintCell();
}

void TrackEditView::moveDown()
{
	if(cursorPos.chord() + 1 < currentTrack->nbStrings()) {
		cursorPos.incChord();
	}
	
	trackView->refreshCursor(cursorPos);
	canvas()->update();
}

void TrackEditView::transposeDown()
{
/*	if (currentTrack->getTabString() > 0)
		moveFinger(currentTrack->getTabString(), -1);*/
	//lastnumber = -1;
}

void TrackEditView::deadNote()
{
	cmdHist->addCommand(new DeadNoteCommand(*this, *currentTrack, cursorPos));
	change = true;
	emit columnChanged();
}

/*
	Check if we delete the bar (because there are no times)
	         we can delete the current time or a chord fretted
			 in the time
*/
void TrackEditView::deleteNote()
{
	TabBar tempBar = currentTrack->bar(cursorPos.bar());
	TabTimes tempTimes = tempBar.times(cursorPos.times());
	
	// Can I delete all the bar ?
	//
	// Check if there is one empty note in the current bar
	//
	if (tempBar.count() == 1 && tempTimes.isEmpty() == false){
		
		// Don't delete the first bar
		//
		cmdHist->addCommand(new DeleteColumnCommand(*this, *currentTrack, cursorPos));
		change = true;
		emit columnChanged();
	} else {
		
		// Check if there is a note on the bar,
		// if there are no notes we can delete
		// the bar
		//
		TabBar tempBar = currentTrack->bar(cursorPos.bar());
		TabTimes tempTimes = tempBar.times(cursorPos.times());
		
		// There is a chord fretted so we can delete this chord
		//
		if (!tempTimes.notes(cursorPos.chord()).isEmpty()) {
			cmdHist->addCommand(new DeleteNoteCommand(*this, *currentTrack, cursorPos));
		} else {
			// There are no chords fretted so we can delete the time
			//
			if (tempTimes.isEmpty() == false)
				cmdHist->addCommand(new DeleteNoteCommand(*this, *currentTrack, cursorPos));
		}
		
		change = true;
		
		emit columnChanged();
	}
	
	repaintCell();
	currentTime();
}

void TrackEditView::deleteColumn()
{
	cmdHist->addCommand(new DeleteColumnCommand(*this, *currentTrack, cursorPos));
	change = true;
	
	repaintCell();
	currentTime();
	
	emit columnChanged();
}

void TrackEditView::deleteColumn(QString /*name*/)
{
// 	cmdHist->addCommand(new DeleteColumnCommand(name, this, currentTrack));
	emit columnChanged();
}

void TrackEditView::insertColumn()
{
	cmdHist->addCommand(new InsertColumnCommand(*this, *currentTrack, cursorPos));
	change = true;
	emit columnChanged();
}

void TrackEditView::insertBar()
{
	cmdHist->addCommand(new InsertColumnCommand(*this, *currentTrack, cursorPos));
	change = true;
	emit columnChanged();
}

void TrackEditView::palmMute()
{
// 	cmdHist->addCommand(new AddFXCommand(*this, *currentTrack, cursorPos, PalmMute));
	change = true;
}

void TrackEditView::dotNote()
{
	cmdHist->addCommand(new DottedNoteCommand(*this, *currentTrack, cursorPos));
	change = true;
}

void TrackEditView::ntupletSelected(int i)
{
	if (i < 0 || i > 11)
		return ;
	
	tuplet = i + 1;
	
/*	if (i == 10)
		// 4:3
	if (i == 11)
		// 7:6*/
}

void TrackEditView::ntuplet()
{
	cmdHist->addCommand(new NTupletCommand(*this, *currentTrack, cursorPos, tuplet));
	change = true;
}

void TrackEditView::keyPlus()
{
	const Duration duration = currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).duration();
	
	if (duration < Whole && duration != NoneDuration) {
		Duration newDuration = NoneDuration;
		
		switch (duration) {
		case Half:
			newDuration = Whole;
			break;
		case Quarter:
			newDuration = Half;
			break;
		case Eighth:
			newDuration = Quarter;
			break;
		case Sixteenth:
			newDuration = Eighth;
			break;
		case ThirthSecond:
			newDuration = Sixteenth;
			break;
		case SixtyFourth:
			newDuration = ThirthSecond;
			break;
		case HundredTwentyEighth:
			newDuration = SixtyFourth;
		case Whole:
		case NoneDuration:
			break;
		}
		
		setLength(newDuration);
		emit statusBar("Time changes");
	}
}

void TrackEditView::keyMinus()
{
	const Duration duration = currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).duration();
	
	if (duration > HundredTwentyEighth && duration != NoneDuration) {
		Duration newDuration = NoneDuration;
		
		switch (duration) {
		case Whole:
			newDuration = Half;
			break;
		case Half:
			newDuration = Quarter;
			break;
		case Quarter:
			newDuration = Eighth;
			break;
		case Eighth:
			newDuration = Sixteenth;
			break;
		case Sixteenth:
			newDuration = ThirthSecond;
			break;
		case ThirthSecond:
			newDuration = SixtyFourth;
			break;
		case SixtyFourth:
			newDuration = HundredTwentyEighth;
		case HundredTwentyEighth:
		case NoneDuration:
			break;
		}
		
		setLength(newDuration);
		emit statusBar("Time changes");
	}
}

void TrackEditView::arrangeTracks()
{
// 	cmdHist->clear();       // because columns will be changed
// 	currentTrack->arrangeBars();
	emit statusBar("");

	repaintCell();

	emit paneChanged();
	emit columnChanged();
}

// Insert a note in the current bar and play this note
//
void TrackEditView::insertTab(int num)
{
	TSE3::PhraseEdit phraseEdit;
	TSE3::Clock time = 0;
	
	if (currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).isEmpty() == false) {
		const uint oldFret = currentTrack->bar(cursorPos.bar()).times(cursorPos.times()).notes(cursorPos.chord()).fret();
		
		if ( 10 * num + oldFret > currentTrack->nbFrets())
			return ;
	}
	
	cmdHist->addCommand(new InsertTabCommand(*this, *currentTrack, num, cursorPos));
	change = true;
	
	TabBar tempBar = currentTrack->bar(cursorPos.bar());
	TabTimes tempTimes = tempBar.times(cursorPos.times());
	
	emit stopMidiPlaying();
	
	if (notePlayer) {
		notePlayer->stopPlaying();
		
		// Check if the old note is played
		//
		while (!notePlayer->finished())
			;
		
		delete notePlayer;
	}
	
	notePlayer = new TabTimesPlayer(tempTimes, tabSong->tempo());
	
	if (currentTrack->mode() == DrumTab)
		notePlayer->drum();
	
	notePlayer->setScheduler(scheduler);
	notePlayer->setTrack(currentTrack);
	notePlayer->start();
	
	repaintCell();
	
	emit statusBar("Note inserted");
	
	// Display the current time
	//
	currentTime();
}

void TrackEditView::contentsMousePressEvent(QMouseEvent *e)
{
	// RightButton pressed
	//
	if (e->button() == RightButton) {
		QWidget *tmpWidget = 0;
		tmpWidget = xmlGUIClient->factory()->container("trackviewpopup", xmlGUIClient);

		if (!tmpWidget || !tmpWidget->inherits("KPopupMenu")) {
			kdDebug() << "TrackEditView::mousePressEvent => wrong container widget" << endl;
			return;
		}

		KPopupMenu *menu(static_cast<KPopupMenu*>(tmpWidget));
		menu->popup(QCursor::pos());
	}
	
	// LeftButton pressed
	//
	if (e->button() == LeftButton) {
		cursorPos = trackView->mousePressEvent(e->pos());
		
		canvas()->update();
		currentTime();
		
		// Center the view
		//
		setContentsPos(0, trackView->cursorPosition().y() - space);
		
		emit statusBar("");
		emit columnChanged();
	}
}

/*
	Change the cursor position and refresh the view
*/
void TrackEditView::setCursor(const TrackPos& position)
{
	cursorPos = position;
	
	trackView->refreshCursor(cursorPos);
	
	repaintCell();
}

void TrackEditView::setPlaybackCursor(bool pc)
{
    playbackCursor = pc;
	repaintCell();
}
	
void TrackEditView::viewScore(bool on)
{
	viewscore = on;
}

uint TrackEditView::getBarNumber()
{
	return currentTrack->count();
}

uint TrackEditView::getBarPosition()
{
	return cursorPos.bar();
}

void TrackEditView::currentTime()
{
	if (currentTrack->mode() == DrumTab)
		return ;
	
	TabBar tempBar = currentTrack->bar(cursorPos.bar());
	TabTimes tempTimes = tempBar.times(cursorPos.times());
	
	if (trackPane)
		trackPane->updateList();
	
	emit timesChanged(tempTimes);
}

void TrackEditView::beginPlaying()
{
	canvas()->setUpdatePeriod(20);
}

void TrackEditView::stopPlaying()
{
	canvas()->setUpdatePeriod(-1);
	
	if (notePlayer) {
		notePlayer->stopPlaying();
		
		// Check if the old note is played
		//
		while (!notePlayer->finished())
			;
		
		delete notePlayer;
		
		notePlayer = 0;
	}
}

void TrackEditView::setTrackPane(TrackPane* pane)
{
	trackPane = pane;
}

/*
	Set the string position of the cursor this function
	is used by songviewcommand when the user change the number
	of strings
*/
void TrackEditView::setString(uint i)
{
	cursorPos.setChord(i);
}

/*
	Return the string position of the cursor this function
	is used by songviewcommand when the user change the number
	of strings
*/
uint TrackEditView::string()
{
	return cursorPos.chord();
}
