/***************************************************************************
 * trackview.cpp: implementation of TrackView class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2004-2006 the KTabEdit development team
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "settings.h"
#include "tabbar.h"
#include "tabsong.h"
#include "tabtimesplayer.h"
#include "timesig.h"
#include "trackbarview.h"
#include "trackcursorview.h"
#include "trackpageview.h"
#include "trackpane.h"
#include "trackpos.h"
#include "trackprint.h"
#include "tracktitleview.h"
#include "tracktuningview.h"
#include "trackview.h"
#include "trackviewcommands.h"
#include "trackwordsmusicview.h"

TrackView::TrackView(QCanvas* canvas, const TabSong *song, int resolutionDPI, double left, double bottom, const QSize& displayResolution, bool landscapeView)
: Unit(resolutionDPI),
  canvasView(canvas),
  tabSong(song),
  currentTrack(0),
  trackResolution(displayResolution),
  landscape(landscapeView),
  leftMargins(cmToPixel(left)),
  bottomMargins(cmToPixel(bottom)),
  trackBody(0, 0, 0, 0),
  titleView(0),
  wordsAndMusicView(0),
  tuningView(0),
  nbPages(1),
  drawEffects(true)
{
	pageViewList.setAutoDelete(true);
	barViewList.setAutoDelete(true);
	
	trackBody = QRect(leftMargins, bottomMargins, trackResolution.width() - (2 * leftMargins), trackResolution.height() - (2 * bottomMargins));
	
	cursorView = new TrackCursorView(canvas);
	
	cursorView->setVisible(true);
	
	// Size of the first bar
	//
	canvas->resize(trackResolution.width(), trackResolution.height());
}

TrackView::~TrackView()
{
	delete titleView;
	delete wordsAndMusicView;
	delete tuningView;
	delete cursorView;
}

void TrackView::drawPages(uint nbPages)
{
	TrackPageView* pageView;
	
	trackPos.setY(nbPages * trackResolution.height() + bottomMargins + cmToPixel(0.5));
	
	// Draw the page and the page number
	//
	pageView = new TrackPageView(nbPages + 1, QRect(leftMargins, nbPages * trackResolution.height() + bottomMargins, 
									trackResolution.width() - (2 * leftMargins), trackResolution.height() - (2 * bottomMargins)),
									DPI(), canvasView);
	
	pageViewList.append(pageView);
}

void TrackView::drawSystem(QPtrList<TrackBarView>& barList, uint newWidth, uint height)
{
	for (uint i = 0; i < barList.count(); i++) {
		// Draw the bar
		//
		TrackBarView* barView = barList.at(i);
		
		barView->setSize(barView->minimalSize() + newWidth);
		
		barView->draw(trackPos);
		
		trackPos.setX(trackPos.x() + barView->size());
	}
	
	barList.clear();
	
	trackPos.setY(trackPos.y() + height + cmToPixel(1.00));
	trackPos.setX(leftMargins);
}

void TrackView::refreshProperties()
{
	QRect pageSize(trackBody);
	
	if (titleView) {
		delete titleView;
		delete wordsAndMusicView;
		delete tuningView;
	}
	
	// Draw the title
	//
	titleView = new TrackTitleView(*tabSong, trackBody, DPI(), canvasView);
	
	pageSize.setTop(titleView->size().bottom());
	
	// Draw words and music
	//
	wordsAndMusicView = new TrackWordsAndMusicView(*tabSong, pageSize, DPI(), canvasView);
	
	pageSize.setTop(wordsAndMusicView->size().bottom());
	
	// Draw the tuning
	//
	tuningView = new TrackTuningView(*currentTrack, pageSize, DPI(), canvasView);
	
	pageSize.setTop(tuningView->size().bottom());
	
	trackPos.setX(pageSize.left());
	trackPos.setY(pageSize.top());
}

void TrackView::refreshView(uint from)
{
	QFont clefFont(Settings::timeSignatureFont(), Settings::timeSignatureSize(), Settings::timeSignatureWeight(), Settings::timeSignatureItalic());
	QFont timeFont(Settings::timeSignatureFont(), Settings::timeSignatureSize(), Settings::timeSignatureWeight(), Settings::timeSignatureItalic());
	
	// Width counter
	//
	long width = 0;
	
	// New width
	//
	uint newWidth = 0;
	
	// Minimal size of a bar
	//
	long minimal = 0;
	
	// First bar of the line
	//
	uint firstBar = 0;
	
	// List of the bars viewer.
	//
	QPtrList<TrackBarView> barList;
	
	bool greaterBar = false;
	
	// Draw the page and the page number
	//
	pageViewList.append(new TrackPageView(1, trackBody, DPI(), canvasView));
	
	for (uint i = from; i < currentTrack->count(); i++) {
		// Create a new bar viewer
		//
		TrackBarView* barView = new TrackBarView(i + 1, currentTrack->bar(i), *currentTrack, currentTrack->nbStrings(), DPI(), canvasView);
		
		// Display partition
		//
		barView->displayPartition(true);
		
		// Display the partition clef
		//
		if (firstBar == i)
			barView->displayPartitionClef(true);

		// Display tablature
		//
		barView->displayTablature(true);
		
		// Display bar number
		//
		barView->displayBarNumber(true);
		
		// Are we drawing guitar Fx
		//
		barView->setDrawEffects(drawEffects);
		
		// Display the word TAB on the first bar of the tablature
		//
		if (i == 0)
			barView->displayTablatureClef(true, clefFont);
		
		// Display the clef on the begining of a system
		//
		if (firstBar == i || i == 0)
			barView->displayClef(true);
		
		// Display the armor :
		//   - at the beginning
		//   - when the armor change
		//
		if (i == 0 || (i > 0 && currentTrack->bar(i - 1).key() != currentTrack->bar(i).key()))
			barView->displayArmor(true);
		
		// Display the time signature :
		//   - at the beginning
		//   - when the time signature change
		//   - at the beginning of a new system
		//
		if (i == 0 || (i > 0 && *dynamic_cast<TimeSignature*>(&currentTrack->bar(i - 1)) != *dynamic_cast<TimeSignature*>(&currentTrack->bar(i))))
			barView->displayTimeSignature(true, timeFont);
		
		// Display the tempo
		//
		if (i == 0 || (i > 0 && currentTrack->bar(i - 1).tempo() != currentTrack->bar(i).tempo()))
			barView->displayTempo(true);
		
		// add the minimal size of the current bar
		//
		width = width + (minimal = barView->minimalSize());
		
		// Special case when the bar is too big to be displayed on a 
		// system so we reduce this bar
		//
		if (minimal > trackBody.width() && barList.count() == 0) {
			barView->reduceToWidth(trackBody.width());
			
			// Append the bar which will be displayed on the system (the current line)
			//
			barList.append(barView);
			barViewList.append(barView);
			
			greaterBar = true;
		}
		
		// Check if it is the last bar
		//
		const bool isLastBar = i + 1 == currentTrack->count();
		
		// Display all the bars in the system (the line)
		//
		if (width > trackBody.width() || isLastBar) {
			bool addLastBar = false;
			
			// Page Jump
			//
			if (trackPos.y() + barView->height() > nbPages * trackResolution.height() - bottomMargins)
				drawPages(nbPages++);
			
			// Add the last bar if the bar list is empty or if the size of
			// the last bar is smaller enough to be added to the bar list
			//
			//
			if ((isLastBar && barList.count() == 0) || (isLastBar && barView->minimalSize() + width < trackBody.width())) {
				// Append the bar which will be displayed on the system (the current line)
				//
				barList.append(barView);
				barViewList.append(barView);
				
				addLastBar = true;
			}
			
			width-=minimal;
			
			// Adapt the bars in the system to use all the space in the system
			//
			if (!isLastBar || (isLastBar && !addLastBar))
				newWidth = (trackBody.width() - width) / barList.count();
			
			if (greaterBar)
				newWidth = 0;
			
			drawSystem(barList, newWidth, barView->height());
			
			if ((!isLastBar && !greaterBar) || (isLastBar && ! addLastBar))
				delete barView;
			
			newWidth = width = 0;
			firstBar = i;
			
			// Decrease for the clef displaying because the clef changes
			// the length
			//
			if (i + 1 != currentTrack->count() || (i + 1 == currentTrack->count() && isLastBar && !addLastBar))
				i--;
			
			if (greaterBar) {
				i++;
				
				greaterBar = false;
			}
			
		} else {
			// Append the bar which will be displayed on the system (the current line)
			//
			barList.append(barView);
			barViewList.append(barView);
		}
	}
	
	canvasView->resize(trackResolution.width() , nbPages * trackResolution.height());
}

void TrackView::refreshCursor(const TrackPos& cursorPos)
{
	if (const TrackBarView* barView = barViewList.at(cursorPos.bar()))
		cursorView->refresh(barView->positionTablature(cursorPos));
	
	canvasView->update();
}

QPoint TrackView::cursorPosition() const
{
	return cursorView->position();
}

void TrackView::setCurrentTrack(const TabTrack* tabTrack)
{
	barViewList.clear();
	
	currentTrack = tabTrack;
	
	refreshProperties();
	refreshView(0);
	
	canvasView->update();
}

// Click event
//
TrackPos TrackView::mousePressEvent(const QPoint &position)
{
	QPtrListIterator<TrackBarView> it(barViewList);
	TrackBarView *barView;
	TrackPos trackPos;
	uint bar = 0;

	// Check in what bar the user click
	//	
	while ((barView = it.current()) != 0) {
		++it;
	
		if (position.x() > barView->positionViewer().x() &&
		    position.x() < barView->positionViewer().x() + barView->size() && 
		    position.y() > barView->positionViewer().y() && 
		    position.y() < barView->positionViewer().y() + barView->height()) {
				break;
			}
		
		bar++;
	}

	// The end of the list
	//	
	if (barView == 0)
		barView = barViewList.last();
	
	trackPos.setBar(bar);
	trackPos.setTimes(0);
	trackPos.setChord(0);
	
	QPoint point = barView->positionTablature(trackPos);
	
	// Check in what times the user click
	//
	while (point.isNull() == false && position.x() > point.x()) {
		trackPos.incTimes();
		point = barView->positionTablature(trackPos);
	}

	// The point passes the position so
	// decrease the times
	//
	if (position.x() < point.x())
		trackPos.decTimes();

	refreshCursor(trackPos);
	
	return trackPos;
}
