/***************************************************************************
* trackviewcommands.cpp: implementation of SetLengthCommand class
 *                                         InsertTabCommand
 *                                         MoveFingerCommand
 *                                         AddFXCommand
 *                                         SetFlagCommand
 *                                         DeleteNoteCommand
 *                                         AddColumnCommand
 *                                         SetTimeSigCommand
 *                                         InsertStrumCommand
 *                                         InsertRythm
 * 
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

/*
  Undo/Redo commands for TrackView
*/

#include "strumlib.h"
#include "tabsong.h"
#include "trackviewcommands.h"

extern strummer lib_strum[];

TrackEditView::DeadNoteCommand::DeadNoteCommand(TrackEditView& tv, TabTrack &trk, TrackPos& pos)
: KNamedCommand(i18n("Dead Note")), currentPos(pos), oldPos(pos), oldtab(trk.bar(pos.bar()).times(pos.times()).notes(pos.chord())),
  trackView(tv), tabTrack(trk)
{
	setName(i18n("Dead note"));
}

void TrackEditView::DeadNoteCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tab = oldtab;
	tab.setDeadNote(!oldtab.isDeadNote());
	
	tempTimes.setNotes(oldPos.chord(), tab);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(true);
}

void TrackEditView::DeadNoteCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tempTimes.setNotes(oldPos.chord(), oldtab);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(false);
}

TrackEditView::RestNoteCommand::RestNoteCommand(TrackEditView& tv, TabTrack &trk, TrackPos& pos)
: KNamedCommand(i18n("Rest Note")), currentPos(pos), oldPos(pos), oldTimes(trk.bar(pos.bar()).times(pos.times())),
  trackView(tv), tabTrack(trk)
{
	setName(i18n("Rest note"));
}

void TrackEditView::RestNoteCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tempTimes.setRest(!oldTimes.isRest());
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(true);
}

void TrackEditView::RestNoteCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	
	tempBar.setTimes(oldPos.times(), oldTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(false);
}

TrackEditView::SetLengthCommand::SetLengthCommand(TrackEditView& tv, TabTrack &trk, TrackPos& pos, Duration newLength)
: KNamedCommand(i18n("Set duration")), currentPos(pos), oldPos(pos), oldLength(trk.bar(pos.bar()).times(pos.times()).duration()), length(newLength), 
  oldTimes(trk.bar(pos.bar()).times(pos.times())), trackView(tv), tabTrack(trk)
{
	QString cmd(i18n("Set duration to %1"));
	QString dur;
	
	switch (length){
	case HundredTwentyEighth:
		dur = "1/128";
		break;
	case SixtyFourth:
		dur = "1/64";
		break;
	case ThirthSecond:  // 1/32
		dur = "1/32";
		break;
	case Sixteenth:  // 1/16
		dur = "1/16";
		break;
	case Eighth:  // 1/8
		dur = "1/8";
		break;
	case Quarter: // 1/4
		dur = "1/4";
		break;
	case Half: // 1/2
		dur = "1/2";
		break;
	case Whole: // whole
		dur = i18n("whole");
		break;
	case NoneDuration:
		dur = i18n("None duration");
		break;
	default:
		dur = i18n("Duration problem");
	}
	
	setName(cmd.arg(dur));
}

void TrackEditView::SetLengthCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	oldLength = tempTimes.duration();
	tempTimes.setDuration(length);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(true);
}

void TrackEditView::SetLengthCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	
	tempBar.setTimes(oldPos.times(), oldTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(false);
}

TrackEditView::DottedNoteCommand::DottedNoteCommand(TrackEditView& tv, TabTrack &trk, TrackPos& pos)
: KNamedCommand(i18n("Dotted Note")), currentPos(pos), oldPos(pos), oldtab(trk.bar(pos.bar()).times(pos.times()).notes(pos.chord())),
  oldtimes(trk.bar(pos.bar()).times(pos.times())), trackView(tv), tabTrack(trk)
{
	setName(i18n("Dotted note"));
}

void TrackEditView::DottedNoteCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tab = oldtab;
	
	if (!oldtimes.isDotted()) {
		tab.setDotted(true);
		tempTimes.setDotted(true);
	} else {
		bool dotted = false;
		
		for (uint i = 0; i < tabTrack.nbStrings(); i++) {
			TabNote tempNote = tempTimes.notes(i);
			
			if (i != oldPos.chord() && tempNote.isDotted()) {
				dotted = true;
				break;
			}
		}
		
		tab.setDotted(!tab.isDotted());
		
		if (!dotted) {
			tempTimes.setDotted(false);
		}
	}
	
	tempTimes.setNotes(oldPos.chord(), tab);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(true);
}

void TrackEditView::DottedNoteCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	
	tempBar.setTimes(oldPos.times(), oldtimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(false);
}

TrackEditView::InsertTabCommand::InsertTabCommand(TrackEditView& tv, TabTrack &trk, int num, TrackPos& pos)
: KNamedCommand(i18n("Insert tab")), currentPos(pos), oldPos(pos), oldtab(trk.bar(pos.bar()).times(pos.times()).notes(pos.chord())),
  trackView(tv), tabTrack(trk)
{
	setName(i18n("Insert tab %1").arg(QString::number(num)));
	
	tab = oldtab;
	tab.setFret(num);
}

void TrackEditView::InsertTabCommand::execute()
{
	int temp = tab.fret();
	
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	if (!oldtab.isEmpty())
		temp = temp + 10 * oldtab.fret();
	
	tab.setFret(temp);
	
	tempTimes.setNotes(oldPos.chord(), tab);
	tempTimes.setRest(false);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(true);
}

void TrackEditView::InsertTabCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tempTimes.setNotes(oldPos.chord(), oldtab);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(false);
}

TrackEditView::BendNoteCommand::BendNoteCommand(TrackEditView& tv, TabTrack &trk, TrackPos& pos, const TabBend& effect)
: KNamedCommand(i18n("Bend Note")), currentPos(pos), oldPos(pos), oldtab(trk.bar(pos.bar()).times(pos.times()).notes(pos.chord())),
  oldtimes(trk.bar(pos.bar()).times(pos.times())), trackView(tv), tabTrack(trk), bend(effect)
{
	setName(i18n("Bend note"));
}

void TrackEditView::BendNoteCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tab = oldtab;
	
	TabBend* tempBend = dynamic_cast<TabBend*>(&tab);
	*tempBend = bend;
	
	tempTimes.setNotes(oldPos.chord(), tab);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(true);
}

void TrackEditView::BendNoteCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	
	tempBar.setTimes(oldPos.times(), oldtimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.repaintCell();
	emit trackView.songChanged(false);
}

TrackEditView::MoveFingerCommand::MoveFingerCommand(TrackView *, TabTrack *, int _from, int _to, int /*_tune*/)
: KNamedCommand(i18n("Transpose"))
{
	kdDebug() << "TODO" << endl;
	
	if (_to < _from) {
		setName(i18n("Transpose down"));
	} else {
		setName(i18n("Transpose up"));
	}
}

void TrackEditView::MoveFingerCommand::execute()
{
}

void TrackEditView::MoveFingerCommand::unexecute()
{
}

TrackEditView::InsertNoteCommand::InsertNoteCommand(TrackEditView & tv, TabTrack& trk, TrackPos& pos)
: KNamedCommand(i18n("Insert note")), trackPos(pos), oldPos(pos),
  trackView(tv), oldBar(trk.bar(pos.bar())), tabTrack(trk)
{
	setName(i18n("Insert note %1").arg(trackPos.times()));
}

void TrackEditView::InsertNoteCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes;
	tempBar.insertTimes(tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	trackPos = oldPos;
	
	trackPos.incTimes();
	
	emit trackView.songChanged(true);
	trackView.repaintCell();
}

void TrackEditView::InsertNoteCommand::unexecute()
{
	tabTrack.setBar(oldPos.bar(), oldBar);
	
	trackPos = oldPos;
	
	emit trackView.songChanged(false);
	trackView.repaintCell();
}

TrackEditView::DeleteNoteCommand::DeleteNoteCommand(TrackEditView & tv, TabTrack& trk, TrackPos& pos)
: KNamedCommand(i18n("Delete note")), currentPos(pos), oldPos(pos),
  trackView(tv), oldBar(trk.bar(pos.bar())), tabTrack(trk)
{
	setName(i18n("Delete note %1").arg(pos.times()));
}

void TrackEditView::DeleteNoteCommand::execute()
{
	currentPos = oldPos;
	
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	bool checkDelete = false;
	
	for (uchar i = 0; i < tabTrack.nbStrings(); i++) {
		if (!tempTimes.notes(i).isEmpty()) {
			checkDelete = true;
			break;
		}
	}
	
	if (checkDelete) {
		TabNote note;
		tempTimes.setNotes(oldPos.chord(), note);
// 		tempTimes.setNotes(oldPos.chord(), 255);
		tempBar.setTimes(oldPos.times(), tempTimes);
	} else {
		tempBar.deleteTimes(oldPos.times());
		
		if (currentPos.times() >= 1) {
			currentPos.decTimes();
		}
	}
// 	TabBar tempBar = tabTrack.bar(oldPos.bar());
// 	tempBar.deleteTimes(oldPos.times());
	tabTrack.setBar(oldPos.bar(), tempBar);
	
/*	if (currentPos.times() >= 1) {
		currentPos.decTimes();
	}*/
	
	emit trackView.songChanged(true);
	trackView.repaintCell();
}

void TrackEditView::DeleteNoteCommand::unexecute()
{
	tabTrack.setBar(oldPos.bar(), oldBar);
	
	currentPos = oldPos;
	
	emit trackView.songChanged(false);
	trackView.repaintCell();
}

TrackEditView::DeleteColumnCommand::DeleteColumnCommand(TrackEditView & tv, TabTrack& trk, TrackPos& pos)
: KNamedCommand(i18n("Delete column")), currentPos(pos), oldPos(pos),
  countBar(trk.count()), oldBar(trk.bar(pos.bar())), trackView(tv), tabTrack(trk)
{
	setName(i18n("Delete %1 columns").arg(QString::number(pos.bar())));
}

void TrackEditView::DeleteColumnCommand::execute()
{
	currentPos = oldPos;
	
	if (countBar == 1) {
		TabBar tempBar;
		
		tempBar.setTimeSignature(0, tabTrack.timeSignature(0));
		tempBar.setTimeSignature(1, tabTrack.timeSignature(1));
		
		// Insert an empty bar
		//
		tabTrack.setBar(oldPos.bar(), tempBar);
		
		currentPos.setTimes(0);
		currentPos.setBar(0);
	} else {
		// Delete the current bar
		//
		tabTrack.deleteBar(currentPos.bar());
		
		// There is a bar beside the current bar so
		// we can decrease the bar number and the
		// the correct time position
		//
		if (currentPos.bar() > 0) {
			currentPos.decBar();
			
			TabBar tempBar = tabTrack.bar(currentPos.bar());
			currentPos.setTimes(tempBar.count() - 1);
		} else {
			currentPos.setTimes(0);
			currentPos.setBar(0);
		}
	}
	
	emit trackView.songChanged(true);
	trackView.update();
	trackView.repaintCell();
}

void TrackEditView::DeleteColumnCommand::unexecute()
{
	currentPos = oldPos;
	
	tabTrack.insertBar(oldPos.bar(), oldBar);
	
	emit trackView.songChanged(false);
	trackView.update();
	trackView.repaintCell();
}

TrackEditView::SetTimeSigCommand::SetTimeSigCommand(TrackEditView & tv, TabTrack& trk, int _bar, bool _toend, int _time1, int _time2)
:KNamedCommand(i18n("Set time sig.")), bar(_bar), time1(_time1), time2(_time2), oldTime1(trk.bar(_bar).timeSignature(0)),   
 oldTime2(trk.bar(_bar).timeSignature(1)), defaultOldTime1(trk.timeSignature(0)),
 defaultOldTime2(trk.timeSignature(0)), toend(_toend), tabTrack(trk), trackView(tv)
{
}

void TrackEditView::SetTimeSigCommand::execute()
{
	if (toend) {
		tabTrack.setTimeSignature(0, time1);
		tabTrack.setTimeSignature(1, time2);
	}
	
	TabBar tempBar = tabTrack.bar(bar);
	
	tempBar.setTimeSignature(0, time1);
	tempBar.setTimeSignature(1, time2);
	
	tabTrack.setBar(bar, tempBar);
	
	emit trackView.songChanged(true);
	trackView.repaintCell();

}

void TrackEditView::SetTimeSigCommand::unexecute()
{
	if (toend) {
		tabTrack.setTimeSignature(0, defaultOldTime1);
		tabTrack.setTimeSignature(1, defaultOldTime2);
	}
	
	TabBar tempBar = tabTrack.bar(bar);
	
	tempBar.setTimeSignature(0, oldTime1);
	tempBar.setTimeSignature(1, oldTime2);
	
	tabTrack.setBar(bar, tempBar);
	
	emit trackView.songChanged(false);
	trackView.update();
	trackView.repaintCell(); //for emit paneChanded
}

TrackEditView::InsertColumnCommand::InsertColumnCommand(TrackEditView & tv, TabTrack& trk, TrackPos& pos)
: KNamedCommand(i18n("Insert column")), trackPos(pos), oldPos(pos),
  tabTrack(trk), trackView(tv)
{
}

void TrackEditView::InsertColumnCommand::execute()
{
	TabBar tempBar;
	TabTimes emptyTimes;
	
	tempBar.setTimeSignature(0, tabTrack.timeSignature(0));
	tempBar.setTimeSignature(1, tabTrack.timeSignature(1));
	
	tempBar.insertTimes(emptyTimes);
	
	trackPos.incBar();
	trackPos.setTimes(0);
	
	tabTrack.insertBar(trackPos.bar(), tempBar);
	
	emit trackView.songChanged(true);
	trackView.repaintCell();
}

void TrackEditView::InsertColumnCommand::unexecute()
{
	tabTrack.deleteBar(oldPos.bar() + 1);
	
	trackPos = oldPos;
	
	trackView.update();
	emit trackView.songChanged(false);
	trackView.repaintCell();
}

TrackEditView::InsertStrumCommand::InsertStrumCommand(TrackEditView & tv, TabTrack& trk, TrackPos& pos, QMemArray<int> &_chord)
: KNamedCommand(i18n("Insert strum")), currentPos(pos), oldPos(pos),
  chord(_chord), oldTimes(trk.bar(pos.bar()).times(pos.times())), trackView(tv), tabTrack(trk)
{
	setName(i18n("Insert Chord"));
}

void TrackEditView::InsertStrumCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	for (unsigned int i = 0; i < tabTrack.nbStrings(); i++) {
		TabNote note;
		
		if (chord[i] == -1)
			note.setDeadNote(true);
		else
			note.setFret(chord[i]);
		
		tempTimes.setNotes(tabTrack.nbStrings() - 1 - i , note);
	}
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.update();
	emit trackView.songChanged(true);
	trackView.repaintCell();
}

void TrackEditView::InsertStrumCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tempBar.setTimes(oldPos.times(), oldTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	trackView.update();
	emit trackView.songChanged(false);
	trackView.repaintCell();
}

TrackEditView::InsertRhythm::InsertRhythm(TrackEditView & tv, TabTrack& trk, QListBox& quantized, TrackPos& pos)
: KNamedCommand(i18n("Insert rhythm")), currentPos(pos), oldPos(pos),
  tabTrack(trk), trackView(tv)
{
	newdur.resize(quantized.count() - 1);
	
	for (uint i = 1; i < quantized.count(); i++)
		newdur[i - 1] = static_cast<Duration>(quantized.text(i).toInt());
}

void TrackEditView::InsertRhythm::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	
	for (uint i = 0; i < newdur.count(); i++) {
		TabTimes tempTimes;
		
		tempTimes.setDuration(newdur[i]);
		tempBar.insertTimes(tempTimes);
	}
	
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	emit trackView.songChanged(true);
	trackView.repaintContents();
}

void TrackEditView::InsertRhythm::unexecute()
{
	tabTrack.setBar(oldPos.bar(), oldBar);
	
	currentPos = oldPos;
	
	emit trackView.songChanged(false);
	trackView.repaintContents();
}

TrackEditView::NTupletCommand::NTupletCommand(TrackEditView& _tv, TabTrack &_trk, TrackPos& pos, uint n)
: KNamedCommand(i18n("NTuplet")), currentPos(pos), oldPos(pos),
  tv(_tv), tabTrack(_trk), oldTime(_trk.bar(pos.bar()).times(pos.times())), ntuplet(n)
{
}

void TrackEditView::NTupletCommand::execute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	TabTimes tempTimes = tempBar.times(oldPos.times());
	
	tempTimes.setNTuplet(ntuplet);
	
	tempBar.setTimes(oldPos.times(), tempTimes);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	emit tv.songChanged(true);
	tv.repaintCell();
}

void TrackEditView::NTupletCommand::unexecute()
{
	TabBar tempBar = tabTrack.bar(oldPos.bar());
	
	tempBar.setTimes(oldPos.times(), oldTime);
	tabTrack.setBar(oldPos.bar(), tempBar);
	
	currentPos = oldPos;
	
	emit tv.songChanged(true);
	tv.repaintCell();
}
