/***************************************************************************
 *   Copyright (C) 2004 by Cyril Bosselut                                  *
 *   bosselut@b1project.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ktagmainwin.h"

Players::Players(){
}

Players::~Players(){
}

void Players::setProperties(const QString& n, const QString& p, const QString& e, const QString& c){
	player = n;
	playCmd = p;
	enqueueCmd = e;
	currentCmd = c;
}

QString Players::getPlayer(){
	return player;
}

QString Players::getPlayCmd(){
	return playCmd;
}

QString Players::getEnqueueCmd(){
	return enqueueCmd;
}

QString Players::getCurrentCmd(){
	return currentCmd;
}

KTagMainWin::KTagMainWin( QWidget* parent, const char* name, WFlags fl ) : DCOPObject( "KTag" ), KMainWindow( parent, name, fl ){
	mConfig = kapp->config();
	setupActions();
	setAcceptDrops(TRUE);
	tabWidget = new KTabWidget( this, "tabWidget" );
	tabWidget->setEnabled( true );
	tabWidget->setHoverCloseButton( TRUE );
	tabWidget->setAutoMask( TRUE );
	tabWidget->setAcceptDrops(TRUE);
	setCentralWidget(tabWidget);
	readProperties();
	connect(kapp,SIGNAL(lastWindowClosed()),qApp,SLOT(quit()));
	connect(tabWidget, SIGNAL(closeRequest(QWidget *)), this, SLOT(closeTab(QWidget *)));
	connect(tabWidget, SIGNAL(currentChanged(QWidget *)), this, SLOT(updateAlbumMenu(QWidget *)));
	connect(tabWidget, SIGNAL(currentChanged(QWidget *)), this, SLOT(updateFileMenu(QWidget *)));
}

KTagMainWin::~KTagMainWin(){
}

void KTagMainWin::setupActions(){
	KStdAction::save(this, SLOT(saveFile()), actionCollection());
	KStdAction::preferences(this, SLOT(config()), actionCollection());
	KStdAction::open(this, SLOT(chooseFile()), actionCollection());
	recent = KStdAction::openRecent(this, SLOT(openFile(const KURL&)), actionCollection());
	mConfig->setGroup("KFileDialog Settings");
	recent->loadEntries(mConfig);
	(void) new KAction(tr2i18n("Save &All"), 0, this, SLOT(saveAll()), actionCollection(), "saveAll");
	//(void) new KAction(tr2i18n("Clean Mime Type"), 0, this, SLOT(cleanMime()), actionCollection(), "cleanMime");
	(void) new KAction(tr2i18n("R&emove tags"), 0, this, SLOT(clearTags()), actionCollection(), "clearTags");
	(void) new KAction(tr2i18n("Download cover"), 0, this, SLOT(getAlbumCover()), actionCollection(), "getCover");
	(void) new KAction(tr2i18n("Create Index"), 0, this, SLOT(createHTML()), actionCollection(), "makeIndex");
	createGUI();
}

void KTagMainWin::saveFile(){
	kdDebug("MainWin") << "Saving current file" << endl;
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	mKtag->writeTags(false);
}

void KTagMainWin::clearTags(){
	int result = KMessageBox::warningYesNo(this,tr2i18n( "Are you sure you want to clear all tags for this song?" ));
	if(result == KMessageBox::Yes){
		Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
		mKtag->clearTags();
	}
}

void KTagMainWin::cleanMime(){
	int result = KMessageBox::warningYesNo(this,tr2i18n( "Are you sure you want to correct Mime Type for this song?" ));
	if(result == KMessageBox::Yes){
		Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
		mKtag->cleanMime();
	}
}

void KTagMainWin::saveAll(){
	if(tabWidget->count() > 0){
		KProgressDialog *p = new KProgressDialog(this, "progress", tr2i18n("Writing tags"), tr2i18n("Saving all files"), true);
		p->showCancelButton(false);
		p->progressBar()->setTotalSteps(tabWidget->count());
		kdDebug("MainWin") << "Saving all tabs " << endl;
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			if(mKtag->modified){
				mKtag->writeTags(true);
			}
			p->progressBar()->setProgress(i+1);
		}
	}
}

void KTagMainWin::closeEvent(QCloseEvent *ce){
	if(tabWidget->count() > 0){
		kdDebug("MainWin") << "Closing tabs " << endl;
		while(tabWidget->count() > 0){
			closeTab(tabWidget->page(0));
		}
	}
	ce->accept();
}

bool KTagMainWin::chooseFile(){
	KURL::List urlList;
	bool b = false;
	urlList = KFileDialog::getOpenURLs(QString::null, "*.mp3 *.ogg *.flac", 0, "Open audio file");
	KURL::List::ConstIterator end( urlList.end() );
	for( KURL::List::ConstIterator url( urlList.begin() ); url != end; ++url ) {
		if( (*url).isLocalFile() ){
			openFile(*url);
			b = true;
		}
	}
	return b;
}

void KTagMainWin::openFile(const KURL &url){
	Ktag* mKtag;
	KIconLoader* ic = new KIconLoader("ktag");
	mKtag = new Ktag(this, url.path());
	if(mKtag->setFile(url.path())){
		kdDebug("MainWin") << "Adding new tab" << endl;
		tabWidget->insertTab( mKtag, "" );
		tabWidget->setTabIconSet( mKtag, ic->loadIconSet( "multimedia", KIcon::Toolbar ) );
		setTabToolTip(mKtag, url.fileName());
		updateTabsLabel();
		connect(mKtag, SIGNAL(tabClosed(QWidget *)), this, SLOT(closeTab(QWidget *)));
		connect(mKtag, SIGNAL(modifiedTag(QWidget *, bool)), this, SLOT(updateTabLabel(QWidget *, bool)));
		connect(mKtag, SIGNAL(modifiedTag(QWidget *, bool)), this, SLOT(updateAlbumMenu(QWidget *)));
		connect(mKtag, SIGNAL(modifiedTag(QWidget *, bool)), this, SLOT(updateFileMenu(QWidget *)));
		connect(mKtag, SIGNAL(fileChanged(QWidget *, const QString&)), this, SLOT(setTabToolTip(QWidget *, const QString&)));
		connect(mKtag, SIGNAL(receivedDropEvent(QDropEvent *)), this, SLOT(dropEvent(QDropEvent *)));
		tabWidget->showPage(mKtag);
		mKtag->setAcceptDrops(TRUE);
		recent->addURL(url);
		recent->saveEntries(mConfig);
	}
	mKtag = 0;
	if(tabWidget->count() == 0){
		kdDebug("MainWin") << "no more tabs, exit" << endl;
		exit(0);
	}
}

void KTagMainWin::openFile(const QString& fp){
	kdDebug("MainWin") << fp << endl;
	KURL url;
	if(fp.contains("%")){
		url.setEncodedPath(QString(fp).remove("file:"));
	}
	else{
		url.setPath(fp);
	}
	openFile(url);
}


void KTagMainWin::closeTab( QWidget* w ){
	kdDebug("MainWin") << "Closing tab " << w->name() << endl;
	w->close();
	tabWidget->removePage(w);
	if(tabWidget->count() == 0){
		kapp->quit();
		return;
	}
	updateTabsLabel();
}

void KTagMainWin::setTabToolTip(QWidget*w, const QString& str){
	tabWidget->setTabToolTip(w, str);
}

void KTagMainWin::updateTabLabel(QWidget*w, bool b){
	Ktag* mKtag = static_cast<Ktag*>(w);
	updateTabsLabel();
	if(b && QFileInfo(mKtag->mAudioFile).isWritable()){
		tabWidget->setTabColor(w, QColor("blue"));
		return;
	}
	if(!QFileInfo(mKtag->mAudioFile).isWritable()){
		tabWidget->setTabColor(w, KGlobalSettings::inactiveTextColor());
		return;
	}
	tabWidget->setTabColor(w, KGlobalSettings::textColor());
}

void KTagMainWin::updateTabsLabel(){
	int numTabs = tabWidget->count();
	if(numTabs > 0){
		int mLength = 0;
		int maxLength = 30;
		if(numTabs > 3 && numTabs < (maxLength/3)){
			maxLength -= numTabs*3;
		}
		else if(numTabs >= (maxLength/3)){
			maxLength = 3;
		}
		for(int i = 0; i <= numTabs-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			QString fileName = mKtag->url().fileName(true);
			mLength = fileName.length();
			if(mLength > maxLength && numTabs > 1){
				fileName = fileName.left( maxLength - 3 ) + "...";
			}
			tabWidget->setTabLabel(mKtag, fileName);
		}
	}
}

void KTagMainWin::dragMoveEvent( QDragMoveEvent *e){
	e->accept(false);
}

void KTagMainWin::dropEvent( QDropEvent *e){
	kdDebug("Ktag") << "File dropped" << endl;
	if(e->provides("text/uri-list")){
		e->accept( true );
		e->encodedData("text/uri-list");
		QStrList lst;
		QString filename;
		QUriDrag::decode(e, lst);
		for ( uint i = 0; i < lst.count(); ++i ) {
			openFile(QString(QDir::convertSeparators(QUriDrag::uriToLocalFile(lst.at(i))).utf8()));
		}
	}
	else{
		e->ignore();
		kdDebug("Ktag") << "Droped file rejected: " << e->format() << endl;
	}
}

void KTagMainWin::config(){
	//An instance of your dialog could be already created and could be cached,
	//in which case you want to display the cached dialog instead of creating
	//another one
	if ( KConfigDialog::showDialog( "configdlg" ) )
		return;

	//KConfigDialog didn't find an instance of this dialog, so lets create it :
	KConfigDialog* dialog = new KConfigDialog( this, "configdlg", Prefs::self(), KDialogBase::IconList, KDialogBase::Ok|KDialogBase::Cancel|KDialogBase::Help );
	patternConfig* patDlg = new patternConfig( 0, "dlPatterns" );
	confSavePatDlg* saveDlg = new confSavePatDlg( 0, "savePatterns" );
	playerConfig* playerDlg = new playerConfig( 0, "playersCmd" );

	dialog->addPage( patDlg, i18n("Cover patterns"), "package_network" );
	dialog->addPage( saveDlg, i18n("File patterns"), "package_editors" );
	dialog->addPage( playerDlg, i18n("Players"), "package_multimedia" );

#if HAVE_MUSICBRAINZ
	mbConfig* mBrainzDlg = new mbConfig( 0, "mbParams" );
	dialog->addPage( mBrainzDlg, i18n("MusicBrainz"), "package_utilities" );
	mBrainzDlg->autoMB->setChecked(mAutoMB);
	QStringList::ConstIterator itl = mbServerList.begin();
	QStringList::ConstIterator enl = mbServerList.end();
	QString str;
	QStringList ls;
	while( itl != enl ){
		str = *itl;
		ls = QStringList::split( ':', str);
		mBrainzDlg->addServer(ls.first(), ls.last().toInt());
		itl++;
	}
	mBrainzDlg->mbServerList->setSelected(mBrainzDlg->mbServerList->findItem(mbServer, 0, ExactMatch), true);
	mBrainzDlg->mProxyServer->setText(mbProxyServer);
	mBrainzDlg->mProxyPort->setText(QString::number(mbProxyPort));

#endif // HAVE_MUSICBRAINZ

	patDlg->addPatternList(mPatterns);
	saveDlg->addPatternList(mSavePatterns);
	QValueList<Players*>::iterator it;
	QValueList<Players*>::iterator end = mPlayers.end();
	for( it = mPlayers.begin(); it != end; ++it ){
		playerDlg->addPlayer((*it)->getPlayer(),(*it)->getPlayCmd(),(*it)->getEnqueueCmd(),(*it)->getCurrentCmd());
	}
	//User edited the configuration - update your local copies of the
	//configuration data
	//connect( dialog, SIGNAL(settingsChanged()), this, SLOT(saveProperties()) );

	//dialog->show();
	if(dialog->exec()){
		kdDebug("Ktag") << "Save config" << endl;
		mPatterns.clear();
		for(int i = 0; i <= patDlg->kcfg_dlPatterns->childCount()-1; i++){
			mPatterns << patDlg->kcfg_dlPatterns->itemAtIndex(i)->text(0);
		}
		mSavePatterns.clear();
		for(int i = 0; i <= saveDlg->kcfg_savePatterns->childCount()-1; i++){
			mSavePatterns << saveDlg->kcfg_savePatterns->itemAtIndex(i)->text(0);
		}
		mPlayers.clear();
		QListViewItemIterator it( playerDlg->mPlayerList );
		while(it.current()){
			QListViewItem *item = it.current();
			if(item->childCount() > 0){
				Players* pl = new Players();
				QString p;
				QString e;
				QString c;
				QListViewItem* cmd1 = item->firstChild();
				if(cmd1->text(0) == tr2i18n("Play")){
					p = cmd1->text(1);
				}
				else if(cmd1->text(0) == tr2i18n("Enqueue")){
					e = cmd1->text(1);
				}
				else if(cmd1->text(0) == tr2i18n("Current")){
					c = cmd1->text(1);
				}
				QListViewItem* cmd2 = cmd1->nextSibling();
				if(cmd2->text(0) == tr2i18n("Play")){
					p = cmd2->text(1);
				}
				else if(cmd2->text(0) == tr2i18n("Enqueue")){
					e = cmd2->text(1);
				}
				else if(cmd2->text(0) == tr2i18n("Current")){
					c = cmd2->text(1);
				}
				QListViewItem* cmd3 = cmd2->nextSibling();
				if(cmd3->text(0) == tr2i18n("Play")){
					p = cmd3->text(1);
				}
				else if(cmd3->text(0) == tr2i18n("Enqueue")){
					e = cmd3->text(1);
				}
				else if(cmd3->text(0) == tr2i18n("Current")){
					c = cmd3->text(1);
				}
				pl->setProperties(item->text(0), p, e, c);
				mPlayers.append(pl);
			}
			it++;
		}
#if HAVE_MUSICBRAINZ
	mAutoMB = mBrainzDlg->autoMB->isChecked();
	if(!mBrainzDlg->mbServerList->selectedItems().isEmpty()){
		mbServer = mBrainzDlg->mbServerList->selectedItem()->text(0);
		mbPort = mBrainzDlg->mbServerList->selectedItem()->text(1).toInt();
	}
	QListViewItemIterator itb( mBrainzDlg->mbServerList );
	mbServerList.clear();
	while(itb.current()){
		mbServerList += (*itb)->text(0)+":"+(*itb)->text(1);
		itb++;
	}
	mbProxyServer = mBrainzDlg->mProxyServer->text();
	mbProxyPort = mBrainzDlg->mProxyPort->text().toInt();
#endif // HAVE_MUSICBRAINZ
		saveProperties();
		readProperties();
	}
	else{
		kdDebug("Ktag") << "Do not save config" << endl;
	}
	delete(dialog);
}

void KTagMainWin::updateStatus(QWidget* w,const QString& str){
	tabWidget->setTabToolTip(w, str);
}

void KTagMainWin::saveProperties(){
	mConfig->setGroup("ktag");
	mConfig->writeEntry("patterns", mPatterns.join(","));
	mConfig->writeEntry("svpatterns", mSavePatterns.join(","));
	mConfig->setGroup("Players");
	QValueList<Players*>::iterator it;
	QValueList<Players*>::iterator end = mPlayers.end();
	QString playersNames = QString::null;
	for( it = mPlayers.begin(); it != end; ++it ){
		if(it != mPlayers.begin()){
			playersNames += ",";
		}
		playersNames += (*it)->getPlayer();
		mConfig->writeEntry(((*it)->getPlayer())+"_play", (*it)->getPlayCmd());
		mConfig->writeEntry(((*it)->getPlayer())+"_enqueue", (*it)->getEnqueueCmd());
		mConfig->writeEntry(((*it)->getPlayer())+"_current", (*it)->getCurrentCmd());
	}
	mConfig->writeEntry("players", playersNames);
#if HAVE_MUSICBRAINZ
	mConfig->setGroup("MusicBrainz");
	mConfig->writeEntry("server", mbServer);
	mConfig->writeEntry("servers", mbServerList.join(","));
	mConfig->writeEntry("port", mbPort);
	mConfig->writeEntry("prxserver", mbProxyServer);
	mConfig->writeEntry("prxport", mbProxyPort);
	if(mAutoMB){
		mConfig->writeEntry("mbAutoSearch", "true");
	}
	else{
		mConfig->writeEntry("mbAutoSearch", "false");
	}
#endif // HAVE_MUSICBRAINZ
	mConfig->sync();
}

void KTagMainWin::readProperties(){
	kdDebug("Ktag") << "Read config" << endl;
	mConfig->setGroup("ktag");
	mPatterns = mConfig->readListEntry("patterns", ',');
	mSavePatterns = mConfig->readListEntry("svpatterns", ',');
	QStringList::ConstIterator it = mSavePatterns.begin();
	QStringList::ConstIterator end = mSavePatterns.end();
	QPopupMenu *saveAsPopup = static_cast<QPopupMenu*>(factory()->container("saveas",this));
	QPopupMenu *saveTrckAsPopup = static_cast<QPopupMenu*>(factory()->container("saveTrckAs",this));
	QPopupMenu *renamePopup = static_cast<QPopupMenu*>(factory()->container("rename",this));
	QPopupMenu *renameTrckAsPopup = static_cast<QPopupMenu*>(factory()->container("renameTrckAs",this));
	saveAsPopup->clear();
	saveTrckAsPopup->clear();
	renamePopup->clear();
	renameTrckAsPopup->clear();
	while( it != end ){
		saveAsPopup->insertItem( *it,  this, SLOT(saveAs(int)));
		saveTrckAsPopup->insertItem( *it,  this, SLOT(saveAllAs(int)));
		renamePopup->insertItem( *it,  this, SLOT(rename(int)));
		renameTrckAsPopup->insertItem( *it,  this, SLOT(renameAll(int)));
		it++;
	}
	mConfig->setGroup("Players");
	mPlayers.clear();
	QStringList players = mConfig->readListEntry("players", ',');
	QStringList::ConstIterator itp = players.begin();
	QStringList::ConstIterator endp = players.end();
	QPopupMenu *playPopup = static_cast<QPopupMenu*>(factory()->container("play",this));
	QPopupMenu *enqueuePopup = static_cast<QPopupMenu*>(factory()->container("enqueue",this));
	QPopupMenu *currentPopup = static_cast<QPopupMenu*>(factory()->container("current",this));
	playPopup->clear();
	enqueuePopup->clear();
	currentPopup->clear();
	while( itp != endp ){
		kdDebug("Ktag") << "Player: " << *itp << endl;
		Players* pl = new Players();
		QString p = mConfig->readEntry((*itp)+"_play", "");
		QString e = mConfig->readEntry((*itp)+"_enqueue", "");
		QString c = mConfig->readEntry((*itp)+"_current", "");
		pl->setProperties(*itp, p, e, c);
		if(!p.isEmpty()){
			playPopup->insertItem( *itp,  this, SLOT(play(int)));
		}
		if(!e.isEmpty()){
			enqueuePopup->insertItem( *itp,  this, SLOT(enqueue(int)));
		}
		if(!c.isEmpty()){
			currentPopup->insertItem( *itp,  this, SLOT(getCurrent(int)));
		}
		mPlayers.append(pl);
		itp++;
	}
#if HAVE_MUSICBRAINZ
	mConfig->setGroup("MusicBrainz");
	mAutoMB = mConfig->readBoolEntry("mbAutoSearch", true);
	mbServer =  mConfig->readEntry("server", "mm.musicbrainz.org");
	mbServerList =  mConfig->readListEntry("servers", ',');
	mbPort =  mConfig->readNumEntry("port", 80);
	mbProxyServer =  mConfig->readEntry("prxserver");
	mbProxyPort =  mConfig->readNumEntry("prxport");
#endif // HAVE_MUSICBRAINZ
	for(int i = 0; i <= tabWidget->count()-1; i++){
		Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
		mKtag->readProperties();
	}
}

void KTagMainWin::saveAs(int i){
	QPopupMenu *saveAsPopup = static_cast<QPopupMenu*>(factory()->container("saveas",this));
	saveAs(saveAsPopup->text(i));
}

void KTagMainWin::saveAllAs(int i){
	QPopupMenu *saveTrckAsPopup = static_cast<QPopupMenu*>(factory()->container("saveTrckAs",this));
	saveAllAs(saveTrckAsPopup->text(i));
}

void KTagMainWin::rename(int i){
	QPopupMenu *renamePopup = static_cast<QPopupMenu*>(factory()->container("rename",this));
	rename(renamePopup->text(i));
}

void KTagMainWin::renameAll(int i){
	QPopupMenu *renameTrckAsPopup = static_cast<QPopupMenu*>(factory()->container("renameTrckAs",this));
	renameAll(renameTrckAsPopup->text(i));
}

void KTagMainWin::saveAs(const QString& p){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	mKtag->saveAs(p);
}

void KTagMainWin::saveAllAs(const QString& p){
	if(tabWidget->count() > 0){
		KProgressDialog *progress = new KProgressDialog(this, "progress", tr2i18n("Copying files"), tr2i18n("Copying files"), true);
		progress->showCancelButton(false);
		progress->progressBar()->setTotalSteps(tabWidget->count());
		kdDebug("MainWin") << "Copying files" << endl;
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mKtag->rename(p);
			progress->progressBar()->setProgress(i+1);
		}
	}
}

void KTagMainWin::rename(const QString& p){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	mKtag->rename(p);
}

void KTagMainWin::renameAll(const QString& p){
	if(tabWidget->count() > 0){
		KProgressDialog *progress = new KProgressDialog(this, "progress", tr2i18n("Renaming files"), tr2i18n("Renaming files"), true);
		progress->showCancelButton(false);
		progress->progressBar()->setTotalSteps(tabWidget->count());
		kdDebug("MainWin") << "Renaming files" << endl;
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mKtag->rename(p);
			progress->progressBar()->setProgress(i+1);
		}
	}
}

void KTagMainWin::updateAlbumMenu(QWidget *w){
	Ktag* mKtag = static_cast<Ktag*>(w);
	if(tabWidget->indexOf(mKtag) == tabWidget->currentPageIndex()){
		QPopupMenu *artistPopup = static_cast<QPopupMenu*>(factory()->container("setArtist",this));
		QPopupMenu *albumPopup = static_cast<QPopupMenu*>(factory()->container("setAlbum",this));
		QPopupMenu *yearPopup = static_cast<QPopupMenu*>(factory()->container("setYear",this));
		QPopupMenu *genrePopup = static_cast<QPopupMenu*>(factory()->container("setGenre",this));
		artistPopup->clear();
		albumPopup->clear();
		yearPopup->clear();
		genrePopup->clear();
		if(!mKtag->mArtist->currentText().isEmpty()){
			artistPopup->insertItem( mKtag->mArtist->currentText().replace("&","&&"),  this, SLOT(setArtist()));
		}
		if(!mKtag->mAlbum->currentText().isEmpty()){
			albumPopup->insertItem( mKtag->mAlbum->currentText().replace("&","&&"),  this, SLOT(setAlbum()));
		}
		if(mKtag->mYear->value() != 0){
			yearPopup->insertItem( QString::number(mKtag->mYear->value()),  this, SLOT(setYear()));
		}
		if(!mKtag->mGenre->currentText().isEmpty()){
			genrePopup->insertItem( mKtag->mGenre->currentText().replace("&","&&"),  this, SLOT(setGenre()));
		}
	}
}

void KTagMainWin::updateFileMenu(QWidget *w){
	Ktag* mKtag = static_cast<Ktag*>(w);
	bool writable = QFileInfo(mKtag->mAudioFile).isWritable();
	KAction *save = actionCollection()->action(KStdAction::stdName(KStdAction::Save));
	save->setEnabled(mKtag->modified && writable);
	//KAction *cleanMime = actionCollection()->action("cleanMime");
	//cleanMime->setEnabled(mKtag->tags->getMimeType() == "text/plain" && writable);
	KAction *clearTags = actionCollection()->action("clearTags");
	clearTags->setEnabled(writable);
	QPopupMenu *renamePopup = static_cast<QPopupMenu*>(factory()->container("rename",this));
	renamePopup->setEnabled(writable);
}

void KTagMainWin::setArtist(){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	setArtist(mKtag->mArtist->currentText().replace("&&","&"));
	tabWidget->setCurrentPage(tabWidget->indexOf(mKtag));
}

void KTagMainWin::setArtist(const QString& p){
	if(tabWidget->count() > 0){
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mKtag->mArtist->setCurrentText(p);
		}
	}
}

void KTagMainWin::setAlbum(){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	setAlbum(mKtag->mAlbum->currentText().replace("&&","&"));
	tabWidget->setCurrentPage(tabWidget->indexOf(mKtag));
}

void KTagMainWin::setAlbum(const QString& p){
	if(tabWidget->count() > 0){
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mKtag->mAlbum->setCurrentText(p);
		}
	}
}

void KTagMainWin::setGenre(){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	setGenre(mKtag->mGenre->currentText().replace("&&","&"));
	tabWidget->setCurrentPage(tabWidget->indexOf(mKtag));
}

void KTagMainWin::setGenre(const QString& p){
	if(tabWidget->count() > 0){
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mKtag->mGenre->setCurrentText(p);
		}
	}
}

void KTagMainWin::setYear(){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	setYear(mKtag->mYear->value());
	tabWidget->setCurrentPage(tabWidget->indexOf(mKtag));
}

void KTagMainWin::setYear(const int y){
	if(tabWidget->count() > 0){
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mKtag->mYear->setValue(y);
		}
	}
}

void KTagMainWin::getAlbumCover(){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	QString folder = KURL(mKtag->mAudioFile).directory(true);
	if(getAlbumCover(mKtag->mArtist->currentText(), mKtag->mAlbum->currentText(), folder)){
		mKtag->mCoverInfo->setEnabled(true);
		mKtag->mCover->setEnabled(true);
		if(QFile::exists(folder + "/cover.png")){
			mKtag->mCover->setPixmap(folder + "/cover.png");
			mKtag->mCoverInfo->setText(tr2i18n("<b>Local file:</b><br>") + folder + "/cover.png");
		}
		else if(QFile::exists(folder + "/cover.jpg")){
			mKtag->mCover->setPixmap(folder + "/cover.jpg");
			mKtag->mCoverInfo->setText(tr2i18n("<b>Local file:</b><br>") + folder + "/cover.jpg");
		}
		else if(QFile::exists(folder + "/cover.jpeg")){
			mKtag->mCover->setPixmap(folder + "/cover.jpeg");
			mKtag->mCoverInfo->setText(tr2i18n("<b>Local file:</b><br>") + folder + "/cover.jpeg");
		}
	}
}

bool KTagMainWin::getAlbumCover(const QString &artist, const QString &album, const QString &folder){
	dlCover* dl = new dlCover();
	Tags *tags = new Tags();
	tags->setAlbum(album);
	tags->setArtist(artist);
	QStringList::Iterator it = mPatterns.begin();
	QStringList::Iterator end = mPatterns.end();
	while( it!= end ){
		KURL url(tags->parsePattern(*it));
		dl->testMimeType(url);
		it++;
	}
	if(dl->coverList->childCount() != 0){
		if(dl->exec() && !dl->coverList->selectedItems().isEmpty()){
			QString nfileName;
			nfileName = KFileDialog::getSaveFileName(folder + "/cover.png", "*.png\n*.jpg *.jpeg", 0, "Save Cover");
			if(nfileName != ""){
				QString ext = "JPEG";
				if(nfileName.contains(".png")){
					ext = "PNG";
				}
				QPixmap* pix = dl->downloadCover(dl->coverList->selectedItem()->text(1));
				pix->save(nfileName, ext, -1);
				delete tags;
				return true;
			}
		}
		delete tags;
		return false;
	}
	KMessageBox::sorry(this, tr2i18n("No cover found for album \"%1\" by \"%2\"").arg(album, artist), QString::null, KMessageBox::AllowLink);
	delete tags;
	return false;
}

album* KTagMainWin::makeAlbum(Ktag* tag){
	album* mAlbum = new album();
	mAlbum->setTitle(tag->mAlbum->currentText());
	mAlbum->setArtist(tag->mArtist->currentText());
	mAlbum->setYear(tag->mYear->value());
	mAlbum->setGenre(tag->mGenre->currentText());
	QString path = KURL(tag->mAudioFile).directory(true);
	if(QFile::exists(path + "/cover.png")){
		mAlbum->setCover(path.append("/cover.png"));
	}
	else if(QFile::exists(path + "/cover.jpg")){
		mAlbum->setCover(path.append("/cover.jpg"));
	}
	else if(QFile::exists(path + "/cover.jpeg")){
		mAlbum->setCover(path.append("/cover.png"));
	}
	if(tabWidget->count() > 0){
		for(int i = 0; i <= tabWidget->count()-1; i++){
			Ktag* mKtag = static_cast<Ktag*>(tabWidget->page(i));
			mAlbum->setTrack(mKtag->mTrack->value(), mKtag->mTitle->currentText(), mKtag->mAudioFile);
		}
	}
	return mAlbum;
}

void KTagMainWin::createHTML(){
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	album* mAlbum = makeAlbum(mKtag);
	KLocale*loc = KGlobal::locale();
	QFile mIndex( KURL(mKtag->mAudioFile).directory(true).append("/index.html") );
	kdDebug() << mIndex.name() << endl;
	QString html;
	QString encoding = loc->encoding();
	html = "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n";
	html += "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\""+ loc->language()+"\" lang=\""+ loc->language()+"\">\n";
	html += "	<head>\n";
	html += "		<meta name=\"lang\" content=\""+ loc->language()+"\" />\n";
	html += "		<meta http-equiv=\"Content-Type\" content=\"text/html; charset="+encoding.replace(" ", "-")+"\" />\n";
	html += "		<title>"+ mAlbum->getTitle() +"</title>\n";
	html += "		<link title=\"Clear\" rel=\"StyleSheet\" href=\""+locate("data", "ktag/default.css")+"\" type=\"text/css\" media=\"all\" rev=\"StyleSheet\" />";
	html += "	</head>\n";
	html += "	<body>\n";
	html += "		<div id=\"album\" class=\""+mAlbum->getGenre().replace(" ","_")+"\">\n";
	html += "			<span id=\"genre\" class=\""+mAlbum->getGenre().replace(" ","_")+"\">"+ mAlbum->getGenre() +"</span>\n";
	html += "			<span id=\"artist\">"+ mAlbum->getArtist() +"</span>\n";
	html += "			<span id=\"title\">"+ mAlbum->getTitle() +"</span>\n";
	html += "			<span id=\"year\">"+ QString::number(mAlbum->getYear()) +"</span>\n";
	if(mAlbum->getCover() != QString::null){
		html += "			<span id=\"cover\"><img src=\""+ mAlbum->getCover().path() +"\" style=\"width: 256px;\" alt=\"cover\" /></span>\n";
	}
	html += "			<ul id=\"tracks\">\n";
	track* tr;
	TrackList tl = mAlbum->getTracks();
	QString trackcss = "track0";
	for(tr = tl.first(); tr; tr = tl.next()){
		html += "				<li class=\""+trackcss+"\" style=\"list-style-type: none;\">\n";
		html += "					<span class=\"pos\">"+ QString::number(tr->getPos()) +"</span>\n";
		html += "					<span class=\"song\"><a href=\"";
		html += tr->getURL().path();
		html += "\">"+ tr->getTitle() +"</a></span>\n";
		QString comment = "&nbsp;";
		if(tr->getComment() != ""){
			comment = tr->getComment();
		}
		html += "					<span class=\"comment\">"+ comment +"</span>\n";
		html += "				</li>\n";
		if(trackcss == "track0"){
			trackcss = "track1";
		}
		else{
			trackcss = "track0";
		}
	}
	html += "			</ul>\n";
	html += "		</div>\n";
	html += "		<div id=\"foot\">\n";
	html += tr2i18n("			Created by <a href=\"http://b1project.com/ktag.php3\">Ktag 0.3</a>\n");
	html += "		</div>\n";
	html += "	</body>\n";
	html += "</html>\n";
	mIndex.open(IO_WriteOnly);
	QTextStream stream( &mIndex );
	stream << html;
	mIndex.close();
}

void KTagMainWin::getCurrent(int i){
	QPopupMenu *currentPopup = static_cast<QPopupMenu*>(factory()->container("current",this));
	kdDebug() << "Getting current from: " << currentPopup->text(i) << endl;
	getCurrent(currentPopup->text(i));
}

void KTagMainWin::getCurrent(const QString& player){
	QValueList<Players*>::iterator it;
	QValueList<Players*>::iterator end = mPlayers.end();
	for( it = mPlayers.begin(); it != end; ++it ){
		if((*it)->getPlayer() == player && (*it)->getCurrentCmd() != QString::null){
			KProcess *proc = new KProcess;
			proc->setUseShell(true);
			*proc << (*it)->getCurrentCmd();
			connect(proc, SIGNAL(receivedStdout(KProcess *, char *, int)), this, SLOT(openFile(KProcess *, char *, int)));
			if(proc->start(KProcIO::NotifyOnExit, KProcIO::Stdout)){
				kdDebug() << "Starting: " << (*it)->getCurrentCmd() << endl;
			}
			return;
		}
	}
}

void KTagMainWin::openFile(KProcess *, char *buffer, int buflen){
	QString myFile = QString::fromLatin1(buffer, buflen -1);
	kdDebug() << "Output from player: " << myFile << endl;
	openFile(myFile);
}

void KTagMainWin::play(int i){
	QPopupMenu *playPopup = static_cast<QPopupMenu*>(factory()->container("play",this));
	kdDebug() << "Play song with: " << playPopup->text(i) << endl;
	play(playPopup->text(i));
}

void KTagMainWin::play(const QString& player){
	QValueList<Players*>::iterator it;
	QValueList<Players*>::iterator end = mPlayers.end();
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	for( it = mPlayers.begin(); it != end; ++it ){
		if((*it)->getPlayer() == player && (*it)->getPlayCmd() != QString::null){
			KProcess *proc = new KProcess;
			proc->setUseShell(true);
			QString cmd = (*it)->getPlayCmd();
			if(cmd.contains("%u")){
				cmd.replace("%u",KURL(mKtag->mAudioFile).url());
			}
			if(cmd.contains("%s")){
				cmd.replace("%s","'"+mKtag->mAudioFile+"'");
			}
			*proc << cmd;
			if(proc->start(KProcIO::DontCare, KProcIO::NoCommunication)){
				kdDebug() << "Starting: " << cmd << endl;
			}
			return;
		}
	}
}

void KTagMainWin::enqueue(int i){
	QPopupMenu *enqueuePopup = static_cast<QPopupMenu*>(factory()->container("enqueue",this));
	kdDebug() << "Enqueue song in: " << enqueuePopup->text(i) << endl;
	enqueue(enqueuePopup->text(i));
}

void KTagMainWin::enqueue(const QString& player){
	QValueList<Players*>::iterator it;
	QValueList<Players*>::iterator end = mPlayers.end();
	Ktag* mKtag = static_cast<Ktag*>(tabWidget->currentPage());
	for( it = mPlayers.begin(); it != end; ++it ){
		if((*it)->getPlayer() == player && (*it)->getEnqueueCmd() != QString::null){
			KProcess *proc = new KProcess;
			proc->setUseShell(true);
			QString cmd = (*it)->getEnqueueCmd();
			if(cmd.contains("%u")){
				cmd.replace("%u",KURL(mKtag->mAudioFile).url());
			}
			if(cmd.contains("%s")){
				cmd.replace("%s","'"+mKtag->mAudioFile+"'");
			}
			*proc << cmd;
			if(proc->start(KProcIO::DontCare, KProcIO::NoCommunication)){
				kdDebug() << "Starting: " << cmd << endl;
			}
			return;
		}
	}
}

#include "ktagmainwin.moc"
