/***************************************************************************
 *   Copyright (C) 2004 by Cyril Bosselut                                  *
 *   bosselut@b1project.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ktag.h"


/**
* @author Cyril Bosselut <bosselut@b1project.com>
* @version 0.5
*/

class KURLRequester;
class QLabel;
class FileCopyJob;

Ktag::Ktag( QWidget* parent, const char* name, WFlags fl )
    : ktagDlg( parent, name, fl )
{
	setAcceptDrops(TRUE);
	mGenre->insertStrList(sorted_genres);
	readProperties();
#if HAVE_MUSICBRAINZ
	KIconLoader* ic = new KIconLoader("ktag");
	mBrainzPopup = new QPopupMenu(mBrainz);
	mBrainzPopup->insertItem( ic->loadIconSet( "reload", KIcon::Small ), tr2i18n("Start Search"),  this, SLOT(mbSearch()));
	connect(mBrainz, SIGNAL(clicked(QPoint)), this, SLOT(mbPopup(QPoint)));
	connect(this, SIGNAL(mbStart(QStringList&)), this, SLOT(updateMbList(QStringList&)));
	connect(this, SIGNAL(mbFound(QStringList&)), this, SLOT(updateMbList(QStringList&)));
	connect(this, SIGNAL(signalStartTRM()), this, SLOT(mbSearch()));
	connect(this, SIGNAL(signalStartQuery(MusicBrainzQuery*)), this, SLOT(mbQuery(MusicBrainzQuery*)));
#endif // HAVE_MUSICBRAINZ
	connect(okButton, SIGNAL(clicked()), this, SLOT(writeTags()));
	connect(saveButton, SIGNAL(clicked()), this, SLOT(saveCover()));
	connect(dlButton, SIGNAL(clicked()), this, SLOT(downloadCover()));
	connect(mArtist, SIGNAL(textChanged(const QString&)), this, SLOT(isModified(const QString&)));
	connect(mAlbum, SIGNAL(textChanged(const QString&)), this, SLOT(isModified(const QString&)));
	connect(mTitle, SIGNAL(textChanged(const QString&)), this, SLOT(isModified(const QString&)));
	connect(mGenre, SIGNAL(textChanged(const QString&)), this, SLOT(isModified(const QString&)));
	connect(mComment, SIGNAL(textChanged(const QString&)), this, SLOT(isModified(const QString&)));
	connect(mNewCover, SIGNAL(textChanged(const QString&)), this, SLOT(isModified(const QString&)));
	connect(mTrack, SIGNAL(valueChanged(int)), this, SLOT(isModified(int)));
	connect(mYear, SIGNAL(valueChanged(int)), this, SLOT(isModified(int)));
	connect(this, SIGNAL(modifiedTag(QWidget *, bool)), this, SLOT(slotUpdateUi(QWidget *, bool)));
}

Ktag::~Ktag()
{
}

void Ktag::readProperties(){
	KConfig* conf = new KConfig("ktagrc", true);
	conf->setGroup("ktag");
	mPatterns = conf->readListEntry("patterns", ',');
#if HAVE_MUSICBRAINZ
	conf->setGroup("MusicBrainz");
	mbServer = conf->readEntry("server", "mm.musicbrainz.org");
	mbPort = conf->readNumEntry("port", 80);
	mbAutoSearch = conf->readBoolEntry("mbAutoSearch", true);
	mbProxyServer =  conf->readEntry("prxserver", "");
	mbProxyPort =  conf->readNumEntry("prxport", 0);
#endif // HAVE_MUSICBRAINZ
}

bool Ktag::setFile( const QString &fp )
{
	mAudioFile = fp;
	tags = new Tags(mAudioFile);
	if(readTags()){
		mURL.setPath(mAudioFile);
		QString path = mURL.directory(true);
		QString filename = mURL.fileName(true);
		KMimeMagicResult *result = KMimeMagic::self()->findFileType(mAudioFile);
		QString mime = result->mimeType();
		mFile->setText(tr2i18n("<b>File:</b> %1<br><b>Path:</b> %2<br><b>MIME Type:</b> %3").arg(filename, path, mime));
		if(!QFileInfo(mAudioFile).isWritable()){
			this->setDisabled(true);
		}
#if HAVE_MUSICBRAINZ
		if(mbAutoSearch){
			emit signalStartTRM();
		}
#endif // HAVE_MUSICBRAINZ
		return true;
	}
	return false;
}

KURL Ktag::url(){
	return mURL;
}

bool Ktag::readTags(){
	if(tags->isMP3()){
		tags->ReadId3();
		coverLabel->setEnabled(true);
		mNewCover->setEnabled(true);
		dlButton->setEnabled(true);
		mNewCover->setFilter("*.png *.jpg *.jpeg *.gif *.bmp *.xbm *.xpm *.pnm");
#if HAVE_MUSICBRAINZ
		mBrainz->setEnabled(true);
#endif // HAVE_MUSICBRAINZ
	}
	else if(tags->isFLAC()){
		tags->ReadFlac();
		coverLabel->setEnabled(true);
		mNewCover->setEnabled(true);
		dlButton->setEnabled(true);
		mNewCover->setFilter("*.png *.jpg *.jpeg *.gif *.bmp *.xbm *.xpm *.pnm");
#if HAVE_MUSICBRAINZ
		mBrainz->setDisabled(true);
#endif // HAVE_MUSICBRAINZ
	}
	else if(tags->isOGG()){
		tags->ReadVorbis();
		coverLabel->setDisabled(true);
		mNewCover->setDisabled(true);
		dlButton->setDisabled(true);
#if HAVE_MUSICBRAINZ
		mBrainz->setEnabled(true);
#endif // HAVE_MUSICBRAINZ
	}
	else if(tags->isOGGFlac()){
		tags->ReadOggFlac();
		coverLabel->setDisabled(true);
		mNewCover->setDisabled(true);
		dlButton->setDisabled(true);
#if HAVE_MUSICBRAINZ
		mBrainz->setDisabled(true);
#endif // HAVE_MUSICBRAINZ
	}
	else{
		kdDebug("Ktag")<< "Not supported filetype" << endl;
		KMessageBox::sorry(this, tr2i18n("%1 is not a supported file type (%2)").arg(mAudioFile,tags->getMimeType()));
		return false;
	}
	KURL *url = new KURL(mAudioFile);
	QString path = url->directory(true);
	mAlbum->clear();
	mArtist->clear();
	mTitle->clear();
	if(!mAlbum->contains(path.section('/', -1, -1))){
		mAlbum->insertURL(path.section('/', -1, -1));
	}
	if(!mArtist->contains(path.section('/', -2, -2))){
		mArtist->insertURL(path.section('/', -2, -2));
	}
	if(!mAlbum->contains(tags->album)){
		mAlbum->insertURL(tags->album, 0);
	}
	if(!mArtist->contains(tags->artist)){
		mArtist->insertURL(tags->artist, 0);
	}
	if(!mTitle->contains(tags->stitle)){
		mTitle->insertURL(tags->stitle, 0);
	}
	mTrack->setValue(tags->track);
	mYear->setValue(tags->year);
	mGenre->setEditURL(tags->genre);
	mComment->setText(tags->comment);
	mCoverInfo->setEnabled(true);
	mCover->setEnabled(true);
	if(tags->haveCover){
		mCover->setPixmap(QPixmap(tags->cover));
		saveButton->setEnabled(true);
		mCoverInfo->setText(tr2i18n("Size: %1 x %2").arg(tags->cover.width()).arg(tags->cover.height()));
	}
	else{
		if(QFile::exists(path + "/cover.png")){
			mCover->setPixmap(path + "/cover.png");
			mCoverInfo->setText(tr2i18n("<b>Local file:</b><br>") + path + "/cover.png");
		}
		else if(QFile::exists(path + "/cover.jpg")){
			mCover->setPixmap(path + "/cover.jpg");
			mCoverInfo->setText(tr2i18n("<b>Local file:</b><br>") + path + "/cover.jpg");
		}
		else if(QFile::exists(path + "/cover.jpeg")){
			mCover->setPixmap(path + "/cover.jpeg");
			mCoverInfo->setText(tr2i18n("<b>Local file:</b><br>") + path + "/cover.jpeg");
		}
		else{
			mCover->setDisabled(true);
			mCoverInfo->setDisabled(true);
			mCoverInfo->setText("");
		}
		saveButton->setDisabled(true);
	}
	emit modifiedTag(this, false);
	return true;
}

#if HAVE_MUSICBRAINZ
void Ktag::mbSearch(){
	if(mbServer != QString::null){
		mBrainz->setMovie(QMovie(locate("data", "ktag/pix/mb.mng")));
		QStringList str;
		str << tr2i18n("Asking MusicBrainz...");
		emit mbStart(str);
		QStringList l;
		l<<mAudioFile;
		MusicBrainzQuery *query = new MusicBrainzQuery( MusicBrainzQuery::File , l );
		query->SetServer(mbServer, mbPort);
		if(mbProxyServer != QString::null){
			query->SetProxy(mbProxyServer, mbProxyPort);
		}
		if(tags->isMP3()){
			query->setFileType("MP3");
		}
		else if(tags->isOGG()){
			query->setFileType("OGG");
		}
		connect( query, SIGNAL(signalDone(const MusicBrainzQuery::TrackList&)), SLOT(updateLists(const MusicBrainzQuery::TrackList&)) );
		connect( query, SIGNAL(signalStatusMsg(const QString&, int)), SLOT(updateTip(const QString&, int)) );
		emit signalStartQuery(query);
	}
	else{
		KMessageBox::sorry(this, tr2i18n("You have not selected a server. Please check your configurations."), QString::null, KMessageBox::AllowLink);
	}
}

void Ktag::mbQuery(MusicBrainzQuery *query){
	query->startQuery();
}

void Ktag::updateLists( const MusicBrainzQuery::TrackList& res ){
	if(!res.isEmpty()){
		QStringList str;
		str << tr2i18n("<b>Click here for MusicBrainz&nbsp; results</b>");
		MusicBrainzQuery::TrackList::ConstIterator it = res.begin();
		MusicBrainzQuery::TrackList::ConstIterator end = res.end();
		while( it != end ){
			if(!mAlbum->contains((*it).album)){
				mAlbum->insertURL((*it).album);
			}
			if(!mArtist->contains((*it).artist)){
				mArtist->insertURL((*it).artist);
			}
			if(!mTitle->contains((*it).name)){
				mTitle->insertURL((*it).name);
			}
			if(mTrack->value() == 0){
				mTrack->setValue((*it).number);
			}
			str << tags->upFirst((*it).artist)+" | "+tags->upFirst((*it).album)+" | "+QString::number((*it).number)+" | "+tags->upFirst((*it).name);
			it++;
		}
		emit mbFound(str);
	}
	mBrainz->setPixmap(QPixmap(locate("data", "ktag/pix/mb.png")));
}

void Ktag::updateMbList( QStringList& list ){
	KIconLoader* ic = new KIconLoader("ktag");
	QStringList::ConstIterator it = list.begin();
	QStringList::ConstIterator end = list.end();
	QString str;
	QToolTip::add( mBrainz, *it );
	mBrainzPopup->clear();
	it++;
	while( it != end ){
		str = *it;
		mBrainzPopup->insertItem( str.replace("&","&&"),  this, SLOT(setFromMb(int)));
		it++;
	}
	mBrainzPopup->insertItem( ic->loadIconSet( "reload", KIcon::Small ), tr2i18n("Refresh"),  this, SLOT(mbSearch()));
}

void Ktag::updateTip(const QString& msg, int){
	QToolTip::add( mBrainz, msg );
}

void Ktag::setFromMb(int i){
	QString str = mBrainzPopup->text(i);
	QStringList ls = QStringList::split( " | ", str);
	mArtist->setCurrentText((*ls.at(0)).replace("&&","&"));
	mAlbum->setCurrentText((*ls.at(1)).replace("&&","&"));
	mTrack->setValue((*ls.at(2)).toInt());
	mTitle->setCurrentText((*ls.at(3)).replace("&&","&"));
}

void Ktag::mbPopup(QPoint p){
	mBrainzPopup->exec(p, 0);
}
#endif // HAVE_MUSICBRAINZ

void Ktag::writeTags(){
	writeTags(false);
}

void Ktag::writeTags(bool noProgress){
	KProgressDialog *p = 0;
	if(!noProgress){
		p = new KProgressDialog(this, "progress", tr2i18n("Writing tags"), tr2i18n("Writing tags to %1").arg(mAudioFile), true);
		p->showCancelButton(false);
		p->progressBar()->setTotalSteps(5);
		p->progressBar()->setProgress(1);
	}
	tags->setInfos(mTrack->value(), mArtist->currentText(), mTitle->currentText(), mAlbum->currentText(), mGenre->currentText(), mYear->value(), mComment->text(),mNewCover->url());
	mAlbum->setDisabled(true);
	mArtist->setDisabled(true);
	mTitle->setDisabled(true);
	mGenre->setDisabled(true);
	mTrack->setDisabled(true);
	mYear->setDisabled(true);
	mCover->setDisabled(true);
	mComment->setDisabled(true);
	okButton->setDisabled(true);
	quitButton->setDisabled(true);
	dlButton->setDisabled(true);
	if(!noProgress){
		p->progressBar()->setProgress(2);
	}
	if(tags->isMP3()){
		tags->WriteId3();
	}
	else if(tags->isFLAC()){
		tags->WriteFlac();
	}
	else if(tags->isOGG()){
		tags->WriteVorbis();
	}
	else if(tags->isOGGFlac()){
		tags->WriteOggFlac();
	}
	if(!noProgress){
		p->progressBar()->setProgress(3);
	}
	mAlbum->setEnabled(true);
	mArtist->setEnabled(true);
	mTitle->setEnabled(true);
	mGenre->setEnabled(true);
	mTrack->setEnabled(true);
	mYear->setEnabled(true);
	mCover->setEnabled(true);
	mComment->setEnabled(true);
	if(tags->isMP3() || tags->isFLAC()){
		mNewCover->setURL("");
	}
	quitButton->setEnabled(true);
	if(!noProgress){
		p->progressBar()->setProgress(4);
	}
	readTags();
	if(!noProgress){
		p->progressBar()->setProgress(5);
	}
	emit modifiedTag(this, false);
}

void Ktag::clearTags(){
	if(tags->isMP3()){
		tags->ClearId3();
	}
	else if(tags->isFLAC()){
		tags->ClearFlac();
	}
	else if(tags->isOGG()){
		tags->ClearVorbis();
	}
	else if(tags->isOGGFlac()){
		tags->ClearOggFlac();
	}
	mGenre->setCurrentText("");
	mCover->setDisabled(true);
	mCoverInfo->setDisabled(true);
	mCoverInfo->setText("");
	setFile(mAudioFile);
	saveButton->setEnabled(false);
	emit modifiedTag(this, false);
}

void Ktag::cleanMime(){
	if(tags->isMP3()){
		tags->correctMimeType();
		setFile(mAudioFile);
		saveButton->setEnabled(false);
		emit modifiedTag(this, false);
	}
}
void Ktag::slotUpdateUi( QWidget*, bool b ){
	modified = b;
	okButton->setEnabled(b);
}

void Ktag::saveCover(){
	QString nfileName;
	nfileName = KFileDialog::getSaveFileName(KURL(mAudioFile).directory(false) + "cover.png", "*.png\n*.jpg *.jpeg", 0, "Save Cover");
	if(!nfileName.isEmpty()){
		QString ext = "JPEG";
		if(nfileName.contains(".png")){
			ext = "PNG";
		}
		(mCover->pixmap())->save(nfileName, ext, -1);
	}
}

void Ktag::isModified(int i){
	isModified(QString::number(i));
}

void Ktag::isModified(const QString&){
	if(QFileInfo(mAudioFile).isWritable()){
		emit modifiedTag(this, true);
		return;
	}
	emit modifiedTag(this, false);
}

void Ktag::closeEvent(QCloseEvent *ce){
	if(modified && QFileInfo(mAudioFile).isWritable()){
		int result = KMessageBox::warningYesNoCancel(this,tr2i18n( "The tags off \"%1\" have changed!\nDo you want to save them?" ).arg(mAudioFile),"SaveDlg");
		if(result == KMessageBox::Yes){
			writeTags();
		}
		else if(result == KMessageBox::Cancel){
			kdDebug("KTag") << "Cancel pressed" << endl;
			ce->ignore();
			return;
		}
	}
	emit tabClosed(this);
	ce->accept();
}

void Ktag::dragMoveEvent( QDragMoveEvent *e){
	e->accept(mFile->rect().contains(e->pos()));
}

void Ktag::dropEvent( QDropEvent *e){
	if(e->provides("text/uri-list") && mFile->rect().contains(e->pos())){
		e->accept( true );
		emit receivedDropEvent(e);
		return;
	}
	e->ignore();
}

void Ktag::downloadCover(){
	if(tags->isMP3()){
		tags->ReadId3();
	}
	else if(tags->isFLAC()){
		tags->ReadFlac();
	}
	else{
		KMessageBox::sorry(this, tr2i18n("%1 is not an mp3 or a flac file").arg(tags->file));
		return;
	}
	dlCover* dl = new dlCover();
	QStringList::ConstIterator it = mPatterns.begin();
	QStringList::ConstIterator end = mPatterns.end();
	while( it != end ){
		KURL url(tags->parsePattern(*it));
		dl->testMimeType(url);
		it++;
	}
	if(dl->coverList->childCount() != 0){
		if(dl->exec() && !dl->coverList->selectedItems().isEmpty()){
			mNewCover->setURL(dl->coverList->selectedItem()->text(1));
			return;
		}
		return;
	}
	KMessageBox::sorry(this, tr2i18n("No cover found for album \"%1\" by \"%2\"").arg(tags->album, tags->artist), QString::null, KMessageBox::AllowLink);
}

QString Ktag::parsefilePattern(const QString &p){
	newFile = KURL(mAudioFile).directory(false);
	newFile += p;
	newFile.replace("%artist", mArtist->currentText());
	newFile.replace("%Artist", tags->upFirst(mArtist->currentText()).replace("_"," "));
	newFile.replace("%title", mTitle->currentText());
	newFile.replace("%Title", tags->upFirst(mTitle->currentText()).replace("_"," "));
	newFile.replace("%album", mAlbum->currentText());
	newFile.replace("%Album", tags->upFirst(mAlbum->currentText()).replace("_"," "));
	newFile.replace("%_artist", mArtist->currentText().replace(" ", "_"));
	newFile.replace("%_Artist", tags->upFirst(mArtist->currentText()).replace(" ", "_"));
	newFile.replace("%_title", mTitle->currentText().replace(" ", "_"));
	newFile.replace("%_Title", tags->upFirst(mTitle->currentText()).replace(" ", "_"));
	newFile.replace("%_album", mAlbum->currentText().replace(" ", "_"));
	newFile.replace("%_Album", tags->upFirst(mAlbum->currentText()).replace(" ", "_"));
	QString trck = QString::number(mTrack->value());
	if(trck.length() < 2){
		trck = "0" + trck;
	}
	newFile.replace("%track", QString::number(mTrack->value()));
	newFile.replace("%Track", trck);
	if(tags->isOGG() || tags->isOGGFlac()){
		newFile.replace("*", "ogg");
	}
	else if(tags->isFLAC()){
		newFile.replace("*", "flac");
	}
	else{
		newFile.replace("*", "mp3");
	}
	return newFile;
}

void Ktag::saveAs(const QString &p){
	newFile = parsefilePattern(p);
	KIO::FileCopyJob *job = 0;
	job = KIO::file_copy(KURL(mAudioFile), KURL(newFile));
	job->setAutoErrorHandlingEnabled(true);
	connect(job, SIGNAL(result(KIO::Job *)), this, SLOT(slotJobResult(KIO::Job *)));
}

void Ktag::rename(const QString &p){
	newFile = parsefilePattern(p);
	KIO::FileCopyJob *job = 0;
	job = KIO::file_move(KURL(mAudioFile), KURL(newFile));
	job->setAutoErrorHandlingEnabled(true);
	connect(job, SIGNAL(result(KIO::Job *)), this, SLOT(slotJobResult(KIO::Job* )));
}

void Ktag::slotJobResult(KIO::Job *job){
	if(job->error() != 0){
		return;
	}
	setFile(newFile);
	emit fileChanged(this, newFile);
	emit modifiedTag(this, false);
}

#include "ktag.moc"
