/* ***************************************************************************   
   * KVideoEncoder - an easy-to-use Video Encoder for KDE                    *
   * Copyright (C) 2005-2006 Philipp Ludwig                                  *
   *                                                                         *
   * This program is free software; you can redistribute it and/or modify it *
   * under the terms of the GNU General Public License as published by the   *
   * Free Software Foundation; either version 2 of the License, or (at your  *
   * option) any later version.                                              *
   *                                                                         *
   * This program is distributed in the hope that it will be useful, but     *
   * WITHOUT ANY WARRANTY; without even the implied warranty of              *
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        *
   * General Public License for more details.                                *
   *                                                                         *
   * You should have received a copy of the GNU General Public License along *
   * with this program; if not, write to the Free Software Foundation, Inc., *
   * 59 cmdle Place, Suite 330, Boston, MA 02111-1301, USA.                  *
   ************************************************************************* */

#include "encoder.h"
#include <qstring.h>
#include "audiocodec_struct.h"
#include "videocodec_struct.h"
#include "kvideoencoder_config.h"

#include <stdio.h>

void Mencoder::BuildCommandLine( void )
{
	config *Config = new config();
	cmd = Config->getMencoder() + " -v ";
	
	
	// Preview?
	if( previewStart != -1 )
		cmd += " -endpos 1 -ss " + QString::number(previewStart);
	
	// Output file
	cmd += " -o '" + outputFile + "'";
	
	// Audio Direct Stream Copy/PCM?
	switch( selectedAudioCodec.Codec )
	{
	case AUDIO_CODEC_NAME_DIRECTCOPY:
		cmd += " -oac copy ";
		break;
	case AUDIO_CODEC_NAME_PCM:
		cmd += " -oac pcm ";
		break;
	default:
		cmd += " -oac lavc ";
	}
	
	// Add the multipass logfile to the command line
	if( (selectedVideoCodec.bitRateMode == BITRATE_MODE_FIRSTPASS) || (selectedVideoCodec.bitRateMode == BITRATE_MODE_SECONDPASS) ) cmd += "-passlogfile '" + selectedVideoCodec.logfile + "' ";
	
	// Add the selected codec to the command line
	switch( selectedVideoCodec.Codec )
	{
	case CODEC_NAME_DIRECTCOPY:
		cmd += " -ovc copy";	
		
		if( selectedVideoCodec.type == 1 ) cmd += " -of mpeg";
		
		// Write new index?
		if( selectedVideoCodec.newIndex ) cmd += " -idx";
		break;
	case CODEC_NAME_DIVX4: 
		cmd += " -ffourcc divx -ovc lavc -lavcopts vcodec=mpeg4:";
		break;
	case CODEC_NAME_MPEG1:
	case CODEC_NAME_MPEG2:
		// ******
		// There is a problem with the mpeg parameter of the different
		// mencoder options. The old versions understand -mpegopts, the new -mpeg
		// We'll use that fix. Hopefully it'll work.
		if( QString(Config->getCommand( "sh -c 'mencoder -mpeg > /tmp/bla 2>&1'" )).find( "mpeg is not an MEncoder option" ) )
			cmd += " -of mpeg -ovc lavc -mpegopts ";
		else
			cmd += " -of mpeg -ovc lavc -mpeg ";
				
		// Size?
		if( videoFilter->scale != NULL )
		{
			cmd += "vwidth="  + QString::number( videoFilter->scale->width )
			       +  ":vheight=" + QString::number( videoFilter->scale->height ) + ":";	
		}	
		
		// Format?
		switch( selectedVideoCodec.mpegType )
		{
		case 0:
			if( selectedVideoCodec.Codec == CODEC_NAME_MPEG1 ) cmd += "format=mpeg1 -lavcopts ";
			if( selectedVideoCodec.Codec == CODEC_NAME_MPEG2 ) cmd += "format=mpeg2 -lavcopts ";
			break;
		case 1:
			cmd += "format=xvcd -lavcopts vrc_buf_size=327:vrc_minrate=1152:vrc_maxrate=1152:";
			break;
		case 2:
			cmd += "format=xsvcd -lavcopts vrc_buf_size=917:vrc_minrate=600:vrc_maxrate=2500:";
			break;
		case 3: cmd += "format=dvd -lavcopts vrc_buf_size=1835:vrc_maxrate=9800:";
			break;
		}				
		
		// The LAVC opts
		if( selectedVideoCodec.Codec == CODEC_NAME_MPEG1 ) cmd += "vcodec=mpeg1video:";
		if( selectedVideoCodec.Codec == CODEC_NAME_MPEG2 ) cmd += "vcodec=mpeg2video:";
		break;		
	case CODEC_NAME_MJPEG:
		cmd += " -ovc lavc -lavcopts vcodec=mjpeg:";
		break;
	case CODEC_NAME_WMV1:
		cmd += " -ovc lavc -lavcopts vcodec=wmv1:";
		break;
	case CODEC_NAME_WMV2:
		cmd += " -ovc lavc -lavcopts vcodec=wmv2:";
		break;
	case CODEC_NAME_XVID:
		if( QString(Config->getCommand( "sh -c 'mencoder -ovc xvidenc > /tmp/bla 2>&1'" )).find( "Unknown suboption xvidenc" ) )
			cmd += " -ovc xvid -xvidencopts ";
		else
			cmd += " -ovc xvidenc -xvidencopts ";
	}
	
	// Add the bitrate to the command line
	if( selectedVideoCodec.Codec != CODEC_NAME_DIRECTCOPY )	
	{
		if( selectedVideoCodec.Codec != CODEC_NAME_XVID )
			switch( selectedVideoCodec.bitRateMode )
			{
		case BITRATE_MODE_CBR:
			cmd += "vbitrate=" + QString::number( selectedVideoCodec.bitRate );
			break;
		case BITRATE_MODE_CQ:
			cmd += "vqscale=" + QString::number( selectedVideoCodec.constantQuantizer );
			break;
		case BITRATE_MODE_FIRSTPASS:
			cmd += "vbitrate=" + QString::number( selectedVideoCodec.bitRate )
			       +  ":vqmin=" + QString::number( selectedVideoCodec.vqmin )
			       +  ":vqmax=" + QString::number( selectedVideoCodec.vqmax )
			       +  ":turbo=" + QString::number( selectedVideoCodec.turbo )
			       +  ":vb_strategy=" + QString::number( selectedVideoCodec.vbstrategy )
			       +  ":vpass=1";
			break;
		case BITRATE_MODE_SECONDPASS:
			cmd += "vbitrate=" + QString::number( selectedVideoCodec.bitRate )
			       +  ":vqmin=" + QString::number( selectedVideoCodec.vqmin )
			       +  ":vqmax=" + QString::number( selectedVideoCodec.vqmax )
			       +  ":vb_strategy=" + QString::number( selectedVideoCodec.vbstrategy )
			       +  ":vpass=2";
			break;
		}
		else
			switch( selectedVideoCodec.bitRateMode )
			{
		case BITRATE_MODE_CBR:
			cmd += "bitrate=" + QString::number( selectedVideoCodec.bitRate );
			break;
		case BITRATE_MODE_CQ:
			cmd += "fixed_quant=" + QString::number( selectedVideoCodec.constantQuantizer );
			break;
		case BITRATE_MODE_FIRSTPASS:
			cmd += "bitrate=" + QString::number( selectedVideoCodec.bitRate )
			       +  ":min_iquant=" + QString::number( selectedVideoCodec.vqmin )
			       +  ":max_iquant=" + QString::number( selectedVideoCodec.vqmax )
			       +  ":pass=1";
			break;
		case BITRATE_MODE_SECONDPASS:
			cmd += "bitrate=" + QString::number( selectedVideoCodec.bitRate )
			       +  ":min_iquant=" + QString::number( selectedVideoCodec.vqmin )
			       +  ":max_iquant=" + QString::number( selectedVideoCodec.vqmax )
			       +  ":pass=2";
			break;
		}
	}
	
	// Add the other video options to the command line
	if( selectedVideoCodec.Codec != CODEC_NAME_DIRECTCOPY )
	{
		if( selectedVideoCodec.Codec != CODEC_NAME_XVID )
		{
			cmd += ":keyint=" + QString::number( selectedVideoCodec.keyframe_interval );
			cmd += ":dia=" + QString::number( selectedVideoCodec.dia );
			
			if( selectedVideoCodec.vhq )       cmd += ":mbd=1";
			if( selectedVideoCodec.v4mv )      cmd += ":v4mv";
			if( selectedVideoCodec.grayScale ) cmd += ":gray";
		}
		else
		{
			cmd += ":max_key_interval=" + QString::number( selectedVideoCodec.keyframe_interval );
			
			int vhq = (selectedVideoCodec.dia + 3) / 3;
			if( selectedVideoCodec.vhq ) vhq++;
			cmd += ":vhq=" + QString::number( vhq );
			
			if( selectedVideoCodec.v4mv )      cmd += ":4mv";
			if( selectedVideoCodec.grayScale ) cmd += ":greyscale";
		}
	}
	
	// Change aspect ratio?
	if( (selectedVideoCodec.Codec != CODEC_NAME_DIRECTCOPY) && (selectedVideoCodec.changeAspect) )
	{
		if( selectedVideoCodec.Codec != CODEC_NAME_XVID )
		{
			cmd += ":aspect=";
			switch( selectedVideoCodec.newAspectRatio )
			{
			case ASPECT_RATIO_VGA11:
				cmd += "1/1";
				break;
			case ASPECT_RATIO_PAL43:
				cmd += "4/3";
				break;
			case ASPECT_RATIO_PAL69:
				cmd += "16/9";
				break;
			case ASPECT_RATIO_OTHER:
				cmd += QString::number( selectedVideoCodec.aspectX ) + "/" + QString::number( selectedVideoCodec.aspectY );
				break;
			}
		}
		else
		{
			cmd += ":par=";
			switch( selectedVideoCodec.newAspectRatio )
			{
			case ASPECT_RATIO_VGA11:
				cmd += "vga11:aspect=1/1";
				break;
			case ASPECT_RATIO_PAL43:
				cmd += "pal43:aspect=4/3";	
				break;
			case ASPECT_RATIO_PAL69:
				cmd += "pal169:aspect=16/9";
				break;
			case ASPECT_RATIO_OTHER:
				cmd += "ext:par_width=" + QString::number( selectedVideoCodec.aspectX ) + ":par_height=" + QString::number( selectedVideoCodec.aspectY );
				cmd += ":aspect:" + QString::number( selectedVideoCodec.aspectX ) + "/" + QString::number( selectedVideoCodec.aspectY );
				break;
			}			
		}
	}
	
	// Add the selected audio codec to the command line
	if( selectedAudioCodec.Codec != AUDIO_CODEC_NAME_DIRECTCOPY )
	{
		if( (selectedVideoCodec.Codec == CODEC_NAME_DIRECTCOPY ) || (selectedVideoCodec.Codec == CODEC_NAME_XVID) ) cmd += " -lavcopts ";
		switch( selectedAudioCodec.Codec )
		{
		case AUDIO_CODEC_NAME_MP2:	
			cmd += ":acodec=mp2:abitrate=" + QString::number( selectedAudioCodec.bitRate );
			break;
		case AUDIO_CODEC_NAME_MP3:
			cmd += ":acodec=mp3:abitrate=" + QString::number( selectedAudioCodec.bitRate );
			break; 
		case AUDIO_CODEC_NAME_IMA:
			cmd += ":acodec=adpcm_ima_wav";
			break;
		}
	}	
	
	// Change sample rate?
	if( (selectedAudioCodec.Codec != AUDIO_CODEC_NAME_DIRECTCOPY ) && ( selectedAudioCodec.changeSampleRate ) )
		cmd += " -srate " + QString::number( selectedAudioCodec.newSampleRate );
	
	// Change Framerate?
	if( (selectedVideoCodec.Codec != CODEC_NAME_DIRECTCOPY) && (selectedVideoCodec.changeFrameRate) ) 
		cmd += " -ofps " + QString::number( selectedVideoCodec.newFrameRate );	
	
	// ***********************************************************
	// Add the filter options fo the command line
	if( !videoFilter->noFilter )
	{
		cmd += " -vf ";
		bool singleFilter = true;
		if( videoFilter->scale != NULL ) 
		{
			singleFilter = false;
			cmd += "scale=" + QString::number( videoFilter->scale->width ) + ":" + QString::number( videoFilter->scale->height );
		}
		if( videoFilter->spp != NULL ) 
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			cmd += "spp=" + QString::number( videoFilter->spp->quality );
		}
		if( videoFilter->denoise3d != NULL )
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			
			if( videoFilter->denoise3d->hq ) cmd += "hqdn3d=";
			else cmd += "denoise3d=";
			
			cmd += QString::number( videoFilter->denoise3d->luma ) + ":" + QString::number( videoFilter->denoise3d->chroma ) + ":" + QString::number( videoFilter->denoise3d->time );
		}
		if( videoFilter->crop != NULL )
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			cmd += "crop=";
			
			cmd += QString::number( videoFilter->crop->width ) + ":" + QString::number( videoFilter->crop->height ) + ":" + QString::number( videoFilter->crop->x ) + ":" + QString::number( videoFilter->crop->y );
		}
		if( videoFilter->eq != NULL )
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			cmd += "eq=" + QString::number( videoFilter->eq->brightness ) + ":" + QString::number( videoFilter->eq->contrast );
		}
		if( videoFilter->filmdint != NULL )
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			
			cmd += "filmdint=luma_only=";
			if( videoFilter->filmdint->luma_copy ) cmd += "1";
			else cmd += "0";
			cmd += "/fast=" + QString::number(videoFilter->filmdint->fast) + "/dint_thres=" + QString::number( videoFilter->filmdint->dint_thres);
		}
		if( selectedVideoCodec.sub != -1 )
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			cmd += "expand=0:0:0:0:1";
		}
		
		// If harddup is selected it must be added at the end of the filter chain
		if( selectedVideoCodec.harddup )
		{
			if( !singleFilter ) cmd += ",";
			else singleFilter = false;
			cmd +="harddup";
		}
		
		// If the scale filter is selected, the flag -sws must be set
		if( videoFilter->scale != NULL ) cmd += " -sws 0";
	}
	else
	{
		if( selectedVideoCodec.sub != -1 )
		{
			cmd += " -vf expand=0:0:0:0:1";
			if (selectedVideoCodec.harddup) cmd += ",harddup";
		}
		else if (selectedVideoCodec.harddup) cmd += " -vf harddup";
	}
	
	// Add the input file to the command line
	switch( inputType )
	{
	case 0: // File
		cmd += " '" + inputFile + "'";
		break;
		
	case 1: // VCD
		cmd += " -cdrom-device " + device;
		cmd += " vcd://" + QString::number( track );
		break;
		
	case 2: // DVD
		cmd += " -dvd-device " + device;
		cmd += " -aid " + aid + " ";		
		cmd += " dvd://" + QString::number(track);
		cmd += " -chapter " + firstCpt + "-" + lastCpt + " ";
		if( selectedVideoCodec.sub >= 0 ) cmd += " -sid " + sid + " ";
		break;
		
	case 3: // Stream
		cmd += " '" + inputFile + "'";
		break;
	}	
	
	delete Config;
}

bool Mencoder::RunEncoder( void )
{
	cmd = "sh -c \"" + cmd + "\" 2>&1";
	DEBUG(cmd);
	handle = popen( cmd.ascii(), "r" );	
	if( handle == NULL ) return false;
	
	// Get the pid
	config *Config = new config();
	pid = Config->getCommand( "pidof mencoder" );
	if( pid.find( " " ) != -1 ) pid = pid.left( pid.find( " " ) );
	pid = pid.stripWhiteSpace();
	delete Config;
	
	// No error yet
	errortype = 0;
	
	return true;
}

int Mencoder::GetProgress( void )
{
	QString temp;
	char buffer[255];
	int n;
	int chars_read = fread( buffer, sizeof(char), 255, handle );
	if( chars_read == 0 ) return -2;
	
	buffer[chars_read - 1] = '\0';
	temp = QString::fromLatin1(buffer);				
	n = temp.find( "%" );
	if( n != -1 )
	{
		temp = temp.mid(n-2,2);
		if( temp == "00" ) temp = "100";
		n = temp.toInt();
		
		if( n != 0 ) return n;
	}
	
	// Error handling
	n = temp.find( "MOV: Found unknown audio atom Fourcc:" );
	if( n != -1 && selectedAudioCodec.Codec == AUDIO_CODEC_NAME_DIRECTCOPY ) errortype = 1;
	
	return -1;
}

int Mencoder::GetFinalSize( void )
{
	QString temp;
	char buffer[255];
	int n;
	int chars_read = fread( buffer, sizeof(char), 255, handle );
	
	buffer[chars_read - 1] = '\0';
	temp = QString::fromLatin1(buffer);
	
	n = temp.find( "mb" );
	if( n != -1 )
	{
		temp = temp.mid( n-10, 10 );
		n = temp.find( "min" );
		temp = temp.mid( n+4, 6 );
		temp = temp.stripWhiteSpace();
		if( temp != "" ) return temp.toInt();
	}
	
	return -1;
}

void Mencoder::KillEncoder( void )
{
	pid = "kill -9 " + pid;
	system( pid.latin1() );	
}
