/***************************************************************************
 *
 *  $Id: kzenconfigdialog.cpp,v 1.17 2005/07/02 17:35:04 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <libnjb.h>

#include <qiconset.h>
#include <qpixmap.h>
#include <qtooltip.h>
#include <qvgroupbox.h>
#include <qcheckbox.h>

#include <kjanuswidget.h>
#include <klocale.h>
#include <kiconloader.h>
#include <klistbox.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kdatetimewidget.h>
#include <qlabel.h>
#include <kdebug.h>
#include <kdirselectdialog.h>

#include <qvbox.h>
#include <qhbox.h>

#include "kzenexplorer.h"
#include "kzencategories.h"
#include "kzenexplorerview.h"
#include "kzentrack.h"
#include "kzenconfig.h"
#include "jukebox.h"
#include "dirselector.h"

#include "kzenconfigdialog.h"

KZenConfigDialog::KZenConfigDialog(QWidget* parent, const char* name)
  :KDialogBase(KDialogBase::IconList,WStyle_DialogBorder, parent, name,
	       true, i18n("Configuration - KZenExplorer"), Ok|Cancel)
{
  fBusy = true;

  fExplorer = (KZenExplorer*) parent;
 
  fJukeboxPage = addVBoxPage(i18n("Jukebox"), i18n("Jukebox"),
			     UserIcon("jukebox", 16));
  
  QString temp;
  QLabel * label;
  temp.sprintf("%s: %s",
	       i18n("Type").latin1(),
	       Jukebox::get()->productName().latin1());
  label = new QLabel(temp, fJukeboxPage);
  
  temp.sprintf("%s: %d.%d.%d   %s: %d.%d.%d",
	       i18n("Hardware version").latin1(),
	       Jukebox::get()->hardwareMajor(),
	       Jukebox::get()->hardwareMinor(),
	       Jukebox::get()->hardwareRelease(),
	       i18n("Firmware version").latin1(),
	       Jukebox::get()->firmwareMajor(),
	       Jukebox::get()->firmwareMinor(),
	       Jukebox::get()->firmwareRelease());
  label = new QLabel(temp, fJukeboxPage);

  QHBox * ownerBox = new QHBox(fJukeboxPage);

  ownerBox->setSpacing(5);
  label = new QLabel(i18n("Owner:"), ownerBox);
  fOwnerEdit = new KLineEdit(ownerBox, "lineedit");
  connect(fOwnerEdit, SIGNAL(textChanged(const QString &)),
	  this, SLOT(ownerChanged(const QString &)));
  fJukeboxPage->setStretchFactor(ownerBox, 0);

  QHBox * clockBox = new QHBox(fJukeboxPage);
  
  clockBox->setSpacing(5);
  label = new QLabel(i18n("Clock:"), clockBox);
  fClockWidget = new KDateTimeWidget(clockBox, "ClockWidget");
  connect(fClockWidget, SIGNAL(valueChanged(const QDateTime &)),
	  this, SLOT(clockChanged(const QDateTime &)));
  fAdjustClock = new KPushButton(i18n("Adjust"), clockBox, "AdjustClock");
  connect(fAdjustClock, SIGNAL(clicked()),
	  this, SLOT(clockAdjust()));
  fJukeboxPage->setStretchFactor(clockBox, 0);

  fSmartImport = new QCheckBox(i18n("Smart import of tracks and playlists"),
			       fJukeboxPage, "SmartImport");

  fJukeboxPage->setStretchFactor(new QWidget(fJukeboxPage),1);

  fCategoryPage = addHBoxPage(i18n("Categories"), i18n("Categories"),
			      UserIcon("category", 16));
  
  QVBox * categoryLists = new QVBox(fCategoryPage);

  fCategoryEdit = new KLineEdit(categoryLists, "CategoryEdit");

  fCategoryListBox = new KListBox(categoryLists, "CategoryListBox");
  fCategoryListBox->setMinimumHeight(150);
  fCategoryListBox->insertStringList(gCategories->categories());

  QVBox * categoryButtons = new QVBox(fCategoryPage);

  fAddCategory = new KPushButton(i18n("Add"), categoryButtons, "AddCategory");
  categoryButtons->setStretchFactor(fAddCategory, 0);
  fModifyCategory = new KPushButton(i18n("Modify"), categoryButtons, "ModifyCategory");
  categoryButtons->setStretchFactor(fModifyCategory, 0);
  fRemoveCategory = new KPushButton(i18n("Remove"), categoryButtons, "RemoveCategory");
  categoryButtons->setStretchFactor(fRemoveCategory, 0);
  categoryButtons->setStretchFactor(new QWidget(categoryButtons), 1);

  connect(fCategoryListBox, SIGNAL(selected(int)), this, SLOT(categorySelected(int)));
  connect(fAddCategory, SIGNAL(clicked()), this, SLOT(addCategory()));
  connect(fModifyCategory, SIGNAL(clicked()), this, SLOT(modifyCategory()));
  connect(fRemoveCategory, SIGNAL(clicked()), this, SLOT(removeCategory()));

  fLibraryPage = addVBoxPage(i18n("Library"), i18n("Local music library"),
			     UserIcon("local_lib", 16));
  
  QHBox * localBox = new QHBox(fLibraryPage);
  
  localBox->setSpacing(5);
  label = new QLabel(i18n("Local library:"), localBox);
  fLocalLibraryEdit = new KLineEdit(localBox, "LocalLibraryEdit");
  label->setBuddy(fLocalLibraryEdit);
  fSelectLocalLibrary = new KPushButton(i18n("Adjust"), localBox, "AdjustClock");
  QIconSet iconSet = SmallIconSet(QString::fromLatin1("fileopen"));
  QPixmap pixMap = iconSet.pixmap(QIconSet::Small, QIconSet::Normal);
  fSelectLocalLibrary->setIconSet(iconSet);
  fSelectLocalLibrary->setFixedSize(pixMap.width()+8, pixMap.height()+8);
  QToolTip::add(fSelectLocalLibrary, i18n("Select local library"));  
  connect(fSelectLocalLibrary,SIGNAL(clicked()),
	  this,SLOT(selectLocalLibrary()));
  fLibraryPage->setStretchFactor(localBox,0);

  QVGroupBox * group = new QVGroupBox( i18n("Path and filename composition"), fLibraryPage);
  group->layout()->setSpacing( KDialog::spacingHint() );
  
  QHBox * savePathBox = new QHBox(group);
  
  label = new QLabel(i18n("Path:"),savePathBox);
  fSavePathEdit = new KLineEdit(group,"SavePathEdit");
  label->setBuddy(fSavePathEdit);

  QHBox * saveNameBox = new QHBox(group);
  
  label = new QLabel(i18n("Filename:"),saveNameBox);
  fSaveNameEdit = new KLineEdit(group,"SaveNameEdit");
  label->setBuddy(fSaveNameEdit);
  
  label = new QLabel(i18n("<b>%artist</b> will be replaced with the name of the artist."), group);
  label = new QLabel(i18n("<b>%album</b> will be replaced with the name of the album."), group);
  label = new QLabel(i18n("<b>%title</b> will be replaced with the title of the track."), group);
  label = new QLabel(i18n("<b>%track</b> will be replaced with the tracknumber."), group);

  fLibraryPage->setStretchFactor(group,0);
  fLibraryPage->setStretchFactor(new QWidget(fLibraryPage),1);

  fSyncPage = addVBoxPage(i18n("Sync"), i18n("Synchronization"),
			  UserIcon("synchronize", 16));
  
  fSyncLibrary = new QCheckBox(i18n("Synchronize Library"),
			       fSyncPage, "SyncLibrary");
  connect(fSyncLibrary, SIGNAL(clicked()), this, SLOT(syncLibraryChanged()));

  label = new QLabel(i18n("Exclude the following directories from sync"), fSyncPage);
  
  fExcludeFromSync = new DirSelector(fSyncPage, "ExcludeFromSync");

  fSyncPlayLists = new QCheckBox(i18n("Synchronize Playlists"),
				  fSyncPage, "SyncPlayLists");
  connect(fSyncPlayLists, SIGNAL(clicked()), this, SLOT(syncPlayListsChanged()));

  fPlayListOverride = new QCheckBox(i18n("Playlists on Jukebox override local playlists"),
				    fSyncPage, "PlayListOverride");
  fCaseSensitiveSearch = new QCheckBox(i18n("Do synchronization and duplicate finding case sensitive"),
				       fSyncPage, "CaseSensitiveSearch");
  fExactSearch = new QCheckBox(i18n("Use exact matching for synchronization and duplicate finding"),
			       fSyncPage, "ExactSearch");
  
  fTimer = new QTimer(this);
  connect(fTimer, SIGNAL(timeout()), this, SLOT(timeout()));
  fTimer->start(1000);
  
  fOwnerStringModified = false;
  fClockModified = false;

  fBusy = false;

  connect (this, SIGNAL(aboutToShowPage(QWidget *)), this, SLOT(aboutToShow(QWidget *)));
}

KZenConfigDialog::~KZenConfigDialog()
{
  
}

void KZenConfigDialog::setup()
{
  fBusy = true;

  fOwnerEdit->setText(fOwnerString);
  fSmartImport->setChecked(KZenConfig::get()->smartImport());
  fLocalLibraryEdit->setText(KZenConfig::get()->localLibrary().path());
  fSavePathEdit->setText(KZenConfig::get()->savePath());
  fSaveNameEdit->setText(KZenConfig::get()->saveName());

  fSyncLibrary->setChecked(KZenConfig::get()->syncLibrary());
  if (KZenConfig::get()->syncLibrary()) {
    fExcludeFromSync->setEnabled(true);
  } else {
    fExcludeFromSync->setEnabled(false);
  }
  fExcludeFromSync->setTopDir(KZenConfig::get()->localLibrary());
  fExcludeFromSync->setDirectories(KZenConfig::get()->excludeFromSync());
  fSyncPlayLists->setChecked(KZenConfig::get()->syncPlayLists());
  fPlayListOverride->setChecked(KZenConfig::get()->playListOverride());
  fCaseSensitiveSearch->setChecked(KZenConfig::get()->caseSensitiveSearch());
  fExactSearch->setChecked(KZenConfig::get()->exactSearch());

  fBusy = false;
}

void KZenConfigDialog::aboutToShow(QWidget * page)
{
  if (page==fSyncPage) {
    KURL url;
    url.setPath(fLocalLibraryEdit->text());
    fExcludeFromSync->setTopDir(url);
    fExcludeFromSync->refresh();
  }
}

void KZenConfigDialog::acceptData()
{
  fOwnerString = fOwnerEdit->text().stripWhiteSpace();

  KZenConfig::get()->setSmartImport(fSmartImport->isChecked());

  KURL url;
  url.setPath(fLocalLibraryEdit->text().stripWhiteSpace());
  if (url.isValid() && url.isLocalFile()) {
    kdDebug() << "Library=" << url.path() << endl;
    KZenConfig::get()->setLocalLibrary(url);
  }
  
  KZenConfig::get()->setSavePath(fSavePathEdit->text().stripWhiteSpace());
  kdDebug() << "SavePath=" << fSavePathEdit->text().stripWhiteSpace() << endl;

  KZenConfig::get()->setSaveName(fSaveNameEdit->text().stripWhiteSpace());
  kdDebug() << "SaveName=" << fSaveNameEdit->text().stripWhiteSpace() << endl;

  KZenConfig::get()->setSyncLibrary(fSyncLibrary->isChecked());
  KZenConfig::get()->setExcludeFromSync(fExcludeFromSync->directories());
  KZenConfig::get()->setSyncPlayLists(fSyncPlayLists->isChecked());
  KZenConfig::get()->setPlayListOverride(fPlayListOverride->isChecked());
  KZenConfig::get()->setCaseSensitiveSearch(fCaseSensitiveSearch->isChecked());
  KZenConfig::get()->setExactSearch(fExactSearch->isChecked());  
}

void KZenConfigDialog::setOwnerString(const QString & owner)
{
  fOwnerString = owner;
}

void KZenConfigDialog::setClock(const QDateTime & datetime)
{
  fBusy = true;
  fClockWidget->setDateTime(datetime);
  fBusy = false;
}

const QDateTime & KZenConfigDialog::KZenConfigDialog::getClock()
{
  fClock = fClockWidget->dateTime();
  return fClock;
}

void KZenConfigDialog::ownerChanged(const QString & o)
{
  if (!fBusy) {
    kdDebug() << "ownerChanged " << o << endl;
    fOwnerStringModified  = true;
  }
}

void KZenConfigDialog::clockChanged(const QDateTime & dt)
{
  if (!fBusy) {
    kdDebug() << "clockChanged " << dt << endl;
    fClockModified = true;
  }
}

void KZenConfigDialog::clockAdjust()
{
  fClock = QDateTime::currentDateTime();
  fClockWidget->setDateTime(fClock);
}

void KZenConfigDialog::timeout()
{
  fClock = fClockWidget->dateTime();
  setClock(fClock.addSecs(1));
}

void KZenConfigDialog::languageChange()
{
  setCaption(i18n("Configuration - KZenExplorer"));
}

void KZenConfigDialog::categorySelected(int idx)
{
  fCategoryEdit->setText(fCategoryListBox->text(idx));
}

void KZenConfigDialog::addCategory()
{
  QString text = fCategoryEdit->text();
  if (text.isNull() || text.isEmpty()) return;
  text = text.stripWhiteSpace();

  fCategoryListBox->insertItem(text);
  gCategories->addCategory(text);
}

void KZenConfigDialog::modifyCategory()
{
  QString text = fCategoryEdit->text();
  if (text.isNull() || text.isEmpty()) return;
  text = text.stripWhiteSpace();
  
  int idx = fCategoryListBox->currentItem();
  if (text!=fCategoryListBox->text(idx)) {
    fCategoryListBox->changeItem(text,idx);
    (gCategories->categories())[idx] = text;

    fExplorer->getView()->refresh();
  }
}

void KZenConfigDialog::removeCategory()
{
  int idx = fCategoryListBox->currentItem();
  fCategoryListBox->removeItem(idx);
  (gCategories->categories()).remove((gCategories->categories())[idx]);

  KZenTrack * track;
  for (uint i=0;i<fExplorer->getTracks()->count();i++) {
    track = fExplorer->getTracks()->at(i);
    track->removeCategory(idx);
  }
  
  fExplorer->getView()->refresh();
}

void KZenConfigDialog::selectLocalLibrary()
{
  KURL url = KDirSelectDialog::selectDirectory(KZenConfig::get()->localLibrary().url(),
					       true);
  if (url.isValid() && url.isLocalFile()) {
    fLocalLibraryEdit->setText(url.path());
  }
}

QStringList & KZenConfigDialog::excludeFromSync()
{
  return fExcludeFromSync->directories();
}

void KZenConfigDialog::syncLibraryChanged()
{
  if (fSyncLibrary->isChecked()) {
    fExcludeFromSync->setEnabled(true);
  } else {
    fExcludeFromSync->setEnabled(false);
  }
}

void KZenConfigDialog::syncPlayListsChanged()
{
  if (fSyncPlayLists->isChecked()) {
    fPlayListOverride->setEnabled(true);
  } else {
    fPlayListOverride->setEnabled(false);
  }
}

#include "kzenconfigdialog.moc"
