/*
 * This is the Loris C++ Class Library, implementing analysis, 
 * manipulation, and synthesis of digitized sounds using the Reassigned 
 * Bandwidth-Enhanced Additive Sound Model.
 *
 * Loris is Copyright (c) 1999-2007 by Kelly Fitz and Lippold Haken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * Synthesizer.C
 *
 * Implementation of class Loris::SynthesizerSynthesizer, a synthesizer of 
 * bandwidth-enhanced Partials.
 *
 * Kelly Fitz, 16 Aug 1999
 * loris@cerlsoundgroup.org
 *
 * http://www.cerlsoundgroup.org/Loris/
 *
 */

#if HAVE_CONFIG_H
	#include "config.h"
#endif

#include "Synthesizer.h"
#include "Oscillator.h"
#include "Breakpoint.h"
#include "BreakpointUtils.h"
#include "Envelope.h"
#include "LorisExceptions.h"
#include "Notifier.h"
#include "Partial.h"

#include <algorithm>
#include <cmath>

#if defined(HAVE_M_PI) && (HAVE_M_PI)
	const double Pi = M_PI;
#else
	const double Pi = 3.14159265358979324;
#endif

//	begin namespace
namespace Loris {

// ---------------------------------------------------------------------------
//	Synthesizer constructor
// ---------------------------------------------------------------------------
//!	Construct a Synthesizer using the specified sampling rate, sample
//!	buffer (a standard library vector), and Partial
//!	fade time (in seconds). Since Partials generated by the Loris Analyzer
//!	generally begin and end at non-zero amplitude, zero-amplitude
//!	Breakpoints are inserted at either end of the Partial, at a temporal
//!	distance equal to the fade time, to reduce turn-on and turn-off
//!	artifacts. If the fade time is unspecified, the default value of one
//!	millisecond (0.001 seconds) is used.
//!
//!	\param	samplerate The rate (Hz) at which to synthesize samples
//!			(must be positive).
//!	\param	buffer The vector (of doubles) into which rendered samples
//!			should be accumulated.
//!	\param	fade The Partial fade time in seconds (must be non-negative).
//!	\throw	InvalidArgument if the specfied sample rate is non-positive.
//!	\throw	InvalidArgument if the specified fade time is negative.
Synthesizer::Synthesizer( double samplerate, std::vector<double> & buffer, 
						  double fade ) :
	sampleBuffer( & buffer ),
	tfade( fade ),
	srate( samplerate )
{
	//	check to make sure that the sample rate is valid:
	if ( srate <= 0. ) 
	{
		Throw( InvalidArgument, "Synthesizer sample rate must be positive." );
	}

	//	check to make sure that the specified fade time
	//	is valid:
	if ( tfade < 0. )
	{
		Throw( InvalidArgument, 
			   "Synthesizer Partial fade time must be non-negative." );
	}
}

// ---------------------------------------------------------------------------
//	synthesize
// ---------------------------------------------------------------------------
//!	Synthesize a bandwidth-enhanced sinusoidal Partial. Zero-amplitude
//!	Breakpoints are inserted at either end of the Partial to reduce
//!	turn-on and turn-off artifacts, as described above. The synthesizer
//!	will resize the buffer as necessary to accommodate all the samples,
//!	including the fade out. Previous contents of the buffer are not
//!	overwritten. Partials with start times earlier than the Partial fade
//!	time will have shorter onset fades. Partials are not rendered at
//! frequencies above the half-sample rate. 
//!
//! \param 	p The Partial to synthesize.
//! \return Nothing.
//!	\pre	The partial must have non-negative start time.
//! \post	This Synthesizer's sample buffer (vector) has been 
//!			resized to accommodate the entire duration of the 
//!			Partial, p, including fade out at the end.
//!	\throw	InvalidPartial if the Partial has negative start time.
//	
void
Synthesizer::synthesize( const Partial & p ) 
{
	if ( p.numBreakpoints() == 0 )
	{
		debugger << "Synthesizer ignoring a partial that contains no Breakpoints" << endl;
		return;
	}
	
	if ( p.startTime() < 0 )
	{
		Throw( InvalidPartial, "Tried to synthesize a Partial having start time less than 0." );
	}

	debugger << "synthesizing Partial from " << p.startTime() * srate 
			 << " to " << p.endTime() * srate << " starting phase "
			 << p.initialPhase() << " starting frequency " 
			 << p.first().frequency() << endl;

	//	resize the sample buffer if necessary:
	typedef unsigned long index_type;
	index_type endSamp = index_type( ( p.endTime() + tfade ) * srate );
	if ( endSamp+1 > sampleBuffer->size() )
	{
		//	pad by one sample:
		sampleBuffer->resize( endSamp+1 );
	}
	
	//	compute the starting time for synthesis of this Partial,
	//	tfade before the Partial's startTime, but not before 0:
	double itime = ( tfade < p.startTime() ) ? ( p.startTime() - tfade ) : 0.;
	index_type currentSamp = index_type( itime * srate );
	
	//	reset the oscillator:
	osc.resetEnvelopes( BreakpointUtils::makeNullBefore( p.first(), p.startTime() - itime ), srate );

	//	cache the previous frequency (in Hz) so that it
	//	can be used to reset the phase when necessary
	//	in the sample computation loop below (this saves
	//	having to recompute from the oscillator's radian
	//	frequency):
	double prevFrequency = p.first().frequency();	
	
	//	better to compute this only once:
	const double OneOverSrate = 1. / srate;
	
	//	synthesize linear-frequency segments until 
	// 	there aren't any more Breakpoints to make segments:
	double * bufferBegin = &( sampleBuffer->front() );
	for ( Partial::const_iterator it = p.begin(); it != p.end(); ++it )
	{
		index_type tgtSamp = index_type( it.time() * srate );
		Assert( tgtSamp >= currentSamp );
		
		//	if the current oscillator amplitude is
		//	zero, and the target Breakpoint amplitude
		//	is not, reset the oscillator phase so that
		//	it matches exactly the target Breakpoint 
		//	phase at tgtSamp:
		if ( osc.amplitude() == 0. )
		{
			//	recompute the phase so that it is correct
			//	at the target Breakpoint (need to do this
			//	because the null Breakpoint phase was computed
			//	from an interval in seconds, not samples, so
			//	it might be inaccurate):
			//
			//	double favg = 0.5 * ( prevFrequency + it.breakpoint().frequency() );
			//	double dphase = 2 * Pi * favg * ( tgtSamp - currentSamp ) / srate;
			//
			double dphase = Pi * ( prevFrequency + it.breakpoint().frequency() ) 
							   * ( tgtSamp - currentSamp ) * OneOverSrate;
			osc.resetPhase( it.breakpoint().phase() - dphase );
		}

		osc.oscillate( bufferBegin + currentSamp, bufferBegin + tgtSamp,
					   it.breakpoint(), srate );
		
		currentSamp = tgtSamp;
		prevFrequency = it.breakpoint().frequency();
	}

	//	render a fade out segment:	
	osc.oscillate( bufferBegin + currentSamp, bufferBegin + endSamp,
				   BreakpointUtils::makeNullAfter( p.last(), tfade ), srate );
	
}
	
// -- access --

// ---------------------------------------------------------------------------
//	fadeTime
// ---------------------------------------------------------------------------
//!	Return this Synthesizer's Partial fade time, in seconds.
double 
Synthesizer::fadeTime( void ) const 
{
	return tfade;
}

// ---------------------------------------------------------------------------
//	sampleRate
// ---------------------------------------------------------------------------
//!	Return the sampling rate (in Hz) for this Synthesizer.
double 
Synthesizer::sampleRate( void ) const 
{
	return srate;
}

// ---------------------------------------------------------------------------
//	samples (const)
// ---------------------------------------------------------------------------
//!	Return a const reference to the sample buffer used (not
//!	owned) by this Synthesizer.
const std::vector<double> &
Synthesizer::samples( void ) const 
{
	return *sampleBuffer;
}

// ---------------------------------------------------------------------------
//	samples (non-const)
// ---------------------------------------------------------------------------
//!	Return a reference to the sample buffer used (not
//!	owned) by this Synthesizer.
std::vector<double> &
Synthesizer::samples( void )  
{
	return *sampleBuffer;
}

// -- mutation --

// ---------------------------------------------------------------------------
//	setFadeTime
// ---------------------------------------------------------------------------
//!	Set this Synthesizer's fade time to the specified value 
//!	(in seconds, must be non-negative).
//!
//!	\param	t The new Partial fade time.
//!	\throw	InvalidArgument if the specified fade time is negative.
void 
Synthesizer::setFadeTime( double partialFadeTime )  
{
	//	check to make sure that the specified fade time
	//	is valid:
	if ( partialFadeTime < 0. )
	{
		Throw( InvalidArgument, "Synthesizer Partial fade time must be non-negative." );
	}

	tfade = partialFadeTime;
}

}	//	end of namespace Loris
